<?php
require_once '../config.php';
requireAdmin();

$today = new DateTimeImmutable('today');
$todayStart = $today->format('Y-m-d 00:00:00');
$chartStartDate = $today->modify('-6 days');
$chartStart = $chartStartDate->format('Y-m-d 00:00:00');

$todayOrdersStmt = $pdo->prepare('
    SELECT
        SUM(CASE WHEN order_type = "purchase" AND status = "completed" THEN total_amount ELSE 0 END) AS sales_amount,
        SUM(CASE WHEN order_type = "purchase" AND status = "completed" THEN 1 ELSE 0 END) AS sales_count,
        SUM(CASE WHEN order_type = "deposit" AND status = "completed" THEN payable_amount ELSE 0 END) AS deposit_amount,
        SUM(CASE WHEN order_type = "deposit" AND status = "completed" THEN commission_amount ELSE 0 END) AS commission_amount
    FROM orders
    WHERE created_at >= :start
');
$todayOrdersStmt->execute([':start' => $todayStart]);
$todayOrders = $todayOrdersStmt->fetch() ?: [];

$todaySalesAmount = (float)($todayOrders['sales_amount'] ?? 0);
$todaySalesCount = (int)($todayOrders['sales_count'] ?? 0);
$todayDepositAmount = (float)($todayOrders['deposit_amount'] ?? 0);
$todayCommissionAmount = (float)($todayOrders['commission_amount'] ?? 0);

$pendingOrdersCount = (int)$pdo->query("SELECT COUNT(*) FROM orders WHERE status = 'pending'")->fetchColumn();

try {
    $openTicketCount = (int)$pdo->query("SELECT COUNT(*) FROM support_tickets WHERE status IN ('open','customer_reply')")->fetchColumn();
} catch (Throwable $th) {
    $openTicketCount = 0;
}

$activeProductsCount = (int)$pdo->query("SELECT COUNT(*) FROM products WHERE is_active = 1")->fetchColumn();

$todayUsersStmt = $pdo->prepare('SELECT COUNT(*) FROM users WHERE created_at >= ?');
$todayUsersStmt->execute([$todayStart]);
$todayNewUsers = (int)$todayUsersStmt->fetchColumn();

$chartStmt = $pdo->prepare('
    SELECT DATE(created_at) AS label,
        SUM(CASE WHEN order_type = "purchase" AND status = "completed" THEN total_amount ELSE 0 END) AS sales_amount,
        SUM(CASE WHEN order_type = "deposit" AND status = "completed" THEN payable_amount ELSE 0 END) AS deposit_amount
    FROM orders
    WHERE created_at >= :start
    GROUP BY DATE(created_at)
    ORDER BY DATE(created_at) ASC
');
$chartStmt->execute([':start' => $chartStart]);
$chartRows = $chartStmt->fetchAll() ?: [];

$chartMap = [];
foreach ($chartRows as $row) {
    $chartMap[$row['label']] = [
        'sales' => (float)$row['sales_amount'],
        'deposit' => (float)$row['deposit_amount'],
    ];
}

$chartLabels = [];
$chartSales = [];
$chartDeposits = [];
for ($i = 0; $i < 7; $i++) {
    $day = $chartStartDate->modify("+{$i} days");
    $key = $day->format('Y-m-d');
    $chartLabels[] = $day->format('d.m');
    $chartSales[] = $chartMap[$key]['sales'] ?? 0.0;
    $chartDeposits[] = $chartMap[$key]['deposit'] ?? 0.0;
}

$chartPayload = [
    'labels' => $chartLabels,
    'sales' => $chartSales,
    'deposits' => $chartDeposits,
];
$chartJson = htmlspecialchars(json_encode($chartPayload, JSON_UNESCAPED_UNICODE | JSON_NUMERIC_CHECK), ENT_QUOTES, 'UTF-8');

$recentOrdersStmt = $pdo->prepare('
    SELECT o.id, o.order_number, o.total_amount, o.status, o.order_type, o.created_at, u.username
    FROM orders o
    LEFT JOIN users u ON u.id = o.user_id
    ORDER BY o.created_at DESC
    LIMIT :limit
');
$recentOrdersStmt->bindValue(':limit', 10, PDO::PARAM_INT);
$recentOrdersStmt->execute();
$recentOrders = $recentOrdersStmt->fetchAll() ?: [];

try {
    $recentTicketsStmt = $pdo->prepare('
        SELECT t.id, t.subject, t.status, t.updated_at, u.username
        FROM support_tickets t
        LEFT JOIN users u ON u.id = t.user_id
        ORDER BY t.updated_at DESC
        LIMIT :limit
    ');
    $recentTicketsStmt->bindValue(':limit', 10, PDO::PARAM_INT);
    $recentTicketsStmt->execute();
    $recentTickets = $recentTicketsStmt->fetchAll() ?: [];
} catch (Throwable $th) {
    $recentTickets = [];
}

$lowStockStmt = $pdo->prepare('
    SELECT id, name, stock
    FROM products
    WHERE is_active = 1 AND stock IS NOT NULL AND stock <= :threshold
    ORDER BY stock ASC, name ASC
    LIMIT :limit
');
$lowStockStmt->bindValue(':threshold', 10, PDO::PARAM_INT);
$lowStockStmt->bindValue(':limit', 10, PDO::PARAM_INT);
$lowStockStmt->execute();
$lowStockProducts = $lowStockStmt->fetchAll() ?: [];

$recentUsersStmt = $pdo->prepare('SELECT username, email, created_at FROM users ORDER BY created_at DESC LIMIT :limit');
$recentUsersStmt->bindValue(':limit', 10, PDO::PARAM_INT);
$recentUsersStmt->execute();
$recentUsers = $recentUsersStmt->fetchAll() ?: [];

$orderStatusLabels = [
    'pending' => 'Beklemede',
    'processing' => 'İşleniyor',
    'completed' => 'Tamamlandı',
    'cancelled' => 'İptal edildi',
];
$orderStatusStyles = [
    'pending' => 'status-chip status-chip--warning',
    'processing' => 'status-chip status-chip--info',
    'completed' => 'status-chip status-chip--success',
    'cancelled' => 'status-chip status-chip--danger',
];

$supportStatusStyles = [
    'open' => 'status-chip status-chip--warning',
    'answered' => 'status-chip status-chip--info',
    'customer_reply' => 'status-chip status-chip--primary',
    'resolved' => 'status-chip status-chip--success',
];

$pageTitle = 'Gösterge Paneli';
include 'includes/admin-header.php';
?>

<div class="admin-page dashboard-page" id="dashboardPage" data-chart='<?= $chartJson ?>'>
    <header class="page-intro glass-surface">
        <div class="page-intro__text">
            <h4>Hoş geldin <?= sanitize($_SESSION['username'] ?? 'Yönetici') ?></h4>
            <p class="page-intro__subtitle">Bugünün performansına hızlı bir bakış</p>
        </div>
        <div class="page-intro__actions">
            <a class="btn btn-outline-primary" href="orders.php"><i class="fas fa-receipt me-2"></i>Siparişler</a>
            <a class="btn btn-primary" href="products.php"><i class="fas fa-plus me-2"></i>Yeni Ürün</a>
        </div>
    </header>

    <section class="metric-grid dashboard-metrics">
        <article class="metric-card">
            <span class="metric-card__label">Bugünkü Satış</span>
            <span class="metric-card__value"><?= formatPrice($todaySalesAmount) ?></span>
            <span class="metric-card__meta"><?= number_format($todaySalesCount) ?> sipariş</span>
        </article>
        <article class="metric-card">
            <span class="metric-card__label">Bugünkü Bakiye Yükleme</span>
            <span class="metric-card__value"><?= formatPrice($todayDepositAmount) ?></span>
            <span class="metric-card__meta">Komisyon: <?= formatPrice($todayCommissionAmount) ?></span>
        </article>
        <article class="metric-card">
            <span class="metric-card__label">Bekleyen Sipariş</span>
            <span class="metric-card__value metric-card__value--warning"><?= number_format($pendingOrdersCount) ?></span>
            <span class="metric-card__meta">Takip edilmesi gereken siparişler</span>
        </article>
        <article class="metric-card">
            <span class="metric-card__label">Açık Destek Talepleri</span>
            <span class="metric-card__value metric-card__value--info"><?= number_format($openTicketCount) ?></span>
            <span class="metric-card__meta">Yanıt bekleyen talepler</span>
        </article>
        <article class="metric-card">
            <span class="metric-card__label">Aktif Ürün</span>
            <span class="metric-card__value"><?= number_format($activeProductsCount) ?></span>
            <span class="metric-card__meta">Mağazada yayınlanan</span>
        </article>
        <article class="metric-card">
            <span class="metric-card__label">Bugün Yeni Üye</span>
            <span class="metric-card__value metric-card__value--positive"><?= number_format($todayNewUsers) ?></span>
            <span class="metric-card__meta">Son 24 saat</span>
        </article>
    </section>

    <section class="glass-surface chart-panel" style="margin-top: 0;">
        <canvas id="dashboardChart" height="140"></canvas>
    </section>

    <div class="panel-grid">
        <section class="card">
            <div class="card-header">
                <div>
                    <h5 class="mb-0"><i class="fas fa-shopping-cart me-2 text-primary"></i>Son Para Yatırmaları</h5>
                    <p class="text-muted mb-0" style="font-size: 13px;">Mağazadaki son hareketler</p>
                </div>
                <a class="btn btn-sm btn-outline-primary" href="orders.php"><i class="fas fa-receipt me-1"></i>Tümünü Gör</a>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0 align-middle">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Müşteri</th>
                                <th>Tutar</th>
                                <th>Tür</th>
                                <th>Durum</th>
                                <th>Tarih</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($recentOrders)): ?>
                                <?php foreach ($recentOrders as $order): ?>
                                    <tr>
                                        <td><strong><?= (int)$order['id'] ?></strong></td>
                                        <td><?= sanitize($order['username'] ?? 'Misafir') ?></td>
                                        <td><strong><?= formatPrice($order['total_amount']) ?></strong></td>
                                        <td>
                                            <span class="badge <?= $order['order_type'] === 'deposit' ? 'bg-info' : 'bg-secondary' ?>">
                                                <?= $order['order_type'] === 'deposit' ? 'Bakiye' : 'Satış' ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php
                                                $status = $order['status'] ?? 'pending';
                                                $label = $orderStatusLabels[$status] ?? ucfirst($status);
                                                $badgeClass = match($status) {
                                                    'completed' => 'bg-success',
                                                    'pending' => 'bg-warning text-dark',
                                                    'processing' => 'bg-info',
                                                    'cancelled' => 'bg-danger',
                                                    default => 'bg-secondary'
                                                };
                                            ?>
                                            <span class="badge <?= $badgeClass ?>"><?= sanitize($label) ?></span>
                                        </td>
                                        <td>
                                            <span class="badge bg-light text-dark">
                                                <i class="far fa-clock me-1"></i><?= date('d.m.Y H:i', strtotime((string)$order['created_at'])) ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="6" class="text-center text-muted py-4">Henüz sipariş bulunmuyor.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </section>

        <section class="card">
            <div class="card-header">
                <div>
                    <h5 class="mb-0"><i class="fas fa-headset me-2 text-info"></i>Destek Kuyruğu</h5>
                    <p class="text-muted mb-0" style="font-size: 13px;">En güncel destek talepleri</p>
                </div>
                <a class="btn btn-sm btn-outline-primary" href="tickets.php"><i class="fas fa-ticket me-1"></i>Destek Paneli</a>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0 align-middle">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Konu</th>
                                <th>Kullanıcı</th>
                                <th>Durum</th>
                                <th>Güncelleme</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($recentTickets)): ?>
                                <?php foreach ($recentTickets as $ticket): ?>
                                    <tr>
                                        <td><strong><?= (int)$ticket['id'] ?></strong></td>
                                        <td><strong><?= sanitize($ticket['subject']) ?></strong></td>
                                        <td><?= sanitize($ticket['username'] ?? 'Bilinmiyor') ?></td>
                                        <td>
                                            <?php
                                                $status = $ticket['status'] ?? 'open';
                                                $label = sanitize(getSupportTicketStatusLabel($status));
                                                $badgeClass = match($status) {
                                                    'resolved' => 'bg-success',
                                                    'open' => 'bg-warning text-dark',
                                                    'answered' => 'bg-info',
                                                    'customer_reply' => 'bg-primary',
                                                    default => 'bg-secondary'
                                                };
                                            ?>
                                            <span class="badge <?= $badgeClass ?>"><?= $label ?></span>
                                        </td>
                                        <td>
                                            <span class="badge bg-light text-dark">
                                                <i class="far fa-clock me-1"></i><?= date('d.m.Y H:i', strtotime((string)$ticket['updated_at'])) ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="5" class="text-center text-muted py-4">Destek talebi bulunmuyor.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </section>
    </div>

    <div class="panel-grid">
        <section class="card">
            <div class="card-header">
                <div>
                    <h5 class="mb-0"><i class="fas fa-exclamation-triangle me-2 text-warning"></i>Düşük Stok Uyarıları</h5>
                    <p class="text-muted mb-0" style="font-size: 13px;">10 adetten az kalan ürünler</p>
                </div>
                <a class="btn btn-sm btn-outline-primary" href="products.php"><i class="fas fa-box me-1"></i>Ürün Yönetimi</a>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0 align-middle">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Ürün Adı</th>
                                <th>Kalan Stok</th>
                                <th class="text-end">İşlem</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($lowStockProducts)): ?>
                                <?php foreach ($lowStockProducts as $product): ?>
                                    <tr>
                                        <td><?= (int)$product['id'] ?></td>
                                        <td><strong><?= sanitize($product['name']) ?></strong></td>
                                        <td>
                                            <span class="badge bg-warning text-dark">
                                                <i class="fas fa-box me-1"></i><?= number_format((int)$product['stock']) ?> adet
                                            </span>
                                        </td>
                                        <td class="text-end">
                                            <a class="btn btn-sm btn-primary" href="product-form.php?id=<?= (int)$product['id'] ?>">
                                                <i class="fas fa-edit me-1"></i>Güncelle
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="4" class="text-center text-muted py-4">
                                        <i class="fas fa-check-circle me-2 text-success"></i>
                                        Tüm ürün stokları yeterli görünüyor.
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </section>

        <section class="card">
            <div class="card-header">
                <div>
                    <h5 class="mb-0"><i class="fas fa-user-plus me-2 text-success"></i>Yeni Kullanıcılar</h5>
                    <p class="text-muted mb-0" style="font-size: 13px;">Son kayıt olan üyeler</p>
                </div>
                <a class="btn btn-sm btn-outline-primary" href="users.php"><i class="fas fa-users me-1"></i>Kullanıcı Listesi</a>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0 align-middle">
                        <thead>
                            <tr>
                                <th>Kullanıcı Adı</th>
                                <th>E-posta</th>
                                <th>Kayıt Tarihi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($recentUsers)): ?>
                                <?php foreach ($recentUsers as $user): ?>
                                    <tr>
                                        <td><strong><?= sanitize($user['username']) ?></strong></td>
                                        <td><?= sanitize($user['email']) ?></td>
                                        <td>
                                            <span class="badge bg-light text-dark">
                                                <i class="far fa-clock me-1"></i><?= date('d.m.Y H:i', strtotime((string)$user['created_at'])) ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="3" class="text-center text-muted py-4">Henüz yeni kayıt bulunmuyor.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </section>
    </div>
</div>

<script>
(function() {
    const container = document.getElementById('dashboardPage');
    const canvas = document.getElementById('dashboardChart');

    if (!container || !canvas || typeof Chart === 'undefined') {
        return;
    }

    let payload = {};
    try {
        payload = JSON.parse(container.dataset.chart || '{}');
    } catch (error) {
        payload = {};
    }

    const styles = getComputedStyle(document.body);
    const axisColor = styles.getPropertyValue('--admin-text-muted') || 'rgba(148,163,184,0.75)';
    const legendColor = styles.getPropertyValue('--admin-text') || '#e2e8f0';
    const gridColor = styles.getPropertyValue('--admin-border') || 'rgba(148,163,184,0.15)';

    const chart = new Chart(canvas, {
        type: 'line',
        data: {
            labels: payload.labels || [],
            datasets: [
                {
                    label: 'Satış',
                    data: payload.sales || [],
                    borderColor: '#6366f1',
                    backgroundColor: 'rgba(99,102,241,0.15)',
                    tension: 0.35,
                    fill: true,
                    borderWidth: 3
                },
                {
                    label: 'Bakiye Yükleme',
                    data: payload.deposits || [],
                    borderColor: '#22d3ee',
                    backgroundColor: 'rgba(34,211,238,0.15)',
                    tension: 0.35,
                    fill: true,
                    borderWidth: 3
                }
            ]
        },
        options: {
            plugins: {
                legend: {
                    labels: {
                        color: legendColor.trim(),
                        usePointStyle: true
                    }
                }
            },
            scales: {
                x: {
                    ticks: { color: axisColor.trim() },
                    grid: { color: gridColor.trim() }
                },
                y: {
                    ticks: { color: axisColor.trim() },
                    grid: { color: gridColor.trim() }
                }
            }
        }
    });

    document.addEventListener('admin-theme-change', () => {
        const newStyles = getComputedStyle(document.body);
        chart.options.plugins.legend.labels.color = (newStyles.getPropertyValue('--admin-text') || '#e2e8f0').trim();
        chart.options.scales.x.ticks.color = (newStyles.getPropertyValue('--admin-text-muted') || 'rgba(148,163,184,0.75)').trim();
        chart.options.scales.y.ticks.color = (newStyles.getPropertyValue('--admin-text-muted') || 'rgba(148,163,184,0.75)').trim();
        chart.options.scales.x.grid.color = (newStyles.getPropertyValue('--admin-border') || 'rgba(148,163,184,0.15)').trim();
        chart.options.scales.y.grid.color = (newStyles.getPropertyValue('--admin-border') || 'rgba(148,163,184,0.15)').trim();
        chart.update();
    });
})();
</script>

<?php include 'includes/admin-footer.php'; ?>
