<?php
require_once '../config.php';

// Zaten admin olarak giriş yaptıysa admin panele yönlendir
if (isLoggedIn() && isAdmin()) {
    redirect('index.php');
}

// Zaten giriş yaptıysa ama admin değilse ana sayfaya
if (isLoggedIn() && !isAdmin()) {
    redirect('../index.php');
}

$error = '';
$recaptchaEnabledAdminLogin = recaptchaEnabled('admin_login');
$recaptchaSiteKey = getSetting('recaptcha_site_key', RECAPTCHA_SITE_KEY);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $error = 'Güvenlik hatası. Lütfen tekrar deneyin.';
    } else {
        // reCAPTCHA kontrolü
        if ($recaptchaEnabledAdminLogin && $recaptchaSiteKey) {
            $token = safe_trim($_POST['g-recaptcha-response'] ?? '');
            if (!verifyRecaptcha($token)) {
                $error = 'Doğrulama başarısız. Lütfen robot olmadığınızı kanıtlayın.';
            }
        }
        
        if ($error === '') {
            $username = safe_trim($_POST['username'] ?? '');
            $password = $_POST['password'] ?? '';

            if ($username === '' || $password === '') {
                $error = 'Kullanıcı adı ve şifre gereklidir.';
            } else {
                $stmt = $pdo->prepare("SELECT * FROM users WHERE (username = ? OR email = ?) AND is_admin = 1");
                $stmt->execute([$username, $username]);
                $user = $stmt->fetch();

                if ($user && password_verify($password, $user['password'])) {
                    if ((int)$user['is_active'] === 0) {
                        $error = 'Hesabınız aktif değil.';
                        logUserActivity($pdo, 'admin_login_blocked', ['username' => $username, 'user_id' => $user['id']]);
                    } else {
                        $_SESSION['user_id'] = $user['id'];
                        $_SESSION['username'] = $user['username'];
                        $_SESSION['email'] = $user['email'];
                        $_SESSION['is_admin'] = $user['is_admin'];
                        $_SESSION['balance'] = $user['balance'];

                        logUserActivity($pdo, 'admin_login_success', ['username' => $username, 'user_id' => $user['id']]);
                        flashMessage('success', 'Hoş geldiniz, ' . sanitize($user['username']) . '!');
                        redirect('index.php');
                    }
                } else {
                    $error = 'Kullanıcı adı veya şifre hatalı veya admin yetkisi yok.';
                    logUserActivity($pdo, 'admin_login_failed', ['username' => $username]);
                }
            }
        }
    }
}

$pageTitle = 'Admin Girişi';
?>
<!DOCTYPE html>
<html lang="tr" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="<?= generateCSRFToken() ?>">
    <title><?= $pageTitle ?> - <?= siteName() ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="<?= publicUrl('assets/css/auth-modern.css') ?>?v=<?= time() ?>">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .admin-login-card {
            max-width: 450px;
            width: 90%;
            background: rgba(255,255,255,0.95);
            border-radius: 20px;
            padding: 3rem;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        }
        .admin-login-header {
            text-align: center;
            margin-bottom: 2rem;
        }
        .admin-login-header i {
            font-size: 3rem;
            color: #6366f1;
            margin-bottom: 1rem;
        }
        .admin-login-header h2 {
            color: #1e293b;
            margin-bottom: 0.5rem;
        }
        .admin-login-header p {
            color: #64748b;
            margin: 0;
        }
    </style>
</head>
<body>
    <div class="admin-login-card">
        <div class="admin-login-header">
            <i class="fas fa-shield-halved"></i>
            <h2>Admin Paneli</h2>
            <p>Yönetim sayfasına erişim</p>
        </div>

        <?php if ($error !== ''): ?>
            <div class="alert alert-danger">
                <i class="fa-solid fa-circle-exclamation me-2"></i><?= sanitize($error) ?>
            </div>
        <?php endif; ?>

        <form method="post"<?= $recaptchaEnabledAdminLogin && $recaptchaSiteKey ? ' data-recaptcha-form' : '' ?> autocomplete="off">
            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
            
            <div class="mb-3">
                <label for="username" class="form-label">Kullanıcı Adı</label>
                <input type="text" id="username" name="username" class="form-control" value="<?= sanitize($_POST['username'] ?? '') ?>" required autofocus>
            </div>
            
            <div class="mb-3">
                <label for="password" class="form-label">Şifre</label>
                <input type="password" id="password" name="password" class="form-control" required>
            </div>
            
            <?php if ($recaptchaEnabledAdminLogin && $recaptchaSiteKey): ?>
                <div class="mb-3" data-recaptcha data-site-key="<?= sanitize($recaptchaSiteKey) ?>"></div>
            <?php endif; ?>
            
            <button type="submit" class="btn btn-primary w-100 mb-3">
                <i class="fa-solid fa-right-to-bracket me-2"></i>Admin Girişi
            </button>
        </form>

        <div class="text-center mt-3">
            <a href="<?= routeUrl('home') ?>" class="text-muted">
                <i class="fas fa-arrow-left me-1"></i>Ana Sayfaya Dön
            </a>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <?php if ($recaptchaEnabledAdminLogin && $recaptchaSiteKey): ?>
        <script>
            // reCAPTCHA için basit render
            var recaptchaContainer = document.querySelector('[data-recaptcha]');
            if (recaptchaContainer && recaptchaContainer.dataset.siteKey) {
                var script = document.createElement('script');
                script.src = 'https://www.google.com/recaptcha/api.js?onload=onRecaptchaLoad&render=explicit';
                script.async = true;
                script.defer = true;
                window.onRecaptchaLoad = function() {
                    grecaptcha.render(recaptchaContainer, {
                        sitekey: recaptchaContainer.dataset.siteKey,
                        theme: 'light'
                    });
                };
                document.head.appendChild(script);
            }
        </script>
    <?php endif; ?>
</body>
</html>

