<?php
require_once '../config.php';
requireAdmin();

if (!function_exists('shortenText')) {
    function shortenText(?string $text, int $limit = 150): string
    {
        $text = trim((string)$text);
        if ($text === '') {
            return '';
        }
        if (mb_strlen($text) <= $limit) {
            return $text;
        }
        return rtrim(mb_substr($text, 0, $limit - 3)) . '...';
    }
}

$games = getMarketplaceGames($pdo, false);
if (empty($games)) {
    flashMessage('warning', 'Önce marketplace oyunları oluşturmalısınız.');
    redirect('marketplace-games.php');
}

$gameMap = [];
foreach ($games as $game) {
    $gameMap[$game['id']] = $game;
}

$selectedGameId = isset($_GET['game']) ? (int)$_GET['game'] : (int)$games[0]['id'];
if (!isset($gameMap[$selectedGameId])) {
    $selectedGameId = (int)$games[0]['id'];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_category'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('marketplace-categories.php?game=' . $selectedGameId);
    }

    $id = isset($_POST['id']) && $_POST['id'] !== '' ? (int)$_POST['id'] : null;
    $marketplaceId = isset($_POST['marketplace_id']) ? (int)$_POST['marketplace_id'] : $selectedGameId;
    if (!isset($gameMap[$marketplaceId])) {
        flashMessage('danger', 'Marketplace oyunu bulunamadı.');
        redirect('marketplace-categories.php?game=' . $selectedGameId);
    }

    $name = trim($_POST['name'] ?? '');
    $providedSlug = trim($_POST['slug'] ?? '');
    $slug = slugify($providedSlug !== '' ? $providedSlug : $name);
    $description = trim($_POST['description'] ?? '');
    $warningText = trim($_POST['warning_text'] ?? '');
    $displayOrder = isset($_POST['display_order']) ? (int)$_POST['display_order'] : 0;
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    $requiresCharacterName = isset($_POST['requires_character_name']) ? 1 : 0;
    $autoApproveOrders = isset($_POST['auto_approve_orders']) ? 1 : 0;
    $customProductColumns = $_POST['custom_product_columns'] ?? '';
    $customCategoryColumns = $_POST['custom_category_columns'] ?? '';
    $customProductColumns = $customProductColumns === '' ? null : max(1, min(6, (int)$customProductColumns));
    $customCategoryColumns = $customCategoryColumns === '' ? null : max(1, min(4, (int)$customCategoryColumns));

    if ($name === '') {
        flashMessage('danger', 'Kategori adı zorunludur.');
        redirect('marketplace-categories.php?game=' . $marketplaceId . ($id ? '&edit=' . $id : ''));
    }

    $currentImage = null;
    $currentBannerImage = null;

    if ($id) {
        $stmt = $pdo->prepare('SELECT * FROM game_marketplace_categories WHERE id = ?');
        $stmt->execute([$id]);
        $existing = $stmt->fetch();
        if (!$existing) {
            flashMessage('danger', 'Kategori bulunamadı.');
            redirect('marketplace-categories.php?game=' . $selectedGameId);
        }
        $currentImage = $existing['image'] ?? null;
        $currentBannerImage = $existing['banner_image'] ?? null;
        $marketplaceId = (int)$existing['marketplace_id'];
    }

    try {
        $currentImage = uploadImageFile($_FILES['image'] ?? [], 'marketplace/categories', $currentImage);
        $currentBannerImage = uploadImageFile($_FILES['banner_image'] ?? [], 'marketplace/banners', $currentBannerImage);
    } catch (Exception $e) {
        flashMessage('danger', 'Kategori görselleri yüklenemedi: ' . $e->getMessage());
        redirect('marketplace-categories.php?game=' . $marketplaceId . ($id ? '&edit=' . $id : ''));
    }

    try {
        if ($id) {
            $stmt = $pdo->prepare('UPDATE game_marketplace_categories SET name = ?, slug = ?, description = ?, warning_text = ?, image = ?, banner_image = ?, requires_character_name = ?, auto_approve_orders = ?, custom_product_columns = ?, custom_category_columns = ?, display_order = ?, is_active = ?, updated_at = NOW() WHERE id = ?');
            $stmt->execute([$name, $slug, $description, $warningText, $currentImage, $currentBannerImage, $requiresCharacterName, $autoApproveOrders, $customProductColumns, $customCategoryColumns, $displayOrder, $isActive, $id]);
            flashMessage('success', 'Kategori güncellendi.');
        } else {
            $stmt = $pdo->prepare('INSERT INTO game_marketplace_categories (marketplace_id, name, slug, description, warning_text, image, banner_image, requires_character_name, auto_approve_orders, custom_product_columns, custom_category_columns, display_order, is_active) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
            $stmt->execute([$marketplaceId, $name, $slug, $description, $warningText, $currentImage, $currentBannerImage, $requiresCharacterName, $autoApproveOrders, $customProductColumns, $customCategoryColumns, $displayOrder, $isActive]);
            flashMessage('success', 'Kategori eklendi.');
        }
        redirect('marketplace-categories.php?game=' . $marketplaceId);
    } catch (PDOException $e) {
        flashMessage('danger', 'İşlem başarısız: ' . $e->getMessage());
        redirect('marketplace-categories.php?game=' . $marketplaceId . ($id ? '&edit=' . $id : ''));
    }
}

if (isset($_GET['delete'])) {
    $deleteId = (int)$_GET['delete'];
    try {
        $stmt = $pdo->prepare('SELECT image, banner_image, marketplace_id FROM game_marketplace_categories WHERE id = ?');
        $stmt->execute([$deleteId]);
        $category = $stmt->fetch();
        if ($category) {
            if (!empty($category['image'])) {
                deleteUploadedFile($category['image']);
            }
            if (!empty($category['banner_image'])) {
                deleteUploadedFile($category['banner_image']);
            }
            $pdo->prepare('DELETE FROM game_marketplace_categories WHERE id = ?')->execute([$deleteId]);
            flashMessage('success', 'Kategori silindi.');
            $selectedGameId = (int)$category['marketplace_id'];
        }
    } catch (PDOException $e) {
        flashMessage('danger', 'Kategori silinirken hata oluştu: ' . $e->getMessage());
    }
    redirect('marketplace-categories.php?game=' . $selectedGameId);
}

// Pagination
$countStmt = $pdo->prepare('SELECT COUNT(*) FROM game_marketplace_categories WHERE marketplace_id = ?');
$countStmt->execute([$selectedGameId]);
$totalCategories = (int)$countStmt->fetchColumn();
$pagination = getPaginationData($totalCategories, 10);

$stmt = $pdo->prepare('SELECT c.*, m.name AS marketplace_name FROM game_marketplace_categories c JOIN game_marketplaces m ON m.id = c.marketplace_id WHERE c.marketplace_id = ? ORDER BY c.display_order ASC, c.name ASC LIMIT ' . $pagination['limit'] . ' OFFSET ' . $pagination['offset']);
$stmt->execute([$selectedGameId]);
$categories = $stmt->fetchAll();

$editCategory = null;
if (isset($_GET['edit'])) {
    $stmt = $pdo->prepare('SELECT * FROM game_marketplace_categories WHERE id = ?');
    $stmt->execute([(int)$_GET['edit']]);
    $editCategory = $stmt->fetch();
    if (!$editCategory) {
        flashMessage('danger', 'Kategori bulunamadı.');
        redirect('marketplace-categories.php?game=' . $selectedGameId);
    }
    $selectedGameId = (int)$editCategory['marketplace_id'];
}

$pageTitle = 'Marketplace Kategorileri';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex flex-wrap align-items-center gap-3 mb-3">
        <h3 class="mb-0"><i class="fas fa-layer-group me-2"></i>Marketplace Kategorileri</h3>
        <form method="get" class="d-flex align-items-center gap-2">
            <label for="game" class="text-muted">Oyun:</label>
            <select name="game" id="game" class="form-select" onchange="this.form.submit()">
                <?php foreach ($games as $game): ?>
                    <option value="<?= $game['id'] ?>" <?= $selectedGameId === (int)$game['id'] ? 'selected' : '' ?>><?= sanitize($game['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </form>
        <a href="marketplace-games.php" class="btn btn-sm btn-outline-secondary"><i class="fas fa-arrow-left me-1"></i>Oyunlara Dön</a>
        <a href="marketplace-servers.php?game=<?= $selectedGameId ?>" class="btn btn-sm btn-outline-primary"><i class="fas fa-box me-1"></i>Ürünleri Yönet</a>
    </div>

    <div class="alert alert-danger d-flex align-items-center gap-2 mb-4" role="alert" style="border-left: 4px solid #dc3545;">
        <i class="fas fa-info-circle"></i>
        <strong>Verileri görmek için yukarıdan oyun seçimi yapınız.</strong>
    </div>

    <div class="row g-4">
        <div class="col-lg-5">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-<?= $editCategory ? 'edit' : 'plus' ?> me-2"></i>
                        <?= $editCategory ? 'Kategori Düzenle' : 'Yeni Kategori' ?>
                    </h5>
                </div>
                <div class="card-body">
                    <form method="post" enctype="multipart/form-data">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="save_category" value="1">
                        <?php if ($editCategory): ?>
                            <input type="hidden" name="id" value="<?= $editCategory['id'] ?>">
                        <?php endif; ?>

                        <div class="mb-3">
                            <label class="form-label">Marketplace Oyunu</label>
                            <select name="marketplace_id" class="form-select" required>
                                <?php foreach ($games as $game): ?>
                                    <option value="<?= $game['id'] ?>" <?= ($editCategory ? (int)$editCategory['marketplace_id'] === (int)$game['id'] : $selectedGameId === (int)$game['id']) ? 'selected' : '' ?>><?= sanitize($game['name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kategori Adı</label>
                            <input type="text" name="name" class="form-control" value="<?= $editCategory ? sanitize($editCategory['name']) : '' ?>" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Slug</label>
                            <input type="text" name="slug" class="form-control" value="<?= $editCategory ? sanitize($editCategory['slug']) : '' ?>" placeholder="örn. knight-gold-bar" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kısa Açıklama</label>
                            <textarea name="description" class="form-control" rows="3" placeholder="Kullanıcıya gösterilecek kısa açıklama."><?= $editCategory ? sanitize($editCategory['description']) : '' ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Uyarı Metni</label>
                            <textarea name="warning_text" class="form-control" rows="3" placeholder="Bize Sat modalında gösterilecek uyarılar."><?= $editCategory ? sanitize($editCategory['warning_text']) : '' ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kart Görseli</label>
                            <?php if ($editCategory && !empty($editCategory['image'])): ?>
                                <div class="mb-2">
                                    <img src="<?= htmlspecialchars(assetUrl($editCategory['image']), ENT_QUOTES, 'UTF-8') ?>" alt="" class="img-fluid rounded shadow-sm" style="max-height: 140px;">
                                </div>
                            <?php endif; ?>
                            <input type="file" name="image" class="form-control" accept="image/*">
                            <small class="text-muted">PNG, JPG veya WEBP (maks. 4MB)</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Üst Banner Görseli</label>
                            <?php if ($editCategory && !empty($editCategory['banner_image'])): ?>
                                <div class="mb-2">
                                    <img src="<?= htmlspecialchars(assetUrl($editCategory['banner_image']), ENT_QUOTES, 'UTF-8') ?>" alt="" class="img-fluid rounded shadow-sm" style="max-height: 160px;">
                                </div>
                            <?php endif; ?>
                            <input type="file" name="banner_image" class="form-control" accept="image/*">
                            <small class="text-muted">Kategori sayfasının üst bölümünde kullanılacak geniş görsel.</small>
                        </div>

                        <div class="form-check form-switch mb-3 ps-0">
                            <input class="form-check-input ms-0" type="checkbox" name="requires_character_name" id="requires_character_name" <?= $editCategory ? ((int)$editCategory['requires_character_name'] ? 'checked' : '') : 'checked' ?>>
                            <label class="form-check-label" for="requires_character_name">Satın alırken karakter adı zorunlu olsun</label>
                            <small class="text-muted d-block">Kapalı olduğunda kullanıcıdan karakter bilgisi istenmez.</small>
                        </div>

                        <div class="form-check form-switch mb-3 ps-0">
                            <input class="form-check-input ms-0" type="checkbox" name="auto_approve_orders" id="auto_approve_orders" <?= $editCategory ? ((int)$editCategory['auto_approve_orders'] ? 'checked' : '') : '' ?>>
                            <label class="form-check-label" for="auto_approve_orders">Stok varsa siparişi otomatik onayla</label>
                            <small class="text-muted d-block">Aktif olduğunda yeterli kod varsa sipariş anında tamamlanır.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Sıra</label>
                            <input type="number" name="display_order" class="form-control" value="<?= $editCategory ? (int)$editCategory['display_order'] : 0 ?>">
                        </div>

                        <div class="form-check form-switch mb-3 ps-0">
                            <input class="form-check-input ms-0" type="checkbox" name="is_active" id="is_active" <?= $editCategory ? ($editCategory['is_active'] ? 'checked' : '') : 'checked' ?>>
                            <label class="form-check-label" for="is_active">Aktif</label>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kategori Kart Sütunları (Oyun Varsayılanı: <?= isset($gameMap[$selectedGameId]['category_grid_columns']) ? (int)$gameMap[$selectedGameId]['category_grid_columns'] : 3 ?>)</label>
                            <select name="custom_category_columns" class="form-select">
                                <option value="">Oyun varsayılanını kullan</option>
                                <?php for ($i = 1; $i <= 4; $i++): ?>
                                    <option value="<?= $i ?>" <?= $editCategory && $editCategory['custom_category_columns'] !== null && (int)$editCategory['custom_category_columns'] === $i ? 'selected' : '' ?>><?= $i ?> kolon</option>
                                <?php endfor; ?>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Ürün Kart Sütunları (Oyun Varsayılanı: <?= isset($gameMap[$selectedGameId]['product_grid_columns']) ? (int)$gameMap[$selectedGameId]['product_grid_columns'] : 4 ?>)</label>
                            <select name="custom_product_columns" class="form-select">
                                <option value="">Oyun varsayılanını kullan</option>
                                <?php for ($i = 2; $i <= 6; $i++): ?>
                                    <option value="<?= $i ?>" <?= $editCategory && $editCategory['custom_product_columns'] !== null && (int)$editCategory['custom_product_columns'] === $i ? 'selected' : '' ?>><?= $i ?> kolon</option>
                                <?php endfor; ?>
                            </select>
                        </div>

                        <div class="d-flex gap-2">
                            <button type="submit" class="btn btn-primary flex-fill">
                                <i class="fas fa-save me-2"></i>Kaydet
                            </button>
                            <?php if ($editCategory): ?>
                                <a href="marketplace-categories.php?game=<?= $selectedGameId ?>" class="btn btn-secondary">
                                    <i class="fas fa-times"></i>
                                </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-7">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-list me-2"></i><?= sanitize($gameMap[$selectedGameId]['name']) ?> Kategorileri</h5>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0 align-middle">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Kategori</th>
                                    <th>Slug</th>
                                    <th>Sıra</th>
                                    <th>Durum</th>
                                    <th>Kurallar</th>
                                    <th class="text-end">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($categories)): ?>
                                <tr>
                                    <td colspan="6" class="text-center text-muted py-4">Bu oyuna ait kategori bulunmuyor.</td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($categories as $category): ?>
                                    <tr>
                                        <td><strong>#<?= $category['id'] ?></strong></td>
                                        <td>
                                            <strong><?= sanitize($category['name']) ?></strong>
                                            <?php if (!empty($category['description'])): ?>
                                                <br><small class="text-muted"><?= sanitize(shortenText($category['description'], 120)) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><code><?= sanitize($category['slug']) ?></code></td>
                                        <td><?= (int)$category['display_order'] ?></td>
                                        <td>
                                            <?php if ($category['is_active']): ?>
                                                <span class="badge bg-success">Aktif</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Pasif</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (!empty($category['requires_character_name'])): ?>
                                                <span class="badge bg-info text-dark">Nick Zorunlu</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Nick Sorulmaz</span>
                                            <?php endif; ?>
                                            <br>
                                            <?php if (!empty($category['auto_approve_orders'])): ?>
                                                <span class="badge bg-success mt-1">Oto Onay</span>
                                            <?php else: ?>
                                                <span class="badge bg-warning text-dark mt-1">Manuel Onay</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="text-end">
                                            <div class="d-inline-flex gap-2">
                                                <a href="marketplace-categories.php?game=<?= $selectedGameId ?>&edit=<?= $category['id'] ?>" class="btn btn-sm btn-outline-primary"><i class="fas fa-edit"></i></a>
                                                <a href="marketplace-categories.php?game=<?= $selectedGameId ?>&delete=<?= $category['id'] ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Bu kategoriyi silmek istediğinize emin misiniz? İlgili server kayıtları da silinecektir.');"><i class="fas fa-trash"></i></a>
                                                <a href="marketplace-servers.php?category=<?= $category['id'] ?>" class="btn btn-sm btn-outline-secondary"><i class="fas fa-server"></i></a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php if ($totalCategories > 10): ?>
                    <div class="card-body border-top">
                        <?= renderPagination($pagination) ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

