<?php
require_once '../config.php';
requireAdmin();

if (!function_exists('shortenText')) {
    function shortenText(?string $text, int $limit = 120): string
    {
        $text = trim((string)$text);
        if ($text === '') {
            return '';
        }
        if (mb_strlen($text) <= $limit) {
            return $text;
        }
        return rtrim(mb_substr($text, 0, $limit - 3)) . '...';
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_game'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('marketplace-games.php');
    }

    $id = isset($_POST['id']) && $_POST['id'] !== '' ? (int)$_POST['id'] : null;
    $name = trim($_POST['name'] ?? '');
    $providedSlug = trim($_POST['slug'] ?? '');
    $slug = slugify($providedSlug !== '' ? $providedSlug : $name);
    $description = trim($_POST['description'] ?? '');
    $displayOrder = isset($_POST['display_order']) ? (int)$_POST['display_order'] : 0;
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    $productColumns = isset($_POST['product_grid_columns']) ? max(1, min(6, (int)$_POST['product_grid_columns'])) : 4;
    $categoryColumns = isset($_POST['category_grid_columns']) ? max(1, min(6, (int)$_POST['category_grid_columns'])) : 3;

    if ($name === '') {
        flashMessage('danger', 'Oyun adı zorunludur.');
        redirect('marketplace-games.php' . ($id ? '?edit=' . $id : ''));
    }

    $currentImage = null;
    $currentBannerImage = null;

    if ($id) {
        $stmt = $pdo->prepare('SELECT image, banner_image FROM game_marketplaces WHERE id = ?');
        $stmt->execute([$id]);
        $existingGame = $stmt->fetch();
        if (!$existingGame) {
            flashMessage('danger', 'Marketplace oyunu bulunamadı.');
            redirect('marketplace-games.php');
        }
        $currentImage = $existingGame['image'] ?? null;
        $currentBannerImage = $existingGame['banner_image'] ?? null;
    }

    try {
        $currentImage = uploadImageFile($_FILES['image'] ?? [], 'marketplace/games', $currentImage);
        $currentBannerImage = uploadImageFile($_FILES['banner_image'] ?? [], 'marketplace/game-banners', $currentBannerImage);
    } catch (Exception $e) {
        flashMessage('danger', 'Oyun görselleri yüklenemedi: ' . $e->getMessage());
        redirect('marketplace-games.php' . ($id ? '?edit=' . $id : ''));
    }

    try {
        if ($id) {
            $stmt = $pdo->prepare('UPDATE game_marketplaces SET name = ?, slug = ?, description = ?, image = ?, banner_image = ?, product_grid_columns = ?, category_grid_columns = ?, display_order = ?, is_active = ?, updated_at = NOW() WHERE id = ?');
            $stmt->execute([$name, $slug, $description, $currentImage, $currentBannerImage, $productColumns, $categoryColumns, $displayOrder, $isActive, $id]);
            flashMessage('success', 'Marketplace oyunu güncellendi.');
        } else {
            $stmt = $pdo->prepare('INSERT INTO game_marketplaces (name, slug, description, image, banner_image, product_grid_columns, category_grid_columns, display_order, is_active) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)');
            $stmt->execute([$name, $slug, $description, $currentImage, $currentBannerImage, $productColumns, $categoryColumns, $displayOrder, $isActive]);
            flashMessage('success', 'Marketplace oyunu eklendi.');
        }
        redirect('marketplace-games.php');
    } catch (PDOException $e) {
        flashMessage('danger', 'İşlem başarısız: ' . $e->getMessage());
        redirect('marketplace-games.php' . ($id ? '?edit=' . $id : ''));
    }
}

if (isset($_GET['delete'])) {
    $deleteId = (int)$_GET['delete'];
    try {
        $stmt = $pdo->prepare('SELECT image, banner_image FROM game_marketplaces WHERE id = ?');
        $stmt->execute([$deleteId]);
        if ($game = $stmt->fetch()) {
            if (!empty($game['image'])) {
                deleteUploadedFile($game['image']);
            }
            if (!empty($game['banner_image'])) {
                deleteUploadedFile($game['banner_image']);
            }
        }

        $stmt = $pdo->prepare('DELETE FROM game_marketplaces WHERE id = ?');
        $stmt->execute([$deleteId]);
        flashMessage('success', 'Marketplace oyunu silindi.');
    } catch (PDOException $e) {
        flashMessage('danger', 'Silme işlemi başarısız oldu: ' . $e->getMessage());
    }
    redirect('marketplace-games.php');
}

// Pagination
$totalGamesStmt = $pdo->query("SELECT COUNT(*) FROM game_marketplaces");
$totalGames = (int)$totalGamesStmt->fetchColumn();
$pagination = getPaginationData($totalGames, 10);

// Get games with pagination
$gamesStmt = $pdo->query("SELECT * FROM game_marketplaces ORDER BY display_order ASC, name ASC LIMIT {$pagination['limit']} OFFSET {$pagination['offset']}");
$games = $gamesStmt->fetchAll();

$editGame = null;
if (isset($_GET['edit'])) {
    $stmt = $pdo->prepare('SELECT * FROM game_marketplaces WHERE id = ?');
    $stmt->execute([(int)$_GET['edit']]);
    $editGame = $stmt->fetch();
    if (!$editGame) {
        flashMessage('danger', 'Marketplace oyunu bulunamadı.');
        redirect('marketplace-games.php');
    }
}

$pageTitle = 'Marketplace Oyunları';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="row g-4">
        <div class="col-lg-4">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-<?= $editGame ? 'edit' : 'plus' ?> me-2"></i>
                        <?= $editGame ? 'Marketplace Oyunu Düzenle' : 'Yeni Marketplace Oyunu' ?>
                    </h5>
                </div>
                <div class="card-body">
                    <form method="post" enctype="multipart/form-data">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="save_game" value="1">
                        <?php if ($editGame): ?>
                            <input type="hidden" name="id" value="<?= $editGame['id'] ?>">
                        <?php endif; ?>

                        <div class="mb-3">
                            <label class="form-label">Oyun Adı</label>
                            <input type="text" name="name" class="form-control" value="<?= $editGame ? sanitize($editGame['name']) : '' ?>" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Slug</label>
                            <input type="text" name="slug" class="form-control" value="<?= $editGame ? sanitize($editGame['slug']) : '' ?>" placeholder="örn. knight-online" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Açıklama</label>
                            <textarea name="description" class="form-control" rows="3" placeholder="İsteğe bağlı açıklama"><?= $editGame ? sanitize($editGame['description']) : '' ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kart Görseli</label>
                            <?php if ($editGame && !empty($editGame['image'])): ?>
                                <div class="mb-2">
                                    <img src="<?= htmlspecialchars(assetUrl($editGame['image']), ENT_QUOTES, 'UTF-8') ?>" alt="<?= sanitize($editGame['name'] ?? '') ?>" class="img-fluid rounded shadow-sm" style="max-height: 140px;">
                                </div>
                            <?php endif; ?>
                            <input type="file" name="image" class="form-control" accept="image/*">
                            <small class="text-muted">PNG, JPG, WEBP veya GIF formatında 4MB'a kadar görsel yükleyebilirsiniz.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Üst Banner Görseli</label>
                            <?php if ($editGame && !empty($editGame['banner_image'])): ?>
                                <div class="mb-2">
                                    <img src="<?= htmlspecialchars(assetUrl($editGame['banner_image']), ENT_QUOTES, 'UTF-8') ?>" alt="<?= sanitize($editGame['name'] ?? '') ?>" class="img-fluid rounded shadow-sm" style="max-height: 160px;">
                                </div>
                            <?php endif; ?>
                            <input type="file" name="banner_image" class="form-control" accept="image/*">
                            <small class="text-muted">Kategori sayfalarında kullanılacak geniş banner görseli.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Ürün Grid Sütun Sayısı</label>
                            <select name="product_grid_columns" class="form-select">
                                <?php for ($i = 2; $i <= 6; $i++): ?>
                                    <option value="<?= $i ?>" <?= $editGame ? ((int)$editGame['product_grid_columns'] === $i ? 'selected' : '') : ($i === 4 ? 'selected' : '') ?>><?= $i ?> kolon</option>
                                <?php endfor; ?>
                            </select>
                            <small class="text-muted">Marketplace kartlarının tek satırda kaç adet görüneceğini belirler.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kategori Kart Sütun Sayısı</label>
                            <select name="category_grid_columns" class="form-select">
                                <?php for ($i = 1; $i <= 4; $i++): ?>
                                    <option value="<?= $i ?>" <?= $editGame ? ((int)$editGame['category_grid_columns'] === $i ? 'selected' : '') : ($i === 3 ? 'selected' : '') ?>><?= $i ?> kolon</option>
                                <?php endfor; ?>
                            </select>
                            <small class="text-muted">Kategori kartlarının tek satırdaki adedini belirtir.</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Sıra</label>
                            <input type="number" name="display_order" class="form-control" value="<?= $editGame ? (int)$editGame['display_order'] : 0 ?>">
                            <small class="text-muted">Küçük değerler menüde önce gösterilir.</small>
                        </div>

                        <div class="form-check form-switch mb-3 ps-0">
                            <input class="form-check-input ms-0" type="checkbox" name="is_active" id="is_active" <?= $editGame ? ($editGame['is_active'] ? 'checked' : '') : 'checked' ?>>
                            <label class="form-check-label" for="is_active">Aktif</label>
                        </div>

                        <div class="d-flex gap-2">
                            <button type="submit" class="btn btn-primary flex-fill">
                                <i class="fas fa-save me-2"></i>Kaydet
                            </button>
                            <?php if ($editGame): ?>
                            <a href="marketplace-games.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i>
                            </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-8">
            <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-list me-2"></i>Marketplace Oyunları</h5>
                    <a href="marketplace-categories.php" class="btn btn-sm btn-outline-primary"><i class="fas fa-layer-group me-1"></i>Kategorileri Yönet</a>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0 align-middle">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Oyun</th>
                                    <th>Slug</th>
                                    <th>Sıra</th>
                                    <th>Durum</th>
                                    <th class="text-end">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($games)): ?>
                                <tr>
                                    <td colspan="6" class="text-center text-muted py-4">Henüz marketplace oyunu eklenmemiş.</td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($games as $game): ?>
                                    <tr>
                                        <td><strong>#<?= $game['id'] ?></strong></td>
                                        <td>
                                            <strong><?= sanitize($game['name']) ?></strong>
                                            <?php if (!empty($game['description'])): ?>
                                                <br><small class="text-muted"><?= sanitize(shortenText($game['description'], 120)) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><code><?= sanitize($game['slug']) ?></code></td>
                                        <td><?= (int)$game['display_order'] ?></td>
                                        <td>
                                            <?php if ($game['is_active']): ?>
                                                <span class="badge bg-success">Aktif</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Pasif</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="text-end">
                                            <div class="d-inline-flex gap-2">
                                                <a href="marketplace-games.php?edit=<?= $game['id'] ?>" class="btn btn-sm btn-outline-primary"><i class="fas fa-edit"></i></a>
                                                <a href="marketplace-games.php?delete=<?= $game['id'] ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Bu oyunu silmek istediğinize emin misiniz? İlgili tüm kategori ve serverlar da silinir.');"><i class="fas fa-trash"></i></a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php if ($totalGames > 10): ?>
                    <div class="card-body border-top">
                        <?= renderPagination($pagination) ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

