<?php
require_once '../config.php';
requireAdmin();

$statusesMap = [
    'pending' => 'Beklemede',
    'processing' => 'İşleniyor',
    'completed' => 'Tamamlandı',
    'cancelled' => 'İptal Edildi'
];

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_status'])) {
    $csrf = $_POST['csrf_token'] ?? '';
    if (!verifyCSRFToken($csrf)) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('marketplace-orders.php');
    }

    $orderId = (int)($_POST['order_id'] ?? 0);
    $newStatus = $_POST['status'] ?? 'pending';
    $adminNote = trim($_POST['admin_note'] ?? '');

    if (!$orderId || !isset($statusesMap[$newStatus])) {
        flashMessage('danger', 'Geçersiz sipariş bilgisi.');
        redirect('marketplace-orders.php');
    }

    try {
        $pdo->beginTransaction();
        $stmt = $pdo->prepare('SELECT * FROM game_marketplace_orders WHERE id = ? FOR UPDATE');
        $stmt->execute([$orderId]);
        $order = $stmt->fetch();
        if (!$order) {
            $pdo->rollBack();
            flashMessage('danger', 'Sipariş bulunamadı.');
            redirect('marketplace-orders.php');
        }

        $oldStatus = $order['status'];
        if ($oldStatus === $newStatus) {
            $pdo->rollBack();
            flashMessage('info', 'Sipariş zaten bu durumda.');
            redirect('marketplace-orders.php');
        }

        $codesToAssign = [];
        $codesToRelease = [];
        $quantityRequired = (int)max(0, ceil((float)$order['quantity']));

        if ($order['order_type'] === 'buy') {
            if ($newStatus === 'completed' && $oldStatus !== 'completed') {
                $codeLookup = $pdo->prepare('SELECT id, code FROM game_marketplace_codes WHERE server_id = ? AND is_sold = 0 ORDER BY id ASC LIMIT ?');
                $codeLookup->execute([$order['server_id'], $quantityRequired]);
                $codesToAssign = $codeLookup->fetchAll();

                if (count($codesToAssign) < $quantityRequired) {
                    $pdo->rollBack();
                    flashMessage('danger', 'Yeterli marketplace kodu bulunamadı. Lütfen stok ekleyin.');
                    redirect('marketplace-orders.php');
                }
            }

            if ($oldStatus === 'completed' && $newStatus !== 'completed') {
                $releaseStmt = $pdo->prepare('SELECT id FROM game_marketplace_codes WHERE order_id = ?');
                $releaseStmt->execute([$order['id']]);
                $codesToRelease = $releaseStmt->fetchAll(PDO::FETCH_COLUMN);
            }
        }

        $update = $pdo->prepare('UPDATE game_marketplace_orders SET status = ?, admin_id = ?, admin_note = ?, processed_at = NOW(), updated_at = NOW() WHERE id = ?');
        $update->execute([$newStatus, $_SESSION['user_id'], $adminNote, $orderId]);

        $sellCreditStates = ['completed'];
        $oldSellCredited = in_array($oldStatus, $sellCreditStates, true);
        $newSellCredited = in_array($newStatus, $sellCreditStates, true);

        if ($order['order_type'] === 'sell' && $newSellCredited && !$oldSellCredited) {
            $pdo->prepare('UPDATE users SET balance = balance + ? WHERE id = ?')->execute([$order['total_amount'], $order['user_id']]);
            $pdo->prepare('UPDATE game_marketplace_servers SET sell_stock = sell_stock + ? WHERE id = ?')->execute([$order['quantity'], $order['server_id']]);

            $userStmt = $pdo->prepare('SELECT email, username FROM users WHERE id = ?');
            $userStmt->execute([$order['user_id']]);
            if ($user = $userStmt->fetch()) {
                $subject = 'Marketplace talebiniz tamamlandı';
                $body = '<p>Merhaba <strong>' . sanitize($user['username']) . '</strong>,</p>';
                $body .= '<p>Bize Sat talebiniz tamamlandı. ' . number_format($order['quantity'], 2, ',', '.') . ' adet için toplam ' . formatPrice($order['total_amount']) . ' bakiyenize eklendi.</p>';
                sendMarketplaceEmail($user['email'], $subject, $body);
            }
        }

        if ($order['order_type'] === 'sell' && !$newSellCredited && $oldSellCredited) {
            $pdo->prepare('UPDATE users SET balance = GREATEST(balance - ?, 0) WHERE id = ?')->execute([$order['total_amount'], $order['user_id']]);
            $pdo->prepare('UPDATE game_marketplace_servers SET sell_stock = GREATEST(sell_stock - ?, 0) WHERE id = ?')->execute([$order['quantity'], $order['server_id']]);
        }

        if (!empty($codesToAssign)) {
            $assignStmt = $pdo->prepare('UPDATE game_marketplace_codes SET is_sold = 1, sold_to_user_id = ?, sold_at = NOW(), order_id = ? WHERE id = ?');
            foreach ($codesToAssign as $codeRow) {
                $assignStmt->execute([$order['user_id'], $order['id'], $codeRow['id']]);
            }
        }

        if (!empty($codesToRelease)) {
            $releaseStmt = $pdo->prepare('UPDATE game_marketplace_codes SET is_sold = 0, sold_to_user_id = NULL, sold_at = NULL, order_id = NULL WHERE id = ?');
            foreach ($codesToRelease as $codeId) {
                $releaseStmt->execute([$codeId]);
            }
        }

        if ($order['order_type'] === 'buy' && $newStatus === 'completed' && $oldStatus !== 'completed') {
            $userStmt = $pdo->prepare('SELECT email, username FROM users WHERE id = ?');
            $userStmt->execute([$order['user_id']]);
            if ($user = $userStmt->fetch()) {
                $subject = 'Marketplace siparişiniz tamamlandı';
                $body = '<p>Merhaba <strong>' . sanitize($user['username']) . '</strong>,</p>';
                $body .= '<p>Satın alım talebiniz tamamlandı. ' . number_format($order['quantity'], 2, ',', '.') . ' adet için toplam ' . formatPrice($order['total_amount']) . ' işlem gerçekleşti.</p>';
                sendMarketplaceEmail($user['email'], $subject, $body);
            }
        }

        // İptal veya reddedilme durumunda stok ve bakiye iadesi
        if ($order['order_type'] === 'buy' && in_array($newStatus, ['cancelled', 'rejected'], true) && !in_array($oldStatus, ['cancelled', 'rejected'], true)) {
            // Stok iadesi
            $pdo->prepare('UPDATE game_marketplace_servers SET sell_stock = sell_stock + ? WHERE id = ?')->execute([$order['quantity'], $order['server_id']]);
            
            // Bakiye iadesi
            $refundAmount = (float)$order['total_amount'];
            $pdo->prepare('UPDATE users SET balance = balance + ? WHERE id = ?')->execute([$refundAmount, $order['user_id']]);
            
            // İade işlemi transaction kaydı
            $refundDesc = 'Marketplace sipariş iadesi #' . $order['id'] . ' - ' . ($newStatus === 'cancelled' ? 'İptal' : 'Red');
            $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description, created_at) VALUES (?, ?, ?, ?, NOW())')
                ->execute([$order['user_id'], 'refund', $refundAmount, $refundDesc]);
        }
        
        // İptal veya reddedilmiş siparişi tekrar tamamlıyorsak, bakiye tekrar düşmeli
        if ($order['order_type'] === 'buy' && $newStatus === 'completed' && in_array($oldStatus, ['cancelled', 'rejected'], true)) {
            // Daha önce iade edilmiş bakiyeyi tekrar düş
            $chargeAmount = (float)$order['total_amount'];
            
            // Kullanıcı bakiyesi yeterli mi kontrol et
            $balanceCheck = $pdo->prepare('SELECT balance FROM users WHERE id = ?');
            $balanceCheck->execute([$order['user_id']]);
            $currentBalance = (float)($balanceCheck->fetchColumn() ?? 0);
            
            if ($currentBalance < $chargeAmount) {
                $pdo->rollBack();
                flashMessage('danger', 'Kullanıcının bakiyesi yetersiz. Gerekli: ' . formatPrice($chargeAmount) . ', Mevcut: ' . formatPrice($currentBalance));
                redirect('marketplace-orders.php');
            }
            
            // Bakiyeyi düş
            $pdo->prepare('UPDATE users SET balance = balance - ? WHERE id = ?')->execute([$chargeAmount, $order['user_id']]);
            
            // İşlem kaydı
            $chargeDesc = 'Marketplace sipariş yeniden tamamlama #' . $order['id'];
            $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description, created_at) VALUES (?, ?, ?, ?, NOW())')
                ->execute([$order['user_id'], 'purchase', -$chargeAmount, $chargeDesc]);
            
            // Stok düş
            $pdo->prepare('UPDATE game_marketplace_servers SET sell_stock = GREATEST(sell_stock - ?, 0) WHERE id = ?')->execute([$order['quantity'], $order['server_id']]);
        }

        $pdo->commit();
        flashMessage('success', 'Sipariş güncellendi.');
    } catch (Throwable $th) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        flashMessage('danger', 'Sipariş güncellenemedi: ' . $th->getMessage());
    }

    redirect('marketplace-orders.php');
}

$statusFilter = $_GET['status'] ?? '';
$typeFilter = $_GET['type'] ?? '';
$queryParts = [];
$params = [];

// Count query for pagination
$countSql = "SELECT COUNT(*) FROM game_marketplace_orders o";
if ($statusFilter !== '' || $typeFilter !== '') {
    $countQueryParts = [];
    if ($statusFilter !== '') {
        $countQueryParts[] = 'o.status = ?';
    }
    if ($typeFilter !== '') {
        $countQueryParts[] = 'o.order_type = ?';
    }
    $countSql .= ' WHERE ' . implode(' AND ', $countQueryParts);
}

$countStmt = $pdo->prepare($countSql);
$countParams = [];
if ($statusFilter !== '') {
    $countParams[] = $statusFilter;
}
if ($typeFilter !== '') {
    $countParams[] = $typeFilter;
}
$countStmt->execute($countParams);
$totalOrders = (int)$countStmt->fetchColumn();

// Pagination
$pagination = getPaginationData($totalOrders, 10);

// Main query
$sql = "SELECT o.*, u.username, m.name AS marketplace_name, c.name AS category_name, s.name AS server_name
        FROM game_marketplace_orders o
        JOIN users u ON u.id = o.user_id
        LEFT JOIN game_marketplaces m ON m.id = o.marketplace_id
        LEFT JOIN game_marketplace_categories c ON c.id = o.category_id
        LEFT JOIN game_marketplace_servers s ON s.id = o.server_id";

if ($statusFilter !== '') {
    $queryParts[] = 'o.status = ?';
    $params[] = $statusFilter;
}

if ($typeFilter !== '') {
    $queryParts[] = 'o.order_type = ?';
    $params[] = $typeFilter;
}

if (!empty($queryParts)) {
    $sql .= ' WHERE ' . implode(' AND ', $queryParts);
}

$sql .= ' ORDER BY o.created_at DESC LIMIT ' . $pagination['limit'] . ' OFFSET ' . $pagination['offset'];

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$orders = $stmt->fetchAll();

$pageTitle = 'Marketplace Siparişleri';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex flex-wrap align-items-center gap-3 mb-4">
        <h3 class="mb-0"><i class="fas fa-clipboard-list me-2"></i>Marketplace Siparişleri</h3>
        <form method="get" class="d-flex flex-wrap gap-2 align-items-center">
            <label class="text-muted" for="type">Tür:</label>
            <select name="type" id="type" class="form-select" onchange="this.form.submit()">
                <option value="">Tümü</option>
                <option value="buy" <?= $typeFilter === 'buy' ? 'selected' : '' ?>>Satın Al</option>
                <option value="sell" <?= $typeFilter === 'sell' ? 'selected' : '' ?>>Bize Sat</option>
            </select>
            <label class="text-muted" for="status">Durum:</label>
            <select name="status" id="status" class="form-select" onchange="this.form.submit()">
                <option value="">Tümü</option>
                <?php foreach ($statusesMap as $key => $label): ?>
                    <option value="<?= $key ?>" <?= $statusFilter === $key ? 'selected' : '' ?>><?= $label ?></option>
                <?php endforeach; ?>
            </select>
        </form>
    </div>

    <div class="card">
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0 align-middle">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Kullanıcı</th>
                            <th>Oyun / Kategori / Server</th>
                            <th>Karakter</th>
                            <th>Tür</th>
                            <th>Miktar</th>
                            <th>Birim Fiyat</th>
                            <th>Toplam</th>
                            <th>Durum</th>
                            <th>Oluşturma</th>
                            <th class="text-end">İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($orders)): ?>
                        <tr>
                            <td colspan="11" class="text-center text-muted py-4">Filtrelerinize uygun sipariş bulunamadı.</td>
                        </tr>
                        <?php else: ?>
                            <?php foreach ($orders as $order): ?>
                            <tr>
                                <td><strong>#<?= $order['id'] ?></strong></td>
                                <td>
                                    <strong><?= sanitize($order['username']) ?></strong>
                                    <br><small class="text-muted">ID: <?= $order['user_id'] ?></small>
                                </td>
                                <td>
                                    <div><strong><?= sanitize($order['marketplace_name'] ?? 'Silinmiş Oyun') ?></strong></div>
                                    <small class="text-muted"><?= sanitize($order['category_name'] ?? '-') ?> / <?= sanitize($order['server_name'] ?? '-') ?></small>
                                </td>
                                <td><?= sanitize($order['character_name'] ?? '-') ?></td>
                                <td>
                                    <?php if ($order['order_type'] === 'buy'): ?>
                                        <span class="badge bg-primary">Satın Al</span>
                                    <?php else: ?>
                                        <span class="badge bg-warning text-dark">Bize Sat</span>
                                    <?php endif; ?>
                                </td>
                                <td><?= number_format((float)$order['quantity'], 2, ',', '.') ?></td>
                                <td><?= formatPrice($order['price_per_unit']) ?></td>
                                <td><?= formatPrice($order['total_amount']) ?></td>
                                <td>
                                    <?php
                                    $statusLabel = $statusesMap[$order['status']] ?? $order['status'];
                                    ?>
                                    <span class="badge bg-secondary text-uppercase"><?= sanitize($statusLabel) ?></span>
                                </td>
                                <td><?= date('d.m.Y H:i', strtotime($order['created_at'])) ?></td>
                                <td class="text-end">
                                    <form method="post" class="d-inline-flex align-items-center gap-2">
                                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                        <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                                        <select name="status" class="form-select form-select-sm">
                                            <?php foreach ($statusesMap as $key => $label): ?>
                                                <option value="<?= $key ?>" <?= $order['status'] === $key ? 'selected' : '' ?>><?= $label ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                        <input type="text" name="admin_note" class="form-control form-control-sm" placeholder="Not" value="<?= sanitize($order['admin_note'] ?? '') ?>">
                                        <button type="submit" name="update_status" class="btn btn-sm btn-primary">Güncelle</button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <?php if ($totalOrders > 10): ?>
            <div class="card-body border-top">
                <?= renderPagination($pagination) ?>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

