<?php
require_once '../config.php';
requireAdmin();

if (!function_exists('shortenText')) {
    function shortenText(?string $text, int $limit = 150): string
    {
        $text = trim((string)$text);
        if ($text === '') {
            return '';
        }
        if (mb_strlen($text) <= $limit) {
            return $text;
        }
        return rtrim(mb_substr($text, 0, $limit - 3)) . '...';
    }
}

$games = getMarketplaceGames($pdo, false);
if (empty($games)) {
    flashMessage('warning', 'Önce marketplace oyunları oluşturmalısınız.');
    redirect('marketplace-games.php');
}

$gameMap = [];
foreach ($games as $game) {
    $gameMap[$game['id']] = $game;
}

$selectedGameId = isset($_GET['game']) ? (int)$_GET['game'] : (int)$games[0]['id'];
if (!isset($gameMap[$selectedGameId])) {
    $selectedGameId = (int)$games[0]['id'];
}

$stmt = $pdo->prepare('SELECT * FROM game_marketplace_categories WHERE marketplace_id = ? ORDER BY display_order ASC, name ASC');
$stmt->execute([$selectedGameId]);
$categories = $stmt->fetchAll();

if (empty($categories)) {
    flashMessage('warning', 'Bu oyun için kategoriler oluşturulmamış. Önce kategori ekleyin.');
    redirect('marketplace-categories.php?game=' . $selectedGameId);
}

$categoryMap = [];
foreach ($categories as $category) {
    $categoryMap[$category['id']] = $category;
}

$selectedCategoryId = isset($_GET['category']) ? (int)$_GET['category'] : (int)$categories[0]['id'];
if (!isset($categoryMap[$selectedCategoryId])) {
    $selectedCategoryId = (int)$categories[0]['id'];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_server'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $selectedCategoryId);
    }

    $id = isset($_POST['id']) && $_POST['id'] !== '' ? (int)$_POST['id'] : null;
    $categoryId = isset($_POST['category_id']) ? (int)$_POST['category_id'] : $selectedCategoryId;

    if (!isset($categoryMap[$categoryId])) {
        $stmt = $pdo->prepare('SELECT c.*, m.name AS marketplace_name FROM game_marketplace_categories c JOIN game_marketplaces m ON m.id = c.marketplace_id WHERE c.id = ?');
        $stmt->execute([$categoryId]);
        $categoryRow = $stmt->fetch();
        if ($categoryRow) {
            $selectedGameId = (int)$categoryRow['marketplace_id'];
        }
        flashMessage('danger', 'Kategori bulunamadı.');
        redirect('marketplace-servers.php?game=' . $selectedGameId);
    }

    $name = trim($_POST['name'] ?? '');
    $providedSlug = trim($_POST['slug'] ?? '');
    $slug = slugify($providedSlug !== '' ? $providedSlug : $name);
    $description = trim($_POST['description'] ?? '');
    $displayOrder = isset($_POST['display_order']) ? (int)$_POST['display_order'] : 0;
    $isActive = isset($_POST['is_active']) ? 1 : 0;

    $buyPrice = (float)($_POST['buy_price'] ?? 0);
    $sellPrice = (float)($_POST['sell_price'] ?? 0);
    $minSellQuantity = (float)($_POST['min_sell_quantity'] ?? 1);
    $maxSellQuantity = $_POST['max_sell_quantity'] !== '' ? (float)$_POST['max_sell_quantity'] : null;
    $minBuyQuantity = (float)($_POST['min_buy_quantity'] ?? 1);
    $maxBuyQuantity = $_POST['max_buy_quantity'] !== '' ? (float)$_POST['max_buy_quantity'] : null;
    $sellStock = (float)($_POST['sell_stock'] ?? 0);

    if ($name === '') {
        flashMessage('danger', 'Ürün adı zorunludur.');
        redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $categoryId . ($id ? '&edit=' . $id : ''));
    }

    if ($minSellQuantity <= 0 || $minBuyQuantity <= 0) {
        flashMessage('danger', 'Minimum miktarlar 0’dan büyük olmalıdır.');
        redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $categoryId . ($id ? '&edit=' . $id : ''));
    }

    if ($maxSellQuantity !== null && $maxSellQuantity < $minSellQuantity) {
        flashMessage('danger', 'Maksimum satış miktarı minimum satış miktarından küçük olamaz.');
        redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $categoryId . ($id ? '&edit=' . $id : ''));
    }

    if ($maxBuyQuantity !== null && $maxBuyQuantity < $minBuyQuantity) {
        flashMessage('danger', 'Maksimum satın alma miktarı minimum satın alma miktarından küçük olamaz.');
        redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $categoryId . ($id ? '&edit=' . $id : ''));
    }

    $currentImage = null;
    if ($id) {
        $stmt = $pdo->prepare('SELECT category_id, image FROM game_marketplace_servers WHERE id = ?');
        $stmt->execute([$id]);
        $existingServer = $stmt->fetch();
        if ($existingServer) {
            $currentImage = $existingServer['image'] ?? null;
        }
    }

    $removeImage = isset($_POST['remove_image']) && $_POST['remove_image'] === '1';
    if ($removeImage && $currentImage) {
        deleteUploadedFile($currentImage);
        $currentImage = null;
    }

    try {
        try {
            $imagePath = uploadImageFile($_FILES['image'] ?? [], 'marketplace/servers', $currentImage);
        } catch (Throwable $imageError) {
            flashMessage('danger', 'Ürün görseli yüklenemedi: ' . $imageError->getMessage());
            redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $categoryId . ($id ? '&edit=' . $id : ''));
        }

        if ($id) {
            $stmt = $pdo->prepare('UPDATE game_marketplace_servers SET name = ?, slug = ?, description = ?, image = ?, buy_price = ?, sell_price = ?, min_sell_quantity = ?, max_sell_quantity = ?, min_buy_quantity = ?, max_buy_quantity = ?, sell_stock = ?, display_order = ?, is_active = ?, updated_at = NOW() WHERE id = ?');
            $stmt->execute([$name, $slug, $description, $imagePath, $buyPrice, $sellPrice, $minSellQuantity, $maxSellQuantity, $minBuyQuantity, $maxBuyQuantity, $sellStock, $displayOrder, $isActive, $id]);
            flashMessage('success', 'Ürün kaydı güncellendi.');
        } else {
            $stmt = $pdo->prepare('INSERT INTO game_marketplace_servers (category_id, name, slug, description, image, buy_price, sell_price, min_sell_quantity, max_sell_quantity, min_buy_quantity, max_buy_quantity, sell_stock, display_order, is_active) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
            $stmt->execute([$categoryId, $name, $slug, $description, $imagePath, $buyPrice, $sellPrice, $minSellQuantity, $maxSellQuantity, $minBuyQuantity, $maxBuyQuantity, $sellStock, $displayOrder, $isActive]);
            flashMessage('success', 'Ürün kaydı eklendi.');
        }
        redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $categoryId);
    } catch (PDOException $e) {
        flashMessage('danger', 'İşlem başarısız: ' . $e->getMessage());
        redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $categoryId . ($id ? '&edit=' . $id : ''));
    }
}

if (isset($_GET['delete'])) {
    $deleteId = (int)$_GET['delete'];
    try {
        $stmt = $pdo->prepare('SELECT category_id, image FROM game_marketplace_servers WHERE id = ?');
        $stmt->execute([$deleteId]);
        $server = $stmt->fetch();
        if ($server) {
            $pdo->prepare('DELETE FROM game_marketplace_servers WHERE id = ?')->execute([$deleteId]);
            if (!empty($server['image'])) {
                deleteUploadedFile($server['image']);
            }
            flashMessage('success', 'Ürün kaydı silindi.');
            $selectedCategoryId = (int)$server['category_id'];
        }
    } catch (PDOException $e) {
        flashMessage('danger', 'Ürün silinirken hata oluştu: ' . $e->getMessage());
    }
    redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $selectedCategoryId);
}

// Pagination
$countStmt = $pdo->prepare('SELECT COUNT(*) FROM game_marketplace_servers WHERE category_id = ?');
$countStmt->execute([$selectedCategoryId]);
$totalServers = (int)$countStmt->fetchColumn();
$pagination = getPaginationData($totalServers, 10);

$stmt = $pdo->prepare('SELECT s.*, c.name AS category_name, m.name AS marketplace_name FROM game_marketplace_servers s JOIN game_marketplace_categories c ON c.id = s.category_id JOIN game_marketplaces m ON m.id = c.marketplace_id WHERE s.category_id = ? ORDER BY s.display_order ASC, s.name ASC LIMIT ' . $pagination['limit'] . ' OFFSET ' . $pagination['offset']);
$stmt->execute([$selectedCategoryId]);
$servers = $stmt->fetchAll();

$editServer = null;
if (isset($_GET['edit'])) {
    $stmt = $pdo->prepare('SELECT s.*, c.marketplace_id FROM game_marketplace_servers s JOIN game_marketplace_categories c ON c.id = s.category_id WHERE s.id = ?');
    $stmt->execute([(int)$_GET['edit']]);
    $editServer = $stmt->fetch();
    if (!$editServer) {
        flashMessage('danger', 'Ürün kaydı bulunamadı.');
        redirect('marketplace-servers.php?game=' . $selectedGameId . '&category=' . $selectedCategoryId);
    }
    $selectedCategoryId = (int)$editServer['category_id'];
    $selectedGameId = (int)$editServer['marketplace_id'];

    $stmt = $pdo->prepare('SELECT * FROM game_marketplace_categories WHERE marketplace_id = ? ORDER BY display_order ASC, name ASC');
    $stmt->execute([$selectedGameId]);
    $categories = $stmt->fetchAll();
    $categoryMap = [];
    foreach ($categories as $category) {
        $categoryMap[$category['id']] = $category;
    }
}

$pageTitle = 'Marketplace Ürünleri';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex flex-wrap align-items-center gap-3 mb-3">
        <h3 class="mb-0"><i class="fas fa-box me-2"></i>Marketplace Ürünleri</h3>
        <form method="get" class="d-flex align-items-center gap-2">
            <label for="game" class="text-muted">Oyun:</label>
            <select name="game" id="game" class="form-select" onchange="this.form.submit()">
                <?php foreach ($games as $game): ?>
                    <option value="<?= $game['id'] ?>" <?= $selectedGameId === (int)$game['id'] ? 'selected' : '' ?>><?= sanitize($game['name']) ?></option>
                <?php endforeach; ?>
            </select>
            <label for="category" class="text-muted">Kategori:</label>
            <select name="category" id="category" class="form-select" onchange="this.form.submit()">
                <?php foreach ($categories as $category): ?>
                    <option value="<?= $category['id'] ?>" <?= $selectedCategoryId === (int)$category['id'] ? 'selected' : '' ?>><?= sanitize($category['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </form>
        <a href="marketplace-categories.php?game=<?= $selectedGameId ?>" class="btn btn-sm btn-outline-secondary"><i class="fas fa-layer-group me-1"></i>Kategorilere Dön</a>
        <a href="marketplace-orders.php" class="btn btn-sm btn-outline-primary"><i class="fas fa-clipboard-list me-1"></i>Siparişler</a>
    </div>

    <div class="alert alert-danger d-flex align-items-center gap-2 mb-4" role="alert" style="border-left: 4px solid #dc3545;">
        <i class="fas fa-info-circle"></i>
        <strong>Verileri görmek için yukarıdan oyun seçimi ve kategori seçimi yapınız.</strong>
    </div>

    <div class="row g-4">
        <div class="col-lg-5">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0">
                        <i class="fas fa-<?= $editServer ? 'edit' : 'plus' ?> me-2"></i>
                        <?= $editServer ? 'Ürün Kaydı Düzenle' : 'Yeni Ürün Kaydı' ?>
                    </h5>
                </div>
                <div class="card-body">
                    <form method="post" enctype="multipart/form-data">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="save_server" value="1">
                        <?php if ($editServer): ?>
                            <input type="hidden" name="id" value="<?= $editServer['id'] ?>">
                        <?php endif; ?>

                        <div class="mb-3">
                            <label class="form-label">Kategori</label>
                            <select name="category_id" class="form-select" required>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?= $category['id'] ?>" <?= ($editServer ? (int)$editServer['category_id'] === (int)$category['id'] : $selectedCategoryId === (int)$category['id']) ? 'selected' : '' ?>><?= sanitize($category['name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Ürün Adı</label>
                            <input type="text" name="name" class="form-control" value="<?= $editServer ? sanitize($editServer['name']) : '' ?>" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Slug</label>
                            <input type="text" name="slug" class="form-control" value="<?= $editServer ? sanitize($editServer['slug']) : '' ?>" placeholder="örn. alastor-server" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Açıklama</label>
                            <textarea name="description" class="form-control" rows="3" placeholder="İsteğe bağlı detaylı açıklama."><?= $editServer ? sanitize($editServer['description']) : '' ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Kart Görseli</label>
                            <?php if ($editServer && !empty($editServer['image'])): ?>
                                <div class="mb-2">
                                    <img src="<?= htmlspecialchars(assetUrl($editServer['image']), ENT_QUOTES, 'UTF-8') ?>" alt="Ürün görseli" class="img-fluid rounded" style="max-height: 160px;">
                                </div>
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" value="1" id="remove_image" name="remove_image">
                                    <label class="form-check-label" for="remove_image">Görseli kaldır</label>
                                </div>
                            <?php endif; ?>
                            <input type="file" name="image" class="form-control" accept="image/*">
                            <small class="text-muted">PNG, JPG, WEBP veya GIF (maks. 4MB)</small>
                        </div>

                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Bizden Al (Satış) Fiyatı</label>
                                <input type="number" step="0.01" min="0" name="sell_price" class="form-control" value="<?= $editServer ? (float)$editServer['sell_price'] : 0 ?>">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Bize Sat Fiyatı</label>
                                <input type="number" step="0.01" min="0" name="buy_price" class="form-control" value="<?= $editServer ? (float)$editServer['buy_price'] : 0 ?>">
                            </div>
                        </div>

                        <div class="row g-3 mt-1">
                            <div class="col-md-6">
                                <label class="form-label">Min Satın Alma Miktarı</label>
                                <input type="number" step="0.01" min="0.01" name="min_buy_quantity" class="form-control" value="<?= $editServer ? (float)$editServer['min_buy_quantity'] : 1 ?>" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Max Satın Alma Miktarı</label>
                                <input type="number" step="0.01" min="0" name="max_buy_quantity" class="form-control" value="<?= $editServer && $editServer['max_buy_quantity'] !== null ? (float)$editServer['max_buy_quantity'] : '' ?>" placeholder="Sınırsız için boş bırakın">
                            </div>
                        </div>

                        <div class="row g-3 mt-1">
                            <div class="col-md-6">
                                <label class="form-label">Min Satış (Bize Sat) Miktarı</label>
                                <input type="number" step="0.01" min="0.01" name="min_sell_quantity" class="form-control" value="<?= $editServer ? (float)$editServer['min_sell_quantity'] : 1 ?>" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Max Satış Miktarı</label>
                                <input type="number" step="0.01" min="0" name="max_sell_quantity" class="form-control" value="<?= $editServer && $editServer['max_sell_quantity'] !== null ? (float)$editServer['max_sell_quantity'] : '' ?>" placeholder="Sınırsız için boş bırakın">
                            </div>
                        </div>

                        <div class="row g-3 mt-1">
                            <div class="col-md-6">
                                <label class="form-label">Satış Stoku (Bizden Al)</label>
                                <input type="number" step="0.01" min="0" name="sell_stock" class="form-control" value="<?= $editServer ? (float)$editServer['sell_stock'] : 0 ?>">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Sıra</label>
                                <input type="number" name="display_order" class="form-control" value="<?= $editServer ? (int)$editServer['display_order'] : 0 ?>">
                            </div>
                        </div>

                        <div class="form-check form-switch mb-3 mt-3 ps-0">
                            <input class="form-check-input ms-0" type="checkbox" name="is_active" id="is_active" <?= $editServer ? ($editServer['is_active'] ? 'checked' : '') : 'checked' ?>>
                            <label class="form-check-label" for="is_active">Aktif</label>
                        </div>

                        <div class="d-flex gap-2">
                            <button type="submit" class="btn btn-primary flex-fill">
                                <i class="fas fa-save me-2"></i>Kaydet
                            </button>
                            <?php if ($editServer): ?>
                                <a href="marketplace-servers.php?game=<?= $selectedGameId ?>&category=<?= $selectedCategoryId ?>" class="btn btn-secondary">
                                    <i class="fas fa-times"></i>
                                </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-7">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-list me-2"></i><?= sanitize($categoryMap[$selectedCategoryId]['name']) ?> Ürünleri</h5>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0 align-middle">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Ürün</th>
                                    <th>Görsel</th>
                                    <th>Satış / Alış</th>
                                    <th>Stok</th>
                                    <th>Durum</th>
                                    <th class="text-end">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($servers)): ?>
                                <tr>
                                    <td colspan="7" class="text-center text-muted py-4">Bu kategoriye ait ürün kaydı yok.</td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($servers as $server): ?>
                                    <tr>
                                        <td><strong>#<?= $server['id'] ?></strong></td>
                                        <td>
                                            <strong><?= sanitize($server['name']) ?></strong>
                                            <?php if (!empty($server['description'])): ?>
                                                <br><small class="text-muted"><?= sanitize(shortenText($server['description'], 120)) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (!empty($server['image'])): ?>
                                                <img src="<?= htmlspecialchars(assetUrl($server['image']), ENT_QUOTES, 'UTF-8') ?>" alt="<?= sanitize($server['name']) ?>" class="img-thumbnail" style="max-height: 60px;">
                                            <?php else: ?>
                                                <span class="text-muted">—</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-primary">Satış: <?= formatPrice($server['sell_price']) ?></span><br>
                                            <span class="badge bg-info text-dark mt-1">Bize Sat: <?= formatPrice($server['buy_price']) ?></span>
                                        </td>
                                        <td><?= number_format((float)$server['sell_stock'], 2, ',', '.') ?></td>
                                        <td>
                                            <?php if ($server['is_active']): ?>
                                                <span class="badge bg-success">Aktif</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Pasif</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="text-end">
                                            <div class="d-inline-flex gap-2">
                                                <a href="marketplace-servers.php?game=<?= $selectedGameId ?>&category=<?= $selectedCategoryId ?>&edit=<?= $server['id'] ?>" class="btn btn-sm btn-outline-primary"><i class="fas fa-edit"></i></a>
                                                <a href="marketplace-servers.php?game=<?= $selectedGameId ?>&category=<?= $selectedCategoryId ?>&delete=<?= $server['id'] ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Bu ürün kaydını silmek istediğinize emin misiniz?');"><i class="fas fa-trash"></i></a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php if ($totalServers > 10): ?>
                    <div class="card-body border-top">
                        <?= renderPagination($pagination) ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

