<?php
require_once '../config.php';
requireAdmin();

$orderId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($orderId <= 0) {
    flashMessage('danger', 'Geçersiz sipariş numarası.');
    redirect('orders.php');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_status'])) {
        $newStatus = $_POST['status'] ?? 'pending';
        if (!in_array($newStatus, ['pending', 'completed', 'cancelled'], true)) {
            $newStatus = 'pending';
        }
        $stmt = $pdo->prepare('UPDATE orders SET status = ?, updated_at = NOW() WHERE id = ?');
        $stmt->execute([$newStatus, $orderId]);
        flashMessage('success', 'Sipariş durumu güncellendi.');
        redirect('order-detail.php?id=' . $orderId);
    }

    if (isset($_POST['add_product_code'])) {
        $orderItemId = (int)($_POST['order_item_id'] ?? 0);
        $productId = (int)($_POST['product_id'] ?? 0);
        $code = safe_trim($_POST['code'] ?? '');
        
        if ($orderItemId > 0 && $productId > 0 && $code !== '') {
            try {
                $pdo->beginTransaction();
                
                // Ürüne bağlı yeni kod ekle
                $insertCodeStmt = $pdo->prepare('INSERT INTO product_codes (product_id, code, is_sold, sold_to_user_id, sold_at) VALUES (?, ?, 1, (SELECT user_id FROM orders WHERE id = ?), NOW())');
                $insertCodeStmt->execute([$productId, $code, $orderId]);
                $codeId = $pdo->lastInsertId();
                
                // Order item'e kod ID'sini bağla
                $updateItemStmt = $pdo->prepare('UPDATE order_items SET product_code_id = ? WHERE id = ?');
                $updateItemStmt->execute([$codeId, $orderItemId]);
                
                $pdo->commit();
                flashMessage('success', 'Kod başarıyla eklendi.');
            } catch (Throwable $th) {
                $pdo->rollBack();
                flashMessage('danger', 'Kod eklenirken hata oluştu: ' . $th->getMessage());
            }
            redirect('order-detail.php?id=' . $orderId);
        } else {
            flashMessage('warning', 'Lütfen tüm alanları doldurun.');
            redirect('order-detail.php?id=' . $orderId);
        }
    }

    if (isset($_POST['update_product_code'])) {
        $codeId = (int)($_POST['code_id'] ?? 0);
        $code = safe_trim($_POST['code'] ?? '');
        
        if ($codeId > 0 && $code !== '') {
            try {
                $updateStmt = $pdo->prepare('UPDATE product_codes SET code = ? WHERE id = ?');
                $updateStmt->execute([$code, $codeId]);
                flashMessage('success', 'Kod başarıyla güncellendi.');
            } catch (Throwable $th) {
                flashMessage('danger', 'Kod güncellenirken hata oluştu: ' . $th->getMessage());
            }
            redirect('order-detail.php?id=' . $orderId);
        } else {
            flashMessage('warning', 'Geçersiz kod bilgisi.');
            redirect('order-detail.php?id=' . $orderId);
        }
    }

    if (isset($_POST['delete_order'])) {
        try {
            $pdo->beginTransaction();
            $pdo->prepare('DELETE FROM transactions WHERE order_id = ?')->execute([$orderId]);
            $pdo->prepare('DELETE FROM order_items WHERE order_id = ?')->execute([$orderId]);
            $pdo->prepare('DELETE FROM orders WHERE id = ?')->execute([$orderId]);
            $pdo->commit();
            flashMessage('success', 'Sipariş silindi.');
            redirect('orders.php');
        } catch (Throwable $th) {
            $pdo->rollBack();
            flashMessage('danger', 'Sipariş silinirken hata oluştu: ' . $th->getMessage());
            redirect('order-detail.php?id=' . $orderId);
        }
    }
}

$orderStmt = $pdo->prepare('
    SELECT o.*, u.username, u.email, u.full_name, u.phone
    FROM orders o
    JOIN users u ON o.user_id = u.id
    WHERE o.id = ?
');
$orderStmt->execute([$orderId]);
$order = $orderStmt->fetch();

if (!$order) {
    flashMessage('danger', 'Sipariş bulunamadı.');
    redirect('orders.php');
}

$itemStmt = $pdo->prepare('
    SELECT oi.*, pc.id AS code_id, pc.code
    FROM order_items oi
    LEFT JOIN product_codes pc ON oi.product_code_id = pc.id
    WHERE oi.order_id = ?
');
$itemStmt->execute([$orderId]);
$orderItems = $itemStmt->fetchAll();

$transactionStmt = $pdo->prepare('SELECT * FROM transactions WHERE order_id = ?');
$transactionStmt->execute([$orderId]);
$transactions = $transactionStmt->fetchAll();

$pageTitle = 'Sipariş Detayı';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h3 class="mb-1">Sipariş #<?= sanitize($order['order_number']) ?></h3>
            <p class="text-muted mb-0"><?= date('d.m.Y H:i', strtotime($order['created_at'])) ?> tarihinde oluşturuldu.</p>
        </div>
        <div class="d-flex gap-2">
            <a href="orders.php" class="btn btn-outline-primary"><i class="fas fa-arrow-left me-2"></i>Listeye Dön</a>
            <form method="post" onsubmit="return confirm('Bu siparişi silmek istediğinizden emin misiniz?');">
                <button type="submit" name="delete_order" class="btn btn-danger"><i class="fas fa-trash me-2"></i>Siparişi Sil</button>
            </form>
        </div>
    </div>

    <div class="row g-4 mb-4">
        <div class="col-lg-4">
            <div class="metric-card h-100">
                <span class="metric-label">Müşteri</span>
                <div class="metric-value" style="font-size:1.4rem;"><?= sanitize($order['username']) ?></div>
                <p class="metric-subtext">E-posta: <?= sanitize($order['email']) ?><br>Ad Soyad: <?= sanitize($order['full_name'] ?? '-') ?><br>Telefon: <?= sanitize($order['phone'] ?? '-') ?></p>
            </div>
        </div>
        <div class="col-lg-4">
            <div class="metric-card h-100">
                <span class="metric-label">Sipariş Özeti</span>
                <div class="metric-value" style="font-size:1.4rem;"><?= formatPrice($order['total_amount']) ?></div>
                <p class="metric-subtext">Ödeme yöntemi: <strong><?= strtoupper(sanitize($order['payment_method'] ?? '-')) ?></strong><br>Tip: <?= $order['order_type'] === 'deposit' ? 'Bakiye Yükleme' : 'Ürün Satışı' ?></p>
            </div>
        </div>
        <?php if ($order['order_type'] === 'deposit'): ?>
        <div class="col-lg-4">
            <div class="metric-card h-100">
                <span class="metric-label">Komisyon</span>
                <div class="metric-value text-success" style="font-size:1.4rem;">+<?= formatPrice($order['commission_amount']) ?></div>
                <p class="metric-subtext">Brüt Tahsilat: <?= formatPrice($order['payable_amount']) ?><br>Komisyon Oranı: %<?= number_format($order['commission_rate'], 2, ',', '.') ?></p>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Sipariş Durumu</h5>
        </div>
        <div class="card-body">
            <form method="post" class="row g-3 align-items-center">
                <div class="col-md-4">
                    <label class="form-label">Durum Güncelle</label>
                    <select name="status" class="form-select">
                        <option value="pending" <?= $order['status'] === 'pending' ? 'selected' : '' ?>>Beklemede</option>
                        <option value="completed" <?= $order['status'] === 'completed' ? 'selected' : '' ?>>Tamamlandı</option>
                        <option value="cancelled" <?= $order['status'] === 'cancelled' ? 'selected' : '' ?>>İptal</option>
                    </select>
                </div>
                <div class="col-md-3 pt-3">
                    <button type="submit" name="update_status" class="btn btn-primary mt-3"><i class="fas fa-save me-2"></i>Kaydet</button>
                </div>
                <div class="col-md-5">
                    <p class="text-muted small mb-0">Oluşturulma: <?= date('d.m.Y H:i', strtotime($order['created_at'])) ?><br>Son Güncelleme: <?= date('d.m.Y H:i', strtotime($order['updated_at'])) ?></p>
                </div>
            </form>
        </div>
    </div>

    <?php if ($order['order_type'] === 'purchase'): ?>
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0"><i class="fas fa-box me-2"></i>Sipariş Ürünleri ve Kodları</h5>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0 align-middle">
                    <thead>
                        <tr>
                            <th>Ürün</th>
                            <th>Fiyat</th>
                            <th>Adet</th>
                            <th style="width: 40%;">Ürün Kodu</th>
                            <th class="text-end">İşlem</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($orderItems as $item): ?>
                        <tr>
                            <td><strong><?= sanitize($item['product_name']) ?></strong></td>
                            <td><?= formatPrice($item['price']) ?></td>
                            <td><span class="badge bg-primary"><?= (int)$item['quantity'] ?></span></td>
                            <td>
                                <?php if (!empty($item['code'])): ?>
                                    <div class="d-flex align-items-center gap-2">
                                        <code class="px-3 py-2 bg-dark text-white rounded flex-fill" style="font-size: 0.9rem;"><?= sanitize($item['code']) ?></code>
                                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="copyToClipboard('<?= sanitize($item['code']) ?>')">
                                            <i class="fas fa-copy"></i>
                                        </button>
                                    </div>
                                <?php else: ?>
                                    <span class="text-muted"><i class="fas fa-info-circle me-1"></i>Kod henüz eklenmedi</span>
                                <?php endif; ?>
                            </td>
                            <td class="text-end">
                                <?php if (!empty($item['code'])): ?>
                                    <button type="button" class="btn btn-sm btn-warning" data-bs-toggle="modal" data-bs-target="#editCodeModal<?= $item['id'] ?>">
                                        <i class="fas fa-edit me-1"></i>Düzenle
                                    </button>
                                <?php else: ?>
                                    <button type="button" class="btn btn-sm btn-success" data-bs-toggle="modal" data-bs-target="#addCodeModal<?= $item['id'] ?>">
                                        <i class="fas fa-plus me-1"></i>Kod Ekle
                                    </button>
                                <?php endif; ?>
                            </td>
                        </tr>

                        <!-- Kod Ekleme Modal -->
                        <div class="modal fade" id="addCodeModal<?= $item['id'] ?>" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <form method="post">
                                        <div class="modal-header">
                                            <h5 class="modal-title"><i class="fas fa-key me-2"></i>Ürün Kodu Ekle</h5>
                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                        </div>
                                        <div class="modal-body">
                                            <input type="hidden" name="order_item_id" value="<?= $item['id'] ?>">
                                            <input type="hidden" name="product_id" value="<?= $item['product_id'] ?>">
                                            <div class="mb-3">
                                                <label class="form-label"><strong>Ürün:</strong> <?= sanitize($item['product_name']) ?></label>
                                                <textarea class="form-control" name="code" rows="4" placeholder="Ürün kodunu buraya girin..." required></textarea>
                                                <small class="text-muted">Müşteri bu kodu profil panelinde görebilecek.</small>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                                            <button type="submit" name="add_product_code" class="btn btn-success">
                                                <i class="fas fa-save me-1"></i>Kodu Kaydet
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>

                        <!-- Kod Düzenleme Modal -->
                        <?php if (!empty($item['code'])): ?>
                        <div class="modal fade" id="editCodeModal<?= $item['id'] ?>" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <form method="post">
                                        <div class="modal-header">
                                            <h5 class="modal-title"><i class="fas fa-edit me-2"></i>Ürün Kodunu Düzenle</h5>
                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                        </div>
                                        <div class="modal-body">
                                            <input type="hidden" name="code_id" value="<?= $item['code_id'] ?>">
                                            <div class="mb-3">
                                                <label class="form-label"><strong>Ürün:</strong> <?= sanitize($item['product_name']) ?></label>
                                                <textarea class="form-control" name="code" rows="4" required><?= sanitize($item['code']) ?></textarea>
                                                <small class="text-muted">Müşteri bu kodu profil panelinde görebilecek.</small>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                                            <button type="submit" name="update_product_code" class="btn btn-warning">
                                                <i class="fas fa-save me-1"></i>Güncelle
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h5 class="mb-0">İşlem Kayıtları</h5>
        </div>
        <div class="card-body p-0">
            <?php if (empty($transactions)): ?>
                <div class="p-4 text-muted">Bu siparişe bağlı işlem kaydı bulunamadı.</div>
            <?php else: ?>
            <div class="table-responsive">
                <table class="table table-borderless mb-0">
                    <thead>
                        <tr>
                            <th>İşlem Tipi</th>
                            <th>Tutar</th>
                            <th>Açıklama</th>
                            <th>Tarih</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($transactions as $txn): ?>
                        <tr>
                            <td><?= ucfirst($txn['type']) ?></td>
                            <td><?= formatPrice($txn['amount']) ?></td>
                            <td><?= sanitize($txn['description'] ?? '-') ?></td>
                            <td><?= date('d.m.Y H:i', strtotime($txn['created_at'])) ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
function copyToClipboard(text) {
    if (navigator.clipboard && window.isSecureContext) {
        navigator.clipboard.writeText(text).then(() => {
            alert('Kod kopyalandı: ' + text);
        }).catch((err) => {
            console.error('Kopyalama hatası:', err);
            fallbackCopyToClipboard(text);
        });
    } else {
        fallbackCopyToClipboard(text);
    }
}

function fallbackCopyToClipboard(text) {
    const textArea = document.createElement("textarea");
    textArea.value = text;
    textArea.style.position = "fixed";
    textArea.style.left = "-999999px";
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();
    try {
        document.execCommand('copy');
        alert('Kod kopyalandı: ' + text);
    } catch (err) {
        console.error('Fallback kopyalama hatası:', err);
        alert('Kopyalama başarısız oldu.');
    }
    document.body.removeChild(textArea);
}
</script>

<?php include 'includes/admin-footer.php'; ?>


