<?php
require_once '../config.php';
requireAdmin();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['bulk_action'])) {
        $selectedIds = array_map('intval', $_POST['selected_ids'] ?? []);
        $selectedIds = array_values(array_filter($selectedIds));

        if (empty($selectedIds)) {
            flashMessage('warning', 'Toplu işlem için sipariş seçmediniz.');
            redirect('orders.php');
        }

        $placeholders = implode(',', array_fill(0, count($selectedIds), '?'));

        try {
            if ($_POST['bulk_action'] === 'complete') {
                $stmt = $pdo->prepare("UPDATE orders SET status = 'completed', updated_at = NOW() WHERE id IN ($placeholders) AND order_type = 'purchase'");
                $stmt->execute($selectedIds);
                flashMessage('success', 'Seçili siparişler tamamlandı olarak işaretlendi.');
            } elseif ($_POST['bulk_action'] === 'delete') {
                $pdo->beginTransaction();
                // Sadece purchase siparişleri silinsin
                $pdo->prepare("DELETE FROM transactions WHERE order_id IN ($placeholders) AND order_id IN (SELECT id FROM orders WHERE order_type = 'purchase')")->execute($selectedIds);
                $pdo->prepare("DELETE FROM order_items WHERE order_id IN ($placeholders)")->execute($selectedIds);
                $pdo->prepare("DELETE FROM orders WHERE id IN ($placeholders) AND order_type = 'purchase'")->execute($selectedIds);
                $pdo->commit();
                flashMessage('success', 'Seçili siparişler silindi.');
            }
        } catch (Throwable $th) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            flashMessage('danger', 'Toplu sipariş işlemi sırasında hata oluştu: ' . $th->getMessage());
        }

        redirect('orders.php');
    }

    if (isset($_POST['update_status'])) {
        $orderId = (int)$_POST['order_id'];
        $newStatus = $_POST['status'] ?? 'pending';
        if (!in_array($newStatus, ['pending', 'completed', 'cancelled'], true)) {
            $newStatus = 'pending';
        }

        $stmt = $pdo->prepare('SELECT order_type, status FROM orders WHERE id = ? AND order_type = "purchase"');
        $stmt->execute([$orderId]);
        $existing = $stmt->fetch();

        if ($existing) {
            $update = $pdo->prepare('UPDATE orders SET status = ?, updated_at = NOW() WHERE id = ? AND order_type = "purchase"');
            $update->execute([$newStatus, $orderId]);
            flashMessage('success', 'Sipariş durumu güncellendi.');
        } else {
            flashMessage('danger', 'Sipariş bulunamadı.');
        }

        redirect('orders.php');
    }

    if (isset($_POST['delete_order'])) {
        $orderId = (int)$_POST['order_id'];
        try {
            // Sadece purchase siparişleri silinebilir
            $checkStmt = $pdo->prepare('SELECT order_type FROM orders WHERE id = ?');
            $checkStmt->execute([$orderId]);
            $orderType = $checkStmt->fetchColumn();
            
            if ($orderType !== 'purchase') {
                flashMessage('warning', 'Bu sipariş türü buradan silinemez.');
                redirect('orders.php');
            }
            
            $pdo->beginTransaction();
            $pdo->prepare('DELETE FROM transactions WHERE order_id = ?')->execute([$orderId]);
            $pdo->prepare('DELETE FROM order_items WHERE order_id = ?')->execute([$orderId]);
            $pdo->prepare('DELETE FROM orders WHERE id = ?')->execute([$orderId]);
            $pdo->commit();
            flashMessage('success', 'Sipariş kaydı silindi.');
        } catch (Throwable $th) {
            $pdo->rollBack();
            flashMessage('danger', 'Sipariş silinirken hata oluştu: ' . $th->getMessage());
        }
        redirect('orders.php');
    }
}

// Pagination (sadece purchase siparişleri, deposit'ler hariç)
$totalOrdersStmt = $pdo->query("SELECT COUNT(DISTINCT o.id) FROM orders o WHERE o.order_type = 'purchase'");
$totalOrders = (int)$totalOrdersStmt->fetchColumn();
$pagination = getPaginationData($totalOrders, 10);

// Siparişleri getir (sadece purchase, havale/EFT ve bakiye yüklemeler hariç)
$orders = $pdo->query("
    SELECT o.*, u.username, u.email,
           COUNT(oi.id) as item_count
    FROM orders o
    JOIN users u ON o.user_id = u.id
    LEFT JOIN order_items oi ON o.id = oi.order_id
    WHERE o.order_type = 'purchase'
    GROUP BY o.id
    ORDER BY o.created_at DESC
    LIMIT {$pagination['limit']} OFFSET {$pagination['offset']}
")->fetchAll();

$pageTitle = 'Sipariş Yönetimi';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h3 class="mb-1"><i class="fas fa-receipt me-2"></i>Sipariş Yönetimi</h3>
            <p class="text-muted mb-0">Tüm siparişleri görüntüleyin, durum güncelleyin veya gerektiğinde silin.</p>
        </div>
        <a href="reports.php" class="btn btn-outline-primary"><i class="fas fa-chart-line me-2"></i>Raporlara Git</a>
    </div>
    
    <div class="card">
        <div class="card-header d-flex flex-wrap justify-content-between align-items-center gap-3">
            <h5 class="mb-0"><i class="fas fa-list-check me-2"></i>Siparişler</h5>
            <form method="post" id="orderBulkForm" class="bulk-action-toolbar" data-bulk-form="orderBulkForm">
                <div class="d-flex align-items-center gap-2">
                    <span class="text-muted small">Seçili siparişler için:</span>
                    <button type="submit" name="bulk_action" value="complete" class="btn btn-sm btn-success" disabled data-bulk-action>
                        <i class="fas fa-check-double me-1"></i>Toplu Onayla
                    </button>
                    <button type="submit" name="bulk_action" value="delete" class="btn btn-sm btn-outline-danger" disabled data-bulk-action data-confirm="Seçili siparişleri silmek istediğinize emin misiniz?">
                        <i class="fas fa-trash-alt me-1"></i>Toplu Sil
                    </button>
                </div>
            </form>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover table-bulk mb-0 align-middle">
                    <thead>
                        <tr>
                            <th class="text-center" style="width:48px;">
                                <input type="checkbox" class="form-check-input" data-bulk-master="orderBulkForm">
                            </th>
                            <th>Sipariş No</th>
                            <th>Kullanıcı</th>
                            <th>Kalem</th>
                            <th>Net Tutar</th>
                            <th>Komisyon</th>
                            <th>Ödeme Yöntemi</th>
                            <th>Durum</th>
                            <th>Tarih</th>
                            <th class="text-end">İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($orders as $order): ?>
                        <tr>
                            <td class="text-center">
                                <input type="checkbox" class="form-check-input" name="selected_ids[]" value="<?= $order['id'] ?>" form="orderBulkForm" data-bulk-target="orderBulkForm">
                            </td>
                            <td><strong><?= sanitize($order['order_number']) ?></strong></td>
                            <td>
                                <strong><?= sanitize($order['username']) ?></strong>
                                <br><small class="text-muted"><?= sanitize($order['email']) ?></small>
                            </td>
                            <td><span class="badge bg-info"><?= $order['item_count'] ?> adet</span></td>
                            <td><strong><?= formatPrice($order['total_amount']) ?></strong></td>
                            <td><?= $order['order_type'] === 'deposit' ? formatPrice($order['commission_amount']) : '-' ?></td>
                            <td><span class="badge bg-primary text-uppercase"><?= sanitize($order['payment_method'] ?? '-') ?></span></td>
                            <td>
                                <form method="post" class="d-flex align-items-center gap-2">
                                    <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                                    <select name="status" class="form-select form-select-sm">
                                        <option value="pending" <?= $order['status'] === 'pending' ? 'selected' : '' ?>>Beklemede</option>
                                        <option value="completed" <?= $order['status'] === 'completed' ? 'selected' : '' ?>>Tamamlandı</option>
                                        <option value="cancelled" <?= $order['status'] === 'cancelled' ? 'selected' : '' ?>>İptal</option>
                                    </select>
                                    <button type="submit" name="update_status" class="btn btn-sm btn-primary">Güncelle</button>
                                </form>
                            </td>
                            <td><?= date('d.m.Y H:i', strtotime($order['created_at'])) ?></td>
                            <td class="text-end">
                                <div class="d-flex justify-content-end gap-2">
                                    <a href="order-detail.php?id=<?= $order['id'] ?>" class="btn btn-sm btn-outline-primary">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <form method="post" onsubmit="return confirm('Bu siparişi silmek istediğinizden emin misiniz?');">
                                        <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                                        <button type="submit" name="delete_order" class="btn btn-sm btn-outline-danger">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php if ($totalOrders > 10): ?>
            <div class="card-body border-top">
                <?= renderPagination($pagination) ?>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>
