<?php
require_once '../config.php';
requireAdmin();

$adminExtraStyles = [
    'https://cdn.quilljs.com/1.3.7/quill.snow.css'
];
$adminExtraScriptsFoot = [
    'https://cdn.quilljs.com/1.3.7/quill.js'
];

$adminId = $_SESSION['user_id'] ?? null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('pages.php');
    }

    $action = $_POST['action'] ?? '';

    if ($action === 'delete_page') {
        $pageId = (int)($_POST['page_id'] ?? 0);
        if ($pageId <= 0) {
            flashMessage('danger', 'Geçersiz sayfa kaydı.');
            redirect('pages.php');
        }

        try {
            $stmt = $pdo->prepare('SELECT image_path FROM pages WHERE id = ?');
            $stmt->execute([$pageId]);
            $page = $stmt->fetch();
            if ($page) {
                if (!empty($page['image_path'])) {
                    deleteUploadedFile($page['image_path']);
                }
                $pdo->prepare('DELETE FROM pages WHERE id = ?')->execute([$pageId]);
                logUserActivity($pdo, 'admin_page_deleted', ['page_id' => $pageId]);
                flashMessage('success', 'Sayfa başarıyla silindi.');
            } else {
                flashMessage('warning', 'Silinecek sayfa bulunamadı.');
            }
        } catch (Throwable $th) {
            flashMessage('danger', 'Sayfa silinirken hata oluştu: ' . sanitize($th->getMessage()));
        }

        redirect('pages.php');
    }

    if ($action === 'save_page') {
        $pageId = isset($_POST['page_id']) && $_POST['page_id'] !== '' ? (int)$_POST['page_id'] : null;
        $title = trim($_POST['title'] ?? '');
        $slugInput = trim($_POST['slug'] ?? '');
        $summary = trim($_POST['summary'] ?? '');
        $contentHtml = trim($_POST['content_html'] ?? '');
        $metaTitle = trim($_POST['meta_title'] ?? '');
        $metaDescription = trim($_POST['meta_description'] ?? '');
        $metaKeywords = trim($_POST['meta_keywords'] ?? '');
        $isPublished = isset($_POST['is_published']) ? 1 : 0;
        $removeImage = isset($_POST['remove_image']);

        if ($title === '') {
            flashMessage('danger', 'Sayfa başlığı zorunludur.');
            redirect('pages.php' . ($pageId ? '?edit=' . $pageId : ''));
        }

        if ($contentHtml === '') {
            flashMessage('danger', 'Sayfa içeriği boş olamaz.');
            redirect('pages.php' . ($pageId ? '?edit=' . $pageId : ''));
        }

        $currentPage = null;
        if ($pageId) {
            $stmt = $pdo->prepare('SELECT * FROM pages WHERE id = ?');
            $stmt->execute([$pageId]);
            $currentPage = $stmt->fetch();
            if (!$currentPage) {
                flashMessage('danger', 'Güncellenecek sayfa bulunamadı.');
                redirect('pages.php');
            }
        }

        $slugBase = $slugInput !== '' ? generateSlug($slugInput) : generateSlug($title);
        $slug = $slugBase;
        $slugIndex = 2;
        $slugStmt = $pdo->prepare('SELECT id FROM pages WHERE slug = ? AND id != ? LIMIT 1');
        while (true) {
            $params = [$slug, $pageId ?? 0];
            $slugStmt->execute($params);
            if (!$slugStmt->fetch()) {
                break;
            }
            $slug = $slugBase . '-' . $slugIndex;
            $slugIndex++;
        }

        $metaDefaults = buildMetaDefaults($title, strip_tags($contentHtml));
        if ($metaTitle === '') {
            $metaTitle = $metaDefaults['meta_title'];
        }
        if ($metaDescription === '') {
            $metaDescription = $metaDefaults['meta_description'];
        }
        if ($metaKeywords === '') {
            $metaKeywords = $metaDefaults['meta_keywords'];
        }

        $imagePath = $currentPage['image_path'] ?? null;
        if ($removeImage && $imagePath) {
            deleteUploadedFile($imagePath);
            $imagePath = null;
        }

        try {
            $imagePath = uploadImageFile($_FILES['image'] ?? [], 'pages', $imagePath);
        } catch (Throwable $th) {
            flashMessage('danger', 'Görsel yüklenemedi: ' . sanitize($th->getMessage()));
            redirect('pages.php' . ($pageId ? '?edit=' . $pageId : ''));
        }

        $publishedAt = $currentPage['published_at'] ?? null;
        if ($isPublished) {
            if (!$publishedAt) {
                $publishedAt = date('Y-m-d H:i:s');
            }
        } else {
            $publishedAt = null;
        }

        try {
            if ($pageId) {
                $stmt = $pdo->prepare('UPDATE pages SET title = :title, slug = :slug, summary = :summary, content = :content, image_path = :image, meta_title = :meta_title, meta_description = :meta_description, meta_keywords = :meta_keywords, is_published = :is_published, published_at = :published_at, updated_at = NOW() WHERE id = :id');
                $stmt->execute([
                    ':title' => $title,
                    ':slug' => $slug,
                    ':summary' => $summary !== '' ? $summary : null,
                    ':content' => $contentHtml,
                    ':image' => $imagePath,
                    ':meta_title' => $metaTitle,
                    ':meta_description' => $metaDescription,
                    ':meta_keywords' => $metaKeywords,
                    ':is_published' => $isPublished,
                    ':published_at' => $publishedAt,
                    ':id' => $pageId,
                ]);
                logUserActivity($pdo, 'admin_page_updated', ['page_id' => $pageId, 'slug' => $slug]);
                flashMessage('success', 'Sayfa başarıyla güncellendi.');
            } else {
                $stmt = $pdo->prepare('INSERT INTO pages (title, slug, summary, content, image_path, meta_title, meta_description, meta_keywords, is_published, published_at, created_by)
                    VALUES (:title, :slug, :summary, :content, :image, :meta_title, :meta_description, :meta_keywords, :is_published, :published_at, :created_by)');
                $stmt->execute([
                    ':title' => $title,
                    ':slug' => $slug,
                    ':summary' => $summary !== '' ? $summary : null,
                    ':content' => $contentHtml,
                    ':image' => $imagePath,
                    ':meta_title' => $metaTitle,
                    ':meta_description' => $metaDescription,
                    ':meta_keywords' => $metaKeywords,
                    ':is_published' => $isPublished,
                    ':published_at' => $isPublished ? $publishedAt : null,
                    ':created_by' => $adminId,
                ]);
                $pageId = (int)$pdo->lastInsertId();
                logUserActivity($pdo, 'admin_page_created', ['page_id' => $pageId, 'slug' => $slug]);
                flashMessage('success', 'Yeni sayfa oluşturuldu.');
            }
        } catch (Throwable $th) {
            flashMessage('danger', 'Sayfa kaydedilemedi: ' . sanitize($th->getMessage()));
            redirect('pages.php' . ($pageId ? '?edit=' . $pageId : ''));
        }

        redirect('pages.php' . ($pageId ? '?edit=' . $pageId : ''));
    }
}

$pagesStmt = $pdo->query('SELECT p.*, u.username AS author_username FROM pages p LEFT JOIN users u ON p.created_by = u.id ORDER BY p.created_at DESC');
$pages = $pagesStmt->fetchAll() ?: [];

$editPage = null;
if (isset($_GET['edit'])) {
    $stmt = $pdo->prepare('SELECT * FROM pages WHERE id = ?');
    $stmt->execute([(int)$_GET['edit']]);
    $editPage = $stmt->fetch();
    if (!$editPage) {
        flashMessage('danger', 'Sayfa bulunamadı.');
        redirect('pages.php');
    }
}

$initialContent = $editPage['content'] ?? '';
$adminExtraScriptsFoot[] = [
    'content' => 'const pageEditorContainer=document.getElementById("pageEditor");if(pageEditorContainer){const quill=new Quill("#pageEditor",{theme:"snow",placeholder:"Sayfa içeriğini buraya yazın"});const initialContent=' . json_encode($initialContent, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . ';if(initialContent){quill.clipboard.dangerouslyPasteHTML(initialContent);}const form=document.getElementById("pageForm");if(form){form.addEventListener("submit",function(){document.getElementById("content_html").value=quill.root.innerHTML.trim();});}}',
    'type' => 'text/javascript'
];

$pageTitle = 'Sayfa Yönetimi';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="row g-4">
        <div class="col-lg-7">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">
                        <i class="fas fa-file-lines me-2"></i><?= $editPage ? 'Sayfayı Düzenle' : 'Yeni Sayfa Oluştur' ?>
                    </h5>
                    <?php if ($editPage): ?>
                        <a href="pages.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-plus me-1"></i>Yeni Sayfa
                        </a>
                    <?php endif; ?>
                </div>
                <div class="card-body">
                    <form method="post" enctype="multipart/form-data" id="pageForm" class="row g-3">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="save_page">
                        <input type="hidden" name="page_id" value="<?= $editPage ? (int)$editPage['id'] : '' ?>">
                        <input type="hidden" name="content_html" id="content_html">
                        <div class="col-12">
                            <label class="form-label">Başlık *</label>
                            <input type="text" name="title" class="form-control" value="<?= sanitize($editPage['title'] ?? ($_POST['title'] ?? '')) ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Slug</label>
                            <input type="text" name="slug" class="form-control" value="<?= sanitize($editPage['slug'] ?? ($_POST['slug'] ?? '')) ?>" placeholder="Otomatik oluşturulur">
                            <small class="text-muted">Boş bırakırsanız otomatik olarak oluşturulur.</small>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Durum</label>
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" role="switch" name="is_published" id="is_published" <?= (($editPage['is_published'] ?? 0) == 1 || (!isset($editPage) && !empty($_POST['is_published']))) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="is_published">Yayınla</label>
                            </div>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Özet</label>
                            <textarea name="summary" class="form-control" rows="2" placeholder="Sayfanın kısa özeti (opsiyonel)"><?= sanitize($editPage['summary'] ?? ($_POST['summary'] ?? '')) ?></textarea>
                        </div>
                        <div class="col-12">
                            <label class="form-label">İçerik *</label>
                            <div id="pageEditor" style="height: 320px; background: #fff; border-radius: 8px;"></div>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Kapak Görseli</label>
                            <input type="file" name="image" class="form-control" accept="image/*">
                            <small class="text-muted">Opsiyonel. Maks 4MB.</small>
                        </div>
                        <?php if (!empty($editPage['image_path'])): ?>
                        <div class="col-md-6 d-flex align-items-center">
                            <img src="../<?= sanitize($editPage['image_path']) ?>" alt="Sayfa görseli" class="img-fluid rounded" style="max-height:120px;">
                        </div>
                        <div class="col-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="remove_image" id="remove_image">
                                <label class="form-check-label" for="remove_image">Mevcut görseli kaldır</label>
                            </div>
                        </div>
                        <?php endif; ?>
                        <div class="col-md-4">
                            <label class="form-label">Meta Başlık</label>
                            <input type="text" name="meta_title" class="form-control" value="<?= sanitize($editPage['meta_title'] ?? ($_POST['meta_title'] ?? '')) ?>" placeholder="Otomatik oluşturulur">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Meta Açıklama</label>
                            <input type="text" name="meta_description" class="form-control" value="<?= sanitize($editPage['meta_description'] ?? ($_POST['meta_description'] ?? '')) ?>" placeholder="Otomatik oluşturulur">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Meta Anahtar Kelimeler</label>
                            <input type="text" name="meta_keywords" class="form-control" value="<?= sanitize($editPage['meta_keywords'] ?? ($_POST['meta_keywords'] ?? '')) ?>" placeholder="Otomatik oluşturulur">
                        </div>
                        <div class="col-12 text-end">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-2"></i><?= $editPage ? 'Sayfayı Güncelle' : 'Sayfayı Oluştur' ?>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        <div class="col-lg-5">
            <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-folder-tree me-2"></i>Sayfalar</h5>
                    <span class="badge bg-secondary">Toplam <?= count($pages) ?> kayıt</span>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Başlık</th>
                                    <th>Slug</th>
                                    <th>Durum</th>
                                    <th>Oluşturan</th>
                                    <th>Tarih</th>
                                    <th class="text-end">İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($pages)): ?>
                                <tr>
                                    <td colspan="7" class="text-center text-muted py-4">Henüz sayfa oluşturulmadı.</td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($pages as $page): ?>
                                    <tr>
                                        <td>#<?= (int)$page['id'] ?></td>
                                        <td><strong><?= sanitize($page['title']) ?></strong></td>
                                        <td><code><?= sanitize($page['slug']) ?></code></td>
                                        <td>
                                            <?php if ((int)$page['is_published'] === 1): ?>
                                                <span class="badge bg-success">Yayında</span>
                                            <?php else: ?>
                                                <span class="badge bg-warning text-dark">Taslak</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= $page['author_username'] ? sanitize($page['author_username']) : '<span class="text-muted">Belirsiz</span>' ?></td>
                                        <td>
                                            <div class="small text-muted">Oluşturma: <?= date('d.m.Y', strtotime($page['created_at'])) ?></div>
                                            <div class="small text-muted">Güncelleme: <?= date('d.m.Y', strtotime($page['updated_at'])) ?></div>
                                        </td>
                                        <td class="text-end">
                                            <a href="pages.php?edit=<?= (int)$page['id'] ?>" class="btn btn-sm btn-outline-primary">
                                                <i class="fas fa-pen"></i>
                                            </a>
                                            <form method="post" class="d-inline" onsubmit="return confirm('Bu sayfayı silmek istediğinize emin misiniz?');">
                                                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                <input type="hidden" name="action" value="delete_page">
                                                <input type="hidden" name="page_id" value="<?= (int)$page['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-outline-danger">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                            <?php if ((int)$page['is_published'] === 1): ?>
                                                <a href="<?= htmlspecialchars(routeUrl('page', ['slug' => $page['slug']]), ENT_QUOTES, 'UTF-8') ?>" class="btn btn-sm btn-link" target="_blank">
                                                    <i class="fas fa-up-right-from-square"></i>
                                                </a>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

