<?php
require_once '../config.php';
requireAdmin();

$errors = [];

try {
    $categories = $pdo->query("SELECT id, name FROM categories WHERE is_active = 1 ORDER BY name ASC")->fetchAll();
} catch (PDOException $e) {
    $categories = [];
}

$id = isset($_GET['id']) ? (int)$_GET['id'] : null;
$product = null;

if ($id) {
    $stmt = $pdo->prepare("SELECT * FROM products WHERE id = ?");
    $stmt->execute([$id]);
    $product = $stmt->fetch();
    if (!$product) {
        flashMessage('danger', 'Ürün bulunamadı.');
        redirect('products.php');
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('product-form.php' . ($id ? '?id=' . $id : ''));
    }

    $id = isset($_POST['id']) && $_POST['id'] !== '' ? (int)$_POST['id'] : null;

    $name = trim($_POST['name'] ?? '');
    $slugInput = trim($_POST['slug'] ?? '');
    $slug = slugify($slugInput !== '' ? $slugInput : $name);
    $categoryId = isset($_POST['category_id']) ? (int)$_POST['category_id'] : 0;
    $price = isset($_POST['price']) ? (float)$_POST['price'] : 0;
    $stock = isset($_POST['stock']) ? (int)$_POST['stock'] : 0;
    $description = trim($_POST['description'] ?? '');
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    $featured = isset($_POST['featured']) ? 1 : 0;

    if ($name === '') {
        $errors[] = 'Ürün adı zorunludur.';
    }

    if ($categoryId === 0) {
        $errors[] = 'Lütfen bir kategori seçin.';
    }

    if ($price <= 0) {
        $errors[] = 'Fiyat 0&#39;dan büyük olmalıdır.';
    }

    if ($slug === '') {
        $slug = uniqid('urun-');
    }

    $currentImage = null;
    if ($id) {
        $stmt = $pdo->prepare("SELECT * FROM products WHERE id = ?");
        $stmt->execute([$id]);
        $product = $stmt->fetch();
        if (!$product) {
            $errors[] = 'Ürün bulunamadı.';
        } else {
            $currentImage = $product['image'] ?? null;
        }
    }

    if (empty($errors)) {
        try {
            $currentImage = uploadImageFile($_FILES['image'] ?? [], 'products', $currentImage);
        } catch (Exception $e) {
            $errors[] = 'Ürün görseli yüklenemedi: ' . $e->getMessage();
        }
    }

    if (empty($errors)) {
        try {
            if ($id) {
                $stmt = $pdo->prepare("UPDATE products SET category_id = ?, name = ?, slug = ?, description = ?, price = ?, stock = ?, image = ?, is_active = ?, featured = ?, updated_at = NOW() WHERE id = ?");
                $stmt->execute([$categoryId, $name, $slug, $description, $price, $stock, $currentImage, $isActive, $featured, $id]);
                flashMessage('success', 'Ürün güncellendi.');
            } else {
                $stmt = $pdo->prepare("INSERT INTO products (category_id, name, slug, description, price, stock, image, is_active, featured, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())");
                $stmt->execute([$categoryId, $name, $slug, $description, $price, $stock, $currentImage, $isActive, $featured]);
                flashMessage('success', 'Ürün eklendi.');
            }
            redirect('products.php');
        } catch (PDOException $e) {
            $errors[] = 'Veritabanı hatası: ' . $e->getMessage();
        }
    }

    // Formu yeniden doldurmak için değerleri sakla
    $product = [
        'id' => $id,
        'category_id' => $categoryId,
        'name' => $name,
        'slug' => $slug,
        'description' => $description,
        'price' => $price,
        'stock' => $stock,
        'image' => $currentImage,
        'is_active' => $isActive,
        'featured' => $featured
    ];
}

$pageTitle = $product && isset($product['id']) ? 'Ürün Düzenle' : 'Yeni Ürün Ekle';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-xl-8 col-lg-9">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">
                        <i class="fa-solid fa-box"></i>
                        <?= $product && isset($product['id']) ? 'Ürün Düzenle' : 'Yeni Ürün Ekle' ?>
                    </h5>
                    <a href="products.php" class="btn btn-sm btn-outline-secondary">
                        <i class="fa-solid fa-arrow-left"></i> Ürün Listesi
                    </a>
                </div>
                <div class="card-body">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <i class="fa-solid fa-circle-exclamation me-2"></i>
                            <?php foreach ($errors as $error): ?>
                                <div><?= sanitize($error) ?></div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>

                    <form method="POST" enctype="multipart/form-data" class="row g-3">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <?php if ($product && isset($product['id'])): ?>
                            <input type="hidden" name="id" value="<?= (int)$product['id'] ?>">
                        <?php endif; ?>

                        <div class="col-md-6">
                            <label class="form-label" for="name">Ürün Adı</label>
                            <input type="text" class="form-control" id="name" name="name" value="<?= isset($product['name']) ? sanitize($product['name']) : '' ?>" required>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label" for="slug">Slug (URL)</label>
                            <input type="text" class="form-control" id="slug" name="slug" value="<?= isset($product['slug']) ? sanitize($product['slug']) : '' ?>" placeholder="boş bırakırsanız otomatik oluşur">
                        </div>

                        <div class="col-md-6">
                            <label class="form-label" for="category_id">Kategori</label>
                            <select id="category_id" name="category_id" class="form-select" required>
                                <option value="">Kategori Seçin</option>
                                <?php foreach ($categories as $cat): ?>
                                    <option value="<?= $cat['id'] ?>" <?= isset($product['category_id']) && (int)$product['category_id'] === (int)$cat['id'] ? 'selected' : '' ?>><?= sanitize($cat['name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="col-md-3">
                            <label class="form-label" for="price">Fiyat</label>
                            <input type="number" step="0.01" min="0" class="form-control" id="price" name="price" value="<?= isset($product['price']) ? sanitize(number_format((float)$product['price'], 2, '.', '')) : '' ?>" required>
                        </div>

                        <div class="col-md-3">
                            <label class="form-label" for="stock">Stok</label>
                            <input type="number" min="0" class="form-control" id="stock" name="stock" value="<?= isset($product['stock']) ? (int)$product['stock'] : 0 ?>" required>
                        </div>

                        <div class="col-12">
                            <label class="form-label" for="description">Açıklama</label>
                            <textarea class="form-control" id="description" name="description" rows="4" placeholder="Ürün açıklaması..."><?= isset($product['description']) ? sanitize($product['description']) : '' ?></textarea>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Ürün Görseli</label>
                            <?php if ($product && !empty($product['image'])): ?>
                                <div class="mb-2">
                                    <img src="../<?= sanitize($product['image']) ?>" alt="Ürün görseli" class="img-fluid rounded shadow-sm" style="max-height: 180px;">
                                </div>
                            <?php endif; ?>
                            <input type="file" class="form-control" name="image" accept="image/*">
                            <small class="text-muted">PNG, JPG, WEBP veya GIF (maks. 4MB)</small>
                        </div>

                        <div class="col-md-6 d-flex align-items-end">
                            <div class="row w-100">
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="is_active" name="is_active" <?= !$product || (isset($product['is_active']) && $product['is_active']) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="is_active">Aktif</label>
                                    </div>
                                </div>
                                <div class="col-12 mt-2">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="featured" name="featured" <?= isset($product['featured']) && $product['featured'] ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="featured">Öne Çıkan</label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-12 d-flex justify-content-end gap-3 mt-4">
                            <a href="products.php" class="btn btn-outline-secondary">İptal</a>
                            <button type="submit" class="btn btn-primary"><i class="fa-solid fa-floppy-disk me-2"></i>Kaydet</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

