<?php
require_once '../config.php';
requireAdmin();

$pageTitle = 'Profil Ayarları';

$admin = getUserInfo($pdo, $_SESSION['user_id']);
$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Güvenlik doğrulaması başarısız.';
    } else {
        $action = $_POST['action'] ?? '';
        
        if ($action === 'update_profile') {
            $email = filter_var(trim($_POST['email'] ?? ''), FILTER_VALIDATE_EMAIL);
            $fullName = trim($_POST['full_name'] ?? '');
            $phone = trim($_POST['phone'] ?? '');
            
            if (!$email) {
                $errors[] = 'Geçerli bir e-posta adresi girin.';
            }
            
            // Email benzersizlik kontrolü
            if (empty($errors)) {
                $emailCheckStmt = $pdo->prepare('SELECT id FROM users WHERE email = ? AND id != ?');
                $emailCheckStmt->execute([$email, $_SESSION['user_id']]);
                if ($emailCheckStmt->fetch()) {
                    $errors[] = 'Bu e-posta adresi başka bir kullanıcı tarafından kullanılıyor.';
                }
            }
            
            if (empty($errors)) {
                try {
                    $updateStmt = $pdo->prepare('
                        UPDATE users 
                        SET email = ?, full_name = ?, phone = ? 
                        WHERE id = ?
                    ');
                    $updateStmt->execute([$email, $fullName, $phone, $_SESSION['user_id']]);
                    $_SESSION['email'] = $email;
                    $success = 'Profil bilgileriniz güncellendi.';
                    $admin = getUserInfo($pdo, $_SESSION['user_id']); // Refresh
                } catch (PDOException $e) {
                    $errors[] = 'Profil güncellenirken hata oluştu.';
                }
            }
        }
        
        elseif ($action === 'change_password') {
            $currentPassword = $_POST['current_password'] ?? '';
            $newPassword = $_POST['new_password'] ?? '';
            $confirmPassword = $_POST['confirm_password'] ?? '';
            
            if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
                $errors[] = 'Tüm şifre alanlarını doldurun.';
            } elseif ($newPassword !== $confirmPassword) {
                $errors[] = 'Yeni şifreler eşleşmiyor.';
            } elseif (strlen($newPassword) < 8) {
                $errors[] = 'Yeni şifre en az 8 karakter olmalıdır.';
            } elseif (!password_verify($currentPassword, $admin['password'])) {
                $errors[] = 'Mevcut şifreniz hatalı.';
            } else {
                try {
                    $newHash = password_hash($newPassword, PASSWORD_DEFAULT);
                    $updateStmt = $pdo->prepare('UPDATE users SET password = ? WHERE id = ?');
                    $updateStmt->execute([$newHash, $_SESSION['user_id']]);
                    $success = 'Şifreniz başarıyla değiştirildi.';
                } catch (PDOException $e) {
                    $errors[] = 'Şifre değiştirilirken hata oluştu.';
                }
            }
        }
    }
}

include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-8">
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-user-circle me-2"></i>Admin Profil Bilgileri</h5>
                </div>
                <div class="card-body">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-circle me-2"></i>
                            <?= implode('<br>', array_map('sanitize', $errors)) ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($success): ?>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle me-2"></i><?= sanitize($success) ?>
                        </div>
                    <?php endif; ?>
                    
                    <form method="post">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="update_profile">
                        
                        <div class="mb-3">
                            <label class="form-label">Kullanıcı Adı</label>
                            <input type="text" class="form-control" value="<?= sanitize($admin['username']) ?>" disabled>
                            <small class="text-muted">Kullanıcı adı değiştirilemez.</small>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">E-posta *</label>
                                <input type="email" name="email" class="form-control" value="<?= sanitize($admin['email']) ?>" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Ad Soyad</label>
                                <input type="text" name="full_name" class="form-control" value="<?= sanitize($admin['full_name'] ?? '') ?>">
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Telefon</label>
                            <input type="text" name="phone" class="form-control" value="<?= sanitize($admin['phone'] ?? '') ?>" placeholder="5XX XXX XX XX">
                        </div>
                        
                        <div class="text-end">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-2"></i>Profil Bilgilerini Kaydet
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-key me-2"></i>Şifre Değiştir</h5>
                </div>
                <div class="card-body">
                    <form method="post">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="change_password">
                        
                        <div class="mb-3">
                            <label class="form-label">Mevcut Şifre *</label>
                            <input type="password" name="current_password" class="form-control" required autocomplete="current-password">
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Yeni Şifre *</label>
                                <input type="password" name="new_password" class="form-control" required autocomplete="new-password" minlength="8">
                                <small class="text-muted">En az 8 karakter</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Yeni Şifre (Tekrar) *</label>
                                <input type="password" name="confirm_password" class="form-control" required autocomplete="new-password" minlength="8">
                            </div>
                        </div>
                        
                        <div class="text-end">
                            <button type="submit" class="btn btn-danger">
                                <i class="fas fa-lock me-2"></i>Şifreyi Değiştir
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

