<?php
require_once '../config.php';
requireAdmin();

function resolveRangeConfiguration(string $range): array
{
    $ranges = [
        'weekly' => ['label' => '7 Gün', 'start' => '-6 days', 'group' => 'DAY'],
        'monthly' => ['label' => '30 Gün', 'start' => '-29 days', 'group' => 'DAY'],
        'quarterly' => ['label' => '3 Ay', 'start' => '-89 days', 'group' => 'WEEK'],
        'halfyear' => ['label' => '6 Ay', 'start' => '-179 days', 'group' => 'WEEK'],
        'yearly' => ['label' => '12 Ay', 'start' => '-364 days', 'group' => 'MONTH'],
    ];

    if (!isset($ranges[$range])) {
        $range = 'monthly';
    }

    return [$range, $ranges[$range], $ranges];
}

function buildSummary(PDO $pdo, DateTimeImmutable $start): array
{
    $startParam = $start->format('Y-m-d H:i:s');

    $ordersStmt = $pdo->prepare('
        SELECT
            SUM(CASE WHEN order_type = "purchase" AND status = "completed" THEN total_amount ELSE 0 END) AS sales_total,
            SUM(CASE WHEN order_type = "purchase" AND status = "completed" THEN 1 ELSE 0 END) AS sales_count,
            SUM(CASE WHEN order_type = "deposit" AND status = "completed" THEN payable_amount ELSE 0 END) AS deposit_gross,
            SUM(CASE WHEN order_type = "deposit" AND status = "completed" THEN total_amount ELSE 0 END) AS deposit_net,
            SUM(CASE WHEN order_type = "deposit" AND status = "completed" THEN commission_amount ELSE 0 END) AS commission_total,
            SUM(CASE WHEN status = "pending" THEN 1 ELSE 0 END) AS pending_orders
        FROM orders
        WHERE created_at >= ?
    ');
    $ordersStmt->execute([$startParam]);
    $orderSummary = $ordersStmt->fetch() ?: [];

    $usersStmt = $pdo->prepare('SELECT COUNT(*) FROM users WHERE created_at >= ?');
    $usersStmt->execute([$startParam]);

    $transactionsStmt = $pdo->prepare('
        SELECT
            SUM(CASE WHEN type = "deposit" THEN amount ELSE 0 END) AS deposit_transactions,
            SUM(CASE WHEN type = "purchase" THEN amount ELSE 0 END) AS purchase_transactions,
            SUM(CASE WHEN type = "withdraw" THEN amount ELSE 0 END) AS withdraw_transactions
        FROM transactions
        WHERE created_at >= ?
    ');
    $transactionsStmt->execute([$startParam]);
    $transactionSummary = $transactionsStmt->fetch() ?: [];

    return [
        'sales_total' => (float)($orderSummary['sales_total'] ?? 0),
        'sales_count' => (int)($orderSummary['sales_count'] ?? 0),
        'deposit_net' => (float)($orderSummary['deposit_net'] ?? 0),
        'deposit_gross' => (float)($orderSummary['deposit_gross'] ?? 0),
        'commission_total' => (float)($orderSummary['commission_total'] ?? 0),
        'pending_orders' => (int)($orderSummary['pending_orders'] ?? 0),
        'new_users' => (int)($usersStmt->fetchColumn() ?: 0),
        'deposit_transactions' => (float)($transactionSummary['deposit_transactions'] ?? 0),
        'purchase_transactions' => (float)($transactionSummary['purchase_transactions'] ?? 0),
        'withdraw_transactions' => (float)($transactionSummary['withdraw_transactions'] ?? 0),
    ];
}

function fetchTrend(PDO $pdo, DateTimeImmutable $start, string $groupInterval): array
{
    switch ($groupInterval) {
        case 'WEEK':
            $labelExpr = "DATE_FORMAT(created_at, '%Y-%u')";
            break;
        case 'MONTH':
            $labelExpr = "DATE_FORMAT(created_at, '%Y-%m')";
            break;
        case 'DAY':
        default:
            $labelExpr = 'DATE(created_at)';
            break;
    }

    $sql = "SELECT {$labelExpr} AS label,
        SUM(CASE WHEN order_type = 'purchase' AND status = 'completed' THEN total_amount ELSE 0 END) AS sales_total,
        SUM(CASE WHEN order_type = 'deposit' AND status = 'completed' THEN payable_amount ELSE 0 END) AS deposit_total,
        SUM(CASE WHEN order_type = 'deposit' AND status = 'completed' THEN commission_amount ELSE 0 END) AS commission_total
    FROM orders
    WHERE created_at >= :start
    GROUP BY label
    ORDER BY label ASC";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([':start' => $start->format('Y-m-d H:i:s')]);

    return $stmt->fetchAll() ?: [];
}

function fetchTopCustomers(PDO $pdo, DateTimeImmutable $start, int $limit = 6): array
{
    $stmt = $pdo->prepare('
        SELECT u.username, u.full_name, COUNT(o.id) AS order_count, SUM(o.total_amount) AS revenue
        FROM orders o
        INNER JOIN users u ON u.id = o.user_id
        WHERE o.status = "completed" AND o.order_type = "purchase" AND o.created_at >= :start
        GROUP BY u.id
        ORDER BY revenue DESC
        LIMIT :limit
    ');
    $stmt->bindValue(':start', $start->format('Y-m-d H:i:s'));
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();

    return $stmt->fetchAll() ?: [];
}

function fetchTopDepositors(PDO $pdo, DateTimeImmutable $start, int $limit = 6): array
{
    $stmt = $pdo->prepare('
        SELECT u.username, u.full_name, SUM(o.total_amount) AS net_amount, SUM(o.commission_amount) AS commission, COUNT(o.id) AS deposit_count
        FROM orders o
        INNER JOIN users u ON u.id = o.user_id
        WHERE o.order_type = "deposit" AND o.created_at >= :start
        GROUP BY u.id
        ORDER BY net_amount DESC
        LIMIT :limit
    ');
    $stmt->bindValue(':start', $start->format('Y-m-d H:i:s'));
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();

    return $stmt->fetchAll() ?: [];
}

$requestedRange = strtolower(trim($_GET['range'] ?? ''));
[$range, $rangeConfig, $rangeOptions] = resolveRangeConfiguration($requestedRange);
$startDate = (new DateTimeImmutable('today'))->modify($rangeConfig['start']);

$summary = buildSummary($pdo, $startDate);
$trendRows = fetchTrend($pdo, $startDate, $rangeConfig['group']);
$topCustomers = fetchTopCustomers($pdo, $startDate);
$topDepositors = fetchTopDepositors($pdo, $startDate);

$chartPayload = [
    'labels' => array_map(static fn($row) => $row['label'], $trendRows),
    'sales' => array_map(static fn($row) => (float)$row['sales_total'], $trendRows),
    'deposits' => array_map(static fn($row) => (float)$row['deposit_total'], $trendRows),
    'commission' => array_map(static fn($row) => (float)$row['commission_total'], $trendRows),
];
$chartJson = htmlspecialchars(json_encode($chartPayload, JSON_UNESCAPED_UNICODE | JSON_NUMERIC_CHECK), ENT_QUOTES, 'UTF-8');

$pageTitle = 'Raporlama';
include 'includes/admin-header.php';
?>

<div class="admin-page reports-page" id="reportsPage" data-chart='<?= $chartJson ?>'>
    <header class="page-intro glass-surface">
        <div class="page-intro__text">
            <h4>Raporlama Merkezi</h4>
            <p class="page-intro__subtitle"><?= sanitize($rangeConfig['label']) ?> dönemi performansı</p>
        </div>
        <form method="get" class="range-switcher">
            <label class="visually-hidden" for="range">Zaman aralığı seçin</label>
            <select id="range" name="range" class="form-select" onchange="this.form.submit()">
                <?php foreach ($rangeOptions as $key => $option): ?>
                    <option value="<?= $key ?>" <?= $key === $range ? 'selected' : '' ?>><?= sanitize($option['label']) ?></option>
                <?php endforeach; ?>
            </select>
        </form>
    </header>

    <section class="metric-grid">
        <article class="metric-card">
            <div class="metric-card__label">Satış Geliri</div>
            <div class="metric-card__value"><?= formatPrice($summary['sales_total']) ?></div>
            <div class="metric-card__meta">Toplam <?= number_format($summary['sales_count']) ?> sipariş</div>
        </article>
        <article class="metric-card">
            <div class="metric-card__label">Net Para Yatırma</div>
            <div class="metric-card__value"><?= formatPrice($summary['deposit_net']) ?></div>
            <div class="metric-card__meta">Brüt: <?= formatPrice($summary['deposit_gross']) ?></div>
        </article>
        <article class="metric-card">
            <div class="metric-card__label">Komisyon Geliri</div>
            <div class="metric-card__value metric-card__value--positive">+<?= formatPrice($summary['commission_total']) ?></div>
            <div class="metric-card__meta">PayTR &amp; Shopier</div>
        </article>
        <article class="metric-card">
            <div class="metric-card__label">Yeni Kullanıcı</div>
            <div class="metric-card__value"><?= number_format($summary['new_users']) ?></div>
            <div class="metric-card__meta">Belirtilen dönemde kayıt</div>
        </article>
        <article class="metric-card">
            <div class="metric-card__label">İşlem Hacmi</div>
            <div class="metric-card__value"><?= formatPrice($summary['deposit_transactions'] + $summary['purchase_transactions']) ?></div>
            <div class="metric-card__meta">Satın alma + bakiye</div>
        </article>
        <article class="metric-card">
            <div class="metric-card__label">Çekim Ödemeleri</div>
            <div class="metric-card__value metric-card__value--negative">-<?= formatPrice($summary['withdraw_transactions']) ?></div>
            <div class="metric-card__meta">Tamamlanan çekimler</div>
        </article>
        <article class="metric-card">
            <div class="metric-card__label">Bekleyen Sipariş</div>
            <div class="metric-card__value metric-card__value--warning"><?= number_format($summary['pending_orders']) ?></div>
            <div class="metric-card__meta">Aksiyon bekleyen işlemler</div>
        </article>
    </section>

    <section class="glass-surface chart-panel">
        <div class="panel-head">
            <div>
                <h5>Performans Trendi</h5>
                <p class="panel-subtitle">Satış, bakiye yükleme ve komisyon dağılımı</p>
            </div>
        </div>
        <canvas id="reportsChart" height="140"></canvas>
    </section>

    <div class="panel-grid">
        <section class="card">
            <div class="card-header">
                <div>
                    <h5 class="mb-0"><i class="fas fa-crown me-2 text-warning"></i>Öne Çıkan Müşteriler</h5>
                    <p class="text-muted mb-0" style="font-size: 13px;">Satış gelirine göre ilk 6 kullanıcı</p>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0 align-middle">
                        <thead>
                            <tr>
                                <th>Kullanıcı</th>
                                <th>Ad Soyad</th>
                                <th>Toplam Gelir</th>
                                <th>Sipariş Sayısı</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($topCustomers)): ?>
                                <?php foreach ($topCustomers as $customer): ?>
                                    <tr>
                                        <td><strong><?= sanitize($customer['username']) ?></strong></td>
                                        <td><?= !empty($customer['full_name']) ? sanitize($customer['full_name']) : '<span class="text-muted">-</span>' ?></td>
                                        <td><strong class="text-success"><?= formatPrice($customer['revenue']) ?></strong></td>
                                        <td>
                                            <span class="badge bg-primary">
                                                <i class="fas fa-shopping-cart me-1"></i><?= number_format((int)$customer['order_count']) ?> sipariş
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="4" class="text-center text-muted py-4">Henüz kayıt bulunmuyor.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </section>

        <section class="card">
            <div class="card-header">
                <div>
                    <h5 class="mb-0"><i class="fas fa-trophy me-2 text-info"></i>Para Yatırma Liderleri</h5>
                    <p class="text-muted mb-0" style="font-size: 13px;">Net bakiye tutarına göre ilk 6 kullanıcı</p>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0 align-middle">
                        <thead>
                            <tr>
                                <th>Kullanıcı</th>
                                <th>Ad Soyad</th>
                                <th>Net Tutar</th>
                                <th>İşlem Sayısı</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($topDepositors)): ?>
                                <?php foreach ($topDepositors as $depositor): ?>
                                    <tr>
                                        <td><strong><?= sanitize($depositor['username']) ?></strong></td>
                                        <td><?= !empty($depositor['full_name']) ? sanitize($depositor['full_name']) : '<span class="text-muted">-</span>' ?></td>
                                        <td><strong class="text-primary"><?= formatPrice($depositor['net_amount']) ?></strong></td>
                                        <td>
                                            <span class="badge bg-info">
                                                <i class="fas fa-wallet me-1"></i><?= number_format((int)$depositor['deposit_count']) ?> yükleme
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="4" class="text-center text-muted py-4">Henüz kayıt bulunmuyor.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </section>
    </div>
</div>

<script>
(function() {
    const container = document.getElementById('reportsPage');
    const canvas = document.getElementById('reportsChart');

    if (!container || !canvas || typeof Chart === 'undefined') {
        return;
    }

    let payload = {};
    try {
        payload = JSON.parse(container.dataset.chart || '{}');
    } catch (error) {
        payload = {};
    }

    const styles = getComputedStyle(document.body);
    const axisColor = styles.getPropertyValue('--admin-text-muted') || 'rgba(148,163,184,0.75)';
    const legendColor = styles.getPropertyValue('--admin-text') || '#e2e8f0';
    const gridColor = styles.getPropertyValue('--admin-border') || 'rgba(148,163,184,0.15)';

    const chart = new Chart(canvas, {
        type: 'line',
        data: {
            labels: payload.labels || [],
            datasets: [
                {
                    label: 'Satış Geliri',
                    data: payload.sales || [],
                    borderColor: '#6366f1',
                    backgroundColor: 'rgba(99,102,241,0.18)',
                    tension: 0.35,
                    fill: true,
                    borderWidth: 3
                },
                {
                    label: 'Brüt Bakiye Yükleme',
                    data: payload.deposits || [],
                    borderColor: '#22d3ee',
                    backgroundColor: 'rgba(34,211,238,0.18)',
                    tension: 0.35,
                    fill: true,
                    borderWidth: 3
                },
                {
                    label: 'Komisyon Geliri',
                    data: payload.commission || [],
                    borderColor: '#f97316',
                    backgroundColor: 'rgba(249,115,22,0.18)',
                    tension: 0.35,
                    fill: true,
                    borderWidth: 3
                }
            ]
        },
        options: {
            plugins: {
                legend: {
                    labels: {
                        color: legendColor.trim(),
                        usePointStyle: true
                    }
                }
            },
            scales: {
                x: {
                    ticks: { color: axisColor.trim() },
                    grid: { color: gridColor.trim() }
                },
                y: {
                    ticks: { color: axisColor.trim() },
                    grid: { color: gridColor.trim() }
                }
            }
        }
    });

    document.addEventListener('admin-theme-change', () => {
        const newStyles = getComputedStyle(document.body);
        chart.options.plugins.legend.labels.color = (newStyles.getPropertyValue('--admin-text') || '#e2e8f0').trim();
        chart.options.scales.x.ticks.color = (newStyles.getPropertyValue('--admin-text-muted') || 'rgba(148,163,184,0.75)').trim();
        chart.options.scales.y.ticks.color = (newStyles.getPropertyValue('--admin-text-muted') || 'rgba(148,163,184,0.75)').trim();
        chart.options.scales.x.grid.color = (newStyles.getPropertyValue('--admin-border') || 'rgba(148,163,184,0.15)').trim();
        chart.options.scales.y.grid.color = (newStyles.getPropertyValue('--admin-border') || 'rgba(148,163,184,0.15)').trim();
        chart.update();
    });
})();
</script>

<?php include 'includes/admin-footer.php'; ?>
