<?php
require_once '../config.php';
requireAdmin();

$prefillIp = trim($_GET['ip'] ?? '');
$prefillDevice = trim($_GET['device'] ?? '');
$prefillUserId = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
$prefillUsername = trim($_GET['username'] ?? '');
$prefillUser = null;

if ($prefillUserId) {
    $userStmt = $pdo->prepare('SELECT id, username, email, last_login_at FROM users WHERE id = ?');
    $userStmt->execute([$prefillUserId]);
    $prefillUser = $userStmt->fetch() ?: null;
} elseif ($prefillUsername !== '') {
    $userStmt = $pdo->prepare('SELECT id, username, email, last_login_at FROM users WHERE username = ?');
    $userStmt->execute([$prefillUsername]);
    $prefillUser = $userStmt->fetch() ?: null;
}

$adminId = $_SESSION['user_id'] ?? null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız. Lütfen tekrar deneyin.');
        redirect('security.php');
    }

    $action = $_POST['action'] ?? '';

    if ($action === 'add_ip_ban') {
        $ipAddress = trim($_POST['ip_address'] ?? '');
        $reason = trim($_POST['reason'] ?? '');
        $notes = trim($_POST['notes'] ?? '');
        $expiresAtInput = trim($_POST['expires_at'] ?? '');

        if (!filter_var($ipAddress, FILTER_VALIDATE_IP)) {
            flashMessage('danger', 'Geçerli bir IP adresi giriniz.');
            redirect('security.php');
        }

        $expiresAt = null;
        if ($expiresAtInput !== '') {
            $expiresDate = DateTime::createFromFormat('Y-m-d\TH:i', $expiresAtInput);
            if ($expiresDate === false) {
                flashMessage('danger', 'Geçerli bir bitiş tarihi giriniz.');
                redirect('security.php');
            }
            $expiresAt = $expiresDate->format('Y-m-d H:i:s');
        }

        try {
            $stmt = $pdo->prepare('INSERT INTO banned_ips (ip_address, reason, notes, expires_at, created_by)
                VALUES (:ip, :reason, :notes, :expires_at, :created_by)
                ON DUPLICATE KEY UPDATE reason = VALUES(reason), notes = VALUES(notes), expires_at = VALUES(expires_at), updated_at = NOW()');
            $stmt->execute([
                ':ip' => $ipAddress,
                ':reason' => $reason !== '' ? $reason : null,
                ':notes' => $notes !== '' ? $notes : null,
                ':expires_at' => $expiresAt,
                ':created_by' => $adminId ?: null,
            ]);

            flashMessage('success', sanitize($ipAddress) . ' adresi başarıyla yasaklandı.');
        } catch (Throwable $th) {
            flashMessage('danger', 'IP yasağı eklenirken bir hata oluştu: ' . sanitize($th->getMessage()));
        }

        redirect('security.php');
    }

    if ($action === 'add_device_ban') {
        $deviceId = trim($_POST['device_id'] ?? '');
        $reason = trim($_POST['reason'] ?? '');
        $notes = trim($_POST['notes'] ?? '');
        $expiresAtInput = trim($_POST['expires_at'] ?? '');

        if ($deviceId === '' || !preg_match('/^[a-f0-9]{32}$/', $deviceId)) {
            flashMessage('danger', 'Geçerli bir cihaz kimliği giriniz.');
            redirect('security.php');
        }

        $expiresAt = null;
        if ($expiresAtInput !== '') {
            $expiresDate = DateTime::createFromFormat('Y-m-d\TH:i', $expiresAtInput);
            if ($expiresDate === false) {
                flashMessage('danger', 'Geçerli bir bitiş tarihi giriniz.');
                redirect('security.php');
            }
            $expiresAt = $expiresDate->format('Y-m-d H:i:s');
        }

        try {
            $stmt = $pdo->prepare('INSERT INTO banned_devices (device_id, reason, notes, expires_at, created_by)
                VALUES (:device, :reason, :notes, :expires_at, :created_by)
                ON DUPLICATE KEY UPDATE reason = VALUES(reason), notes = VALUES(notes), expires_at = VALUES(expires_at), updated_at = NOW()');
            $stmt->execute([
                ':device' => $deviceId,
                ':reason' => $reason !== '' ? $reason : null,
                ':notes' => $notes !== '' ? $notes : null,
                ':expires_at' => $expiresAt,
                ':created_by' => $adminId ?: null,
            ]);

            flashMessage('success', 'Cihaz ID ' . sanitize($deviceId) . ' başarıyla yasaklandı.');
        } catch (Throwable $th) {
            flashMessage('danger', 'Cihaz yasağı eklenirken bir hata oluştu: ' . sanitize($th->getMessage()));
        }

        redirect('security.php');
    }

    if ($action === 'delete_ip_ban') {
        $id = (int)($_POST['ban_id'] ?? 0);
        if ($id <= 0) {
            flashMessage('danger', 'Geçersiz IP yasağı kaydı.');
            redirect('security.php');
        }

        try {
            $stmt = $pdo->prepare('DELETE FROM banned_ips WHERE id = ?');
            $stmt->execute([$id]);
            flashMessage('success', 'IP yasağı kaldırıldı.');
        } catch (Throwable $th) {
            flashMessage('danger', 'IP yasağı kaldırılırken hata oluştu: ' . sanitize($th->getMessage()));
        }

        redirect('security.php');
    }

    if ($action === 'delete_device_ban') {
        $id = (int)($_POST['ban_id'] ?? 0);
        if ($id <= 0) {
            flashMessage('danger', 'Geçersiz cihaz yasağı kaydı.');
            redirect('security.php');
        }

        try {
            $stmt = $pdo->prepare('DELETE FROM banned_devices WHERE id = ?');
            $stmt->execute([$id]);
            flashMessage('success', 'Cihaz yasağı kaldırıldı.');
        } catch (Throwable $th) {
            flashMessage('danger', 'Cihaz yasağı kaldırılırken hata oluştu: ' . sanitize($th->getMessage()));
        }

        redirect('security.php');
    }
}

$ipBans = [];
$deviceBans = [];

try {
    $ipBansStmt = $pdo->query('SELECT b.*, u.username AS admin_username
        FROM banned_ips b
        LEFT JOIN users u ON b.created_by = u.id
        ORDER BY b.created_at DESC');
    $ipBans = $ipBansStmt->fetchAll() ?: [];
} catch (Throwable $th) {
    // Tablo yoksa veya hata varsa boş array döndür
    $ipBans = [];
}

try {
    $deviceBansStmt = $pdo->query('SELECT b.*, u.username AS admin_username
        FROM banned_devices b
        LEFT JOIN users u ON b.created_by = u.id
        ORDER BY b.created_at DESC');
    $deviceBans = $deviceBansStmt->fetchAll() ?: [];
} catch (Throwable $th) {
    // Tablo yoksa veya hata varsa boş array döndür
    $deviceBans = [];
}

$pageTitle = 'Güvenlik & Yasaklamalar';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <?php if ($prefillIp !== '' || $prefillDevice !== '' || $prefillUser): ?>
    <div class="alert alert-info d-flex flex-column gap-1">
        <strong><i class="fas fa-circle-info me-1"></i>Seçilen kullanıcı bilgileri</strong>
        <?php if ($prefillUser): ?>
            <div>Kullanıcı: <strong><?= sanitize($prefillUser['username']) ?></strong> (ID #<?= (int)$prefillUser['id'] ?><?= !empty($prefillUser['email']) ? ' • ' . sanitize($prefillUser['email']) : '' ?>)</div>
        <?php elseif ($prefillUsername !== ''): ?>
            <div>Kullanıcı: <strong><?= sanitize($prefillUsername) ?></strong></div>
        <?php endif; ?>
        <?php if ($prefillIp !== ''): ?>
            <div>IP Adresi: <code><?= sanitize($prefillIp) ?></code></div>
        <?php endif; ?>
        <?php if ($prefillDevice !== ''): ?>
            <div>Cihaz ID: <code><?= sanitize($prefillDevice) ?></code></div>
        <?php endif; ?>
    </div>
    <?php endif; ?>

    <div class="row g-4">
        <div class="col-lg-6">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-shield-halved me-2"></i>IP Yasağı Oluştur</h5>
                </div>
                <div class="card-body">
                    <form method="post" class="row g-3">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="add_ip_ban">
                        <div class="col-12">
                            <label class="form-label">IP Adresi *</label>
                            <input type="text" name="ip_address" class="form-control" placeholder="Örn: 185.23.45.10" value="<?= sanitize($prefillIp) ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Bitiş Tarihi</label>
                            <input type="datetime-local" name="expires_at" class="form-control" placeholder="Opsiyonel">
                            <small class="text-muted">Boş bırakırsanız kalıcı olur.</small>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Yasak Nedeni</label>
                            <input type="text" name="reason" class="form-control" placeholder="Örn: Dolandırıcılık şüphesi">
                        </div>
                        <div class="col-12">
                            <label class="form-label">Not</label>
                            <textarea name="notes" class="form-control" rows="2" placeholder="İç notlar (opsiyonel)"></textarea>
                        </div>
                        <div class="col-12 text-end">
                            <button type="submit" class="btn btn-danger">
                                <i class="fas fa-ban me-2"></i>IP Adresini Yasakla
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        <div class="col-lg-6">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-mobile-screen-button me-2"></i>Cihaz Yasağı Oluştur</h5>
                </div>
                <div class="card-body">
                    <form method="post" class="row g-3">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="add_device_ban">
                        <div class="col-12">
                            <label class="form-label">Cihaz Kimliği *</label>
                            <input type="text" name="device_id" class="form-control" placeholder="32 karakterlik cihaz ID" value="<?= sanitize($prefillDevice) ?>" required>
                            <small class="text-muted">Cihaz ID, log kayıtlarından kopyalanabilir.</small>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Bitiş Tarihi</label>
                            <input type="datetime-local" name="expires_at" class="form-control" placeholder="Opsiyonel">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Yasak Nedeni</label>
                            <input type="text" name="reason" class="form-control" placeholder="Örn: Çoklu hesap">
                        </div>
                        <div class="col-12">
                            <label class="form-label">Not</label>
                            <textarea name="notes" class="form-control" rows="2" placeholder="İç notlar (opsiyonel)"></textarea>
                        </div>
                        <div class="col-12 text-end">
                            <button type="submit" class="btn btn-warning">
                                <i class="fas fa-user-lock me-2"></i>Cihazı Yasakla
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-3">
            <h5 class="mb-0"><i class="fas fa-list me-2"></i>Aktif IP Yasakları</h5>
            <span class="badge bg-secondary">Toplam <?= count($ipBans) ?> kayıt</span>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>IP Adresi</th>
                            <th>Neden</th>
                            <th>Oluşturan</th>
                            <th>Bitiş</th>
                            <th>Kayıt Tarihi</th>
                            <th class="text-end">İşlem</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($ipBans)): ?>
                        <tr>
                            <td colspan="7" class="text-center text-muted py-4">Aktif IP yasağı bulunmuyor.</td>
                        </tr>
                        <?php else: ?>
                            <?php foreach ($ipBans as $ban): ?>
                            <?php
                                $expires = $ban['expires_at'] ?? null;
                                $isActive = !$expires || strtotime((string)$expires) > time();
                            ?>
                            <tr class="<?= $isActive ? '' : 'table-secondary' ?>">
                                <td>#<?= (int)$ban['id'] ?></td>
                                <td><strong><?= sanitize($ban['ip_address']) ?></strong></td>
                                <td>
                                    <?php if (!empty($ban['reason'])): ?>
                                        <span class="badge bg-danger-subtle text-danger"><?= sanitize($ban['reason']) ?></span>
                                    <?php endif; ?>
                                    <?php if (!empty($ban['notes'])): ?>
                                        <div class="text-muted small"><?= sanitize($ban['notes']) ?></div>
                                    <?php endif; ?>
                                </td>
                                <td><?= $ban['admin_username'] ? sanitize($ban['admin_username']) : '<span class="text-muted">Sistem</span>' ?></td>
                                <td>
                                    <?php if ($expires): ?>
                                        <?= date('d.m.Y H:i', strtotime($expires)) ?>
                                        <?php if (!$isActive): ?><span class="badge bg-secondary ms-1">Süresi Doldu</span><?php endif; ?>
                                    <?php else: ?>
                                        <span class="badge bg-primary">Süresiz</span>
                                    <?php endif; ?>
                                </td>
                                <td><?= date('d.m.Y H:i', strtotime($ban['created_at'])) ?></td>
                                <td class="text-end">
                                    <form method="post" class="d-inline" onsubmit="return confirm('Bu IP yasağını kaldırmak istediğinize emin misiniz?');">
                                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                        <input type="hidden" name="action" value="delete_ip_ban">
                                        <input type="hidden" name="ban_id" value="<?= (int)$ban['id'] ?>">
                                        <button type="submit" class="btn btn-sm btn-outline-success">
                                            <i class="fas fa-unlock"></i>
                                        </button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-3">
            <h5 class="mb-0"><i class="fas fa-mobile-screen-button me-2"></i>Aktif Cihaz Yasakları</h5>
            <span class="badge bg-secondary">Toplam <?= count($deviceBans) ?> kayıt</span>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Cihaz ID</th>
                            <th>Neden</th>
                            <th>Oluşturan</th>
                            <th>Bitiş</th>
                            <th>Kayıt Tarihi</th>
                            <th class="text-end">İşlem</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($deviceBans)): ?>
                        <tr>
                            <td colspan="7" class="text-center text-muted py-4">Aktif cihaz yasağı bulunmuyor.</td>
                        </tr>
                        <?php else: ?>
                            <?php foreach ($deviceBans as $ban): ?>
                            <?php
                                $expires = $ban['expires_at'] ?? null;
                                $isActive = !$expires || strtotime((string)$expires) > time();
                            ?>
                            <tr class="<?= $isActive ? '' : 'table-secondary' ?>">
                                <td>#<?= (int)$ban['id'] ?></td>
                                <td><code><?= sanitize($ban['device_id']) ?></code></td>
                                <td>
                                    <?php if (!empty($ban['reason'])): ?>
                                        <span class="badge bg-warning text-dark"><?= sanitize($ban['reason']) ?></span>
                                    <?php endif; ?>
                                    <?php if (!empty($ban['notes'])): ?>
                                        <div class="text-muted small"><?= sanitize($ban['notes']) ?></div>
                                    <?php endif; ?>
                                </td>
                                <td><?= $ban['admin_username'] ? sanitize($ban['admin_username']) : '<span class="text-muted">Sistem</span>' ?></td>
                                <td>
                                    <?php if ($expires): ?>
                                        <?= date('d.m.Y H:i', strtotime($expires)) ?>
                                        <?php if (!$isActive): ?><span class="badge bg-secondary ms-1">Süresi Doldu</span><?php endif; ?>
                                    <?php else: ?>
                                        <span class="badge bg-primary">Süresiz</span>
                                    <?php endif; ?>
                                </td>
                                <td><?= date('d.m.Y H:i', strtotime($ban['created_at'])) ?></td>
                                <td class="text-end">
                                    <form method="post" class="d-inline" onsubmit="return confirm('Bu cihaz yasağını kaldırmak istediğinize emin misiniz?');">
                                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                        <input type="hidden" name="action" value="delete_device_ban">
                                        <input type="hidden" name="ban_id" value="<?= (int)$ban['id'] ?>">
                                        <button type="submit" class="btn btn-sm btn-outline-success">
                                            <i class="fas fa-unlock"></i>
                                        </button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

