<?php
require_once '../config.php';
requireAdmin();

$pageTitle = 'Ödeme Ayarları';

$paytrSettings = [
    'enabled' => getBoolSetting('paytr_enabled', true),
    'merchant_id' => getSetting('paytr_merchant_id', PAYTR_MERCHANT_ID),
    'merchant_key' => getSetting('paytr_merchant_key', PAYTR_MERCHANT_KEY),
    'merchant_salt' => getSetting('paytr_merchant_salt', PAYTR_MERCHANT_SALT),
    'test_mode' => getBoolSetting('paytr_test_mode', PAYTR_TEST_MODE),
    'commission' => (float)getCommissionRate('paytr'),
    'ok_url' => getSetting('paytr_ok_url', SITE_URL . '/balance.php?payment=success'),
    'fail_url' => getSetting('paytr_fail_url', SITE_URL . '/balance.php?payment=failed'),
    'callback_url' => getSetting('paytr_callback_url', SITE_URL . '/payment/paytr-callback.php'),
];

$shopierSettings = [
    'enabled' => getBoolSetting('shopier_enabled', true),
    'api_key' => getSetting('shopier_api_key', SHOPIER_API_KEY),
    'api_secret' => getSetting('shopier_api_secret', SHOPIER_API_SECRET),
    'website_index' => getSetting('shopier_website_index', '0'),
    'test_mode' => getBoolSetting('shopier_test_mode', SHOPIER_TEST_MODE),
    'commission' => (float)getCommissionRate('shopier'),
];

$bankTransferSettings = [
    'enabled' => getBoolSetting('bank_transfer_enabled', false),
    'bank_name' => getSetting('bank_transfer_bank_name', ''),
    'account_holder' => getSetting('bank_transfer_account_holder', ''),
    'iban' => getSetting('bank_transfer_iban', ''),
    'commission' => (float)getCommissionRate('bank_transfer'),
];

$errors = [];

$normalizePercentage = static function ($value): float {
    $value = str_replace(',', '.', (string)$value);
    $number = (float)$value;
    if (!is_finite($number) || $number < 0) {
        return 0.0;
    }
    return round($number, 2);
};

$normalizeUrl = static function (?string $value, string $fallback, string $label, array &$errors): string {
    $value = trim((string)$value);
    if ($value === '') {
        return $fallback;
    }
    if (!filter_var($value, FILTER_VALIDATE_URL)) {
        $errors[] = $label . ' için geçerli bir URL giriniz.';
        return $fallback;
    }
    return $value;
};

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Güvenlik doğrulaması başarısız. Lütfen formu yeniden gönderin.';
    } else {
        $paytrSettings['enabled'] = isset($_POST['paytr_enabled']);
        $paytrSettings['merchant_id'] = trim($_POST['paytr_merchant_id'] ?? $paytrSettings['merchant_id']);
        $paytrSettings['merchant_key'] = trim($_POST['paytr_merchant_key'] ?? $paytrSettings['merchant_key']);
        $paytrSettings['merchant_salt'] = trim($_POST['paytr_merchant_salt'] ?? $paytrSettings['merchant_salt']);
        $paytrSettings['test_mode'] = isset($_POST['paytr_test_mode']);
        $paytrSettings['commission'] = $normalizePercentage($_POST['paytr_commission_rate'] ?? $paytrSettings['commission']);
        $paytrSettings['ok_url'] = $normalizeUrl($_POST['paytr_ok_url'] ?? $paytrSettings['ok_url'], $paytrSettings['ok_url'], 'PayTR başarılı dönüş adresi', $errors);
        $paytrSettings['fail_url'] = $normalizeUrl($_POST['paytr_fail_url'] ?? $paytrSettings['fail_url'], $paytrSettings['fail_url'], 'PayTR başarısız dönüş adresi', $errors);
        $paytrSettings['callback_url'] = $normalizeUrl($_POST['paytr_callback_url'] ?? $paytrSettings['callback_url'], $paytrSettings['callback_url'], 'PayTR callback adresi', $errors);

        $shopierSettings['enabled'] = isset($_POST['shopier_enabled']);
        $shopierSettings['api_key'] = trim($_POST['shopier_api_key'] ?? $shopierSettings['api_key']);
        $shopierSettings['api_secret'] = trim($_POST['shopier_api_secret'] ?? $shopierSettings['api_secret']);
        $shopierSettings['website_index'] = trim($_POST['shopier_website_index'] ?? $shopierSettings['website_index']);
        $shopierSettings['test_mode'] = isset($_POST['shopier_test_mode']);
        $shopierSettings['commission'] = $normalizePercentage($_POST['shopier_commission_rate'] ?? $shopierSettings['commission']);

        $bankTransferSettings['enabled'] = isset($_POST['bank_transfer_enabled']);
        $bankTransferSettings['bank_name'] = trim($_POST['bank_transfer_bank_name'] ?? $bankTransferSettings['bank_name']);
        $bankTransferSettings['account_holder'] = trim($_POST['bank_transfer_account_holder'] ?? $bankTransferSettings['account_holder']);
        $bankTransferSettings['iban'] = trim($_POST['bank_transfer_iban'] ?? $bankTransferSettings['iban']);
        $bankTransferSettings['commission'] = $normalizePercentage($_POST['bank_transfer_commission_rate'] ?? $bankTransferSettings['commission']);

        if ($paytrSettings['merchant_id'] === '' || $paytrSettings['merchant_key'] === '' || $paytrSettings['merchant_salt'] === '') {
            $errors[] = 'PayTR için Merchant ID, Key ve Salt alanları zorunludur.';
        }
        if ($shopierSettings['api_key'] === '' || $shopierSettings['api_secret'] === '') {
            $errors[] = 'Shopier için API Key ve Secret alanları zorunludur.';
        }
        if ($bankTransferSettings['enabled'] && ($bankTransferSettings['iban'] === '' || $bankTransferSettings['account_holder'] === '')) {
            $errors[] = 'Havale/EFT için IBAN ve hesap sahibi bilgileri zorunludur.';
        }

        if (empty($errors)) {
            try {
                $pdo->beginTransaction();

                saveSetting('paytr_enabled', $paytrSettings['enabled'] ? '1' : '0', false);
                saveSetting('paytr_merchant_id', $paytrSettings['merchant_id'], false);
                saveSetting('paytr_merchant_key', $paytrSettings['merchant_key'], false);
                saveSetting('paytr_merchant_salt', $paytrSettings['merchant_salt'], false);
                saveSetting('paytr_test_mode', $paytrSettings['test_mode'] ? '1' : '0', false);
                saveSetting('paytr_ok_url', $paytrSettings['ok_url'], false);
                saveSetting('paytr_fail_url', $paytrSettings['fail_url'], false);
                saveSetting('paytr_callback_url', $paytrSettings['callback_url'], false);
                saveSetting('paytr_commission_rate', (string)$paytrSettings['commission'], false);

                saveSetting('shopier_enabled', $shopierSettings['enabled'] ? '1' : '0', false);
                saveSetting('shopier_api_key', $shopierSettings['api_key'], false);
                saveSetting('shopier_api_secret', $shopierSettings['api_secret'], false);
                saveSetting('shopier_website_index', $shopierSettings['website_index'], false);
                saveSetting('shopier_test_mode', $shopierSettings['test_mode'] ? '1' : '0', false);
                saveSetting('shopier_commission_rate', (string)$shopierSettings['commission'], false);

                saveSetting('bank_transfer_enabled', $bankTransferSettings['enabled'] ? '1' : '0', false);
                saveSetting('bank_transfer_bank_name', $bankTransferSettings['bank_name'], false);
                saveSetting('bank_transfer_account_holder', $bankTransferSettings['account_holder'], false);
                saveSetting('bank_transfer_iban', $bankTransferSettings['iban'], false);
                saveSetting('bank_transfer_commission_rate', (string)$bankTransferSettings['commission'], false);

                $pdo->commit();
            } catch (Throwable $th) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $errors[] = 'Ödeme ayarları kaydedilemedi: ' . $th->getMessage();
            }
        }

        if (empty($errors)) {
            loadSettings(true);
            flashMessage('success', 'Ödeme ayarları başarıyla güncellendi.');
            redirect('settings-payments.php');
        }
    }
}

include 'includes/admin-header.php';
?>

<div class="admin-page settings-page">
    <?php if (!empty($errors)): ?>
        <div class="admin-alerts">
            <?php foreach ($errors as $error): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i><?= sanitize($error) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <form method="post" class="card">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fab fa-cc-paypal me-2 text-primary"></i>PayTR Ayarları</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">PayTR ödeme entegrasyonu için gerekli bilgileri girin</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-4">
                <div class="col-12">
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" id="paytrEnabled" name="paytr_enabled" <?= $paytrSettings['enabled'] ? 'checked' : '' ?>>
                        <label class="form-check-label fw-semibold" for="paytrEnabled">
                            <i class="fas fa-power-off me-1"></i>PayTR Ödeme Sistemini Aktif Et
                        </label>
                    </div>
                    <small class="text-muted d-block mt-2">Aktif olduğunda kullanıcılar bakiye yüklerken PayTR seçeneğini görebilir</small>
                </div>
                <div class="col-lg-4">
                    <label class="form-label">Merchant ID *</label>
                    <input type="text" name="paytr_merchant_id" class="form-control" required value="<?= sanitize($paytrSettings['merchant_id']) ?>">
                    <small class="text-muted">PayTR panelinden alınan Merchant ID</small>
                </div>
                <div class="col-lg-4">
                    <label class="form-label">Merchant Key *</label>
                    <input type="text" name="paytr_merchant_key" class="form-control" required value="<?= sanitize($paytrSettings['merchant_key']) ?>">
                    <small class="text-muted">PayTR panelinden alınan Merchant Key</small>
                </div>
                <div class="col-lg-4">
                    <label class="form-label">Merchant Salt *</label>
                    <input type="text" name="paytr_merchant_salt" class="form-control" required value="<?= sanitize($paytrSettings['merchant_salt']) ?>">
                    <small class="text-muted">PayTR panelinden alınan Merchant Salt</small>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Başarılı Dönüş URL</label>
                    <input type="url" name="paytr_ok_url" class="form-control" value="<?= sanitize($paytrSettings['ok_url']) ?>">
                    <small class="text-muted">Ödeme başarılı olduğunda yönlendirilecek sayfa</small>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Başarısız Dönüş URL</label>
                    <input type="url" name="paytr_fail_url" class="form-control" value="<?= sanitize($paytrSettings['fail_url']) ?>">
                    <small class="text-muted">Ödeme başarısız olduğunda yönlendirilecek sayfa</small>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Callback URL</label>
                    <input type="url" name="paytr_callback_url" class="form-control" value="<?= sanitize($paytrSettings['callback_url']) ?>">
                    <small class="text-muted">Ödeme sonucu bildirimi için callback adresi</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Komisyon Oranı (%)</label>
                    <input type="number" step="0.01" min="0" name="paytr_commission_rate" class="form-control" value="<?= htmlspecialchars((string)$paytrSettings['commission'], ENT_QUOTES, 'UTF-8') ?>">
                    <small class="text-muted">Her işlemden alınacak komisyon yüzdesi</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Test Modu</label>
                    <div class="form-check form-switch mt-2">
                        <input class="form-check-input" type="checkbox" id="paytrTestMode" name="paytr_test_mode" <?= $paytrSettings['test_mode'] ? 'checked' : '' ?>>
                        <label class="form-check-label" for="paytrTestMode">Test modunu etkinleştir</label>
                    </div>
                    <small class="text-muted d-block mt-2">Canlı ortamda kapatılmalıdır</small>
                </div>
            </div>
        </div>
        <div class="card-footer d-flex justify-content-end gap-2 bg-transparent border-top">
            <button type="reset" class="btn btn-secondary">
                <i class="fas fa-undo me-2"></i>Sıfırla
            </button>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save me-2"></i>PayTR Ayarlarını Kaydet
            </button>
        </div>
    </form>

    <form method="post" class="card">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fas fa-shopping-cart me-2 text-success"></i>Shopier Ayarları</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">Shopier ödeme entegrasyonunu yapılandırın</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-4">
                <div class="col-12">
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" id="shopierEnabled" name="shopier_enabled" <?= $shopierSettings['enabled'] ? 'checked' : '' ?>>
                        <label class="form-check-label fw-semibold" for="shopierEnabled">
                            <i class="fas fa-power-off me-1"></i>Shopier Ödeme Sistemini Aktif Et
                        </label>
                    </div>
                    <small class="text-muted d-block mt-2">Aktif olduğunda kullanıcılar bakiye yüklerken Shopier seçeneğini görebilir</small>
                </div>
                <div class="col-lg-5">
                    <label class="form-label">API Key *</label>
                    <input type="text" name="shopier_api_key" class="form-control" required value="<?= sanitize($shopierSettings['api_key']) ?>">
                    <small class="text-muted">Shopier panelinden alınan API Key</small>
                </div>
                <div class="col-lg-5">
                    <label class="form-label">API Secret *</label>
                    <input type="text" name="shopier_api_secret" class="form-control" required value="<?= sanitize($shopierSettings['api_secret']) ?>">
                    <small class="text-muted">Shopier panelinden alınan API Secret</small>
                </div>
                <div class="col-lg-2">
                    <label class="form-label">Website Index</label>
                    <input type="text" name="shopier_website_index" class="form-control" value="<?= sanitize($shopierSettings['website_index']) ?>">
                    <small class="text-muted">Varsayılan: 0</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Komisyon Oranı (%)</label>
                    <input type="number" step="0.01" min="0" name="shopier_commission_rate" class="form-control" value="<?= htmlspecialchars((string)$shopierSettings['commission'], ENT_QUOTES, 'UTF-8') ?>">
                    <small class="text-muted">Her işlemden alınacak komisyon yüzdesi</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Test Modu</label>
                    <div class="form-check form-switch mt-2">
                        <input class="form-check-input" type="checkbox" id="shopierTestMode" name="shopier_test_mode" <?= $shopierSettings['test_mode'] ? 'checked' : '' ?>>
                        <label class="form-check-label" for="shopierTestMode">Test modunu etkinleştir</label>
                    </div>
                    <small class="text-muted d-block mt-2">Canlı ortamda kapatılmalıdır</small>
                </div>
            </div>
        </div>
        <div class="card-footer d-flex justify-content-end gap-2 bg-transparent border-top">
            <button type="reset" class="btn btn-secondary">
                <i class="fas fa-undo me-2"></i>Sıfırla
            </button>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save me-2"></i>Shopier Ayarlarını Kaydet
            </button>
        </div>
    </form>

    <form method="post" class="card">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fas fa-university me-2 text-warning"></i>Havale / EFT Ayarları</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">Banka hesap bilgilerinizi girin ve manuel ödeme yöntemini aktifleştirin</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-4">
                <div class="col-12">
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" id="bankTransferEnabled" name="bank_transfer_enabled" <?= $bankTransferSettings['enabled'] ? 'checked' : '' ?>>
                        <label class="form-check-label fw-semibold" for="bankTransferEnabled">
                            <i class="fas fa-power-off me-1"></i>Havale/EFT Ödeme Yöntemini Aktif Et
                        </label>
                    </div>
                    <small class="text-muted d-block mt-2">Aktif olduğunda kullanıcılar bakiye yüklerken Havale/EFT seçeneğini görebilir</small>
                </div>
                <div class="col-lg-4">
                    <label class="form-label">Banka Adı</label>
                    <input type="text" name="bank_transfer_bank_name" class="form-control" value="<?= sanitize($bankTransferSettings['bank_name']) ?>" placeholder="Örn: Ziraat Bankası">
                    <small class="text-muted">Kullanıcılara gösterilecek banka adı</small>
                </div>
                <div class="col-lg-4">
                    <label class="form-label">Hesap Sahibi <span class="text-danger">*</span></label>
                    <input type="text" name="bank_transfer_account_holder" class="form-control" value="<?= sanitize($bankTransferSettings['account_holder']) ?>" placeholder="Örn: AHMET YILMAZ" required>
                    <small class="text-muted">IBAN'a kayıtlı isim soyisim</small>
                </div>
                <div class="col-lg-4">
                    <label class="form-label">IBAN Numarası <span class="text-danger">*</span></label>
                    <input type="text" name="bank_transfer_iban" class="form-control" value="<?= sanitize($bankTransferSettings['iban']) ?>" placeholder="TR00 0000 0000 0000 0000 0000 00" required maxlength="32">
                    <small class="text-muted">Havale yapılacak IBAN numarası</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Komisyon Oranı (%)</label>
                    <input type="number" step="0.01" min="0" name="bank_transfer_commission_rate" class="form-control" value="<?= htmlspecialchars((string)$bankTransferSettings['commission'], ENT_QUOTES, 'UTF-8') ?>">
                    <small class="text-muted">Manuel havale işlemlerinden alınacak komisyon yüzdesi (varsayılan: 0)</small>
                </div>
                <div class="col-12">
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Bilgi:</strong> Kullanıcı havale/EFT seçeneğini seçtiğinde yukarıdaki bilgiler gösterilecek ve ödeme yaptıktan sonra admin onayı için bekleyecektir.
                    </div>
                </div>
            </div>
        </div>
        <div class="card-footer d-flex justify-content-end gap-2 bg-transparent border-top">
            <button type="reset" class="btn btn-secondary">
                <i class="fas fa-undo me-2"></i>Sıfırla
            </button>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save me-2"></i>Havale/EFT Ayarlarını Kaydet
            </button>
        </div>
    </form>
</div>

<?php include 'includes/admin-footer.php'; ?>
