<?php
require_once '../config.php';
requireAdmin();

$pageTitle = 'SEO & Indexleme';

$robotsTxtPath = __DIR__ . '/../robots.txt';
$robotsTxtSetting = getSetting('seo_robots_content', '');
$robotsTxtContent = $robotsTxtSetting !== '' ? (string)$robotsTxtSetting : (is_file($robotsTxtPath) ? (string)file_get_contents($robotsTxtPath) : defaultRobotsTxt());
$robotsTxtContent = str_replace(["\r\n", "\r"], "\n", $robotsTxtContent);

$sitemapOptions = [
    'home' => getBoolSetting('sitemap_include_home', true),
    'products' => getBoolSetting('sitemap_include_products', true),
    'categories' => getBoolSetting('sitemap_include_categories', true),
    'pages' => getBoolSetting('sitemap_include_pages', true),
    'blog' => getBoolSetting('sitemap_include_blog', true),
    'marketplace' => getBoolSetting('sitemap_include_marketplace', true),
    'marketplace_items' => getBoolSetting('sitemap_include_marketplace_items', true),
    'support' => getBoolSetting('sitemap_include_support', true),
];

$sitemapPath = __DIR__ . '/../sitemap.xml';
$sitemapLastBuilt = is_file($sitemapPath) ? filemtime($sitemapPath) : null;
$robotsLastUpdated = is_file($robotsTxtPath) ? filemtime($robotsTxtPath) : null;

$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $formAction = $_POST['form_action'] ?? '';
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Güvenlik doğrulaması başarısız. Lütfen formu yeniden gönderin.';
    } elseif ($formAction === 'save_robots') {
        $robotsTxtContent = isset($_POST['robots_txt_content']) ? str_replace(["\r\n", "\r"], "\n", (string)$_POST['robots_txt_content']) : $robotsTxtContent;
        $robotsTxtContent = trim($robotsTxtContent);
        if ($robotsTxtContent === '') {
            $robotsTxtContent = defaultRobotsTxt();
        }
        if (!str_ends_with($robotsTxtContent, "\n")) {
            $robotsTxtContent .= "\n";
        }

        try {
            saveSetting('seo_robots_content', $robotsTxtContent, false);
            if (@file_put_contents($robotsTxtPath, $robotsTxtContent, LOCK_EX) === false) {
                throw new RuntimeException('robots.txt dosyası yazılamadı. Dosya izinlerini kontrol edin.');
            }
            loadSettings(true);
            $robotsLastUpdated = time();
            flashMessage('success', 'robots.txt içeriği güncellendi.');
            redirect('settings-seo.php');
        } catch (Throwable $th) {
            $errors[] = 'robots.txt kaydedilemedi: ' . $th->getMessage();
        }
    } elseif ($formAction === 'generate_sitemap') {
        $sitemapOptions['home'] = isset($_POST['sitemap_include_home']);
        $sitemapOptions['products'] = isset($_POST['sitemap_include_products']);
        $sitemapOptions['categories'] = isset($_POST['sitemap_include_categories']);
        $sitemapOptions['pages'] = isset($_POST['sitemap_include_pages']);
        $sitemapOptions['blog'] = isset($_POST['sitemap_include_blog']);
        $sitemapOptions['marketplace'] = isset($_POST['sitemap_include_marketplace']);
        $sitemapOptions['marketplace_items'] = isset($_POST['sitemap_include_marketplace_items']);
        $sitemapOptions['support'] = isset($_POST['sitemap_include_support']);

        try {
            $pdo->beginTransaction();
            saveSetting('sitemap_include_home', $sitemapOptions['home'] ? '1' : '0', false);
            saveSetting('sitemap_include_products', $sitemapOptions['products'] ? '1' : '0', false);
            saveSetting('sitemap_include_categories', $sitemapOptions['categories'] ? '1' : '0', false);
            saveSetting('sitemap_include_pages', $sitemapOptions['pages'] ? '1' : '0', false);
            saveSetting('sitemap_include_blog', $sitemapOptions['blog'] ? '1' : '0', false);
            saveSetting('sitemap_include_marketplace', $sitemapOptions['marketplace'] ? '1' : '0', false);
            saveSetting('sitemap_include_marketplace_items', $sitemapOptions['marketplace_items'] ? '1' : '0', false);
            saveSetting('sitemap_include_support', $sitemapOptions['support'] ? '1' : '0', false);
            $pdo->commit();
        } catch (Throwable $th) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $errors[] = 'Sitemap ayarları kaydedilemedi: ' . $th->getMessage();
        }

        if (empty($errors)) {
            $result = generateSitemapXml($pdo, $sitemapOptions);
            if (empty($result['count'])) {
                flashMessage('warning', 'Sitemap oluşturulacak URL bulunamadı.');
            } else {
                if (@file_put_contents($sitemapPath, $result['xml'], LOCK_EX) === false) {
                    $errors[] = 'sitemap.xml dosyası yazılamadı. Dosya izinlerini kontrol edin.';
                } else {
                    $sitemapLastBuilt = time();
                    flashMessage('success', 'Sitemap başarıyla oluşturuldu (' . $result['count'] . ' URL).');
                }
            }

            if (empty($errors)) {
                redirect('settings-seo.php');
            }
        }
    }
}

include 'includes/admin-header.php';
?>

<div class="admin-page settings-page">
    <?php if (!empty($errors)): ?>
        <div class="admin-alerts">
            <?php foreach ($errors as $error): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i><?= sanitize($error) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <form method="post" class="glass-surface">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <input type="hidden" name="form_action" value="save_robots">
        <div class="panel-head">
            <div>
                <h5>robots.txt İçeriği</h5>
                <p class="panel-subtitle">Arama motorlarının indekslemesini özelleştirin</p>
            </div>
            <div class="page-intro__actions">
                <button type="submit" class="btn btn-primary"><i class="fas fa-save me-2"></i>robots.txt Kaydet</button>
            </div>
        </div>
        <div class="mb-3">
            <label class="form-label">robots.txt</label>
            <textarea name="robots_txt_content" rows="12" class="form-control" spellcheck="false"><?= htmlspecialchars($robotsTxtContent, ENT_QUOTES, 'UTF-8') ?></textarea>
        </div>
        <?php if ($robotsLastUpdated): ?>
            <p class="text-muted small mb-0">Son güncelleme: <?= date('d.m.Y H:i', $robotsLastUpdated) ?></p>
        <?php endif; ?>
    </form>

    <form method="post" class="glass-surface">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <input type="hidden" name="form_action" value="generate_sitemap">
        <div class="panel-head">
            <div>
                <h5>Sitemap Oluştur</h5>
                <p class="panel-subtitle">Site haritasına dahil edilecek içerikleri seçin</p>
            </div>
            <div class="page-intro__actions">
                <button type="submit" class="btn btn-primary"><i class="fas fa-sitemap me-2"></i>Sitemap Oluştur</button>
            </div>
        </div>
        <div class="row g-3">
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="sitemapHome" name="sitemap_include_home" <?= $sitemapOptions['home'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="sitemapHome">Ana sayfa</label>
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="sitemapProducts" name="sitemap_include_products" <?= $sitemapOptions['products'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="sitemapProducts">Ürünler</label>
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="sitemapCategories" name="sitemap_include_categories" <?= $sitemapOptions['categories'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="sitemapCategories">Kategoriler</label>
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="sitemapPages" name="sitemap_include_pages" <?= $sitemapOptions['pages'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="sitemapPages">Statik sayfalar</label>
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="sitemapBlog" name="sitemap_include_blog" <?= $sitemapOptions['blog'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="sitemapBlog">Blog yazıları</label>
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="sitemapMarket" name="sitemap_include_marketplace" <?= $sitemapOptions['marketplace'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="sitemapMarket">Marketplace içerikleri</label>
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="sitemapMarketItems" name="sitemap_include_marketplace_items" <?= $sitemapOptions['marketplace_items'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="sitemapMarketItems">Marketplace ürünleri</label>
                </div>
            </div>
            <div class="col-md-4">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="sitemapSupport" name="sitemap_include_support" <?= $sitemapOptions['support'] ? 'checked' : '' ?>>
                    <label class="form-check-label" for="sitemapSupport">Destek merkezi</label>
                </div>
            </div>
        </div>
        <?php if ($sitemapLastBuilt): ?>
            <p class="text-muted small mt-3 mb-0">Son oluşturulma: <?= date('d.m.Y H:i', $sitemapLastBuilt) ?></p>
        <?php endif; ?>
    </form>
</div>

<?php include 'includes/admin-footer.php'; ?>
