<?php
require_once '../config.php';
requireAdmin();

$pageTitle = 'Site Ayarları';

$generalSettings = [
    'site_name' => siteName(),
    'site_description' => siteDescription(),
    'site_keywords' => siteKeywords(),
    'site_base_url' => siteBaseUrl(),
];

$brandingSettings = [
    'mode' => strtolower((string)getSetting('branding_logo_mode', 'text')),
    'text' => getSetting('branding_logo_text', ''),
    'tagline' => getSetting('branding_logo_tagline', ''),
    'logo' => getSetting('branding_logo_image', ''),
    'favicon' => getSetting('branding_favicon_image', ''),
    'width' => (int)getSetting('branding_logo_width', 0),
    'height' => (int)getSetting('branding_logo_height', 0),
];
if (!in_array($brandingSettings['mode'], ['text', 'image'], true)) {
    $brandingSettings['mode'] = 'text';
}

$headerSettings = [
    'top_phone' => getSetting('header_top_phone', ''),
    'top_email' => getSetting('header_top_email', ''),
];

$smtpSettings = [
    'host' => getSetting('smtp_host', SMTP_HOST),
    'port' => (int)getSetting('smtp_port', SMTP_PORT),
    'username' => getSetting('smtp_username', SMTP_USERNAME),
    'password' => getSetting('smtp_password', SMTP_PASSWORD),
    'encryption' => getSetting('smtp_encryption', SMTP_ENCRYPTION),
    'from_email' => getSetting('smtp_from_email', SMTP_FROM_EMAIL),
    'from_name' => getSetting('smtp_from_name', SMTP_FROM_NAME),
    'test_mode' => getBoolSetting('smtp_test_mode', SMTP_TEST_MODE),
    'base_url' => getSetting('email_base_url', EMAIL_BASE_URL),
    'reset_template' => getSetting('email_template_password_reset', ''),
];

$recaptchaSettings = [
    'site_key' => getSetting('recaptcha_site_key', RECAPTCHA_SITE_KEY),
    'secret_key' => getSetting('recaptcha_secret_key', RECAPTCHA_SECRET_KEY),
    'enable_login' => getBoolSetting('recaptcha_enable_login', false),
    'enable_register' => getBoolSetting('recaptcha_enable_register', false),
    'enable_forgot' => getBoolSetting('recaptcha_enable_forgot', false),
    'enable_admin_login' => getBoolSetting('recaptcha_enable_admin_login', false),
];

$emailVerificationRequired = getBoolSetting('auth_require_email_verification', false);

$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $formAction = $_POST['form_action'] ?? 'save';
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Güvenlik doğrulaması başarısız. Lütfen formu yeniden gönderin.';
    } elseif ($formAction === 'smtp_test') {
        $testEmail = trim($_POST['smtp_test_email'] ?? '');
        if ($testEmail === '' || !filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Geçerli bir test e-posta adresi giriniz.';
        } else {
            $sent = sendEmail($testEmail, 'SMTP Testi', '<p>Merhaba,<br>SMTP ayarlarınız başarıyla çalışıyor.</p>', 'SMTP ayar test mesajı');
            if ($sent) {
                flashMessage('success', 'Test e-postası başarıyla gönderildi. Gelen kutunuzu kontrol edin.');
            } else {
                $errorDetails = getLastEmailError();
                $message = 'Test e-postası gönderilemedi. SMTP ayarlarınızı kontrol edin.';
                if ($errorDetails !== '') {
                    $message .= ' (' . $errorDetails . ')';
                }
                flashMessage('danger', $message);
            }
            redirect('settings-site.php');
        }
    } else {
        // Sadece POST'ta varsa güncelle (form bazlı kontrol)
        if (isset($_POST['site_name'])) {
            $generalSettings['site_name'] = trim($_POST['site_name'] ?? '');
            if ($generalSettings['site_name'] === '') {
                $errors[] = 'Site adı zorunludur.';
            }
        }
        if (isset($_POST['site_description'])) {
            $generalSettings['site_description'] = trim($_POST['site_description'] ?? $generalSettings['site_description']);
        }
        if (isset($_POST['site_keywords'])) {
            $generalSettings['site_keywords'] = trim($_POST['site_keywords'] ?? $generalSettings['site_keywords']);
        }
        if (isset($_POST['site_base_url'])) {
            $generalSettings['site_base_url'] = trim($_POST['site_base_url'] ?? $generalSettings['site_base_url']);
            if ($generalSettings['site_base_url'] === '') {
                $generalSettings['site_base_url'] = SITE_URL;
            }
            if (!filter_var($generalSettings['site_base_url'], FILTER_VALIDATE_URL)) {
                $errors[] = 'Site temel URL alanı için geçerli bir adres girin.';
            } else {
                $generalSettings['site_base_url'] = rtrim($generalSettings['site_base_url'], '/');
            }
        }

        $brandingSettings['mode'] = isset($_POST['branding_logo_mode']) && $_POST['branding_logo_mode'] === 'image' ? 'image' : 'text';
        $brandingSettings['text'] = trim($_POST['branding_logo_text'] ?? $brandingSettings['text']);
        $brandingSettings['tagline'] = trim($_POST['branding_logo_tagline'] ?? $brandingSettings['tagline']);
        $brandingSettings['width'] = isset($_POST['branding_logo_width']) ? max(0, (int)$_POST['branding_logo_width']) : $brandingSettings['width'];
        $brandingSettings['height'] = isset($_POST['branding_logo_height']) ? max(0, (int)$_POST['branding_logo_height']) : $brandingSettings['height'];

        try {
            if (isset($_FILES['branding_logo_image']) && $_FILES['branding_logo_image']['error'] !== UPLOAD_ERR_NO_FILE) {
                $brandingSettings['logo'] = uploadImageFile($_FILES['branding_logo_image'], 'branding', $brandingSettings['logo']);
            }
        } catch (Throwable $th) {
            $errors[] = 'Logo yüklenirken hata oluştu: ' . $th->getMessage();
        }

        if (isset($_POST['remove_branding_logo_image']) && $_POST['remove_branding_logo_image'] === '1') {
            deleteUploadedFile($brandingSettings['logo']);
            $brandingSettings['logo'] = '';
        }

        try {
            if (isset($_FILES['branding_favicon']) && $_FILES['branding_favicon']['error'] !== UPLOAD_ERR_NO_FILE) {
                $brandingSettings['favicon'] = uploadImageFile($_FILES['branding_favicon'], 'branding', $brandingSettings['favicon']);
            }
        } catch (Throwable $th) {
            $errors[] = 'Favicon yüklenirken hata oluştu: ' . $th->getMessage();
        }

        if (isset($_POST['remove_branding_favicon']) && $_POST['remove_branding_favicon'] === '1') {
            deleteUploadedFile($brandingSettings['favicon']);
            $brandingSettings['favicon'] = '';
        }

        $headerSettings['top_phone'] = trim($_POST['header_top_phone'] ?? $headerSettings['top_phone']);
        $headerSettings['top_email'] = trim($_POST['header_top_email'] ?? $headerSettings['top_email']);
        if ($headerSettings['top_email'] !== '' && !filter_var($headerSettings['top_email'], FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Header e-posta alanı için geçerli bir adres girin.';
        }

        $smtpSettings['host'] = trim($_POST['smtp_host'] ?? $smtpSettings['host']);
        $smtpSettings['port'] = (int)($_POST['smtp_port'] ?? $smtpSettings['port']);
        $smtpSettings['username'] = trim($_POST['smtp_username'] ?? $smtpSettings['username']);
        // Şifre alanı boşsa mevcut şifreyi koru (password input'lar boş gelir)
        $newPassword = trim($_POST['smtp_password'] ?? '');
        if ($newPassword !== '') {
            $smtpSettings['password'] = $newPassword;
        }
        // Eğer şifre boşsa, mevcut şifreyi koru ($smtpSettings['password'] zaten mevcut değeri içeriyor)
        $smtpSettings['encryption'] = strtolower(trim($_POST['smtp_encryption'] ?? $smtpSettings['encryption']));
        $smtpSettings['from_email'] = trim($_POST['smtp_from_email'] ?? $smtpSettings['from_email']);
        $smtpSettings['from_name'] = trim($_POST['smtp_from_name'] ?? $smtpSettings['from_name']);
        $smtpSettings['test_mode'] = isset($_POST['smtp_test_mode']);
        $smtpSettings['base_url'] = trim($_POST['email_base_url'] ?? $smtpSettings['base_url']);
        $smtpSettings['reset_template'] = isset($_POST['email_template_password_reset']) ? (string)$_POST['email_template_password_reset'] : $smtpSettings['reset_template'];

        $recaptchaSettings['site_key'] = trim($_POST['recaptcha_site_key'] ?? $recaptchaSettings['site_key']);
        $recaptchaSettings['secret_key'] = trim($_POST['recaptcha_secret_key'] ?? $recaptchaSettings['secret_key']);
        $recaptchaSettings['enable_login'] = isset($_POST['recaptcha_enable_login']);
        $recaptchaSettings['enable_register'] = isset($_POST['recaptcha_enable_register']);
        $recaptchaSettings['enable_forgot'] = isset($_POST['recaptcha_enable_forgot']);
        $recaptchaSettings['enable_admin_login'] = isset($_POST['recaptcha_enable_admin_login']);

        $allowedEnc = ['tls', 'ssl', 'none', ''];
        if (!in_array($smtpSettings['encryption'], $allowedEnc, true)) {
            $smtpSettings['encryption'] = 'tls';
        }
        if ($smtpSettings['encryption'] === 'none') {
            $smtpSettings['encryption'] = '';
        }
        if ($smtpSettings['from_email'] !== '' && !filter_var($smtpSettings['from_email'], FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'SMTP Gönderici e-posta adresi geçerli değil.';
        }
        if ($smtpSettings['base_url'] !== '' && !filter_var($smtpSettings['base_url'], FILTER_VALIDATE_URL)) {
            $errors[] = 'Şifre sıfırlama bağlantısı için geçerli bir temel URL girin.';
        }
        if ($smtpSettings['port'] <= 0) {
            $smtpSettings['port'] = 587;
        }

        $emailVerificationRequired = isset($_POST['auth_require_email_verification']);

        if ($brandingSettings['mode'] === 'text' && $brandingSettings['text'] === '') {
            $brandingSettings['text'] = $generalSettings['site_name'] !== '' ? $generalSettings['site_name'] : siteName();
        }

        if (empty($errors)) {
            try {
                $pdo->beginTransaction();

                saveSetting('site_name', $generalSettings['site_name'], false);
                saveSetting('site_description', $generalSettings['site_description'], false);
                saveSetting('site_keywords', $generalSettings['site_keywords'], false);
                saveSetting('site_base_url', $generalSettings['site_base_url'], false);

                saveSetting('branding_logo_mode', $brandingSettings['mode'], false);
                saveSetting('branding_logo_text', $brandingSettings['text'], false);
                saveSetting('branding_logo_tagline', $brandingSettings['tagline'], false);
                saveSetting('branding_logo_image', $brandingSettings['logo'], false);
                saveSetting('branding_favicon_image', $brandingSettings['favicon'], false);
                saveSetting('branding_logo_width', (string)$brandingSettings['width'], false);
                saveSetting('branding_logo_height', (string)$brandingSettings['height'], false);

                saveSetting('header_top_phone', $headerSettings['top_phone'], false);
                saveSetting('header_top_email', $headerSettings['top_email'], false);

                saveSetting('smtp_host', $smtpSettings['host'], false);
                saveSetting('smtp_port', (string)$smtpSettings['port'], false);
                saveSetting('smtp_username', $smtpSettings['username'], false);
                saveSetting('smtp_password', $smtpSettings['password'], false);
                saveSetting('smtp_encryption', $smtpSettings['encryption'], false);
                saveSetting('smtp_from_email', $smtpSettings['from_email'], false);
                saveSetting('smtp_from_name', $smtpSettings['from_name'], false);
                saveSetting('smtp_test_mode', $smtpSettings['test_mode'] ? '1' : '0', false);
                saveSetting('email_base_url', $smtpSettings['base_url'], false);
                saveSetting('email_template_password_reset', $smtpSettings['reset_template'], false);

                saveSetting('recaptcha_site_key', $recaptchaSettings['site_key'], false);
                saveSetting('recaptcha_secret_key', $recaptchaSettings['secret_key'], false);
                saveSetting('recaptcha_enable_login', $recaptchaSettings['enable_login'] ? '1' : '0', false);
                saveSetting('recaptcha_enable_register', $recaptchaSettings['enable_register'] ? '1' : '0', false);
                saveSetting('recaptcha_enable_forgot', $recaptchaSettings['enable_forgot'] ? '1' : '0', false);
                saveSetting('recaptcha_enable_admin_login', $recaptchaSettings['enable_admin_login'] ? '1' : '0', false);

                saveSetting('auth_require_email_verification', $emailVerificationRequired ? '1' : '0', false);

                // Save show_subcategories_on_home setting
                $showSubcategories = isset($_POST['show_subcategories_on_home']) ? '1' : '0';
                saveSetting('show_subcategories_on_home', $showSubcategories, false);

                $pdo->commit();
            } catch (Throwable $th) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $errors[] = 'Ayarlar kaydedilemedi: ' . $th->getMessage();
            }
        }

        if (empty($errors)) {
            loadSettings(true);
            siteBaseUrl(true);
            siteBasePath(true);
            flashMessage('success', 'Site ayarları başarıyla güncellendi.');
            redirect('settings-site.php');
        }
    }
}

include 'includes/admin-header.php';
?>

<div class="admin-page settings-page">
    <?php if (!empty($errors)): ?>
        <div class="admin-alerts">
            <?php foreach ($errors as $error): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i><?= sanitize($error) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <form method="post" class="card" enctype="multipart/form-data">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <input type="hidden" name="form_action" value="save">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fas fa-globe me-2 text-primary"></i>Genel Bilgiler</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">Site adı, açıklama ve temel URL ayarları</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-4">
                <div class="col-lg-6">
                    <label class="form-label">Site Adı *</label>
                    <input type="text" name="site_name" class="form-control" required value="<?= sanitize($generalSettings['site_name']) ?>">
                    <small class="text-muted">Tarayıcı başlığında ve logoda görünür</small>
                </div>
                <div class="col-lg-6">
                    <label class="form-label">Site Temel URL *</label>
                    <input type="url" name="site_base_url" class="form-control" required value="<?= sanitize($generalSettings['site_base_url']) ?>">
                    <small class="text-muted">Ör: https://orneksite.com</small>
                </div>
                <div class="col-12">
                    <label class="form-label">Site Açıklaması</label>
                    <textarea name="site_description" class="form-control" rows="3"><?= sanitize($generalSettings['site_description']) ?></textarea>
                    <small class="text-muted">Meta açıklama olarak kullanılır (SEO için önemli)</small>
                </div>
                <div class="col-12">
                    <label class="form-label">Site Anahtar Kelimeleri</label>
                    <textarea name="site_keywords" class="form-control" rows="2" placeholder="Kelime1, Kelime2, ..."><?= sanitize($generalSettings['site_keywords']) ?></textarea>
                    <small class="text-muted">Virgülle ayrılmış anahtar kelimeler (SEO için)</small>
                </div>
                
                <div class="col-12">
                    <hr class="my-2">
                    <div class="form-check form-switch">
                        <input class="form-check-input" 
                               type="checkbox" 
                               name="show_subcategories_on_home" 
                               id="show_subcategories_on_home" 
                               <?= getBoolSetting('show_subcategories_on_home', true) ? 'checked' : '' ?>>
                        <label class="form-check-label fw-semibold" for="show_subcategories_on_home">
                            Anasayfada Alt Kategorileri Göster
                        </label>
                    </div>
                    <small class="text-muted d-block mt-1">
                        Açık olduğunda, anasayfa kategoriler bölümünde alt kategoriler de gösterilir
                    </small>
                </div>
            </div>
        </div>
        <div class="card-footer d-flex justify-content-end gap-2 bg-transparent border-top">
            <button type="reset" class="btn btn-secondary">
                <i class="fas fa-undo me-2"></i>Sıfırla
            </button>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save me-2"></i>Genel Ayarları Kaydet
            </button>
        </div>
    </form>

    <form method="post" class="card" enctype="multipart/form-data">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <input type="hidden" name="form_action" value="save">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fas fa-palette me-2 text-warning"></i>Marka Kimliği</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">Logo, favicon ve metinsel kimlik ayarları</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-4">
                <div class="col-md-4">
                    <label class="form-label">Logo Modu</label>
                    <select name="branding_logo_mode" class="form-select">
                        <option value="text"<?= $brandingSettings['mode'] === 'text' ? ' selected' : '' ?>>Metin</option>
                        <option value="image"<?= $brandingSettings['mode'] === 'image' ? ' selected' : '' ?>>Görsel</option>
                    </select>
                    <small class="text-muted">Logo türünü seçin</small>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Logo Genişliği (px)</label>
                    <input type="number" name="branding_logo_width" class="form-control" value="<?= (int)$brandingSettings['width'] ?>" min="0">
                    <small class="text-muted">Boş bırakırsanız otomatik olur</small>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Logo Yüksekliği (px)</label>
                    <input type="number" name="branding_logo_height" class="form-control" value="<?= (int)$brandingSettings['height'] ?>" min="0">
                    <small class="text-muted">Boş bırakırsanız otomatik olur</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Metin Logo</label>
                    <input type="text" name="branding_logo_text" class="form-control" value="<?= sanitize($brandingSettings['text']) ?>">
                    <small class="text-muted">Metin modu seçildiğinde gösterilir</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Slogan</label>
                    <input type="text" name="branding_logo_tagline" class="form-control" value="<?= sanitize($brandingSettings['tagline']) ?>">
                    <small class="text-muted">Logo altında gösterilir (opsiyonel)</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Logo Görseli</label>
                    <input type="file" name="branding_logo_image" class="form-control" accept="image/*">
                    <small class="text-muted">PNG veya SVG önerilir (şeffaf arka plan)</small>
                    <?php if ($brandingSettings['logo']): ?>
                        <div class="mt-3 p-3 bg-light rounded d-flex align-items-center gap-3">
                            <img src="<?= htmlspecialchars(assetUrl($brandingSettings['logo']), ENT_QUOTES, 'UTF-8') ?>" alt="Logo" style="height:48px;border-radius:8px;" class="img-thumbnail">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" value="1" id="removeLogo" name="remove_branding_logo_image">
                                <label class="form-check-label" for="removeLogo">Mevcut logoyu sil</label>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Favicon</label>
                    <input type="file" name="branding_favicon" class="form-control" accept="image/*">
                    <small class="text-muted">16x16 veya 32x32 boyutunda .ico veya .png</small>
                    <?php if ($brandingSettings['favicon']): ?>
                        <div class="mt-3 p-3 bg-light rounded d-flex align-items-center gap-3">
                            <img src="<?= htmlspecialchars(assetUrl($brandingSettings['favicon']), ENT_QUOTES, 'UTF-8') ?>" alt="Favicon" style="height:32px;border-radius:6px;" class="img-thumbnail">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" value="1" id="removeFavicon" name="remove_branding_favicon">
                                <label class="form-check-label" for="removeFavicon">Mevcut faviconu sil</label>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <div class="card-footer d-flex justify-content-end gap-2 bg-transparent border-top">
            <button type="reset" class="btn btn-secondary">
                <i class="fas fa-undo me-2"></i>Sıfırla
            </button>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save me-2"></i>Marka Ayarlarını Kaydet
            </button>
        </div>
    </form>

    <form method="post" class="card">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <input type="hidden" name="form_action" value="save">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fas fa-phone me-2 text-info"></i>Üst Bar İletişim</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">Header üst bölümünde gösterilecek iletişim bilgileri</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-4">
                <div class="col-md-6">
                    <label class="form-label">Üst Bar Telefon</label>
                    <input type="text" name="header_top_phone" class="form-control" value="<?= sanitize($headerSettings['top_phone']) ?>" placeholder="+90 555 123 45 67">
                    <small class="text-muted">Site üst barında gösterilecek telefon numarası</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Üst Bar E-posta</label>
                    <input type="email" name="header_top_email" class="form-control" value="<?= sanitize($headerSettings['top_email']) ?>" placeholder="info@orneksite.com">
                    <small class="text-muted">Site üst barında gösterilecek e-posta adresi</small>
                </div>
                <div class="col-12">
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" id="emailVerify" name="auth_require_email_verification" <?= $emailVerificationRequired ? 'checked' : '' ?>>
                        <label class="form-check-label" for="emailVerify">Yeni kayıt olan kullanıcılar için e-posta doğrulaması zorunlu olsun</label>
                    </div>
                    <small class="text-muted d-block mt-2">Etkinleştirildiğinde, kullanıcılar e-posta adreslerini doğrulayana kadar giriş yapamazlar</small>
                </div>
            </div>
        </div>
        <div class="card-footer d-flex justify-content-end gap-2 bg-transparent border-top">
            <button type="reset" class="btn btn-secondary">
                <i class="fas fa-undo me-2"></i>Sıfırla
            </button>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save me-2"></i>İletişim Ayarlarını Kaydet
            </button>
        </div>
    </form>

    <form method="post" class="card">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <input type="hidden" name="form_action" value="save">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fas fa-envelope me-2 text-danger"></i>SMTP & E-posta Ayarları</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">Bildirim e-postalarının gönderim ayarları</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-4">
                <div class="col-lg-4">
                    <label class="form-label">SMTP Sunucusu</label>
                    <input type="text" name="smtp_host" class="form-control" value="<?= sanitize($smtpSettings['host']) ?>" placeholder="smtp.gmail.com">
                    <small class="text-muted">E-posta sağlayıcınızın SMTP sunucusu</small>
                </div>
                <div class="col-lg-2">
                    <label class="form-label">Port</label>
                    <input type="number" name="smtp_port" class="form-control" value="<?= (int)$smtpSettings['port'] ?>" placeholder="587">
                    <small class="text-muted">TLS: 587, SSL: 465</small>
                </div>
                <div class="col-lg-3">
                    <label class="form-label">Şifreleme</label>
                    <select name="smtp_encryption" class="form-select">
                        <option value="tls"<?= $smtpSettings['encryption'] === 'tls' ? ' selected' : '' ?>>TLS</option>
                        <option value="ssl"<?= $smtpSettings['encryption'] === 'ssl' ? ' selected' : '' ?>>SSL</option>
                        <option value="none"<?= $smtpSettings['encryption'] === '' ? ' selected' : '' ?>>Şifreleme Yok</option>
                    </select>
                    <small class="text-muted">TLS önerilir</small>
                </div>
                <div class="col-lg-3">
                    <label class="form-label">Test Modu</label>
                    <div class="form-check form-switch mt-2">
                        <input class="form-check-input" type="checkbox" name="smtp_test_mode" id="smtpTestMode" <?= $smtpSettings['test_mode'] ? 'checked' : '' ?>>
                        <label class="form-check-label" for="smtpTestMode">Sertifika doğrulama</label>
                    </div>
                    <small class="text-muted d-block mt-2">Sorun yaşarsanız kapatın</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">SMTP Kullanıcı Adı</label>
                    <input type="text" name="smtp_username" class="form-control" value="<?= sanitize($smtpSettings['username']) ?>" placeholder="kullanici@gmail.com">
                    <small class="text-muted">E-posta sağlayıcınızdaki kullanıcı adı</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">SMTP Şifresi</label>
                    <input type="password" name="smtp_password" class="form-control" value="<?= sanitize($smtpSettings['password']) ?>" placeholder="********">
                    <small class="text-muted">E-posta sağlayıcınızdaki şifre veya uygulama şifresi</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Gönderici E-posta</label>
                    <input type="email" name="smtp_from_email" class="form-control" value="<?= sanitize($smtpSettings['from_email']) ?>" placeholder="noreply@orneksite.com">
                    <small class="text-muted">E-postalar bu adresten gönderilir</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Gönderici Adı</label>
                    <input type="text" name="smtp_from_name" class="form-control" value="<?= sanitize($smtpSettings['from_name']) ?>" placeholder="Örnek Site">
                    <small class="text-muted">E-postalarda görünecek gönderici adı</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">E-posta Temel URL</label>
                    <input type="url" name="email_base_url" class="form-control" value="<?= sanitize($smtpSettings['base_url']) ?>" placeholder="https://orneksite.com">
                    <small class="text-muted">E-posta içerisindeki linklerde kullanılır</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Şifre Sıfırlama Şablonu</label>
                    <textarea name="email_template_password_reset" class="form-control" rows="4" placeholder="HTML şablonu (opsiyonel)"><?= htmlspecialchars($smtpSettings['reset_template'], ENT_QUOTES, 'UTF-8') ?></textarea>
                    <small class="text-muted">Boş bırakılırsa varsayılan şablon kullanılır</small>
                </div>
            </div>
        </div>
        <div class="card-footer d-flex justify-content-end gap-2 bg-transparent border-top">
            <button type="reset" class="btn btn-secondary">
                <i class="fas fa-undo me-2"></i>Sıfırla
            </button>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save me-2"></i>SMTP Ayarlarını Kaydet
            </button>
        </div>
    </form>

    <!-- Test Email Formu - Ana formdan ayrı -->
    <form method="post" class="card">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <input type="hidden" name="form_action" value="smtp_test">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fas fa-paper-plane me-2 text-info"></i>Test E-posta Gönder</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">SMTP ayarlarınızı test etmek için bir test e-postası gönderin</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-3 align-items-end">
                <div class="col-md-8">
                    <label class="form-label">Test E-posta Adresi</label>
                    <input type="email" name="smtp_test_email" class="form-control" placeholder="test@orneksite.com" required>
                    <small class="text-muted">Test e-postasının gönderileceği e-posta adresi</small>
                </div>
                <div class="col-md-4">
                    <button type="submit" class="btn btn-outline-primary w-100">
                        <i class="fas fa-paper-plane me-2"></i>Test Gönder
                    </button>
                </div>
            </div>
        </div>
    </form>

    <form method="post" class="card">
        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
        <input type="hidden" name="form_action" value="save">
        <div class="card-header">
            <div>
                <h5 class="mb-1"><i class="fab fa-google me-2 text-success"></i>Google reCAPTCHA</h5>
                <p class="text-muted mb-0" style="font-size: 13px;">Formlarda bot koruması için reCAPTCHA anahtarları</p>
            </div>
        </div>
        <div class="card-body">
            <div class="row g-4">
                <div class="col-md-6">
                    <label class="form-label">Site Anahtarı</label>
                    <input type="text" name="recaptcha_site_key" class="form-control" value="<?= sanitize($recaptchaSettings['site_key']) ?>" placeholder="6Lc...">
                    <small class="text-muted">Google reCAPTCHA'dan alınan site key</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">Gizli Anahtar</label>
                    <input type="text" name="recaptcha_secret_key" class="form-control" value="<?= sanitize($recaptchaSettings['secret_key']) ?>" placeholder="6Lc...">
                    <small class="text-muted">Google reCAPTCHA'dan alınan secret key</small>
                </div>
                <div class="col-12">
                    <label class="form-label mb-3">reCAPTCHA Kullanım Alanları</label>
                    <div class="row g-3">
                        <div class="col-md-4">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="recaptchaLogin" name="recaptcha_enable_login" <?= $recaptchaSettings['enable_login'] ? 'checked' : '' ?>>
                                <label class="form-check-label" for="recaptchaLogin">Giriş formunda kullan</label>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="recaptchaRegister" name="recaptcha_enable_register" <?= $recaptchaSettings['enable_register'] ? 'checked' : '' ?>>
                                <label class="form-check-label" for="recaptchaRegister">Kayıt formunda kullan</label>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="recaptchaForgot" name="recaptcha_enable_forgot" <?= $recaptchaSettings['enable_forgot'] ? 'checked' : '' ?>>
                                <label class="form-check-label" for="recaptchaForgot">Şifre sıfırlamada kullan</label>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="recaptchaAdminLogin" name="recaptcha_enable_admin_login" <?= $recaptchaSettings['enable_admin_login'] ? 'checked' : '' ?>>
                                <label class="form-check-label" for="recaptchaAdminLogin">Admin girişinde kullan</label>
                            </div>
                        </div>
                    </div>
                    <div class="alert alert-info mt-3">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Güvenlik Önerisi:</strong> Admin paneli girişi için reCAPTCHA kullanılması önerilir.
                    </div>
                </div>
                <div class="col-12">
                    <div class="alert alert-info mb-0">
                        <i class="fas fa-info-circle me-2"></i>
                        Anahtarlar boş olduğunda reCAPTCHA devre dışı kalır. Anahtarlar için <a href="https://www.google.com/recaptcha/admin" target="_blank" rel="noopener" class="alert-link">Google reCAPTCHA konsolunu</a> kullanın.
                    </div>
                </div>
            </div>
        </div>
        <div class="card-footer d-flex justify-content-end gap-2 bg-transparent border-top">
            <button type="reset" class="btn btn-secondary">
                <i class="fas fa-undo me-2"></i>Sıfırla
            </button>
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-save me-2"></i>reCAPTCHA Ayarlarını Kaydet
            </button>
        </div>
    </form>
</div>

<?php include 'includes/admin-footer.php'; ?>
