<?php
require_once '../config.php';
requireAdmin();

$ticketId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($ticketId <= 0) {
    flashMessage('danger', 'Geçersiz destek talebi.');
    redirect('tickets.php');
}

$statusMap = getSupportTicketStatusMap();
$statusBadgeMap = [
    'open' => 'badge bg-warning text-dark',
    'answered' => 'badge bg-primary',
    'customer_reply' => 'badge bg-info text-dark',
    'resolved' => 'badge bg-success',
];

$ticketStmt = $pdo->prepare('SELECT t.*, u.username, u.email FROM support_tickets t JOIN users u ON u.id = t.user_id WHERE t.id = :id LIMIT 1');
$ticketStmt->execute([':id' => $ticketId]);
$ticket = $ticketStmt->fetch();

if (!$ticket) {
    flashMessage('danger', 'Destek talebi bulunamadı.');
    redirect('tickets.php');
}

$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $errors[] = 'Güvenlik doğrulaması başarısız oldu.';
    } else {
        $action = $_POST['action'] ?? '';

        if ($action === 'admin_reply') {
            $messageBody = trim($_POST['message'] ?? '');
            $nextStatus = $_POST['next_status'] ?? 'answered';
            if (!array_key_exists($nextStatus, $statusMap)) {
                $nextStatus = 'answered';
            }

            if ($messageBody === '') {
                $errors[] = 'Yanıt mesajı boş olamaz.';
            }

            $attachmentPath = null;
            if (empty($errors) && isset($_FILES['attachment'])) {
                try {
                    $attachmentPath = uploadImageFile($_FILES['attachment'], 'tickets');
                } catch (Throwable $th) {
                    $errors[] = 'Dosya yüklenemedi: ' . $th->getMessage();
                }
            }

            if (empty($errors)) {
                $currentStatus = $ticket['status'];
                try {
                    $pdo->beginTransaction();

                    $messageStmt = $pdo->prepare('INSERT INTO support_ticket_messages (ticket_id, sender_type, sender_id, message, attachment_path) VALUES (:ticket_id, :sender_type, :sender_id, :message, :attachment)');
                    $messageStmt->execute([
                        ':ticket_id' => $ticketId,
                        ':sender_type' => 'admin',
                        ':sender_id' => $_SESSION['user_id'],
                        ':message' => $messageBody,
                        ':attachment' => $attachmentPath,
                    ]);

                    $updateSql = 'UPDATE support_tickets SET admin_id = :admin_id, last_message_at = NOW(), updated_at = NOW(), status = :status';
                    $params = [
                        ':admin_id' => $_SESSION['user_id'],
                        ':status' => $nextStatus,
                        ':id' => $ticketId,
                    ];
                    if ($nextStatus !== $currentStatus) {
                        $updateSql .= ', last_status_changed_at = NOW()';
                    }
                    $updateSql .= ' WHERE id = :id';
                    $updateStmt = $pdo->prepare($updateSql);
                    $updateStmt->execute($params);

                    $pdo->commit();

                    logUserActivity($pdo, 'support_ticket_admin_reply', ['ticket_id' => $ticketId, 'status' => $nextStatus]);
                    flashMessage('success', 'Yanıtınız gönderildi.');
                    redirect('ticket-detail.php?id=' . $ticketId);
                } catch (Throwable $th) {
                    $pdo->rollBack();
                    $errors[] = 'Yanıt gönderilemedi: ' . $th->getMessage();
                }
            }
        }

        if ($action === 'update_status') {
            $newStatus = $_POST['status'] ?? '';
            if (!array_key_exists($newStatus, $statusMap)) {
                $errors[] = 'Geçersiz durum seçimi.';
            } else {
                try {
                    $updateStmt = $pdo->prepare('UPDATE support_tickets SET status = :status, last_status_changed_at = NOW(), updated_at = NOW() WHERE id = :id');
                    $updateStmt->execute([':status' => $newStatus, ':id' => $ticketId]);
                    logUserActivity($pdo, 'support_ticket_status_updated', ['ticket_id' => $ticketId, 'status' => $newStatus]);
                    flashMessage('success', 'Durum başarıyla güncellendi.');
                    redirect('ticket-detail.php?id=' . $ticketId);
                } catch (Throwable $th) {
                    $errors[] = 'Durum güncellenemedi: ' . $th->getMessage();
                }
            }
        }
    }
}

$messagesStmt = $pdo->prepare('SELECT m.*, u.username AS admin_username FROM support_ticket_messages m LEFT JOIN users u ON (m.sender_type = "admin" AND u.id = m.sender_id) WHERE m.ticket_id = :ticket_id ORDER BY m.created_at ASC');
$messagesStmt->execute([':ticket_id' => $ticketId]);
$messages = $messagesStmt->fetchAll() ?: [];

$pageTitle = 'Destek Talebi #' . $ticketId;
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <?php if (!empty($errors)): ?>
        <div class="alert alert-danger">
            <ul class="mb-0">
                <?php foreach ($errors as $error): ?>
                    <li><?= sanitize($error) ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <div class="row g-4">
        <div class="col-xl-4">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-ticket me-2"></i>Talep Bilgileri</h5>
                </div>
                <div class="card-body">
                    <dl class="row mb-0">
                        <dt class="col-5">Talep No</dt>
                        <dd class="col-7">#<?= (int)$ticket['id'] ?></dd>
                        <dt class="col-5">Konu</dt>
                        <dd class="col-7"><?= sanitize($ticket['subject']) ?></dd>
                        <dt class="col-5">Kullanıcı</dt>
                        <dd class="col-7">
                            <strong><?= sanitize($ticket['username']) ?></strong><br>
                            <span class="text-muted small"><?= sanitize($ticket['email'] ?? '') ?></span>
                        </dd>
                        <dt class="col-5">Durum</dt>
                        <dd class="col-7">
                            <span class="<?= $statusBadgeMap[$ticket['status']] ?? 'badge bg-secondary' ?>"><?= sanitize(getSupportTicketStatusLabel($ticket['status'])) ?></span>
                        </dd>
                        <dt class="col-5">Oluşturulma</dt>
                        <dd class="col-7"><?= date('d.m.Y H:i', strtotime($ticket['created_at'])) ?></dd>
                        <dt class="col-5">Son Güncelleme</dt>
                        <dd class="col-7"><?= date('d.m.Y H:i', strtotime($ticket['updated_at'])) ?></dd>
                    </dl>

                    <form method="post" class="mt-3">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="update_status">
                        <label class="form-label">Durumu Güncelle</label>
                        <select name="status" class="form-select mb-2">
                            <?php foreach ($statusMap as $key => $label): ?>
                                <option value="<?= $key ?>" <?= $ticket['status'] === $key ? 'selected' : '' ?>><?= sanitize($label) ?></option>
                            <?php endforeach; ?>
                        </select>
                        <button type="submit" class="btn btn-outline-primary w-100"><i class="fas fa-arrows-rotate me-2"></i>Durumu Kaydet</button>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-xl-8">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-comments me-2"></i>Mesajlar</h5>
                </div>
                <div class="card-body">
                    <div class="ticket-thread mb-4" style="max-height:420px; overflow-y:auto;">
                        <?php if (empty($messages)): ?>
                            <div class="text-center text-muted py-5">Henüz mesaj bulunmuyor.</div>
                        <?php else: ?>
                            <?php foreach ($messages as $message): ?>
                                <?php
                                    $isAdmin = $message['sender_type'] === 'admin';
                                    $bubbleClass = $isAdmin ? 'bg-primary text-white' : 'bg-light';
                                    $author = $isAdmin ? ($message['admin_username'] ?? 'Admin') : $ticket['username'];
                                    $authorClass = $isAdmin ? 'text-primary' : '';
                                ?>
                                <div class="mb-3">
                                    <div class="d-flex justify-content-between align-items-center mb-1">
                                        <strong class="<?= $authorClass ?>"><?= sanitize($author) ?></strong>
                                        <span class="text-muted small"><?= date('d.m.Y H:i', strtotime($message['created_at'])) ?></span>
                                    </div>
                                    <div class="p-3 rounded <?= $bubbleClass ?> mt-1" style="border: 1px solid var(--color-border-primary);">
                                        <p class="mb-0" style="<?= !$isAdmin ? 'color: var(--color-text-primary);' : '' ?>"><?= nl2br(sanitize($message['message'])) ?></p>
                                        <?php if (!empty($message['attachment_path'])): ?>
                                            <?php $attachmentUrl = assetUrl($message['attachment_path']); ?>
                                            <div class="mt-2">
                                                <a href="<?= htmlspecialchars($attachmentUrl, ENT_QUOTES, 'UTF-8') ?>" target="_blank" class="btn btn-sm <?= $isAdmin ? 'btn-outline-light' : 'btn-outline-primary' ?>"><i class="fas fa-image me-1"></i>Ekli Görsel</a>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>

                    <form method="post" enctype="multipart/form-data">
                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                        <input type="hidden" name="action" value="admin_reply">
                        <div class="mb-3">
                            <label class="form-label">Yanıtınız</label>
                            <textarea name="message" class="form-control" rows="4" required><?= sanitize($_POST['message'] ?? '') ?></textarea>
                        </div>
                        <div class="row g-3 align-items-end">
                            <div class="col-md-6">
                                <label class="form-label">Görsel (isteğe bağlı)</label>
                                <input type="file" name="attachment" class="form-control" accept="image/png,image/jpeg,image/webp,image/gif">
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Yanıt sonrası durum</label>
                                <select name="next_status" class="form-select">
                                    <option value="answered">Yanıtlandı</option>
                                    <option value="resolved">Çözüldü</option>
                                </select>
                            </div>
                        </div>
                        <div class="d-flex justify-content-end gap-2 mt-3">
                            <a href="tickets.php" class="btn btn-outline-secondary"><i class="fas fa-arrow-left me-2"></i>Listeye Dön</a>
                            <button type="submit" class="btn btn-primary"><i class="fas fa-paper-plane me-2"></i>Yanıt Gönder</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>


