<?php
require_once '../config.php';
requireAdmin();

$userId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($userId <= 0) {
    flashMessage('danger', 'Geçersiz kullanıcı.');
    redirect('users.php');
}

$userStmt = $pdo->prepare('SELECT * FROM users WHERE id = ?');
$userStmt->execute([$userId]);
$user = $userStmt->fetch();

if (!$user) {
    flashMessage('danger', 'Kullanıcı bulunamadı.');
    redirect('users.php');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('user-detail.php?id=' . $userId);
    }
    
    if (isset($_POST['toggle_status'])) {
        $newStatus = $user['is_active'] ? 0 : 1;
        $pdo->prepare('UPDATE users SET is_active = ?, updated_at = NOW() WHERE id = ?')->execute([$newStatus, $userId]);
        flashMessage('success', 'Kullanıcı durumu güncellendi.');
        redirect('user-detail.php?id=' . $userId);
    }

    if (isset($_POST['update_balance'])) {
        $newBalance = (float)($_POST['balance'] ?? $user['balance']);
        $pdo->prepare('UPDATE users SET balance = ?, updated_at = NOW() WHERE id = ?')->execute([$newBalance, $userId]);
        flashMessage('success', 'Kullanıcı bakiyesi güncellendi.');
        redirect('user-detail.php?id=' . $userId);
    }
    
    if (isset($_POST['promote_to_admin'])) {
        if ($userId === $_SESSION['user_id']) {
            flashMessage('warning', 'Kendi yetkinizi değiştiremezsiniz.');
        } else {
            try {
                $pdo->prepare('UPDATE users SET is_admin = 1 WHERE id = ?')->execute([$userId]);
                flashMessage('success', 'Kullanıcı admin yetkisi aldı.');
                logUserActivity($pdo, 'admin_user_promoted', ['target_user_id' => $userId, 'by_admin_id' => $_SESSION['user_id']]);
            } catch (PDOException $e) {
                flashMessage('danger', 'Yetki güncellenirken hata oluştu.');
            }
        }
        redirect('user-detail.php?id=' . $userId);
    }
    
    if (isset($_POST['demote_from_admin'])) {
        if ($userId === $_SESSION['user_id']) {
            flashMessage('warning', 'Kendi yetkinizi değiştiremezsiniz.');
        } else {
            try {
                $pdo->prepare('UPDATE users SET is_admin = 0 WHERE id = ?')->execute([$userId]);
                flashMessage('success', 'Kullanıcının admin yetkisi kaldırıldı.');
                logUserActivity($pdo, 'admin_user_demoted', ['target_user_id' => $userId, 'by_admin_id' => $_SESSION['user_id']]);
            } catch (PDOException $e) {
                flashMessage('danger', 'Yetki güncellenirken hata oluştu.');
            }
        }
        redirect('user-detail.php?id=' . $userId);
    }
}

$orderSummaryStmt = $pdo->prepare('SELECT COUNT(*) AS order_count, COALESCE(SUM(total_amount),0) AS total_spent FROM orders WHERE user_id = ? AND order_type = "purchase" AND status = "completed"');
$orderSummaryStmt->execute([$userId]);
$orderSummary = $orderSummaryStmt->fetch();

$depositSummaryStmt = $pdo->prepare('SELECT COUNT(*) AS deposit_count, COALESCE(SUM(total_amount),0) AS net_amount, COALESCE(SUM(commission_amount),0) AS commission_amount FROM orders WHERE user_id = ? AND order_type = "deposit" AND status = "completed"');
$depositSummaryStmt->execute([$userId]);
$depositSummary = $depositSummaryStmt->fetch();

$recentOrdersStmt = $pdo->prepare('SELECT * FROM orders WHERE user_id = ? AND order_type = "purchase" ORDER BY created_at DESC LIMIT 8');
$recentOrdersStmt->execute([$userId]);
$recentOrders = $recentOrdersStmt->fetchAll();

$recentDepositsStmt = $pdo->prepare('SELECT * FROM orders WHERE user_id = ? AND order_type = "deposit" ORDER BY created_at DESC LIMIT 8');
$recentDepositsStmt->execute([$userId]);
$recentDeposits = $recentDepositsStmt->fetchAll();

$transactionsStmt = $pdo->prepare('SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 12');
$transactionsStmt->execute([$userId]);
$transactions = $transactionsStmt->fetchAll();

$latestLog = null;
try {
    $logStmt = $pdo->prepare('SELECT * FROM user_activity_logs WHERE user_id = ? ORDER BY created_at DESC LIMIT 1');
    $logStmt->execute([$userId]);
    $latestLog = $logStmt->fetch() ?: null;
} catch (Throwable $th) {
    $latestLog = null;
}

$pageTitle = 'Kullanıcı Detayı';
include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h3 class="mb-1"><?= sanitize($user['username']) ?></h3>
            <p class="text-muted mb-0"><?= sanitize($user['email']) ?> &bull; ID: #<?= $user['id'] ?></p>
        </div>
        <div class="d-flex gap-2">
            <form method="post">
                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                <button type="submit" name="toggle_status" class="btn btn-outline-primary">
                    <i class="fas fa-user-shield me-2"></i><?= $user['is_active'] ? 'Pasif Yap' : 'Aktif Yap' ?>
                </button>
            </form>
            
            <?php if ((int)$user['is_admin'] === 0): ?>
                <form method="post" class="d-inline" onsubmit="return confirm('Bu kullanıcıya admin yetkisi vermek istediğinize emin misiniz?')">
                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                    <button type="submit" name="promote_to_admin" class="btn btn-warning">
                        <i class="fas fa-crown me-2"></i>Admin Yap
                    </button>
                </form>
            <?php elseif ($userId !== $_SESSION['user_id']): ?>
                <form method="post" class="d-inline" onsubmit="return confirm('Bu kullanıcının admin yetkisini kaldırmak istediğinize emin misiniz?')">
                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                    <button type="submit" name="demote_from_admin" class="btn btn-secondary">
                        <i class="fas fa-user me-2"></i>Admin Yetkisini Kaldır
                    </button>
                </form>
            <?php endif; ?>
            <a href="users.php" class="btn btn-outline-secondary"><i class="fas fa-arrow-left me-2"></i>Listeye Dön</a>
        </div>
    </div>

    <div class="row g-4 mb-4">
        <div class="col-lg-3">
            <div class="metric-card h-100">
                <span class="metric-label">Bakiye</span>
                <div class="metric-value" style="font-size:1.5rem;">
                    <?= formatPrice($user['balance']) ?>
                </div>
                <form method="post" class="mt-3">
                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                    <label class="form-label">Bakiyeyi Güncelle</label>
                    <div class="input-group">
                        <input type="number" step="0.01" name="balance" value="<?= htmlspecialchars(number_format($user['balance'], 2, '.', ''), ENT_QUOTES, 'UTF-8') ?>" class="form-control">
                        <button type="submit" name="update_balance" class="btn btn-primary">Kaydet</button>
                    </div>
                </form>
            </div>
        </div>
        <div class="col-lg-3">
            <div class="metric-card h-100">
                <span class="metric-label">Satın Alma Siparişleri</span>
                <div class="metric-value" style="font-size:1.5rem;"><?= number_format($orderSummary['order_count'] ?? 0) ?></div>
                <p class="metric-subtext">Toplam Harcama: <?= formatPrice($orderSummary['total_spent'] ?? 0) ?></p>
            </div>
        </div>
        <div class="col-lg-3">
            <div class="metric-card h-100">
                <span class="metric-label">Bakiye Yüklemeleri</span>
                <div class="metric-value" style="font-size:1.5rem;"><?= number_format($depositSummary['deposit_count'] ?? 0) ?></div>
                <p class="metric-subtext">Net: <?= formatPrice($depositSummary['net_amount'] ?? 0) ?><br>Komisyon: <?= formatPrice($depositSummary['commission_amount'] ?? 0) ?></p>
            </div>
        </div>
        <div class="col-lg-3">
            <div class="metric-card h-100">
                <span class="metric-label">Hesap Durumu</span>
                <div class="metric-value" style="font-size:1.5rem;">
                    <?php if ($user['is_active']): ?>
                        <span class="badge bg-success">Aktif</span>
                    <?php else: ?>
                        <span class="badge bg-danger">Pasif</span>
                    <?php endif; ?>
                </div>
                <p class="metric-subtext">Kayıt Tarihi: <?= date('d.m.Y H:i', strtotime($user['created_at'])) ?></p>
            </div>
        </div>
    </div>

    <div class="row g-4 mb-4">
        <div class="col-lg-6">
            <div class="card h-100">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-location-dot me-2"></i>Son Oturum Bilgisi</h5>
                </div>
                <div class="card-body">
                    <?php if ($latestLog): ?>
                        <dl class="row mb-0">
                            <dt class="col-sm-4 text-muted">IP Adresi</dt>
                            <dd class="col-sm-8"><code><?= sanitize($latestLog['ip_address'] ?? '-') ?></code></dd>

                            <dt class="col-sm-4 text-muted">Konum</dt>
                            <dd class="col-sm-8">
                                <?= sanitize(trim(($latestLog['city'] ?? '') . ', ' . ($latestLog['country'] ?? ''))) ?: '<span class="text-muted">Bilinmiyor</span>' ?>
                            </dd>

                            <dt class="col-sm-4 text-muted">Cihaz ID</dt>
                            <dd class="col-sm-8">
                                <?= !empty($latestLog['device_id']) ? '<code>' . sanitize($latestLog['device_id']) . '</code>' : '<span class="text-muted">Yok</span>' ?>
                            </dd>

                            <dt class="col-sm-4 text-muted">Tarayıcı</dt>
                            <dd class="col-sm-8">
                                <?= sanitize(mb_strimwidth($latestLog['user_agent'] ?? '-', 0, 160, '…', 'UTF-8')) ?>
                            </dd>

                            <dt class="col-sm-4 text-muted">Kaydedildi</dt>
                            <dd class="col-sm-8"><?= date('d.m.Y H:i', strtotime($latestLog['created_at'])) ?></dd>
                        </dl>
                    <?php else: ?>
                        <p class="text-muted mb-0">Kullanıcıya ait log bulunamadı.</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <div class="row g-4 mb-4">
        <div class="col-lg-6">
            <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-shopping-cart me-2"></i>Son Siparişler</h5>
                    <a href="orders.php" class="btn btn-sm btn-outline-primary">Siparişlere Git</a>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>Sipariş No</th>
                                    <th>Toplam</th>
                                    <th>Durum</th>
                                    <th>Tarih</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recentOrders as $orderRow): ?>
                                <tr>
                                    <td><strong><?= sanitize($orderRow['order_number']) ?></strong></td>
                                    <td><?= formatPrice($orderRow['total_amount']) ?></td>
                                    <td>
                                        <?php if ($orderRow['status'] === 'completed'): ?>
                                            <span class="badge bg-success">Tamamlandı</span>
                                        <?php elseif ($orderRow['status'] === 'pending'): ?>
                                            <span class="badge bg-warning text-dark">Beklemede</span>
                                        <?php else: ?>
                                            <span class="badge bg-danger">İptal</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= date('d.m.Y H:i', strtotime($orderRow['created_at'])) ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-lg-6">
            <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-wallet me-2"></i>Para Yatırmaları</h5>
                    <a href="deposits.php" class="btn btn-sm btn-outline-primary">Detay</a>
                </div>
                <div class="card-body p-0">
                    <?php if (empty($recentDeposits)): ?>
                        <div class="p-4 text-muted">Bu kullanıcıya ait para yatırma kaydı bulunmuyor.</div>
                    <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead>
                                <tr>
                                    <th>Tutar</th>
                                    <th>Komisyon</th>
                                    <th>Durum</th>
                                    <th>Tarih</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recentDeposits as $depositRow): ?>
                                <tr>
                                    <td><?= formatPrice($depositRow['total_amount']) ?></td>
                                    <td class="text-success">+<?= formatPrice($depositRow['commission_amount']) ?></td>
                                    <td>
                                        <?php if ($depositRow['status'] === 'completed'): ?>
                                            <span class="badge bg-success">Tamamlandı</span>
                                        <?php elseif ($depositRow['status'] === 'pending'): ?>
                                            <span class="badge bg-warning text-dark">Beklemede</span>
                                        <?php else: ?>
                                            <span class="badge bg-danger">İptal</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= date('d.m.Y H:i', strtotime($depositRow['created_at'])) ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <div class="card">
        <div class="card-header">
            <h5 class="mb-0"><i class="fas fa-history me-2"></i>Hareket Geçmişi</h5>
        </div>
        <div class="card-body p-0">
            <?php if (empty($transactions)): ?>
                <div class="p-4 text-muted">Herhangi bir işlem kaydı bulunmuyor.</div>
            <?php else: ?>
            <div class="table-responsive">
                <table class="table table-borderless mb-0">
                    <thead>
                        <tr>
                            <th>Tür</th>
                            <th>Tutar</th>
                            <th>Açıklama</th>
                            <th>Tarih</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($transactions as $txn): ?>
                        <tr>
                            <td><?= ucfirst($txn['type']) ?></td>
                            <td><?= formatPrice($txn['amount']) ?></td>
                            <td><?= sanitize($txn['description'] ?? '-') ?></td>
                            <td><?= date('d.m.Y H:i', strtotime($txn['created_at'])) ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>


