<?php
require_once '../config.php';
requireAdmin();

$pageTitle = 'Bakiye Çekimleri';

$allowedStatuses = ['pending', 'approved', 'rejected', 'all'];
$statusFilter = $_GET['status'] ?? 'pending';
if (!in_array($statusFilter, $allowedStatuses, true)) {
    $statusFilter = 'pending';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $postedFilter = $_POST['status_filter'] ?? $statusFilter;
    if (in_array($postedFilter, $allowedStatuses, true)) {
        $statusFilter = $postedFilter;
    }

    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız. Lütfen tekrar deneyin.');
        redirect('withdrawals.php?status=' . urlencode($statusFilter));
    }

    $requestId = (int)($_POST['request_id'] ?? 0);
    $action = $_POST['action'] ?? '';
    $adminNote = trim($_POST['admin_note'] ?? '');
    if ($adminNote !== '') {
        $adminNote = mb_substr($adminNote, 0, 250);
    }

    if ($requestId <= 0) {
        flashMessage('danger', 'Geçersiz talep bilgisi.');
        redirect('withdrawals.php?status=' . urlencode($statusFilter));
    }

    try {
        $pdo->beginTransaction();

        $withdrawStmt = $pdo->prepare('SELECT * FROM withdraw_requests WHERE id = ? FOR UPDATE');
        $withdrawStmt->execute([$requestId]);
        $withdrawRequest = $withdrawStmt->fetch();

        if (!$withdrawRequest) {
            $pdo->rollBack();
            flashMessage('danger', 'Çekim talebi bulunamadı.');
            redirect('withdrawals.php?status=' . urlencode($statusFilter));
        }

        $currentStatus = strtolower((string)$withdrawRequest['status']);

        $userStmt = $pdo->prepare('SELECT id, username, balance FROM users WHERE id = ? FOR UPDATE');
        $userStmt->execute([$withdrawRequest['user_id']]);
        $userRow = $userStmt->fetch();

        if (!$userRow) {
            $pdo->rollBack();
            flashMessage('danger', 'Kullanıcı bilgisine erişilemedi.');
            redirect('withdrawals.php?status=' . urlencode($statusFilter));
        }

        $amount = (float)$withdrawRequest['amount'];
        $adminId = $_SESSION['user_id'];

        if ($action === 'approve') {
            if ($currentStatus !== 'pending') {
                $pdo->rollBack();
                flashMessage('info', 'Sadece bekleyen talepler onaylanabilir.');
                redirect('withdrawals.php?status=' . urlencode($statusFilter));
            }

            $updateWithdraw = $pdo->prepare('UPDATE withdraw_requests SET status = "approved", admin_id = ?, admin_note = ?, processed_at = NOW() WHERE id = ?');
            $updateWithdraw->execute([$adminId, $adminNote, $requestId]);

            $pdo->commit();
            flashMessage('success', 'Çekim talebi onaylandı.');
        } elseif ($action === 'reject') {
            if ($currentStatus !== 'pending' && $currentStatus !== 'approved') {
                $pdo->rollBack();
                flashMessage('info', 'Bu talep zaten işlenmiş.');
                redirect('withdrawals.php?status=' . urlencode($statusFilter));
            }

            // Eğer approved durumdan rejected'a alınıyorsa bakiyeyi geri ekle
            if ($currentStatus === 'approved') {
                $updateBalance = $pdo->prepare('UPDATE users SET balance = balance + ? WHERE id = ?');
                $updateBalance->execute([number_format($amount, 2, '.', ''), $userRow['id']]);

                $transactionStmt = $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description, created_at) VALUES (?, "refund", ?, ?, NOW())');
                $description = 'İptal edilen çekim talebi #' . $requestId . ' iadesi';
                $transactionStmt->execute([$userRow['id'], number_format($amount, 2, '.', ''), $description]);
            }
            // Eğer pending durumdan rejected'a alınıyorsa bakiyeyi geri ekle
            elseif ($currentStatus === 'pending') {
                $updateBalance = $pdo->prepare('UPDATE users SET balance = balance + ? WHERE id = ?');
                $updateBalance->execute([number_format($amount, 2, '.', ''), $userRow['id']]);

                $transactionStmt = $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description, created_at) VALUES (?, "refund", ?, ?, NOW())');
                $description = 'Reddedilen çekim talebi #' . $requestId . ' iadesi';
                $transactionStmt->execute([$userRow['id'], number_format($amount, 2, '.', ''), $description]);
            }

            $updateWithdraw = $pdo->prepare('UPDATE withdraw_requests SET status = "rejected", admin_id = ?, admin_note = ?, processed_at = NOW() WHERE id = ?');
            $updateWithdraw->execute([$adminId, $adminNote, $requestId]);

            $pdo->commit();
            flashMessage('success', 'Çekim talebi reddedildi ve bakiye iade edildi.');
        } else {
            $pdo->rollBack();
            flashMessage('danger', 'Geçersiz işlem.');
        }
    } catch (Throwable $th) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        flashMessage('danger', 'İşlem sırasında hata oluştu: ' . $th->getMessage());
    }

    redirect('withdrawals.php?status=' . urlencode($statusFilter));
}

$summaryStmt = $pdo->query('SELECT status, COUNT(*) AS total_requests, SUM(amount) AS total_amount FROM withdraw_requests GROUP BY status');
$summaryRows = $summaryStmt ? $summaryStmt->fetchAll(PDO::FETCH_ASSOC) : [];
$summary = [
    'pending' => ['count' => 0, 'total' => 0.0],
    'approved' => ['count' => 0, 'total' => 0.0],
    'rejected' => ['count' => 0, 'total' => 0.0],
];
foreach ($summaryRows as $row) {
    $statusKey = strtolower((string)$row['status']);
    if (isset($summary[$statusKey])) {
        $summary[$statusKey]['count'] = (int)($row['total_requests'] ?? 0);
        $summary[$statusKey]['total'] = (float)($row['total_amount'] ?? 0);
    }
}

$listSql = 'SELECT wr.*, u.username, u.email, u.payout_account_name, u.payout_bank_name, u.payout_iban, au.username AS admin_username
            FROM withdraw_requests wr
            JOIN users u ON u.id = wr.user_id
            LEFT JOIN users au ON au.id = wr.admin_id';
$listParams = [];
if ($statusFilter !== 'all') {
    $listSql .= ' WHERE wr.status = ?';
    $listParams[] = $statusFilter;
}
$listSql .= ' ORDER BY wr.created_at DESC LIMIT 100';

$listStmt = $pdo->prepare($listSql);
$listStmt->execute($listParams);
$withdrawals = $listStmt->fetchAll(PDO::FETCH_ASSOC);

include 'includes/admin-header.php';
?>

<div class="dashboard-container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4 class="mb-1"><i class="fas fa-money-bill-transfer me-2"></i>Bakiye Çekim Talepleri</h4>
            <p class="text-muted mb-0">Kullanıcıların banka hesaplarına aktarmak istediği bakiyeleri yönetin.</p>
        </div>
    </div>

    <div class="row g-3 mb-4">
        <div class="col-md-4">
            <div class="card h-100">
                <div class="card-body">
                    <h6 class="text-muted">Bekleyen Talepler</h6>
                    <div class="metric-value text-warning"><?= number_format($summary['pending']['count'], 0, ',', '.') ?></div>
                    <div class="small text-muted">Toplam Tutar: <?= formatPrice($summary['pending']['total']) ?></div>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card h-100">
                <div class="card-body">
                    <h6 class="text-muted">Onaylanan Talepler</h6>
                    <div class="metric-value text-success"><?= number_format($summary['approved']['count'], 0, ',', '.') ?></div>
                    <div class="small text-muted">Toplam Ödeme: <?= formatPrice($summary['approved']['total']) ?></div>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card h-100">
                <div class="card-body">
                    <h6 class="text-muted">Reddedilen Talepler</h6>
                    <div class="metric-value text-danger"><?= number_format($summary['rejected']['count'], 0, ',', '.') ?></div>
                    <div class="small text-muted">Toplam Tutar: <?= formatPrice($summary['rejected']['total']) ?></div>
                </div>
            </div>
        </div>
    </div>

    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <div>
                <h5 class="mb-0">Çekim Talepleri</h5>
                <small class="text-muted">Filtreleyerek bekleyen talepleri hızla yönetin.</small>
            </div>
            <div class="btn-group">
                <?php foreach ($allowedStatuses as $statusOption): ?>
                    <?php $labelMap = ['all' => 'Tümü', 'pending' => 'Bekleyen', 'approved' => 'Onaylanan', 'rejected' => 'Reddedilen']; ?>
                    <a href="withdrawals.php?status=<?= urlencode($statusOption) ?>" class="btn btn-sm btn<?= $statusFilter === $statusOption ? '' : '-outline' ?>-primary">
                        <?= sanitize($labelMap[$statusOption] ?? ucfirst($statusOption)) ?>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
        <div class="card-body p-0">
            <?php if (empty($withdrawals)): ?>
                <div class="p-4 text-center text-muted">Seçili kritere göre görüntülenecek çekim talebi bulunamadı.</div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>#</th>
                                <th>Kullanıcı</th>
                                <th>Talep Tarihi</th>
                                <th>Miktar</th>
                                <th>Banka Bilgisi</th>
                                <th>Durum</th>
                                <th>İşlem</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($withdrawals as $withdrawal): ?>
                                <?php
                                    $status = strtolower((string)$withdrawal['status']);
                                    $badgeClass = 'bg-warning text-dark';
                                    $statusLabel = 'Beklemede';
                                    if ($status === 'approved') {
                                        $badgeClass = 'bg-success';
                                        $statusLabel = 'Tamamlandı';
                                    } elseif ($status === 'rejected') {
                                        $badgeClass = 'bg-danger';
                                        $statusLabel = 'Reddedildi';
                                    }
                                    $ibanDisplay = $withdrawal['payout_iban'] ? trim(chunk_split(preg_replace('/[^A-Z0-9]/', '', strtoupper((string)$withdrawal['payout_iban'])), 4, ' ')) : '-';
                                ?>
                                <tr>
                                    <td><?= (int)$withdrawal['id'] ?></td>
                                    <td>
                                        <strong><?= sanitize($withdrawal['username']) ?></strong>
                                        <div class="small text-muted"><?= sanitize($withdrawal['email']) ?></div>
                                    </td>
                                    <td><?= date('d.m.Y H:i', strtotime($withdrawal['created_at'])) ?></td>
                                    <td><strong><?= formatPrice($withdrawal['amount']) ?></strong></td>
                                    <td>
                                        <div class="small"><?= sanitize($withdrawal['account_name'] ?? $withdrawal['payout_account_name'] ?? '-') ?></div>
                                        <div class="small text-muted"><?= sanitize($withdrawal['bank_name'] ?? $withdrawal['payout_bank_name'] ?? '-') ?></div>
                                        <div class="small text-muted"><?= sanitize($ibanDisplay ?: '-') ?></div>
                                    </td>
                                    <td>
                                        <span class="badge <?= $badgeClass ?>"><?= sanitize($statusLabel) ?></span>
                                        <?php if (!empty($withdrawal['processed_at'])): ?>
                                            <div class="small text-muted"><?= date('d.m.Y H:i', strtotime($withdrawal['processed_at'])) ?></div>
                                        <?php endif; ?>
                                        <?php if (!empty($withdrawal['admin_username'])): ?>
                                            <div class="small text-muted">İşlem: <?= sanitize($withdrawal['admin_username']) ?></div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($status === 'pending' || $status === 'approved'): ?>
                                            <form method="post" class="d-flex gap-2 align-items-center">
                                                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                <input type="hidden" name="request_id" value="<?= (int)$withdrawal['id'] ?>">
                                                <input type="hidden" name="status_filter" value="<?= sanitize($statusFilter) ?>">
                                                <input type="text" name="admin_note" class="form-control form-control-sm" placeholder="Not" value="<?= sanitize($withdrawal['admin_note'] ?? '') ?>">
                                                <div class="btn-group">
                                                    <?php if ($status === 'pending'): ?>
                                                        <button type="submit" name="action" value="approve" class="btn btn-sm btn-success" onclick="return confirm('Bu talebi onaylamak istediğinize emin misiniz?');">
                                                            <i class="fas fa-check"></i>
                                                        </button>
                                                    <?php endif; ?>
                                                    <button type="submit" name="action" value="reject" class="btn btn-sm btn-danger" onclick="return confirm('Bu talebi reddetmek istediğinize emin misiniz? <?= $status === 'approved' ? 'Bakiye kullanıcıya iade edilecektir.' : '' ?>');">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                </div>
                                            </form>
                                        <?php else: ?>
                                            <div class="small text-muted">Talep kapatıldı.</div>
                                            <?php if (!empty($withdrawal['admin_note'])): ?>
                                                <div class="small">Not: <?= sanitize($withdrawal['admin_note']) ?></div>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'includes/admin-footer.php'; ?>

