document.addEventListener('DOMContentLoaded', function() {
    initThemeToggle();
    // initMobileNav(); // Header.php'de zaten var
    initHomeSlider();
    initCarousels();
    initMarketplaceDropdowns();
    initMarketplaceModals();
    initMarketplaceUserNotifications();
    initMarketplaceOrderDetails();
    scheduleNavBalanceRefresh();
    initCopyButtons();
    initBalanceForm();

    var recaptchaContainer = document.querySelector('[data-recaptcha-site-key]');
    if (recaptchaContainer) {
        var siteKey = recaptchaContainer.dataset.recaptchaSiteKey;
        if (siteKey) {
            loadRecaptchaScript(siteKey);
            document.addEventListener('captcha-script-loaded', renderRecaptchaWidgets);
        }
    }
});

var marketplaceUserAudioContext;

function initThemeToggle() {
    var toggles = document.querySelectorAll('[data-theme-toggle]');
    
    if (!toggles || toggles.length === 0) {
        return;
    }

    var root = document.body;
    var stored = localStorage.getItem('site-theme');
    var initialTheme = root.getAttribute('data-theme') || 'light';
    var preferred = stored || initialTheme;

    function updateButtonUI(theme) {
        for (var i = 0; i < toggles.length; i++) {
            var btn = toggles[i];
            var icon = btn.querySelector('i');
            var text = btn.querySelector('span');
            if (icon) {
                icon.className = 'fa-solid ' + (theme === 'dark' ? 'fa-sun' : 'fa-moon');
            }
            if (text) {
                text.textContent = theme === 'dark' ? 'Light' : 'Dark';
            }
        }
    }

    function applyTheme(theme) {
        var nextTheme = theme === 'dark' ? 'dark' : 'light';
        root.setAttribute('data-theme', nextTheme);
        root.dataset.theme = nextTheme;
        
        try {
            localStorage.setItem('site-theme', nextTheme);
        } catch (e) {
            // LocalStorage hatası sessizce geçiliyor
        }
        
        document.cookie = 'site_theme=' + nextTheme + ';path=/;max-age=' + (60 * 60 * 24 * 365);
        updateButtonUI(nextTheme);
    }

    applyTheme(preferred);

    for (var i = 0; i < toggles.length; i++) {
        toggles[i].addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            var currentTheme = root.getAttribute('data-theme') || 'light';
            var nextTheme = currentTheme === 'dark' ? 'light' : 'dark';
            applyTheme(nextTheme);
        }, true);
    }
}

function initCopyButtons() {
    const copyButtons = document.querySelectorAll('[data-copy-code]');
    if (!copyButtons.length) {
        return;
    }

    copyButtons.forEach((button) => {
        button.addEventListener('click', () => {
            const code = button.getAttribute('data-copy-code');
            if (!code) {
                return;
            }
            navigator.clipboard.writeText(code).then(() => {
                const original = button.innerHTML;
                button.innerHTML = '<i class="fa-solid fa-check"></i> Kopyalandı';
                button.classList.remove('btn-outline-primary');
                button.classList.add('btn-success');
                setTimeout(() => {
                    button.innerHTML = original;
                    button.classList.remove('btn-success');
                    button.classList.add('btn-outline-primary');
                }, 2000);
            });
        });
    });
}

function initBalanceForm() {
    const balanceForm = document.querySelector('[data-balance-form]');
    if (!balanceForm) {
        return;
    }

    const amountInput = balanceForm.querySelector('[name="amount"]');
    const paymentSelect = balanceForm.querySelector('[name="payment_method"]');
    const submitButton = balanceForm.querySelector('[data-balance-submit]');
    const summaryContainer = balanceForm.querySelector('[data-payment-summary]');
    const summaryBase = summaryContainer ? summaryContainer.querySelector('[data-summary-base]') : null;
    const summaryRate = summaryContainer ? summaryContainer.querySelector('[data-summary-rate]') : null;
    const summaryCommission = summaryContainer ? summaryContainer.querySelector('[data-summary-commission]') : null;
    const summaryTotal = summaryContainer ? summaryContainer.querySelector('[data-summary-total]') : null;
    const paytrCommission = Number(balanceForm.dataset.commissionPaytr || 0);
    const shopierCommission = Number(balanceForm.dataset.commissionShopier || 0);
    const bankCommission = Number(balanceForm.dataset.commissionBank || 0);

    const formatCurrency = (value) => {
        return new Intl.NumberFormat('tr-TR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        }).format(value) + ' TL';
    };

    const getCommissionRateForMethod = (method) => {
        if (method === 'paytr') {
            return paytrCommission;
        }
        if (method === 'shopier') {
            return shopierCommission;
        }
        if (method === 'bank_transfer') {
            return bankCommission;
        }
        return 0;
    };

    const calculatePayable = (amount, method) => {
        const rate = getCommissionRateForMethod(method);
        const commissionAmount = amount * rate / 100;
        const payable = amount + commissionAmount;
        return {
            rate,
            commissionAmount,
            payable
        };
    };

    const updateSummary = () => {
        if (!summaryContainer || !amountInput || !paymentSelect) {
            return;
        }

        const rawAmount = Number(amountInput.value || 0);
        const method = paymentSelect.value;

        if (!method || rawAmount <= 0) {
            summaryContainer.style.display = 'none';
            summaryContainer.dataset.payable = '';
            return;
        }

        const { rate, commissionAmount, payable } = calculatePayable(rawAmount, method);

        if (rawAmount < 1) {
            summaryContainer.style.display = 'none';
            summaryContainer.dataset.payable = '';
            return;
        }

        summaryContainer.style.display = 'block';
        summaryContainer.dataset.payable = payable.toFixed(2);
        summaryContainer.dataset.rate = rate.toFixed(2);

        if (summaryBase) {
            summaryBase.textContent = formatCurrency(rawAmount);
        }
        if (summaryRate) {
            const rateText = rate === 0 ? '%0' : `%${rate.toFixed(2).replace(/\.00$/, '')}`;
            summaryRate.textContent = rateText;
        }
        if (summaryCommission) {
            summaryCommission.textContent = formatCurrency(commissionAmount);
        }
        if (summaryTotal) {
            summaryTotal.textContent = formatCurrency(payable);
        }
    };

    balanceForm.querySelectorAll('[data-set-amount]').forEach((button) => {
        button.addEventListener('click', () => {
            const value = button.getAttribute('data-set-amount');
            if (amountInput && value) {
                amountInput.value = value;
                updateSummary();
            }
        });
    });

    if (amountInput) {
        amountInput.addEventListener('input', updateSummary);
    }

    if (paymentSelect) {
        paymentSelect.addEventListener('change', updateSummary);
    }

    if (!submitButton) {
        return;
    }

    submitButton.addEventListener('click', (event) => {
        event.preventDefault();
        if (!amountInput || !paymentSelect) {
            return;
        }

        const amount = Number(amountInput.value || 0);
        const method = paymentSelect.value;

        if (Number.isNaN(amount) || amount < 1) {
            alert('Minimum yükleme tutarı 1 TL\'dir.');
            return;
        }

        if (!method) {
            alert('Lütfen ödeme yöntemi seçiniz.');
            return;
        }

        if (method === 'bank_transfer') {
            showBankTransferModal(amount);
            return;
        }

        const { commissionAmount, payable } = calculatePayable(amount, method);
        const message = `${formatCurrency(amount)} bakiye yüklemek için ${formatCurrency(payable)} ödemeyi onaylıyor musunuz?`;
        if (confirm(message)) {
            const params = new URLSearchParams();
            params.set('method', method);
            params.set('amount', amount.toFixed(2));
            if (commissionAmount > 0) {
                params.set('charge', payable.toFixed(2));
            }
            window.location.href = `payment/process.php?${params.toString()}`;
        }
    });

    updateSummary();
}


function showBankTransferModal(amount) {
    // Bootstrap modalını kapat
    const mainModal = document.querySelector('#addBalanceModal');
    if (mainModal) {
        const bsModal = bootstrap.Modal.getInstance(mainModal);
        if (bsModal) {
            bsModal.hide();
        }
    }

    // Banka bilgilerini al (config'den)
    fetch('payment/get-bank-info.php')
        .then(response => response.json())
        .then(data => {
            if (!data.success) {
                alert('Banka bilgileri alınamadı: ' + data.message);
                return;
            }

            const bankInfo = data.data;
            const commissionRate = data.commission_rate || 0;
            const commission = (amount * commissionRate) / 100;
            const total = amount + commission;

            // Modal HTML
            const modalHTML = `
                <div class="modal fade" id="bankTransferModal" tabindex="-1" aria-hidden="true">
                    <div class="modal-dialog modal-dialog-centered modal-lg">
                        <div class="modal-content">
                            <div class="modal-header border-0">
                                <h5 class="modal-title"><i class="fa-solid fa-university me-2"></i>Havale / EFT Bilgileri</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <div class="alert alert-info">
                                    <i class="fa-solid fa-info-circle me-2"></i>
                                    Lütfen aşağıdaki banka hesabına ${total.toFixed(2)} TL havale/EFT yapınız. İşlem sonrası bilgilerinizi girin ve onay bekleyin.
                                </div>

                                <div class="card mb-4">
                                    <div class="card-body">
                                        <h6 class="card-title mb-3"><i class="fa-solid fa-building-columns me-2"></i>Hesap Bilgileri</h6>
                                        ${bankInfo.bank_name ? `
                                        <div class="mb-3">
                                            <label class="text-muted small">Banka Adı</label>
                                            <div class="fw-bold">${bankInfo.bank_name}</div>
                                        </div>` : ''}
                                        <div class="mb-3">
                                            <label class="text-muted small">Hesap Sahibi</label>
                                            <div class="fw-bold">${bankInfo.account_holder}</div>
                                        </div>
                                        <div class="mb-3">
                                            <label class="text-muted small">IBAN</label>
                                            <div class="input-group">
                                                <input type="text" class="form-control" value="${bankInfo.iban}" readonly>
                                                <button class="btn btn-outline-primary" type="button" onclick="copyToClipboard('${bankInfo.iban}')">
                                                    <i class="fa-solid fa-copy"></i>
                                                </button>
                                            </div>
                                        </div>
                                        <div class="alert alert-warning mb-0">
                                            <strong>Ödenecek Tutar:</strong> ${total.toFixed(2)} TL
                                            ${commission > 0 ? `<br><small>(${amount.toFixed(2)} TL + ${commission.toFixed(2)} TL komisyon)</small>` : ''}
                                        </div>
                                    </div>
                                </div>

                                <form id="bankTransferForm">
                                    <input type="hidden" name="amount" value="${amount}">
                                    <div class="mb-3">
                                        <label class="form-label">Gönderen Ad Soyad <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="sender_name" required 
                                               placeholder="Havale yapan kişinin adı soyadı">
                                        <small class="text-muted">Banka dekontunda görünen ad soyad</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">Açıklama (Opsiyonel)</label>
                                        <textarea class="form-control" name="transfer_note" rows="2" 
                                                  placeholder="Ek açıklama veya referans numarası"></textarea>
                                    </div>
                                    <button type="submit" class="btn btn-primary w-100">
                                        <i class="fa-solid fa-check-circle me-2"></i>Havale Yaptım, Onay Bekliyor
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            // Eski modalı temizle ve yenisini ekle
            const oldModal = document.querySelector('#bankTransferModal');
            if (oldModal) {
                oldModal.remove();
            }
            document.body.insertAdjacentHTML('beforeend', modalHTML);

            // Modalı göster
            const bankModal = new bootstrap.Modal(document.querySelector('#bankTransferModal'));
            bankModal.show();

            // Form submit
            document.querySelector('#bankTransferForm').addEventListener('submit', function(e) {
                e.preventDefault();
                processBankTransfer(new FormData(this));
            });
        })
        .catch(err => {
            alert('Bir hata oluştu: ' + err.message);
        });
}

function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        alert('IBAN kopyalandı!');
    });
}

function processBankTransfer(formData) {
    const submitBtn = document.querySelector('#bankTransferForm button[type="submit"]');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>İşleniyor...';

    // CSRF token ekle
    const csrfMeta = document.querySelector('meta[name="csrf-token"]');
    if (csrfMeta) {
        formData.append('csrf_token', csrfMeta.getAttribute('content'));
    }

    fetch('payment/bank-transfer-init.php', {
        method: 'POST',
        body: formData
    })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert(data.message);
                window.location.href = data.redirect || '/balance.php?payment=pending';
            } else {
                alert('Hata: ' + data.message);
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fa-solid fa-check-circle me-2"></i>Havale Yaptım, Onay Bekliyor';
            }
        })
        .catch(err => {
            alert('Bir hata oluştu: ' + err.message);
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fa-solid fa-check-circle me-2"></i>Havale Yaptım, Onay Bekliyor';
        });
}

function simulateDemoPayment(amount) {
    const message = amount + ' TL bakiye yüklemek istediğinizden emin misiniz?';
    if (!confirm(message)) {
        return;
    }

    fetch('api/add-balance.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({ amount })
    })
        .then((response) => response.json())
        .then((data) => {
            if (data.success) {
                alert('Bakiye başarıyla yüklendi!');
                window.location.reload();
            } else {
                alert('Bir hata oluştu: ' + (data.message || 'Bilinmeyen hata'));
            }
        })
        .catch(() => {
            alert('İşlem sırasında bir hata oluştu.');
        });
}

function initMobileNav() {
    var panel = document.querySelector('[data-mobile-nav]');
    var overlay = document.querySelector('[data-nav-overlay]');
    var toggleButtons = document.querySelectorAll('[data-nav-toggle]');
    var closeButtons = document.querySelectorAll('[data-nav-close]');

    if (!panel || !overlay || toggleButtons.length === 0) {
        return;
    }

    function openNav() {
        panel.classList.add('open');
        overlay.classList.add('active');
        document.body.classList.add('nav-open');
        
        // Hamburger ikonunu X'e çevir
        for (var i = 0; i < toggleButtons.length; i++) {
            var icon = toggleButtons[i].querySelector('i');
            if (icon) {
                icon.classList.remove('fa-bars');
                icon.classList.add('fa-xmark');
            }
            toggleButtons[i].setAttribute('aria-label', 'Menüyü kapat');
        }
    }

    function closeNav() {
        panel.classList.remove('open');
        overlay.classList.remove('active');
        document.body.classList.remove('nav-open');
        
        // X ikonunu hamburger'e çevir
        for (var i = 0; i < toggleButtons.length; i++) {
            var icon = toggleButtons[i].querySelector('i');
            if (icon) {
                icon.classList.remove('fa-xmark');
                icon.classList.add('fa-bars');
            }
            toggleButtons[i].setAttribute('aria-label', 'Menüyü aç');
        }
    }

    function toggleNav() {
        if (panel.classList.contains('open')) {
            closeNav();
        } else {
            openNav();
        }
    }

    for (var i = 0; i < toggleButtons.length; i++) {
        toggleButtons[i].addEventListener('click', toggleNav);
    }
    
    for (var i = 0; i < closeButtons.length; i++) {
        closeButtons[i].addEventListener('click', closeNav);
    }
    
    overlay.addEventListener('click', closeNav);

    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape' && panel.classList.contains('open')) {
            closeNav();
        }
    });

    var dropdownTriggers = panel.querySelectorAll('[data-mobile-dropdown]');
    
    for (var i = 0; i < dropdownTriggers.length; i++) {
        dropdownTriggers[i].addEventListener('click', function(event) {
            event.preventDefault();
            var group = this.closest('.mobile-nav-group');
            if (!group) return;
            var isOpen = group.classList.contains('open');
            var openGroups = panel.querySelectorAll('.mobile-nav-group.open');
            for (var j = 0; j < openGroups.length; j++) {
                openGroups[j].classList.remove('open');
            }
            if (!isOpen) {
                group.classList.add('open');
            }
        });
    }
}

function initMarketplaceDropdowns() {
    var dropdowns = document.querySelectorAll('[data-dropdown]');
    
    for (var i = 0; i < dropdowns.length; i++) {
        (function() {
            var dropdown = dropdowns[i];
            var menu = dropdown.querySelector('.dropdown-menu');
            if (!menu) {
                return;
            }

            function open() {
                dropdown.classList.add('open');
            }
            
            function close() {
                dropdown.classList.remove('open');
            }

            dropdown.addEventListener('mouseenter', open);
            dropdown.addEventListener('mouseleave', close);
            dropdown.addEventListener('click', function(event) {
                event.preventDefault();
                dropdown.classList.toggle('open');
            });
        })();
    }

    document.addEventListener('click', function(event) {
        for (var i = 0; i < dropdowns.length; i++) {
            if (!dropdowns[i].contains(event.target)) {
                dropdowns[i].classList.remove('open');
            }
        }
    });
}

function renderRecaptchaWidgets() {
    if (typeof grecaptcha === 'undefined') {
        return;
    }
    document.querySelectorAll('[data-recaptcha]').forEach((container) => {
        // Eğer zaten render edilmişse, önce temizle
        if (container.dataset.widgetId) {
            try {
                grecaptcha.reset(parseInt(container.dataset.widgetId));
            } catch (e) {
                // Widget bulunamadıysa yeniden render et
                container.innerHTML = '';
                delete container.dataset.widgetId;
            }
        }
        
        // İlk kez render ediliyorsa
        if (!container.dataset.widgetId) {
            const siteKey = container.dataset.siteKey;
            if (!siteKey) {
                return;
            }
            try {
                const widgetId = grecaptcha.render(container, {
                    sitekey: siteKey,
                    theme: document.body.dataset.theme === 'dark' ? 'dark' : 'light'
                });
                container.dataset.widgetId = widgetId;
            } catch (e) {
                // reCAPTCHA render hatası sessizce geçiliyor
            }
        }
    });
}

function loadRecaptchaScript(siteKey) {
    if (!siteKey || document.getElementById('recaptcha-script')) {
        return;
    }
    const script = document.createElement('script');
    script.id = 'recaptcha-script';
    script.src = `https://www.google.com/recaptcha/api.js?onload=__recaptchaOnLoad&render=explicit`;
    script.async = true;
    script.defer = true;
    window.__recaptchaOnLoad = () => {
        document.dispatchEvent(new Event('captcha-script-loaded'));
        renderRecaptchaWidgets();
    };
    document.head.appendChild(script);
}

function initMarketplaceModals() {
    if (window && window.__MARKETPLACE_MODAL_OVERRIDE) {
        return;
    }
    const modal = document.getElementById('marketplaceModal');
    if (!modal) return;

    const escapeHtml = (value = '') => String(value)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');

    const modalInstance = new bootstrap.Modal(modal);
    const modalTitle = modal.querySelector('[data-modal-title]');
    const modalTypeBadge = modal.querySelector('[data-modal-type]');
    const modalForm = modal.querySelector('form');
    const modalWarning = modal.querySelector('[data-modal-warning]');
    const quantityInput = modal.querySelector('[data-modal-quantity]');
    const quantityMin = modal.querySelector('[data-modal-min]');
    const quantityMax = modal.querySelector('[data-modal-max]');
    const priceDisplay = modal.querySelector('[data-modal-price]');
    const totalDisplay = modal.querySelector('[data-modal-total]');
    const alertBox = modal.querySelector('[data-modal-alert]');
    const csrfInput = modalForm?.querySelector('[name="csrf_token"]');
    const actionInput = modalForm?.querySelector('[name="action"]');
    const serverInput = modalForm?.querySelector('[name="server_id"]');
    const characterInput = modalForm?.querySelector('[name="character_name"]');
    const characterGroup = modal.querySelector('[data-modal-character-group]');
    const characterLabel = modal.querySelector('[data-modal-character-label]');
    const characterHint = modal.querySelector('[data-modal-character-hint]');
    const submitButton = modalForm?.querySelector('button[type="submit"]');
    const currencyFormatter = new Intl.NumberFormat('tr-TR', { style: 'currency', currency: 'TRY' });

    let activeConfig = null;

    const resetAlert = () => {
        if (!alertBox) return;
        alertBox.classList.add('d-none');
        alertBox.classList.remove('alert-success', 'alert-danger');
        alertBox.innerHTML = '';
    };

    const showAlert = (type, message, codes) => {
        if (!alertBox) {
            if (message) {
                window.alert(message);
            }
            return;
        }
        alertBox.classList.remove('d-none', 'alert-success', 'alert-danger');
        alertBox.classList.add(type === 'success' ? 'alert-success' : 'alert-danger');
        let html = message ? `<div>${escapeHtml(message)}</div>` : '';
        if (type === 'success' && Array.isArray(codes) && codes.length) {
            const codeHtml = codes.map((code) => `<code class="marketplace-code">${escapeHtml(code)}</code>`).join('<br>');
            html += `<div class="mt-2"><strong>Teslim edilen kodlar</strong></div><div class="marketplace-code-list mt-1">${codeHtml}</div>`;
        }
        alertBox.innerHTML = html;
    };

    const triggerButtons = document.querySelectorAll('[data-marketplace-action]');
    triggerButtons.forEach((btn) => {
        btn.addEventListener('click', () => {
            const data = btn.dataset;
            const action = data.marketplaceAction === 'sell' ? 'sell' : 'buy';

            resetAlert();

            activeConfig = {
                serverId: parseInt(data.marketplaceServerId || '0', 10) || 0,
                action,
                name: data.marketplaceName || 'Marketplace İşlemi',
                minSell: parseFloat(data.marketplaceMinSell || '0') || 1,
                maxSell: data.marketplaceMaxSell !== '' && data.marketplaceMaxSell !== undefined ? parseFloat(data.marketplaceMaxSell) : null,
                minBuy: parseFloat(data.marketplaceMinBuy || '0') || 1,
                maxBuy: data.marketplaceMaxBuy !== '' && data.marketplaceMaxBuy !== undefined ? parseFloat(data.marketplaceMaxBuy) : null,
                buyPrice: parseFloat(data.marketplaceBuyPrice || '0') || 0,
                sellPrice: parseFloat(data.marketplaceSellPrice || '0') || 0,
                requiresCharacter: data.marketplaceRequiresCharacter === '1',
                autoApprove: data.marketplaceAutoApprove === '1'
            };

            const isSell = action === 'sell';
            modalTypeBadge.textContent = isSell ? 'Bize Sat' : 'Satın Al';
            modalTypeBadge.className = isSell ? 'badge bg-warning text-dark' : 'badge bg-primary';
            modalTitle.textContent = data.marketplaceName || 'Marketplace İşlemi';
            if (submitButton) {
                submitButton.textContent = isSell ? 'Onayla' : 'Sepete Ekle';
                submitButton.classList.toggle('btn-warning', isSell);
                submitButton.classList.toggle('btn-primary', !isSell);
            }
            if (modalWarning) {
                const warning = data.marketplaceWarning || '';
                modalWarning.textContent = warning;
                modalWarning.style.display = warning ? 'block' : 'none';
            }

            if (characterGroup) {
                const requiresCharacter = !!activeConfig.requiresCharacter;
                characterGroup.style.display = requiresCharacter ? '' : 'none';
                if (characterInput) {
                    characterInput.value = '';
                }
                if (characterHint) {
                    if (requiresCharacter) {
                        characterHint.textContent = 'Karakter adınızı eksiksiz giriniz.';
                        characterHint.style.display = 'block';
                    } else {
                        characterHint.textContent = '';
                        characterHint.style.display = 'none';
                    }
                }
                if (characterLabel) {
                    characterLabel.textContent = requiresCharacter ? 'Karakter Adı' : 'Karakter Adı (İsteğe bağlı)';
                }
            }

            const pricePerUnit = isSell ? activeConfig.buyPrice : activeConfig.sellPrice;
            const min = isSell ? activeConfig.minSell : activeConfig.minBuy;
            const max = isSell ? activeConfig.maxSell : activeConfig.maxBuy;

            if (serverInput) {
                serverInput.value = String(activeConfig.serverId);
            }
            if (actionInput) {
                actionInput.value = isSell ? '' : 'add_marketplace';
            }

            if (quantityInput) {
                quantityInput.value = Math.ceil(min);
                quantityInput.min = Math.ceil(min);
                quantityInput.step = '1';
                quantityInput.max = max !== null ? Math.floor(max) : '';
                if (quantityInput.__handler) {
                    quantityInput.removeEventListener('input', quantityInput.__handler);
                }
            }
            if (quantityMin) quantityMin.textContent = Math.ceil(min).toLocaleString('tr-TR');
            const maxWrapper = modal.querySelector('[data-modal-max-wrapper]');
            if (quantityMax) {
                if (max !== null) {
                    quantityMax.textContent = Math.floor(max).toLocaleString('tr-TR');
                    if (maxWrapper) maxWrapper.style.display = '';
                } else {
                    if (maxWrapper) maxWrapper.style.display = 'none';
                }
            }
            priceDisplay.textContent = new Intl.NumberFormat('tr-TR', { style: 'currency', currency: 'TRY' }).format(pricePerUnit);

            const updateTotal = () => {
                if (!activeConfig) {
                    return;
                }
                const isSellActive = activeConfig.action === 'sell';
                const minLimit = isSellActive ? activeConfig.minSell : activeConfig.minBuy;
                const maxLimit = isSellActive ? activeConfig.maxSell : activeConfig.maxBuy;
                const unitPrice = isSellActive ? activeConfig.buyPrice : activeConfig.sellPrice;
                const qty = parseFloat(quantityInput.value || '0');
                if (qty < minLimit) quantityInput.value = minLimit;
                if (maxLimit !== null && qty > maxLimit) quantityInput.value = maxLimit;
                const amount = parseFloat(quantityInput.value || '0');
                totalDisplay.textContent = new Intl.NumberFormat('tr-TR', { style: 'currency', currency: 'TRY' }).format(amount * unitPrice);
            };

            if (quantityInput) {
                quantityInput.addEventListener('input', updateTotal);
                quantityInput.__handler = updateTotal;
            }
            updateTotal();

            modalInstance.show();
        });
    });

    modal.addEventListener('hidden.bs.modal', () => {
        if (quantityInput && quantityInput.__handler) {
            quantityInput.removeEventListener('input', quantityInput.__handler);
            quantityInput.__handler = null;
        }
        resetAlert();
        if (characterInput) {
            characterInput.value = '';
        }
        if (quantityInput) {
            quantityInput.value = quantityInput.min || 1;
        }
        if (characterGroup) {
            characterGroup.style.display = '';
        }
        if (characterHint) {
            characterHint.style.display = '';
            characterHint.textContent = 'Karakter adınızı eksiksiz giriniz.';
        }
        if (characterLabel) {
            characterLabel.textContent = 'Karakter Adı';
        }
        activeConfig = null;
    });

    modalForm?.addEventListener('submit', (event) => {
        if (!activeConfig || !activeConfig.serverId) {
            event.preventDefault();
            showAlert('danger', 'İşlem bilgileri bulunamadı. Lütfen sayfayı yenileyin.');
            return;
        }

        const action = activeConfig.action;
        const minLimit = action === 'sell' ? activeConfig.minSell : activeConfig.minBuy;
        const maxLimit = action === 'sell' ? activeConfig.maxSell : activeConfig.maxBuy;
        const price = action === 'sell' ? activeConfig.buyPrice : activeConfig.sellPrice;
        const quantity = parseFloat(quantityInput.value || '0');
        const requiresCharacter = !!activeConfig.requiresCharacter;
        const characterName = characterInput ? characterInput.value.trim() : '';

        resetAlert();

        if (requiresCharacter && !characterName) {
            event.preventDefault();
            showAlert('danger', 'Lütfen karakter adınızı girin.');
            return;
        }

        if (quantity < minLimit) {
            event.preventDefault();
            showAlert('danger', `Minimum miktar ${minLimit.toLocaleString('tr-TR')} olmalıdır.`);
            return;
        }

        if (maxLimit !== null && quantity > maxLimit) {
            event.preventDefault();
            showAlert('danger', `Maksimum miktar ${maxLimit.toLocaleString('tr-TR')} değerini aşamaz.`);
            return;
        }

        if (action === 'buy') {
            if (serverInput) {
                serverInput.value = String(activeConfig.serverId);
            }
            if (actionInput) {
                actionInput.value = 'add_marketplace';
            }
            return;
        }

        event.preventDefault();
        if (action === 'sell') {
            const confirmMessage = `${activeConfig.name} ilanını bize satmak üzeresiniz. Miktar: ${quantity.toLocaleString('tr-TR')}. Onaylıyor musunuz?`;
            if (!window.confirm(confirmMessage)) {
                return;
            }

            const payload = {
                csrf_token: csrfInput ? csrfInput.value : '',
                server_id: activeConfig.serverId,
                order_type: action,
                quantity,
                character_name: characterName
            };

            if (submitButton) {
                submitButton.disabled = true;
                submitButton.dataset.originalText = submitButton.dataset.originalText || submitButton.innerHTML;
                submitButton.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Gönderiliyor';
            }

            fetch('marketplace/api/create-order.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'include',
                body: JSON.stringify(payload)
            })
                .then((res) => res.json())
                .then((data) => {
                    if (submitButton) {
                        submitButton.disabled = false;
                        submitButton.innerHTML = submitButton.dataset.originalText || 'Onayla';
                    }

                    if (data && data.summary && typeof data.summary.balance !== 'undefined') {
                        updateNavBalanceDisplay(parseFloat(data.summary.balance) || 0);
                    }

                    if (alertBox) {
                        alertBox.textContent = data.message || '';
                        alertBox.classList.remove('d-none');
                        alertBox.classList.add(data.success ? 'alert-success' : 'alert-danger');
                    } else if (data.message) {
                        alert(data.message);
                    }

                    const deliveredCodes = Array.isArray(data.delivered_codes) ? data.delivered_codes : [];
                    showAlert(data.success ? 'success' : 'danger', data.message || '', deliveredCodes);

                    if (!data.success) {
                        return;
                    }

                    if (characterInput) {
                        characterInput.value = '';
                    }
                    if (quantityInput) {
                        quantityInput.value = minLimit;
                    }

                    if (!deliveredCodes.length) {
                        setTimeout(() => {
                            modalInstance.hide();
                        }, 1400);
                    }
                })
                .catch(() => {
                    if (submitButton) {
                        submitButton.disabled = false;
                        submitButton.innerHTML = submitButton.dataset.originalText || 'Onayla';
                    }
                    showAlert('danger', 'İşlem sırasında bir hata oluştu.');
                });
        }
    });
}

function initMarketplaceUserNotifications() {
    ensureUserAudioUnlock();

    let lastCheck = new Date().toISOString();
    
    // Get API endpoint from global config
    var pollEndpoint = 'marketplace/api/poll-user.php';
    if (window.APP_CONFIG && window.APP_CONFIG.apiEndpoints && window.APP_CONFIG.apiEndpoints.marketplacePoll) {
        pollEndpoint = window.APP_CONFIG.apiEndpoints.marketplacePoll;
    }

    const poll = () => {
        fetch(`${pollEndpoint}?since=${encodeURIComponent(lastCheck)}`, {
            credentials: 'include'
        })
            .then((res) => res.json())
            .then((data) => {
                if (!data || !data.success) {
                    return;
                }
                lastCheck = data.timestamp || new Date().toISOString();
                if (data.count && data.count > 0) {
                    playUserNotificationTone();
                }
            })
            .catch(() => {});
    };

    poll();
    setInterval(poll, 6000);
}

function getUserAudioContext() {
    if (typeof window.AudioContext === 'undefined' && typeof window.webkitAudioContext === 'undefined') {
        return null;
    }
    if (!marketplaceUserAudioContext) {
        const Ctx = window.AudioContext || window.webkitAudioContext;
        marketplaceUserAudioContext = new Ctx();
    }
    return marketplaceUserAudioContext;
}

function ensureUserAudioUnlock() {
    const ctx = getUserAudioContext();
    if (!ctx) {
        return;
    }
    const unlock = () => {
        if (ctx.state === 'suspended') {
            ctx.resume().catch(() => {});
        }
    };
    document.addEventListener('pointerdown', unlock, { once: true });
    document.addEventListener('keydown', unlock, { once: true });
}

function playUserNotificationTone() {
    const ctx = getUserAudioContext();
    if (!ctx) {
        return;
    }

    if (ctx.state === 'suspended') {
        ctx.resume().catch(() => {});
    }

    const duration = 4.2;
    const oscillator = ctx.createOscillator();
    const gain = ctx.createGain();

    oscillator.type = 'sine';
    oscillator.connect(gain);
    gain.connect(ctx.destination);

    const now = ctx.currentTime;
    oscillator.frequency.setValueAtTime(420, now);
    oscillator.frequency.linearRampToValueAtTime(560, now + 1.8);
    oscillator.frequency.linearRampToValueAtTime(480, now + 3.6);

    gain.gain.setValueAtTime(0.0001, now);
    gain.gain.exponentialRampToValueAtTime(0.18, now + 0.2);
    gain.gain.exponentialRampToValueAtTime(0.0001, now + duration);

    oscillator.start(now);
    oscillator.stop(now + duration);
}

function initMarketplaceOrderDetails() {
    const modalEl = document.getElementById('marketplaceOrderModal');
    if (!modalEl) return;
    if (typeof bootstrap === 'undefined' || !bootstrap.Modal) {
        return;
    }
    const modal = bootstrap.Modal.getOrCreateInstance(modalEl);
    const fieldMap = {
        id: modalEl.querySelector('[data-field="id"]'),
        date: modalEl.querySelector('[data-field="date"]'),
        game: modalEl.querySelector('[data-field="game"]'),
        category: modalEl.querySelector('[data-field="category"]'),
        server: modalEl.querySelector('[data-field="server"]'),
        character: modalEl.querySelector('[data-field="character"]'),
        type: modalEl.querySelector('[data-field="type"]'),
        status: modalEl.querySelector('[data-field="status"]'),
        quantity: modalEl.querySelector('[data-field="quantity"]'),
        price: modalEl.querySelector('[data-field="price"]'),
        total: modalEl.querySelector('[data-field="total"]'),
        codes: modalEl.querySelector('[data-field="codes"]')
    };

    const typeBadgeClass = {
        buy: 'badge bg-primary',
        sell: 'badge bg-warning text-dark'
    };

    const statusBadgeClass = {
        pending: 'badge bg-warning text-dark',
        awaiting_payment: 'badge bg-info text-dark',
        processing: 'badge bg-info text-dark',
        approved: 'badge bg-success',
        completed: 'badge bg-success',
        cancelled: 'badge bg-danger',
        rejected: 'badge bg-danger'
    };

    document.querySelectorAll('[data-order-detail]').forEach((button) => {
        button.addEventListener('click', () => {
            Object.keys(fieldMap).forEach((key) => {
                if (fieldMap[key]) {
                    let value = button.dataset[`order${key.charAt(0).toUpperCase()}${key.slice(1)}`];
                    if (!value || value.trim() === '' || value === '-') {
                        value = '—';
                    }
                    if (key === 'codes' && value !== '—') {
                        value = value.replace(/,\s*/g, '\n');
                    }
                    fieldMap[key].textContent = value;
                }
            });

            const orderType = (button.dataset.orderType || '').toLowerCase();
            if (fieldMap.type) {
                fieldMap.type.className = typeBadgeClass[orderType] || 'badge bg-secondary';
                fieldMap.type.textContent = orderType === 'sell' ? 'Bize Sat' : 'Satın Al';
            }

            if (fieldMap.status) {
                const statusText = button.dataset.orderStatus || '';
                const statusKey = statusText.toLowerCase();
                fieldMap.status.className = statusBadgeClass[statusKey] || 'badge bg-secondary';
                fieldMap.status.textContent = statusText.toUpperCase();
            }

            modal.show();
        });
    });
}

function formatCurrency(value) {
    return new Intl.NumberFormat('tr-TR', { style: 'currency', currency: 'TRY' }).format(value);
}

function updateNavBalanceDisplay(amount, highlight = true) {
    const balanceEl = document.getElementById('nav-balance-display');
    if (!balanceEl) return;
    balanceEl.innerHTML = `<i class="fa-solid fa-wallet"></i>${formatCurrency(amount)}`;
    if (highlight) {
        balanceEl.classList.add('nav-updating-balance');
        setTimeout(() => balanceEl.classList.remove('nav-updating-balance'), 1800);
    }
}

function updateCartCount(count) {
    const badge = document.querySelector('[data-cart-count]');
    const mobile = document.querySelector('[data-cart-count-mobile]');
    const value = parseInt(count, 10) || 0;
    if (badge) {
        badge.textContent = value;
        if (value > 0) {
            badge.classList.remove('d-none');
        } else {
            badge.classList.add('d-none');
        }
    }
    if (mobile) {
        mobile.textContent = value;
    }
}

function scheduleNavBalanceRefresh() {
    if (typeof window.fetch !== 'function') return;
    
    // Get API endpoint from global config
    var balanceEndpoint = 'api/balance.php';
    if (window.APP_CONFIG && window.APP_CONFIG.apiEndpoints && window.APP_CONFIG.apiEndpoints.balance) {
        balanceEndpoint = window.APP_CONFIG.apiEndpoints.balance;
    }
    
    const refresh = () => {
        fetch(balanceEndpoint, { credentials: 'include' })
            .then((res) => res.json())
            .then((data) => {
                if (data && typeof data.balance !== 'undefined') {
                    updateNavBalanceDisplay(parseFloat(data.balance) || 0, false);
                }
            })
            .catch(() => {});
    };
    refresh();
    setInterval(refresh, 30000);
}

function initHomeSlider() {
    if (typeof Swiper === 'undefined') {
        setTimeout(initHomeSlider, 100);
        return;
    }
    
    const swiperElement = document.querySelector('.fp-swiper-home');
    if (!swiperElement) return;
    
    const sliderArea = document.querySelector('.fp-slider-area');
    const autoplayEnabled = sliderArea ? sliderArea.dataset.autoplay === 'true' : true;
    const autoplayDelay = sliderArea ? (parseInt(sliderArea.dataset.delay) || 5000) : 5000;
    const slides = swiperElement.querySelectorAll('.swiper-slide');
    const totalSlides = slides.length;
    
    if (totalSlides == 1) {
        new Swiper(".fp-swiper-home", {
            spaceBetween: 24,
            navigation: {
                nextEl: ".fp-swiper-home .swiper-button-next",
                prevEl: ".fp-swiper-home .swiper-button-prev",
            }
        });
    } else {
        new Swiper(".fp-swiper-home", {
            spaceBetween: 24,
            autoplay: autoplayEnabled ? {
                delay: autoplayDelay,
                disableOnInteraction: false,
                pauseOnMouseEnter: true
            } : false,
            navigation: {
                nextEl: ".fp-swiper-home .swiper-button-next",
                prevEl: ".fp-swiper-home .swiper-button-prev",
            },
            loop: true
        });
    }
}

// Carousel Navigation (Popüler Kategoriler)
function initCarousels() {
    var carousels = document.querySelectorAll('[id$="-categories"]');
    
    for (var i = 0; i < carousels.length; i++) {
        (function() {
            var carousel = carousels[i];
            var carouselId = carousel.id;
            var prevBtn = document.querySelector('.carousel-nav--prev[data-carousel="' + carouselId + '"]');
            var nextBtn = document.querySelector('.carousel-nav--next[data-carousel="' + carouselId + '"]');
            
            if (!prevBtn || !nextBtn) return;
            
            function updateButtons() {
                var scrollLeft = carousel.scrollLeft;
                var maxScroll = carousel.scrollWidth - carousel.clientWidth;
                prevBtn.disabled = scrollLeft <= 0;
                nextBtn.disabled = scrollLeft >= maxScroll - 5;
            }
            
            function scroll(direction) {
                var scrollAmount = carousel.clientWidth * 0.8;
                var targetScroll = carousel.scrollLeft + (direction === 'next' ? scrollAmount : -scrollAmount);
                carousel.scrollTo({ left: targetScroll, behavior: 'smooth' });
                setTimeout(updateButtons, 300);
            }
            
            prevBtn.addEventListener('click', function(e) {
                e.preventDefault();
                scroll('prev');
            });
            nextBtn.addEventListener('click', function(e) { 
                e.preventDefault();
                scroll('next');
            });
            carousel.addEventListener('scroll', updateButtons);
            updateButtons();
        })();
    }
}

// Advertisement Click Tracking
document.addEventListener("DOMContentLoaded", function() {
    document.querySelectorAll(".advertisement-link[data-ad-id]").forEach(function(link) {
        link.addEventListener("click", function() {
            var adId = this.getAttribute("data-ad-id");
            if (adId) {
                // Track click asynchronously
                fetch(BASE_URL + "api/track-ad-click.php", {
                    method: "POST",
                    headers: {
                        "Content-Type": "application/json"
                    },
                    body: JSON.stringify({ ad_id: adId })
                }).catch(function() {
                    // Silent fail
                });
            }
        });
    });
});
