(function () {
    document.addEventListener('DOMContentLoaded', () => {
        const modalEl = document.getElementById('marketplaceModal');
        if (!modalEl) {
            return;
        }
        if (typeof bootstrap === 'undefined' || !bootstrap.Modal) {
            return;
        }

        const modal = bootstrap.Modal.getOrCreateInstance(modalEl);
        const form = modalEl.querySelector('form');
        if (!form) {
            return;
        }

        const csrfInput = form.querySelector('[name="csrf_token"]');
        const actionInput = form.querySelector('[name="action"]');
        const serverInput = form.querySelector('[name="server_id"]');
        const quantityInput = form.querySelector('[name="quantity"]');
        const characterInput = form.querySelector('[name="character_name"]');
        const submitButton = form.querySelector('button[type="submit"]');

        const alertBox = modalEl.querySelector('[data-modal-alert]');
        const warningBox = modalEl.querySelector('[data-modal-warning]');
        const minDisplay = modalEl.querySelector('[data-modal-min]');
        const maxDisplay = modalEl.querySelector('[data-modal-max]');
        const priceDisplay = modalEl.querySelector('[data-modal-price]');
        const totalDisplay = modalEl.querySelector('[data-modal-total]');
        const typeBadge = modalEl.querySelector('[data-modal-type]');
        const titleEl = modalEl.querySelector('[data-modal-title]');
        const characterGroup = modalEl.querySelector('[data-modal-character-group]');
        const characterLabel = modalEl.querySelector('[data-modal-character-label]');
        const characterHint = modalEl.querySelector('[data-modal-character-hint]');

        const currencyFormatter = new Intl.NumberFormat('tr-TR', { style: 'currency', currency: 'TRY' });
        const escapeHtml = (value) => String(value)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;');

        let currentConfig = null;
        let currentAction = 'buy';

        const resetAlert = () => {
            if (!alertBox) return;
            alertBox.classList.add('d-none');
            alertBox.classList.remove('alert-success', 'alert-danger');
            alertBox.innerHTML = '';
        };

        const showAlert = (type, message) => {
            if (!alertBox) {
                window.alert(message);
                return;
            }
            alertBox.classList.remove('d-none', 'alert-success', 'alert-danger');
            alertBox.classList.add(type === 'success' ? 'alert-success' : 'alert-danger');
            alertBox.innerHTML = message ? `<div>${message}</div>` : '';
        };

        const updateTotals = () => {
            if (!currentConfig || !totalDisplay) {
                return;
            }
            const unitPrice = currentAction === 'sell' ? currentConfig.buyPrice : currentConfig.sellPrice;
            const qty = parseFloat(quantityInput.value || '0');
            totalDisplay.textContent = currencyFormatter.format(qty * unitPrice);
        };

        const applyConfig = (config) => {
            currentConfig = config;
            currentAction = config.action;

            resetAlert();
            if (warningBox) {
                const warning = config.warning || '';
                warningBox.textContent = warning;
                warningBox.style.display = warning ? 'block' : 'none';
            }

            if (serverInput) {
                serverInput.value = config.serverId.toString();
            }

            const requiresCharacter = !!config.requiresCharacter;
            if (characterGroup) {
                characterGroup.style.display = requiresCharacter ? '' : 'none';
            }
            if (characterInput) {
                characterInput.value = '';
                characterInput.required = requiresCharacter;
            }
            if (characterLabel) {
                characterLabel.textContent = requiresCharacter ? 'Karakter Adı' : 'Karakter Adı (İsteğe bağlı)';
            }
            if (characterHint) {
                characterHint.style.display = requiresCharacter ? 'block' : 'none';
            }

            const typeText = currentAction === 'sell' ? 'Bize Sat' : 'Satın Al';
            if (typeBadge) {
                typeBadge.textContent = typeText;
                typeBadge.className = currentAction === 'sell' ? 'badge bg-warning text-dark' : 'badge bg-primary';
            }
            if (titleEl) {
                titleEl.textContent = config.name || 'Marketplace İşlemi';
            }

            const unitPrice = currentAction === 'sell' ? config.buyPrice : config.sellPrice;
            if (priceDisplay) {
                priceDisplay.textContent = currencyFormatter.format(unitPrice);
            }

            const minLimit = currentAction === 'sell' ? config.minSell : config.minBuy;
            const maxLimit = currentAction === 'sell' ? config.maxSell : config.maxBuy;

            if (quantityInput) {
                quantityInput.min = Math.ceil(minLimit);
                quantityInput.step = '1';
                quantityInput.value = Math.ceil(minLimit);
                quantityInput.max = maxLimit !== null ? Math.floor(maxLimit) : '';
            }

            if (minDisplay) {
                minDisplay.textContent = Math.ceil(minLimit).toLocaleString('tr-TR');
            }
            const maxWrapper = modalEl.querySelector('[data-modal-max-wrapper]');
            if (maxDisplay) {
                if (maxLimit !== null) {
                    maxDisplay.textContent = Math.floor(maxLimit).toLocaleString('tr-TR');
                    if (maxWrapper) maxWrapper.style.display = '';
                } else {
                    if (maxWrapper) maxWrapper.style.display = 'none';
                }
            }

            updateTotals();

            if (submitButton) {
                submitButton.textContent = currentAction === 'sell' ? 'Onayla' : 'Sepete Ekle';
                submitButton.classList.toggle('btn-warning', currentAction === 'sell');
                submitButton.classList.toggle('btn-primary', currentAction !== 'sell');
            }

            if (actionInput) {
                actionInput.value = currentAction === 'sell' ? '' : 'add_marketplace';
            }
            if (form) {
                // Get cart URL from modal data attribute
                const cartUrl = modalEl.dataset.cartUrl || 'cart.php';
                form.setAttribute('action', currentAction === 'sell' ? '#' : cartUrl);
            }
        };

    document.querySelectorAll('[data-marketplace-action]').forEach((btn) => {
        btn.addEventListener('click', (event) => {
            event.preventDefault();
            const data = btn.dataset;
            const action = data.marketplaceAction === 'sell' ? 'sell' : 'buy';
            const config = {
                action,
                serverId: parseInt(data.marketplaceServerId || '0', 10) || 0,
                name: data.marketplaceName || 'Marketplace İşlemi',
                warning: data.marketplaceWarning || '',
                buyPrice: parseFloat(data.marketplaceBuyPrice || '0') || 0,
                sellPrice: parseFloat(data.marketplaceSellPrice || '0') || 0,
                minSell: parseFloat(data.marketplaceMinSell || '0') || 0,
                maxSell: data.marketplaceMaxSell ? parseFloat(data.marketplaceMaxSell) : null,
                minBuy: parseFloat(data.marketplaceMinBuy || '0') || 0,
                maxBuy: data.marketplaceMaxBuy ? parseFloat(data.marketplaceMaxBuy) : null,
                requiresCharacter: data.marketplaceRequiresCharacter === '1',
                autoApprove: data.marketplaceAutoApprove === '1'
            };

            applyConfig(config);
            modal.show();
        });
    });

        if (quantityInput) {
            quantityInput.addEventListener('input', () => {
                if (!currentConfig) {
                    return;
                }
                let value = parseInt(quantityInput.value || '0', 10);
                if (Number.isNaN(value) || value <= 0) {
                    value = Math.ceil(currentAction === 'sell' ? currentConfig.minSell : currentConfig.minBuy);
                }
                const minLimit = Math.ceil(currentAction === 'sell' ? currentConfig.minSell : currentConfig.minBuy);
                const maxLimit = currentAction === 'sell' ? currentConfig.maxSell : currentConfig.maxBuy;
                const maxLimitInt = maxLimit !== null ? Math.floor(maxLimit) : null;
                
                if (value < minLimit) {
                    value = minLimit;
                }
                if (maxLimitInt !== null && value > maxLimitInt) {
                    value = maxLimitInt;
                }
                quantityInput.value = value;
                updateTotals();
            });
        }

        modalEl.addEventListener('hidden.bs.modal', () => {
            currentConfig = null;
            if (quantityInput) {
                quantityInput.value = '1';
            }
            if (characterInput) {
                characterInput.value = '';
            }
            resetAlert();
        });

        form.addEventListener('submit', (event) => {
            if (!currentConfig) {
                event.preventDefault();
                showAlert('danger', 'İşlem yapılamadı. Sayfayı yenileyip tekrar deneyin.');
                return;
            }

            if (currentAction === 'buy') {
                return; // normal form submission to cart.php
            }

            event.preventDefault();

            if (!csrfInput) {
                showAlert('danger', 'Güvenlik anahtarı bulunamadı.');
                return;
            }

            const quantity = parseFloat(quantityInput.value || '0');
            const requiresCharacter = !!currentConfig.requiresCharacter;
            const characterName = (characterInput?.value || '').trim();

            if (requiresCharacter && characterName === '') {
                showAlert('danger', 'Lütfen karakter adınızı girin.');
                return;
            }

            const minLimit = currentAction === 'sell' ? currentConfig.minSell : currentConfig.minBuy;
            const maxLimit = currentAction === 'sell' ? currentConfig.maxSell : currentConfig.maxBuy;
            if (quantity < minLimit) {
                showAlert('danger', `Minimum miktar ${minLimit.toLocaleString('tr-TR')} olmalıdır.`);
                return;
            }
            if (maxLimit !== null && quantity > maxLimit) {
                showAlert('danger', `Maksimum miktar ${maxLimit.toLocaleString('tr-TR')} değerini aşamaz.`);
                return;
            }

            if (submitButton) {
                submitButton.disabled = true;
                submitButton.dataset.originalText = submitButton.dataset.originalText || submitButton.textContent;
                submitButton.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Gönderiliyor';
            }

            const payload = {
                csrf_token: csrfInput.value,
                server_id: currentConfig.serverId,
                order_type: currentAction,
                quantity,
                character_name: characterName
            };

            // Get API URL from modal data attribute
            const apiUrl = modalEl.dataset.apiUrl || 'marketplace/api/create-order.php';

            fetch(apiUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'include',
                body: JSON.stringify(payload)
            })
                .then((res) => res.json())
                .then((data) => {
                    if (submitButton) {
                        submitButton.disabled = false;
                        submitButton.innerHTML = submitButton.dataset.originalText || 'Onayla';
                    }

                    if (data && data.success) {
                        showAlert('success', data.message || 'İşleminiz başarıyla alındı.');
                        if (Array.isArray(data.delivered_codes) && data.delivered_codes.length) {
                            alertBox.innerHTML += '<div class="mt-2"><strong>Teslim edilen kodlar</strong></div>' + data.delivered_codes.map((code) => `<code class="marketplace-code">${escapeHtml(code)}</code>`).join('<br>');
                        }
                        if (data.summary && typeof data.summary.balance !== 'undefined') {
                            const newBalance = parseFloat(data.summary.balance) || 0;
                            if (typeof updateNavBalanceDisplay === 'function') {
                                updateNavBalanceDisplay(newBalance);
                            }
                        }
                        setTimeout(() => modal.hide(), 1500);
                    } else {
                        showAlert('danger', (data && data.message) || 'İşlem sırasında bir hata oluştu.');
                    }
                })
                .catch(() => {
                    if (submitButton) {
                        submitButton.disabled = false;
                        submitButton.innerHTML = submitButton.dataset.originalText || 'Onayla';
                    }
                    showAlert('danger', 'İşlem sırasında bir hata oluştu.');
                });
        });
    });
})();
