<?php
require_once 'config.php';
requireLogin();

$user = getUserInfo($pdo, $_SESSION['user_id']);

// Shopier'den dönüş kontrolü
$shopierRedirect = isset($_GET['shopier_redirect']) && $_GET['shopier_redirect'] == '1';
if ($shopierRedirect && isset($_GET['payment']) && $_GET['payment'] === 'success') {
    // Son yapılan Shopier ödemesini kontrol et
    $checkStmt = $pdo->prepare('
        SELECT id, order_number, status, created_at 
        FROM orders 
        WHERE user_id = ? 
        AND payment_method = "shopier" 
        AND created_at > DATE_SUB(NOW(), INTERVAL 10 MINUTE)
        ORDER BY created_at DESC 
        LIMIT 1
    ');
    $checkStmt->execute([$_SESSION['user_id']]);
    $recentOrder = $checkStmt->fetch();
    
    if ($recentOrder) {
        if ($recentOrder['status'] === 'completed') {
            // Ödeme tamamlandı
            flashMessage('success', 'Ödemeniz başarıyla tamamlandı. Bakiyeniz güncellendi!');
            header('Location: balance.php');
            exit;
        } elseif ($recentOrder['status'] === 'pending') {
            // Hala pending - 2 saniye bekle ve tekrar kontrol et
            sleep(2);
            
            $recheckStmt = $pdo->prepare('SELECT status FROM orders WHERE id = ?');
            $recheckStmt->execute([$recentOrder['id']]);
            $updatedOrder = $recheckStmt->fetch();
            
            if ($updatedOrder && $updatedOrder['status'] === 'completed') {
                // Şimdi tamamlandı
                flashMessage('success', 'Ödemeniz başarıyla tamamlandı. Bakiyeniz güncellendi!');
                header('Location: balance.php');
                exit;
            } else {
                // Hala pending - otomatik yenileme için parametreyi temizle
                // Pending payment özelliği devreye girecek
                flashMessage('info', 'Ödemeniz işleniyor... Sayfa otomatik olarak güncellenecek.');
                header('Location: balance.php');
                exit;
            }
        } elseif ($recentOrder['status'] === 'cancelled') {
            // İptal edildi
            flashMessage('danger', 'Ödeme işlemi başarısız oldu.');
            header('Location: balance.php');
            exit;
        }
    }
    
    // Sipariş bulunamadı veya diğer durumlar
    header('Location: balance.php');
    exit;
}

// Sayfalama için değişkenler
$perPage = 10;
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$offset = ($page - 1) * $perPage;

// Toplam transaction sayısı
$countStmt = $pdo->prepare('SELECT COUNT(*) FROM transactions WHERE user_id = ?');
$countStmt->execute([$_SESSION['user_id']]);
$totalTransactions = (int)$countStmt->fetchColumn();
$totalPages = max(1, ceil($totalTransactions / $perPage));

// Sayfa geçerli mi kontrol
if ($page > $totalPages) {
    $page = $totalPages;
    $offset = ($page - 1) * $perPage;
}

$txnStmt = $pdo->prepare('
    SELECT t.*, t.created_at AS txn_created_at, o.status AS order_status
    FROM transactions t
    LEFT JOIN orders o ON t.order_id = o.id
    WHERE t.user_id = ?
    ORDER BY t.created_at DESC
    LIMIT ? OFFSET ?
');
$txnStmt->bindValue(1, $_SESSION['user_id'], PDO::PARAM_INT);
$txnStmt->bindValue(2, $perPage, PDO::PARAM_INT);
$txnStmt->bindValue(3, $offset, PDO::PARAM_INT);
$txnStmt->execute();
$transactions = $txnStmt->fetchAll();

// Son 10 dakikada pending ödeme var mı kontrol et
$pendingCheckStmt = $pdo->prepare('
    SELECT COUNT(*) 
    FROM orders 
    WHERE user_id = ? 
    AND status = "pending" 
    AND payment_method IN ("shopier", "paytr")
    AND created_at > DATE_SUB(NOW(), INTERVAL 10 MINUTE)
');
$pendingCheckStmt->execute([$_SESSION['user_id']]);
$hasPendingPayment = (int)$pendingCheckStmt->fetchColumn() > 0;

$displayTransactions = [];
foreach ($transactions as $transRow) {
    $isDeposit = ($transRow['type'] ?? '') === 'deposit';
    $hasOrder = !empty($transRow['order_id']);
    $orderStatus = $transRow['order_status'] ?? null;
    // Sadece cancelled durumundaki deposit işlemlerini gizle
    // pending ve completed durumundakileri göster
    if ($isDeposit && $hasOrder && $orderStatus === 'cancelled') {
        continue;
    }
    $displayTransactions[] = $transRow;
}

// Hediye kodu action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'use_gift_code') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('balance.php');
    }
    
    $giftCode = strtoupper(trim($_POST['gift_code'] ?? ''));
    if ($giftCode === '') {
        flashMessage('danger', 'Lütfen hediye kodunu girin.');
        redirect('balance.php');
    }
    
    try {
        $pdo->beginTransaction();
        
        // Kodu kontrol et ve kilitle
        $stmt = $pdo->prepare('SELECT * FROM gift_codes WHERE code = ? AND is_active = 1 FOR UPDATE');
        $stmt->execute([$giftCode]);
        $gift = $stmt->fetch();
        
        if (!$gift) {
            throw new Exception('Geçersiz hediye kodu.');
        }
        
        if ($gift['is_used']) {
            throw new Exception('Bu hediye kodu zaten kullanılmış.');
        }
        
        $now = new DateTime();
        if ($gift['start_date'] && new DateTime($gift['start_date']) > $now) {
            throw new Exception('Bu hediye kodu henüz aktif değil.');
        }
        if ($gift['end_date'] && new DateTime($gift['end_date']) < $now) {
            throw new Exception('Bu hediye kodunun süresi dolmuş.');
        }
        
        $amount = (float)$gift['amount'];
        
        // Bakiyeyi güncelle
        $updateBalance = $pdo->prepare('UPDATE users SET balance = balance + ? WHERE id = ?');
        $updateBalance->execute([$amount, $_SESSION['user_id']]);
        
        // Hediye kodunu güncelle
        $updateGift = $pdo->prepare('UPDATE gift_codes SET is_used = 1, used_by_user_id = ?, used_at = NOW(), used_count = used_count + 1 WHERE id = ?');
        $updateGift->execute([$_SESSION['user_id'], $gift['id']]);
        
        // Kullanım kaydı oluştur
        $insertUsage = $pdo->prepare('INSERT INTO gift_code_usage (gift_code_id, user_id, amount, ip_address) VALUES (?, ?, ?, ?)');
        $insertUsage->execute([$gift['id'], $_SESSION['user_id'], $amount, $_SERVER['REMOTE_ADDR'] ?? null]);
        
        // Transaction kaydı
        $insertTxn = $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description) VALUES (?, "deposit", ?, ?)');
        $insertTxn->execute([$_SESSION['user_id'], $amount, 'Hediye kodu kullanımı: ' . $giftCode]);
        
        $pdo->commit();
        
        // Session bakiyesini güncelle
        $balanceStmt = $pdo->prepare('SELECT balance FROM users WHERE id = ?');
        $balanceStmt->execute([$_SESSION['user_id']]);
        $_SESSION['balance'] = (float)$balanceStmt->fetchColumn();
        
        flashMessage('success', 'Hediye kodu başarıyla kullanıldı! ' . formatPrice($amount) . ' bakiyenize eklendi.');
    } catch (Exception $e) {
        $pdo->rollBack();
        flashMessage('danger', $e->getMessage());
    }
    
    redirect('balance.php');
}

$paytrCommissionRate = (float)getCommissionRate('paytr');
$shopierCommissionRate = (float)getCommissionRate('shopier');
$bankTransferCommissionRate = (float)getCommissionRate('bank_transfer');

$credentialIsValid = static function ($value): bool {
    $trimmed = trim((string)$value);
    if ($trimmed === '') {
        return false;
    }
    return !preg_match('/^x+$/i', $trimmed);
};

$paytrEnabled = getBoolSetting('paytr_enabled', true);
$paytrConfigured = $paytrEnabled
    && $credentialIsValid(getSetting('paytr_merchant_id', PAYTR_MERCHANT_ID))
    && $credentialIsValid(getSetting('paytr_merchant_key', PAYTR_MERCHANT_KEY))
    && $credentialIsValid(getSetting('paytr_merchant_salt', PAYTR_MERCHANT_SALT));

$shopierEnabled = getBoolSetting('shopier_enabled', true);
$shopierConfigured = $shopierEnabled
    && $credentialIsValid(getSetting('shopier_api_key', SHOPIER_API_KEY))
    && $credentialIsValid(getSetting('shopier_api_secret', SHOPIER_API_SECRET));

$bankTransferEnabled = getBoolSetting('bank_transfer_enabled', false);
$bankTransferConfigured = $bankTransferEnabled 
    && trim(getSetting('bank_transfer_iban', '')) !== ''
    && trim(getSetting('bank_transfer_account_holder', '')) !== '';

$pageTitle = 'Bakiye';
include 'includes/header.php';
?>

<section class="account-section">
    <div class="container-wide">
        <div class="section-heading">
            <h2 class="section-title">Bakiye Yönetimi</h2>
            <p class="section-subtitle">Bakiyenizi görüntüleyin, hızlıca yükleyin ve en güncel işlemlerinizi takip edin.</p>
        </div>

        <div class="account-grid">
            <div class="balance-summary card-glass">
                <span class="balance-label">Mevcut Bakiyeniz</span>
                <div class="balance-amount"><?= formatPrice($user['balance']) ?></div>
                <div class="summary-actions mt-3">
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addBalanceModal">
                        <i class="fa-solid fa-circle-plus me-2"></i>Bakiye Yükle
                    </button>
                    <a href="<?= routeUrl('orders') ?>" class="btn btn-outline-primary">
                        <i class="fa-solid fa-receipt me-2"></i>Siparişlerim
                    </a>
                </div>
            </div>

            <div class="transactions-card">
                <div class="card-head">
                    <h5 class="mb-0"><i class="fa-solid fa-clock-rotate-left me-2"></i>İşlem Geçmişi</h5>
                </div>
                <div class="table-responsive">
                    <?php if (empty($displayTransactions)): ?>
                        <div class="p-4 text-center text-muted">Henüz işlem kaydınız bulunmuyor.</div>
                    <?php else: ?>
                        <table class="transactions-table">
                            <thead>
                                <tr>
                                    <th>Tarih</th>
                                    <th>Tip</th>
                                    <th>Açıklama</th>
                                    <th class="text-end">Tutar</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($displayTransactions as $trans): ?>
                                <tr>
                                    <?php $createdAt = $trans['txn_created_at'] ?? $trans['created_at'] ?? null; ?>
                                    <td><?= $createdAt ? date('d.m.Y H:i', strtotime($createdAt)) : '-' ?></td>
                                    <td>
                                        <?php
                                            $txnType = $trans['type'] ?? '';
                                            switch ($txnType) {
                                                case 'deposit':
                                                    echo '<span class="tag-pill"><i class="fa-solid fa-arrow-down"></i>Yükleme</span>';
                                                    break;
                                                case 'withdraw':
                                                    echo '<span class="tag-pill"><i class="fa-solid fa-arrow-up"></i>Çekim</span>';
                                                    break;
                                                case 'refund':
                                                    echo '<span class="tag-pill"><i class="fa-solid fa-rotate-left"></i>İade</span>';
                                                    break;
                                                default:
                                                    echo '<span class="tag-pill"><i class="fa-solid fa-cart-shopping"></i>Alışveriş</span>';
                                                    break;
                                            }
                                        ?>
                                    </td>
                                    <td><?= sanitize($trans['description']) ?></td>
                                    <td class="text-end">
                                        <?php
                                        $isDeposit = $txnType === 'deposit';
                                        $orderStatus = $trans['order_status'] ?? null;
                                        ?>
                                        <?php if ($isDeposit): ?>
                                            <?php if ($orderStatus === 'completed' || $orderStatus === null): ?>
                                                <strong class="text-success">+<?= formatPrice($trans['amount']) ?></strong>
                                            <?php elseif ($orderStatus === 'cancelled'): ?>
                                                <span class="text-warning">İptal</span>
                                            <?php else: ?>
                                                <span class="text-warning">Beklemede</span>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <?php
                                            $nonDepositPositive = $txnType === 'refund';
                                            $nonDepositClass = $nonDepositPositive ? 'text-success' : 'text-danger';
                                            $nonDepositSign = $nonDepositPositive ? '+' : '-';
                                            ?>
                                            <strong class="<?= $nonDepositClass ?>"><?= $nonDepositSign ?><?= formatPrice($trans['amount']) ?></strong>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
                
                <?php if ($totalPages > 1): ?>
                <div class="mt-4">
                    <nav aria-label="İşlem geçmişi sayfalama">
                        <ul class="pagination justify-content-center">
                            <!-- Önceki sayfa -->
                            <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                                <a class="page-link" href="<?= $page <= 1 ? '#' : 'balance.php?page=' . ($page - 1) ?>">
                                    <i class="fa-solid fa-chevron-left"></i>
                                </a>
                            </li>
                            
                            <?php
                            // Gösterilecek sayfa numaralarını hesapla
                            $startPage = max(1, $page - 2);
                            $endPage = min($totalPages, $page + 2);
                            
                            // İlk sayfa
                            if ($startPage > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="balance.php?page=1">1</a>
                                </li>
                                <?php if ($startPage > 2): ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">...</span>
                                    </li>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <!-- Sayfa numaraları -->
                            <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                                <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                    <a class="page-link" href="balance.php?page=<?= $i ?>"><?= $i ?></a>
                                </li>
                            <?php endfor; ?>
                            
                            <!-- Son sayfa -->
                            <?php if ($endPage < $totalPages): ?>
                                <?php if ($endPage < $totalPages - 1): ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">...</span>
                                    </li>
                                <?php endif; ?>
                                <li class="page-item">
                                    <a class="page-link" href="balance.php?page=<?= $totalPages ?>"><?= $totalPages ?></a>
                                </li>
                            <?php endif; ?>
                            
                            <!-- Sonraki sayfa -->
                            <li class="page-item <?= $page >= $totalPages ? 'disabled' : '' ?>">
                                <a class="page-link" href="<?= $page >= $totalPages ? '#' : 'balance.php?page=' . ($page + 1) ?>">
                                    <i class="fa-solid fa-chevron-right"></i>
                                </a>
                            </li>
                        </ul>
                    </nav>
                    
                    <div class="text-center text-muted small mt-2">
                        Sayfa <?= $page ?> / <?= $totalPages ?> (Toplam <?= $totalTransactions ?> işlem)
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>

<div class="modal fade" id="addBalanceModal" tabindex="-1" aria-labelledby="addBalanceLabel" aria-hidden="true" data-bs-theme="dark">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content" style="background-color: var(--bs-dark, #1e1e1e); color: #fff;">
            <div class="modal-header border-0" style="background-color: #1e1e1e; color: #fff; border-bottom: 1px solid #333;">
                <h5 class="modal-title" id="addBalanceLabel" style="color: #fff !important; font-weight: 600;"><i class="fa-solid fa-wallet me-2" style="color: #0089FF;"></i>Bakiye Yükle</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Kapat" style="filter: invert(1);"></button>
            </div>
            <div class="modal-body" style="background-color: #1e1e1e; color: #fff;">
                <!-- Tab Navigation -->
                <ul class="nav nav-tabs mb-4" id="balanceTab" role="tablist" style="border-bottom-color: #333;">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="payment-tab" data-bs-toggle="tab" data-bs-target="#payment-content" type="button" role="tab" aria-controls="payment-content" aria-selected="true">
                            <i class="fa-solid fa-credit-card me-2"></i>Ödeme Yap
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="gift-tab" data-bs-toggle="tab" data-bs-target="#gift-content" type="button" role="tab" aria-controls="gift-content" aria-selected="false">
                            <i class="fa-solid fa-gift me-2"></i>Hediye Kodu
                        </button>
                    </li>
                </ul>
                
                <!-- Tab Content -->
                <div class="tab-content" id="balanceTabContent">
                    <!-- Ödeme Yap Tab -->
                    <div class="tab-pane fade show active" id="payment-content" role="tabpanel" aria-labelledby="payment-tab">
                        <form data-balance-form data-commission-paytr="<?= sanitize($paytrCommissionRate) ?>" data-commission-shopier="<?= sanitize($shopierCommissionRate) ?>" data-commission-bank="<?= sanitize($bankTransferCommissionRate) ?>">
                            <div class="mb-4">
                                <label class="form-label" style="color: #fff !important; font-weight: 600;">Yüklenecek Tutar</label>
                                <div class="input-group input-group-lg">
                                    <input type="number" class="form-control" name="amount" min="1" step="1" value="100" required>
                                    <span class="input-group-text"><?= CURRENCY ?></span>
                                </div>
                            </div>
                            <div class="quick-amounts mb-4">
                                <button type="button" class="btn btn-outline-primary" data-set-amount="50">50 TL</button>
                                <button type="button" class="btn btn-outline-primary" data-set-amount="100">100 TL</button>
                                <button type="button" class="btn btn-outline-primary" data-set-amount="250">250 TL</button>
                                <button type="button" class="btn btn-outline-primary" data-set-amount="500">500 TL</button>
                                <button type="button" class="btn btn-outline-primary" data-set-amount="1000">1000 TL</button>
                            </div>
                            <div class="mb-4">
                                <label class="form-label" style="color: #fff !important; font-weight: 600;">Ödeme Yöntemi</label>
                                <select class="form-select" name="payment_method" required>
                                    <option value="">Seçiniz...</option>
                                    <?php if ($paytrConfigured): ?>
                                        <option value="paytr">PayTR - Kredi Kartı</option>
                                    <?php endif; ?>
                                    <?php if ($shopierConfigured): ?>
                                        <option value="shopier">Shopier - Kredi Kartı</option>
                                    <?php endif; ?>
                                    <?php if ($bankTransferConfigured): ?>
                                        <option value="bank_transfer">Havale / EFT (Manuel)</option>
                                    <?php endif; ?>
                                    <?php if (!$paytrConfigured && !$shopierConfigured && !$bankTransferConfigured): ?>
                                        <option value="" disabled>Önce ödeme yöntemi ayarlarını tamamlayın</option>
                                    <?php endif; ?>
                                </select>
                            </div>
                            <div class="mb-4 payment-summary" data-payment-summary style="display:none;">
                                <div class="d-flex justify-content-between small text-muted mb-1">
                                    <span>Yüklenecek Tutar</span>
                                    <strong data-summary-base>0,00 TL</strong>
                                </div>
                                <div class="d-flex justify-content-between small text-muted mb-1">
                                    <span>Komisyon (<span data-summary-rate>%0</span>)</span>
                                    <strong class="text-warning" data-summary-commission>0,00 TL</strong>
                                </div>
                                <div class="d-flex justify-content-between">
                                    <span>Ödenecek Tutar</span>
                                    <strong class="text-info" data-summary-total>0,00 TL</strong>
                                </div>
                            </div>
                            <button type="button" class="btn btn-primary w-100" data-balance-submit>
                                <i class="fa-solid fa-check-circle me-2"></i>Ödemeyi Tamamla
                            </button>
                        </form>
                    </div>
                    
                    <!-- Hediye Kodu Tab -->
                    <div class="tab-pane fade" id="gift-content" role="tabpanel" aria-labelledby="gift-tab">
                        <form method="post" action="balance.php">
                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                            <input type="hidden" name="action" value="use_gift_code">
                            <div class="mb-4">
                                <label class="form-label" style="color: #fff !important; font-weight: 600;">Hediye Kodunuz</label>
                                <div class="input-group input-group-lg">
                                    <input type="text" class="form-control text-uppercase" name="gift_code" placeholder="XXXXXX" maxlength="20" required style="letter-spacing: 2px;">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fa-solid fa-check me-2"></i>Kullan
                                    </button>
                                </div>
                            </div>
                            <div class="alert alert-info mb-0" style="background-color: rgba(0, 137, 255, 0.1); border-color: rgba(0, 137, 255, 0.3); color: #fff;">
                                <i class="fa-solid fa-info-circle me-2"></i>
                                <small>Hediye kodunuzu girin ve kullanın. Kod geçerliyse bakiyenize otomatik olarak eklenecektir.</small>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php if ($hasPendingPayment): ?>
<script>
// Beklemedeki ödeme varsa 3 saniyede bir sayfa otomatik yenilensin
(function() {
    let checkCount = 0;
    const maxChecks = 40; // 3 saniye x 40 = 2 dakika
    
    const checkPaymentStatus = setInterval(function() {
        checkCount++;
        
        // 2 dakika sonra otomatik yenilemeyi durdur
        if (checkCount >= maxChecks) {
            clearInterval(checkPaymentStatus);
            const alertDiv = document.querySelector('.payment-pending-alert');
            if (alertDiv) {
                alertDiv.innerHTML = `
                    <i class="fa-solid fa-exclamation-triangle me-2"></i>
                    <strong>Ödemeniz hala işleniyor.</strong> Lütfen birkaç dakika sonra tekrar kontrol edin.
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                `;
                alertDiv.className = 'alert alert-warning alert-dismissible fade show position-fixed payment-pending-alert';
            }
            return;
        }
        
        // Sayfayı yenile
        window.location.reload();
    }, 3000); // 3 saniye
    
    // Bilgilendirme mesajı
    const alertDiv = document.createElement('div');
    alertDiv.className = 'alert alert-info alert-dismissible fade show position-fixed payment-pending-alert';
    alertDiv.style.cssText = 'top: 20px; right: 20px; z-index: 9999; max-width: 400px; box-shadow: 0 4px 12px rgba(0,0,0,0.15);';
    alertDiv.innerHTML = `
        <i class="fa-solid fa-spinner fa-spin me-2"></i>
        <strong>Ödemeniz işleniyor...</strong> Sayfa 3 saniyede bir otomatik güncelleniyor.
        <button type="button" class="btn-close" data-bs-dismiss="alert" onclick="clearInterval(checkPaymentStatus)"></button>
    `;
    document.body.appendChild(alertDiv);
})();
</script>
<?php endif; ?>

<?php include 'includes/footer.php'; ?>
