<?php
require_once 'config.php';
requireLogin();

$userId = (int)$_SESSION['user_id'];
$isAjax = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower((string)$_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
$action = $_POST['action'] ?? null;

function cartJson(array $payload): void
{
    header('Content-Type: application/json');
    echo json_encode($payload);
    exit;
}

function fetchCartData(PDO $pdo, int $userId): array
{
    $productsStmt = $pdo->prepare('
        SELECT ci.*, p.name, p.price AS current_price, p.stock, p.image, c.name AS category_name
        FROM cart_items ci
        JOIN products p ON ci.reference_id = p.id
        JOIN categories c ON p.category_id = c.id
        WHERE ci.user_id = ? AND ci.item_type = "product"
        ORDER BY ci.updated_at DESC
    ');
    $productsStmt->execute([$userId]);
    $products = $productsStmt->fetchAll();

    $marketStmt = $pdo->prepare('
        SELECT ci.*, s.name AS server_name, s.sell_price AS current_price, s.sell_stock, s.category_id,
               c.name AS category_name, m.name AS marketplace_name
        FROM cart_items ci
        JOIN game_marketplace_servers s ON ci.reference_id = s.id
        JOIN game_marketplace_categories c ON s.category_id = c.id
        JOIN game_marketplaces m ON c.marketplace_id = m.id
        WHERE ci.user_id = ? AND ci.item_type = "marketplace"
        ORDER BY ci.updated_at DESC
    ');
    $marketStmt->execute([$userId]);
    $marketplace = $marketStmt->fetchAll();

    return ['products' => $products, 'marketplace' => $marketplace];
}

function checkoutCart(PDO $pdo, int $userId): array
{
    $cart = fetchCartData($pdo, $userId);
    $productItems = $cart['products'];
    $marketItems = $cart['marketplace'];

    if (empty($productItems) && empty($marketItems)) {
        return ['success' => false, 'message' => 'Sepetiniz boş.'];
    }

    try {
        $pdo->beginTransaction();

        $balanceStmt = $pdo->prepare('SELECT balance FROM users WHERE id = ? FOR UPDATE');
        $balanceStmt->execute([$userId]);
        $balance = (float)($balanceStmt->fetchColumn() ?? 0);

        $productOperations = [];
        $productTotal = 0.0;

        foreach ($productItems as $item) {
            $quantity = max(1, (int)$item['quantity']);
            $productStmt = $pdo->prepare('SELECT p.id, p.name, p.price, p.stock, p.category_id, c.delivery_type FROM products p LEFT JOIN categories c ON p.category_id = c.id WHERE p.id = ? FOR UPDATE');
            $productStmt->execute([$item['reference_id']]);
            $product = $productStmt->fetch();
            if (!$product) {
                throw new Exception('Sepetteki bir ürün artık mevcut değil.');
            }
            if ((int)$product['stock'] < $quantity) {
                throw new Exception('"' . $product['name'] . '" ürünü için yeterli stok bulunmuyor.');
            }

            $deliveryType = $product['delivery_type'] ?? 'auto';
            $codes = [];
            
            // Otomatik teslimat için kod kontrolü yap
            if ($deliveryType === 'auto') {
                $codeStmt = $pdo->prepare('SELECT id, code FROM product_codes WHERE product_id = ? AND is_sold = 0 LIMIT ' . $quantity . ' FOR UPDATE');
                $codeStmt->execute([$product['id']]);
                $codes = $codeStmt->fetchAll();
                if (count($codes) < $quantity) {
                    throw new Exception('"' . $product['name'] . '" ürünü için stok kodu bulunamadı.');
                }
            }
            // Manuel teslimat için kod kontrolü yapmıyoruz, admin sonra ekleyecek

            $lineTotal = (float)$product['price'] * $quantity;
            $productTotal += $lineTotal;

            $productOperations[] = [
                'cart_id' => (int)$item['id'],
                'product_id' => (int)$product['id'],
                'name' => $product['name'],
                'quantity' => $quantity,
                'unit_price' => (float)$product['price'],
                'codes' => $codes,
                'delivery_type' => $deliveryType
            ];
        }

        $marketOperations = [];
        $marketTotal = 0.0;

        foreach ($marketItems as $item) {
            $quantity = max(0.01, (float)$item['quantity']);
            $serverStmt = $pdo->prepare('SELECT s.*, s.delivery_type, c.name AS category_name, c.marketplace_id FROM game_marketplace_servers s JOIN game_marketplace_categories c ON s.category_id = c.id WHERE s.id = ? FOR UPDATE');
            $serverStmt->execute([$item['reference_id']]);
            $server = $serverStmt->fetch();
            if (!$server) {
                throw new Exception('Sepetteki bir ilan artık mevcut değil.');
            }

            $price = (float)$server['sell_price'];
            if ($price <= 0) {
                throw new Exception('"' . $server['name'] . '" ilanı için geçerli satış fiyatı bulunamadı.');
            }

            $minQty = (float)$server['min_buy_quantity'];
            $maxQty = $server['max_buy_quantity'] !== null ? (float)$server['max_buy_quantity'] : null;
            if ($quantity < $minQty) {
                throw new Exception('"' . $server['name'] . '" ilanı için minimum ' . number_format($minQty, 2, ',', '.') . ' miktar satın alınabilir.');
            }
            if ($maxQty !== null && $quantity > $maxQty) {
                $quantity = $maxQty;
            }

            $existingStock = (float)$server['sell_stock'];
            if ($existingStock > 0 && $quantity > $existingStock) {
                throw new Exception('"' . $server['name'] . '" ilanı için yeterli stok bulunmuyor.');
            }

            $marketTotal += $price * $quantity;
            $deliveryType = $server['delivery_type'] ?? 'auto';
            $marketOperations[] = [
                'cart_id' => (int)$item['id'],
                'server' => $server,
                'quantity' => $quantity,
                'unit_price' => $price,
                'character_name' => $item['character_name'] !== '' ? $item['character_name'] : null,
                'auto_approve' => (int)$server['auto_approve_orders'] === 1 && $deliveryType === 'auto',
                'delivery_type' => $deliveryType
            ];
        }

        $subtotal = $productTotal + $marketTotal;
        
        // İndirim kodu uygula
        $discountAmount = 0.0;
        $discountCodeId = null;
        if (!empty($_SESSION['discount_code'])) {
            $discountStmt = $pdo->prepare('SELECT * FROM discount_codes WHERE code = ? AND is_active = 1 FOR UPDATE');
            $discountStmt->execute([$_SESSION['discount_code']]);
            $discountCode = $discountStmt->fetch();
            
            if ($discountCode) {
                $now = new DateTime();
                $isValid = true;
                
                if ($discountCode['start_date'] && new DateTime($discountCode['start_date']) > $now) {
                    $isValid = false;
                }
                if ($discountCode['end_date'] && new DateTime($discountCode['end_date']) < $now) {
                    $isValid = false;
                }
                if ($discountCode['usage_limit'] && $discountCode['used_count'] >= $discountCode['usage_limit']) {
                    $isValid = false;
                }
                if ($subtotal < (float)$discountCode['min_purchase_amount']) {
                    throw new Exception('Minimum sepet tutarı ' . formatPrice($discountCode['min_purchase_amount']) . ' olmalıdır.');
                }
                
                if ($isValid) {
                    if ($discountCode['discount_type'] === 'percentage') {
                        $discountAmount = $subtotal * ((float)$discountCode['discount_value'] / 100);
                        if ($discountCode['max_discount_amount']) {
                            $discountAmount = min($discountAmount, (float)$discountCode['max_discount_amount']);
                        }
                    } else {
                        $discountAmount = (float)$discountCode['discount_value'];
                    }
                    $discountAmount = min($discountAmount, $subtotal);
                    $discountCodeId = (int)$discountCode['id'];
                }
            }
        }
        
        $grandTotal = $subtotal - $discountAmount;
        
        if ($grandTotal > $balance + 0.0001) {
            throw new Exception('Bakiyeniz yetersiz. Gerekli tutar: ' . formatPrice($grandTotal));
        }

        $updatedBalance = $balance;

        if ($productTotal > 0) {
            $orderNumber = generateOrderNumber();
            $orderStmt = $pdo->prepare('INSERT INTO orders (user_id, order_number, total_amount, payable_amount, commission_amount, commission_rate, status, payment_method, order_type, created_at, updated_at) VALUES (?, ?, ?, ?, 0, 0, "completed", "balance", "purchase", NOW(), NOW())');
            $orderStmt->execute([$userId, $orderNumber, $productTotal, $productTotal]);
            $orderId = (int)$pdo->lastInsertId();

            $insertItem = $pdo->prepare('INSERT INTO order_items (order_id, product_id, product_code_id, product_name, price, quantity) VALUES (?, ?, ?, ?, ?, 1)');
            $updateCode = $pdo->prepare('UPDATE product_codes SET is_sold = 1, sold_to_user_id = ?, sold_at = NOW() WHERE id = ?');
            $updateProductStock = $pdo->prepare('UPDATE products SET stock = stock - ?, sales_count = sales_count + ? WHERE id = ?');

            foreach ($productOperations as $operation) {
                $deliveryType = $operation['delivery_type'] ?? 'auto';
                
                // Otomatik teslimat: Kodları ata ve işaretle
                if ($deliveryType === 'auto' && !empty($operation['codes'])) {
                    foreach ($operation['codes'] as $codeRow) {
                        $insertItem->execute([$orderId, $operation['product_id'], $codeRow['id'], $operation['name'], $operation['unit_price']]);
                        $updateCode->execute([$userId, $codeRow['id']]);
                    }
                } else {
                    // Manuel teslimat: Kod olmadan sipariş item'ı oluştur (admin sonra ekleyecek)
                    for ($i = 0; $i < $operation['quantity']; $i++) {
                        $insertItem->execute([$orderId, $operation['product_id'], null, $operation['name'], $operation['unit_price']]);
                    }
                }
                
                // Her durumda stoktan düş (hem auto hem manual için)
                $updateProductStock->execute([$operation['quantity'], $operation['quantity'], $operation['product_id']]);
            }

            $deductBalance = $pdo->prepare('UPDATE users SET balance = balance - ? WHERE id = ?');
            $deductBalance->execute([$productTotal, $userId]);
            $updatedBalance -= $productTotal;

            $transactionStmt = $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description, order_id, created_at) VALUES (?, "purchase", ?, ?, ?, NOW())');
            $transactionStmt->execute([$userId, $productTotal, 'Sepet ürün satın alımı #' . $orderNumber, $orderId]);
        }

        if ($marketTotal > 0) {
            foreach ($marketOperations as $operation) {
                $server = $operation['server'];
                $quantity = $operation['quantity'];
                $totalAmount = $operation['unit_price'] * $quantity;

                if ($updatedBalance + 0.0001 < $totalAmount) {
                    throw new Exception('Marketplace işlemleri için yeterli bakiye bulunmuyor.');
                }

                if ((float)$server['sell_stock'] > 0) {
                    $updateStock = $pdo->prepare('UPDATE game_marketplace_servers SET sell_stock = GREATEST(sell_stock - ?, 0) WHERE id = ?');
                    $updateStock->execute([$quantity, $server['id']]);
                }

                $deliveryType = $operation['delivery_type'] ?? 'auto';
                $initialStatus = ($deliveryType === 'manual') ? 'completed' : 'pending';
                
                $orderStmt = $pdo->prepare('INSERT INTO game_marketplace_orders (user_id, marketplace_id, category_id, server_id, order_type, status, quantity, price_per_unit, total_amount, currency, character_name, created_at, updated_at) VALUES (?, ?, ?, ?, "buy", ?, ?, ?, ?, "TRY", ?, NOW(), NOW())');
                $orderStmt->execute([
                    $userId,
                    $server['marketplace_id'],
                    $server['category_id'],
                    $server['id'],
                    $initialStatus,
                    $quantity,
                    $operation['unit_price'],
                    $totalAmount,
                    $operation['character_name']
                ]);
                $marketOrderId = (int)$pdo->lastInsertId();

                $deductBalance = $pdo->prepare('UPDATE users SET balance = balance - ? WHERE id = ?');
                $deductBalance->execute([$totalAmount, $userId]);
                $updatedBalance -= $totalAmount;

                $transactionStmt = $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description, order_id, created_at) VALUES (?, "purchase", ?, ?, NULL, NOW())');
                $transactionStmt->execute([$userId, $totalAmount, 'Marketplace satın alma #' . $marketOrderId]);

                // Otomatik teslimat ve auto_approve aktifse kodları ata
                if ($operation['auto_approve'] && $deliveryType === 'auto') {
                    $codeAmount = (int)max(1, ceil($quantity));
                    $codeLookup = $pdo->prepare('SELECT id, code FROM game_marketplace_codes WHERE server_id = ? AND is_sold = 0 ORDER BY id ASC LIMIT ' . $codeAmount . ' FOR UPDATE');
                    $codeLookup->execute([$server['id']]);
                    $codes = $codeLookup->fetchAll();
                    if (count($codes) < $codeAmount) {
                        throw new Exception('Marketplace kod stoğu yetersiz olduğu için işlem tamamlanamadı.');
                    }
                    $assign = $pdo->prepare('UPDATE game_marketplace_codes SET is_sold = 1, sold_to_user_id = ?, sold_at = NOW(), order_id = ? WHERE id = ?');
                    foreach ($codes as $codeRow) {
                        $assign->execute([$userId, $marketOrderId, $codeRow['id']]);
                    }
                    $pdo->prepare('UPDATE game_marketplace_orders SET status = "completed", processed_at = NOW(), updated_at = NOW() WHERE id = ?')->execute([$marketOrderId]);
                }
                // Manuel teslimat için kod atama yapılmıyor, admin sonra ekleyecek
            }
        }

        // İndirim kodu kullanıldıysa kaydet
        if ($discountCodeId && $discountAmount > 0) {
            $pdo->prepare('UPDATE discount_codes SET used_count = used_count + 1 WHERE id = ?')->execute([$discountCodeId]);
            $pdo->prepare('INSERT INTO discount_code_usage (discount_code_id, user_id, discount_amount, order_total, used_at) VALUES (?, ?, ?, ?, NOW())')->execute([
                $discountCodeId, $userId, $discountAmount, $subtotal
            ]);
        }
        
        $pdo->prepare('DELETE FROM cart_items WHERE user_id = ?')->execute([$userId]);

        $pdo->commit();

        $balanceAfterStmt = $pdo->prepare('SELECT balance FROM users WHERE id = ?');
        $balanceAfterStmt->execute([$userId]);
        $_SESSION['balance'] = (float)($balanceAfterStmt->fetchColumn() ?? $updatedBalance);
        refreshCartCount($pdo, $userId);

        return ['success' => true, 'message' => 'Siparişiniz başarıyla oluşturuldu.'];
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        return ['success' => false, 'message' => $e->getMessage() ?: 'Sepet onaylanırken bir hata oluştu.'];
    }
}

if ($action === 'add_product') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        if ($isAjax) {
            cartJson(['success' => false, 'message' => 'Güvenlik doğrulaması başarısız.']);
        }
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect($_POST['redirect'] ?? 'cart.php');
    }
    $productId = (int)($_POST['product_id'] ?? 0);
    $quantity = max(1, (int)($_POST['quantity'] ?? 1));
    $result = addProductToCart($pdo, $userId, $productId, $quantity);
    if ($isAjax) {
        cartJson(['success' => $result['success'], 'message' => $result['message'], 'cart_count' => $_SESSION['cart_count'] ?? 0]);
    }
    flashMessage($result['success'] ? 'success' : 'danger', $result['message']);
    $redirect = $_POST['redirect'] ?? 'cart.php';
    redirect($redirect ?: 'cart.php');
}

if ($action === 'add_marketplace') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        if ($isAjax) {
            cartJson(['success' => false, 'message' => 'Güvenlik doğrulaması başarısız.']);
        }
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('marketplace.php');
    }
    $serverId = (int)($_POST['server_id'] ?? 0);
    $quantity = (float)($_POST['quantity'] ?? 0);
    $character = $_POST['character_name'] ?? '';
    $result = addMarketplaceItemToCart($pdo, $userId, $serverId, $quantity, $character);
    if ($isAjax) {
        cartJson(['success' => $result['success'], 'message' => $result['message'], 'cart_count' => $_SESSION['cart_count'] ?? 0]);
    }
    flashMessage($result['success'] ? 'success' : 'danger', $result['message']);
    redirect('cart.php');
}

if ($action === 'remove_item') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('cart.php');
    }
    $itemId = (int)($_POST['item_id'] ?? 0);
    if ($itemId > 0) {
        $del = $pdo->prepare('DELETE FROM cart_items WHERE id = ? AND user_id = ?');
        $del->execute([$itemId, $userId]);
    }
    refreshCartCount($pdo, $userId);
    flashMessage('success', 'Ürün sepetten kaldırıldı.');
    redirect('cart.php');
}

if ($action === 'update_item') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('cart.php');
    }
    $itemId = (int)($_POST['item_id'] ?? 0);
    $newQuantity = max(0.01, (float)($_POST['quantity'] ?? 1));

    $stmt = $pdo->prepare('SELECT * FROM cart_items WHERE id = ? AND user_id = ?');
    $stmt->execute([$itemId, $userId]);
    $item = $stmt->fetch();
    if ($item) {
        if ($item['item_type'] === 'product') {
            $productStmt = $pdo->prepare('SELECT price, stock, name FROM products WHERE id = ?');
            $productStmt->execute([$item['reference_id']]);
            $product = $productStmt->fetch();
            if (!$product) {
                flashMessage('danger', 'Ürün bulunamadı.');
                redirect('cart.php');
            }
            $newQuantity = max(1, (int)$newQuantity);
            $stock = (int)$product['stock'];
            if ($newQuantity > $stock) {
                flashMessage('danger', '"' . $product['name'] . '" ürününün stoğu "' . $stock . '" adet. Lütfen stok miktarını aşmayın.');
                redirect('cart.php');
            }
            $upd = $pdo->prepare('UPDATE cart_items SET quantity = ?, unit_price = ?, updated_at = NOW() WHERE id = ?');
            $upd->execute([$newQuantity, $product['price'], $itemId]);
            flashMessage('success', 'Ürün miktarı güncellendi.');
        } else {
            $serverStmt = $pdo->prepare('SELECT sell_price, sell_stock, min_buy_quantity, max_buy_quantity, name FROM game_marketplace_servers WHERE id = ?');
            $serverStmt->execute([$item['reference_id']]);
            $server = $serverStmt->fetch();
            if (!$server) {
                flashMessage('danger', 'Marketplace ilanı bulunamadı.');
                redirect('cart.php');
            }
            $price = (float)$server['sell_price'];
            if ($price <= 0) {
                flashMessage('danger', 'İlan fiyatı tanımlanmamış.');
                redirect('cart.php');
            }
            $minQty = (float)$server['min_buy_quantity'];
            $maxQty = $server['max_buy_quantity'] !== null ? (float)$server['max_buy_quantity'] : null;
            $newQuantity = max($minQty, (float)$newQuantity);
            if ($maxQty !== null && $newQuantity > $maxQty) {
                flashMessage('danger', '"' . $server['name'] . '" ilanı için maksimum miktar: ' . number_format($maxQty, 2, ',', '.'));
                redirect('cart.php');
            }
            $sellStock = (float)$server['sell_stock'];
            if ($sellStock > 0 && $newQuantity > $sellStock) {
                flashMessage('danger', '"' . $server['name'] . '" ilanının stoğu "' . number_format($sellStock, 2, ',', '.') . '" adet. Lütfen stok miktarını aşmayın.');
                redirect('cart.php');
            }
            $upd = $pdo->prepare('UPDATE cart_items SET quantity = ?, unit_price = ?, updated_at = NOW() WHERE id = ?');
            $upd->execute([$newQuantity, $price, $itemId]);
            flashMessage('success', 'İlan miktarı güncellendi.');
        }
    }
    refreshCartCount($pdo, $userId);
    redirect('cart.php');
}

if ($action === 'apply_discount') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('cart.php');
    }
    
    $code = strtoupper(trim($_POST['discount_code'] ?? ''));
    if ($code === '') {
        flashMessage('danger', 'Lütfen bir indirim kodu girin.');
        redirect('cart.php');
    }
    
    // Kodu kontrol et
    $stmt = $pdo->prepare('SELECT * FROM discount_codes WHERE code = ? AND is_active = 1');
    $stmt->execute([$code]);
    $discount = $stmt->fetch();
    
    if (!$discount) {
        flashMessage('danger', 'Geçersiz indirim kodu.');
        redirect('cart.php');
    }
    
    $now = new DateTime();
    if ($discount['start_date'] && new DateTime($discount['start_date']) > $now) {
        flashMessage('danger', 'Bu indirim kodu henüz aktif değil.');
        redirect('cart.php');
    }
    if ($discount['end_date'] && new DateTime($discount['end_date']) < $now) {
        flashMessage('danger', 'Bu indirim kodunun süresi dolmuş.');
        redirect('cart.php');
    }
    if ($discount['usage_limit'] && $discount['used_count'] >= $discount['usage_limit']) {
        flashMessage('danger', 'Bu indirim kodunun kullanım limiti dolmuş.');
        redirect('cart.php');
    }
    
    $_SESSION['discount_code'] = $code;
    flashMessage('success', 'İndirim kodu uygulandı!');
    redirect('cart.php');
}

if ($action === 'remove_discount') {
    unset($_SESSION['discount_code']);
    flashMessage('success', 'İndirim kodu kaldırıldı.');
    redirect('cart.php');
}

if ($action === 'checkout') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        flashMessage('danger', 'Güvenlik doğrulaması başarısız.');
        redirect('cart.php');
    }
    $result = checkoutCart($pdo, $userId);
    flashMessage($result['success'] ? 'success' : 'danger', $result['message']);
    if ($result['success']) {
        // İndirim kodu kullanıldıysa session'dan temizle
        unset($_SESSION['discount_code']);
        redirect(routeUrl('orders'));
    }
    redirect('cart.php');
}

$cartData = fetchCartData($pdo, $userId);
$products = $cartData['products'];
$marketItems = $cartData['marketplace'];

$productTotal = 0.0;
foreach ($products as $item) {
    $productTotal += (float)$item['unit_price'] * (int)$item['quantity'];
}
$marketTotal = 0.0;
foreach ($marketItems as $item) {
    $marketTotal += (float)$item['unit_price'] * (float)$item['quantity'];
}
$subtotal = $productTotal + $marketTotal;

// İndirim kodu kontrolü
$discountAmount = 0.0;
$discountCode = null;
if (!empty($_SESSION['discount_code'])) {
    $stmt = $pdo->prepare('SELECT * FROM discount_codes WHERE code = ? AND is_active = 1');
    $stmt->execute([$_SESSION['discount_code']]);
    $discountCode = $stmt->fetch();
    
    if ($discountCode) {
        // Geçerlilik kontrolü
        $now = new DateTime();
        $isValid = true;
        
        if ($discountCode['start_date'] && new DateTime($discountCode['start_date']) > $now) {
            $isValid = false;
        }
        if ($discountCode['end_date'] && new DateTime($discountCode['end_date']) < $now) {
            $isValid = false;
        }
        if ($discountCode['usage_limit'] && $discountCode['used_count'] >= $discountCode['usage_limit']) {
            $isValid = false;
        }
        if ($subtotal < (float)$discountCode['min_purchase_amount']) {
            $isValid = false;
        }
        
        if ($isValid) {
            // İndirim hesapla
            if ($discountCode['discount_type'] === 'percentage') {
                $discountAmount = $subtotal * ((float)$discountCode['discount_value'] / 100);
                if ($discountCode['max_discount_amount']) {
                    $discountAmount = min($discountAmount, (float)$discountCode['max_discount_amount']);
                }
            } else {
                $discountAmount = (float)$discountCode['discount_value'];
            }
            $discountAmount = min($discountAmount, $subtotal);
        } else {
            unset($_SESSION['discount_code']);
            $discountCode = null;
        }
    } else {
        unset($_SESSION['discount_code']);
    }
}

$grandTotal = $subtotal - $discountAmount;

$pageTitle = 'Sepetim';
include 'includes/header.php';
?>

<section class="section-padding section-bg-dark">
    <div class="container-wide">
        <div class="section-heading">
            <div>
                <h2 class="section-title"><i class="fa-solid fa-cart-shopping me-2"></i>Sepetim</h2>
                <p class="section-subtitle">Marketplace ilanlarınızı ve ürünlerinizi buradan yönetin.</p>
            </div>
            <div>
                <a href="<?= routeUrl('products') ?>" class="btn btn-outline-light"><i class="fa-solid fa-shop me-2"></i>Alışverişe devam et</a>
            </div>
        </div>

        <?php $cartMessage = getFlashMessage(); ?>
        <?php if ($cartMessage): ?>
            <div class="alert alert-<?= sanitize($cartMessage['type']) ?>">
                <i class="fa-solid fa-circle-info me-2"></i><?= sanitize($cartMessage['message']) ?>
            </div>
        <?php endif; ?>

        <?php if (empty($products) && empty($marketItems)): ?>
            <div class="empty-state mt-4">
                <i class="fa-solid fa-cart-shopping"></i>
                <h4>Sepetiniz boş</h4>
                <p>Ürünleri sepetinize ekledikten sonra burada sipariş oluşturabilirsiniz.</p>
                <a href="<?= routeUrl('products') ?>" class="btn btn-primary mt-3"><i class="fa-solid fa-shop me-2"></i>Ürünleri keşfet</a>
            </div>
        <?php else: ?>
            <div class="row g-4">
                <div class="col-lg-8">
                    <?php if (!empty($products)): ?>
                        <div class="card mb-4">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="mb-0"><i class="fa-solid fa-boxes-stacked me-2"></i>Ürünler</h5>
                                <span class="badge bg-primary"><?= count($products) ?> ürün</span>
                            </div>
                            <div class="card-body p-0">
                                <div class="table-responsive">
                                    <table class="table mb-0">
                                        <thead>
                                            <tr>
                                                <th>Ürün</th>
                                                <th class="text-center">Adet</th>
                                                <th class="text-end">Birim Fiyat</th>
                                                <th class="text-end">Tutar</th>
                                                <th class="text-end">İşlemler</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($products as $item): ?>
                                                <?php $lineTotal = (float)$item['unit_price'] * (int)$item['quantity']; ?>
                                                <tr>
                                                    <td>
                                                        <strong><?= sanitize($item['name']) ?></strong>
                                                        <div class="text-muted small">Kategori: <?= sanitize($item['category_name']) ?></div>
                                                    </td>
                                                    <td class="text-center">
                                                        <form method="post" class="d-inline-flex align-items-center justify-content-center gap-2">
                                                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                            <input type="hidden" name="action" value="update_item">
                                                            <input type="hidden" name="item_id" value="<?= (int)$item['id'] ?>">
                                                            <input type="number" name="quantity" min="1" class="form-control form-control-sm" value="<?= (int)$item['quantity'] ?>" style="width:80px;">
                                                            <button type="submit" class="btn btn-sm btn-outline-primary">Güncelle</button>
                                                        </form>
                                                    </td>
                                                    <td class="text-end"><?= formatPrice($item['unit_price']) ?></td>
                                                    <td class="text-end"><strong><?= formatPrice($lineTotal) ?></strong></td>
                                                    <td class="text-end">
                                                        <form method="post" class="d-inline">
                                                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                            <input type="hidden" name="action" value="remove_item">
                                                            <input type="hidden" name="item_id" value="<?= (int)$item['id'] ?>">
                                                            <button type="submit" class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-trash"></i></button>
                                                        </form>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($marketItems)): ?>
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="mb-0"><i class="fa-solid fa-store me-2"></i>Marketplace İlanları</h5>
                                <span class="badge bg-primary"><?= count($marketItems) ?> ilan</span>
                            </div>
                            <div class="card-body p-0">
                                <div class="table-responsive">
                                    <table class="table mb-0">
                                        <thead>
                                            <tr>
                                                <th>İlan</th>
                                                <th>Karakter</th>
                                                <th class="text-center">Miktar</th>
                                                <th class="text-end">Birim Fiyat</th>
                                                <th class="text-end">Tutar</th>
                                                <th class="text-end">İşlemler</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($marketItems as $item): ?>
                                                <?php $lineTotal = (float)$item['unit_price'] * (float)$item['quantity']; ?>
                                                <tr>
                                                    <td>
                                                        <strong><?= sanitize($item['server_name']) ?></strong>
                                                        <div class="text-muted small">Kategori: <?= sanitize($item['category_name']) ?></div>
                                                    </td>
                                                    <td>
                                                        <?php if ($item['character_name'] !== ''): ?>
                                                            <?= sanitize($item['character_name']) ?>
                                                        <?php else: ?>
                                                            <span class="text-muted">Belirtilmedi</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td class="text-center">
                                                        <form method="post" class="d-inline-flex align-items-center justify-content-center gap-2">
                                                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                            <input type="hidden" name="action" value="update_item">
                                                            <input type="hidden" name="item_id" value="<?= (int)$item['id'] ?>">
                                                            <input type="number" name="quantity" min="0.01" step="0.01" class="form-control form-control-sm" value="<?= (float)$item['quantity'] ?>" style="width:100px;">
                                                            <button type="submit" class="btn btn-sm btn-outline-primary">Güncelle</button>
                                                        </form>
                                                    </td>
                                                    <td class="text-end"><?= formatPrice($item['unit_price']) ?></td>
                                                    <td class="text-end"><strong><?= formatPrice($lineTotal) ?></strong></td>
                                                    <td class="text-end">
                                                        <form method="post" class="d-inline">
                                                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                                            <input type="hidden" name="action" value="remove_item">
                                                            <input type="hidden" name="item_id" value="<?= (int)$item['id'] ?>">
                                                            <button type="submit" class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-trash"></i></button>
                                                        </form>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>

                <div class="col-lg-4">
                    <div class="card sticky-top" style="top: 120px;">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fa-solid fa-receipt me-2"></i>Sipariş Özeti</h5>
                        </div>
                        <div class="card-body">
                            <!-- İndirim Kodu -->
                            <?php if ($discountCode): ?>
                                <div class="alert alert-success mb-3">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div>
                                            <i class="fa-solid fa-tag me-2"></i>
                                            <strong><?= sanitize($discountCode['code']) ?></strong>
                                            <?php if ($discountCode['discount_type'] === 'percentage'): ?>
                                                <span class="badge bg-success ms-2">%<?= number_format($discountCode['discount_value'], 0) ?></span>
                                            <?php endif; ?>
                                        </div>
                                        <form method="post" class="d-inline">
                                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                            <input type="hidden" name="action" value="remove_discount">
                                            <button type="submit" class="btn btn-sm btn-outline-danger">
                                                <i class="fa-solid fa-times"></i>
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            <?php else: ?>
                                <div class="mb-3">
                                    <label class="form-label small text-muted">İndirim Kodu</label>
                                    <form method="post" class="input-group">
                                        <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                        <input type="hidden" name="action" value="apply_discount">
                                        <input type="text" name="discount_code" class="form-control text-uppercase" placeholder="Kodunuzu girin">
                                        <button type="submit" class="btn btn-outline-primary">
                                            <i class="fa-solid fa-check"></i>
                                        </button>
                                    </form>
                                </div>
                            <?php endif; ?>
                            
                            <hr>
                            
                            <div class="d-flex justify-content-between mb-2">
                                <span>Ürünler</span>
                                <strong><?= formatPrice($productTotal) ?></strong>
                            </div>
                            <div class="d-flex justify-content-between mb-2">
                                <span>Marketplace</span>
                                <strong><?= formatPrice($marketTotal) ?></strong>
                            </div>
                            
                            <?php if ($discountAmount > 0): ?>
                                <div class="d-flex justify-content-between mb-2 text-success">
                                    <span><i class="fa-solid fa-tag me-1"></i>İndirim</span>
                                    <strong>-<?= formatPrice($discountAmount) ?></strong>
                                </div>
                            <?php endif; ?>
                            
                            <hr>
                            
                            <div class="d-flex justify-content-between align-items-center mb-4">
                                <span class="text-muted">Toplam Tutar</span>
                                <h4 class="mb-0"><?= formatPrice($grandTotal) ?></h4>
                            </div>
                            <form method="post">
                                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                <input type="hidden" name="action" value="checkout">
                                <button type="submit" class="btn btn-primary w-100"<?= $grandTotal <= 0 ? ' disabled' : '' ?>>
                                    <i class="fa-solid fa-circle-check me-2"></i>Siparişi Tamamla
                                </button>
                            </form>
                            <p class="text-muted small mt-3 mb-0">Siparişiniz, bakiye kontrolü ve stok uygunluğu doğrulandıktan sonra oluşturulur.</p>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>

<?php include 'includes/footer.php'; ?>
