<?php
// config.php - Konfigürasyon Dosyası

// Hata raporlama - Production için optimize edildi
// Deprecated uyarılarını kapat, sadece kritik hataları göster
error_reporting(E_ALL & ~E_DEPRECATED & ~E_STRICT & ~E_NOTICE & ~E_WARNING);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/logs/php-errors.log');

if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
}

if (!function_exists('str_starts_with')) {
    function str_starts_with(string $haystack, string $needle): bool
    {
        return $needle === '' || strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}

// PHP 8.1+ uyumlu güvenli trim fonksiyonları
if (!function_exists('safe_trim')) {
    function safe_trim($value, string $characters = " \n\r\t\v\0"): string
    {
        return trim((string)$value, $characters);
    }
}

if (!function_exists('safe_ltrim')) {
    function safe_ltrim($value, string $characters = " \n\r\t\v\0"): string
    {
        return ltrim((string)$value, $characters);
    }
}

if (!function_exists('safe_rtrim')) {
    function safe_rtrim($value, string $characters = " \n\r\t\v\0"): string
    {
        return rtrim((string)$value, $characters);
    }
}

// Veritabanı Ayarları
define('DB_HOST', 'localhost');
define('DB_NAME', 'asfasfa');
define('DB_USER', 'asfafa');
define('DB_PASS', '{{afasfasf');
define('DB_CHARSET', 'utf8mb4');

// Site Ayarları
define('SITE_URL', 'https://epincibaba.com');
define('BASE_URL', SITE_URL . '/'); // Script yolları için
define('SITE_NAME', 'Digital Shop');
define('SITE_DESCRIPTION', 'Dijital ürünler, oyun kodları ve pazaryeri ilanlarını tek platformda sunan güvenilir mağaza.');
define('SITE_KEYWORDS', 'dijital ürünler, oyun kodları, hediye kartları, epin, pazaryeri');
define('CURRENCY', 'TL');
define('PAYTR_COMMISSION_RATE_DEFAULT', 0.0);
define('SHOPIER_COMMISSION_RATE_DEFAULT', 0.0);

// Upload Ayarları
define('UPLOAD_DIR', __DIR__ . '/uploads');
define('UPLOAD_URL', 'uploads');

if (!is_dir(UPLOAD_DIR)) {
    mkdir(UPLOAD_DIR, 0775, true);
}

// PayTR Ayarları
define('PAYTR_MERCHANT_ID', 'XXXXXX'); // PayTR Merchant ID
define('PAYTR_MERCHANT_KEY', 'XXXXXXXXXXXXXXXX'); // PayTR Merchant Key
define('PAYTR_MERCHANT_SALT', 'XXXXXXXXXXXXXXXX'); // PayTR Merchant Salt
define('PAYTR_TEST_MODE', true); // Test modu: true, Canlı: false

// Shopier Ayarları
define('SHOPIER_API_KEY', 'XXXXXXXXXXXXXXXXXXXXXXXX'); // Shopier API Key
define('SHOPIER_API_SECRET', 'XXXXXXXXXXXXXXXXXXXXXXXX'); // Shopier API Secret
define('SHOPIER_TEST_MODE', true); // Test modu: true, Canlı: false

// SMTP Defaults (override via settings table)
define('SMTP_HOST', 'smtp.example.com');
define('SMTP_PORT', 587);
define('SMTP_USERNAME', 'no-reply@example.com');
define('SMTP_PASSWORD', 'your_smtp_password');
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_FROM_EMAIL', 'no-reply@example.com');
define('SMTP_FROM_NAME', 'Market Notification');
define('SMTP_TEST_MODE', true);
define('EMAIL_BASE_URL', 'http://localhost/digital-shop');
define('RECAPTCHA_SITE_KEY', '');
define('RECAPTCHA_SECRET_KEY', '');

// Session Ayarları
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);

$isHttpsEnvironment = (stripos(SITE_URL, 'https://') === 0)
    || (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
    || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443);

if (!defined('IS_HTTPS_ENVIRONMENT')) {
    define('IS_HTTPS_ENVIRONMENT', $isHttpsEnvironment);
}

if (PHP_VERSION_ID >= 70300) {
    ini_set('session.cookie_samesite', $isHttpsEnvironment ? 'None' : 'Lax');
}
ini_set('session.cookie_secure', $isHttpsEnvironment ? 1 : 0);

// Timezone
date_default_timezone_set('Europe/Istanbul');

// Session başlat
if (!defined('SKIP_SESSION_START') || SKIP_SESSION_START !== true) {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
}

// PDO Veritabanı Bağlantısı
try {
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
    $options = [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,
    ];
    $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
} catch (PDOException $e) {
    die("Veritabanı bağlantı hatası: " . $e->getMessage());
}

ensureSchemaUpgrades($pdo);

// Uygulama ayar cache'i
$appSettingsCache = null;
$seoRoutesCache = null;

function loadSettings(bool $force = false): array {
    global $pdo, $appSettingsCache;

    if ($force || $appSettingsCache === null) {
        $appSettingsCache = [];
        try {
            $stmt = $pdo->query("SELECT setting_key, setting_value FROM settings");
            foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
                $appSettingsCache[$row['setting_key']] = $row['setting_value'];
            }
        } catch (Throwable $th) {
            // Ayarlar tabloya erişilemiyorsa cache boş kalır
            $appSettingsCache = [];
        }
    }

    return $appSettingsCache;
}

function getSeoRoutes(): array
{
    global $seoRoutesCache;
    if ($seoRoutesCache === null) {
        $routesFile = __DIR__ . '/seo_routes.php';
        $seoRoutesCache = is_file($routesFile) ? require $routesFile : [];
    }
    return $seoRoutesCache;
}

function siteBaseUrl(bool $forceRefresh = false): string
{
    static $baseUrlCache = null;
    if ($forceRefresh || $baseUrlCache === null) {
        $saved = trim((string)getSetting('site_base_url', SITE_URL));
        if ($saved === '') {
            $saved = SITE_URL;
        }
        if (!preg_match('#^https?://#i', $saved)) {
            $saved = SITE_URL;
        }
        $baseUrlCache = rtrim($saved, '/');
        if ($baseUrlCache === '') {
            $baseUrlCache = rtrim(SITE_URL, '/');
        }
    }
    return $baseUrlCache;
}

function siteBasePath(bool $forceRefresh = false): string
{
    static $basePathCache = null;
    if ($forceRefresh || $basePathCache === null) {
        $parsed = parse_url(siteBaseUrl($forceRefresh), PHP_URL_PATH);
        $path = $parsed !== null ? '/' . safe_trim($parsed, '/') : '';
        if ($path === '/' || $path === '//') {
            $path = '';
        } else {
            $path = safe_rtrim($path, '/');
        }
        $basePathCache = $path;
    }
    return $basePathCache;
}

function buildRelativeUrl(string $path): string
{
    $path = safe_trim($path);
    if ($path === '') {
        $path = '/';
    }
    if ($path[0] !== '/') {
        $path = '/' . $path;
    }
    $basePath = siteBasePath();
    if ($basePath === '' || $path === $basePath || str_starts_with($path, $basePath . '/') || str_starts_with($path, $basePath . '?')) {
        // Path already contains base path or equals it
        return $path;
    }
    if ($path === '/') {
        return $basePath === '' ? '/' : $basePath . '/';
    }
    return $basePath . $path;
}

function routeUrl(string $name, array $params = [], bool $absolute = false): string
{
    $routes = getSeoRoutes();
    if (!isset($routes[$name])) {
        $path = '/' . ltrim($name, '/');
        $relative = buildRelativeUrl($path);
        if ($absolute) {
            $baseUrl = siteBaseUrl();
            $basePath = siteBasePath();
            $tail = $relative;
            if ($basePath !== '' && str_starts_with($relative, $basePath)) {
                $tail = substr($relative, strlen($basePath));
                if ($tail === '' || $tail === false) {
                    $tail = '/';
                }
            }
            return rtrim($baseUrl, '/') . ($tail === '/' ? '/' : $tail);
        }
        return $relative;
    }
    $pattern = $routes[$name]['pattern'] ?? '/';
    $url = $pattern;
    if (preg_match_all('/\{([a-zA-Z0-9_]+)\}/', $pattern, $matches)) {
        foreach ($matches[1] as $paramName) {
            $value = $params[$paramName] ?? '';
            if ($value === null) {
                $value = '';
            }
            $value = (string)$value;
            $url = str_replace('{' . $paramName . '}', rawurlencode($value), $url);
        }
    }
    $url = preg_replace('#//+#', '/', $url);
    if ($url === '') {
        $url = '/';
    }
    $relative = buildRelativeUrl($url);
    if ($absolute) {
        $baseUrl = siteBaseUrl();
        $basePath = siteBasePath();
        $tail = $relative;
        if ($basePath !== '' && str_starts_with($relative, $basePath)) {
            $tail = substr($relative, strlen($basePath));
            if ($tail === '' || $tail === false) {
                $tail = '/';
            }
        }
        return rtrim($baseUrl, '/') . ($tail === '/' ? '/' : $tail);
    }
    return $relative;
}

function seoFriendlyUrl(string $url, bool $absolute = false): string
{
    if ($url === null || $url === '') {
        return $absolute ? siteBaseUrl() . '/' : buildRelativeUrl('/');
    }
    $url = safe_trim($url);
    if ($url === '') {
        return $absolute ? siteBaseUrl() . '/' : buildRelativeUrl('/');
    }
    if (preg_match('#^https?://#i', $url)) {
        return $url;
    }

    $parts = parse_url($url);
    $path = ltrim($parts['path'] ?? $url, '/');
    $query = [];
    if (!empty($parts['query'])) {
        parse_str($parts['query'], $query);
    }

    $routes = getSeoRoutes();
    foreach ($routes as $name => $route) {
        $legacy = $route['legacy'] ?? null;
        if (!$legacy) {
            continue;
        }
        $legacyPath = ltrim($legacy['path'] ?? '', '/');
        if ($legacyPath !== $path) {
            continue;
        }

        $requiredQuery = $legacy['query'] ?? [];
        $params = [];
        $remainingQuery = $query;
        $matches = true;
        if (!empty($requiredQuery)) {
            foreach ($requiredQuery as $legacyKey => $paramName) {
                if (is_int($legacyKey)) {
                    $legacyKey = $paramName;
                }
                if (!array_key_exists($legacyKey, $query)) {
                    $matches = false;
                    break;
                }
                $params[$paramName] = $query[$legacyKey];
                unset($remainingQuery[$legacyKey]);
            }
        }
        if (!$matches) {
            continue;
        }

        $friendly = routeUrl($name, $params, $absolute);
        if (!empty($remainingQuery)) {
            $friendly .= (strpos($friendly, '?') === false ? '?' : '&') . http_build_query($remainingQuery);
        }
        if (!empty($parts['fragment'])) {
            $friendly .= '#' . $parts['fragment'];
        }
        $relative = buildRelativeUrl($friendly);
        if ($absolute) {
            $baseUrl = siteBaseUrl();
            $basePath = siteBasePath();
            $tail = $relative;
            if ($basePath !== '' && str_starts_with($relative, $basePath)) {
                $tail = substr($relative, strlen($basePath));
                if ($tail === '' || $tail === false) {
                    $tail = '/';
                }
            }
            return rtrim($baseUrl, '/') . ($tail === '/' ? '/' : $tail);
        }
        return $relative;
    }

    $normalized = '/' . $path;
    if (!empty($parts['query'])) {
        $normalized .= '?' . $parts['query'];
    }
    if (!empty($parts['fragment'])) {
        $normalized .= '#' . $parts['fragment'];
    }

    $relative = buildRelativeUrl($normalized);
    if ($absolute) {
        $baseUrl = siteBaseUrl();
        $basePath = siteBasePath();
        $tail = $relative;
        if ($basePath !== '' && str_starts_with($relative, $basePath)) {
            $tail = substr($relative, strlen($basePath));
            if ($tail === '' || $tail === false) {
                $tail = '/';
            }
        }
        return rtrim($baseUrl, '/') . ($tail === '/' ? '/' : $tail);
    }
    return $relative;
}

function defaultRobotsTxt(): string
{
    $lines = [
        'User-agent: *',
        'Disallow: ' . buildRelativeUrl('/admin/'),
        'Disallow: ' . buildRelativeUrl('/payment/'),
        'Disallow: ' . buildRelativeUrl('/api/'),
        'Disallow: ' . buildRelativeUrl('/marketplace/api/'),
        'Sitemap: ' . siteBaseUrl() . '/sitemap.xml',
    ];

    return implode(PHP_EOL, $lines) . PHP_EOL;
}

function generateSitemapXml(PDO $pdo, array $options = []): array
{
    $defaults = [
        'home' => true,
        'products' => true,
        'categories' => true,
        'pages' => true,
        'blog' => true,
        'marketplace' => true,
        'marketplace_items' => true,
        'support' => true,
    ];

    $normalized = [];
    foreach ($options as $key => $value) {
        $normalized[strtolower((string)$key)] = (bool)$value;
    }
    $options = array_merge($defaults, $normalized);

    $entries = [];
    $seen = [];
    $addEntry = static function (string $loc, ?string $lastmod = null, string $changefreq = 'weekly', string $priority = '0.5') use (&$entries, &$seen): void {
        $loc = safe_trim($loc);
        if ($loc === '' || isset($seen[$loc])) {
            return;
        }
        $seen[$loc] = true;
        $entries[] = [
            'loc' => $loc,
            'lastmod' => $lastmod,
            'changefreq' => $changefreq,
            'priority' => $priority,
        ];
    };

    if (!empty($options['home'])) {
        $addEntry(routeUrl('home', [], true), gmdate('c'), 'daily', '1.0');
    }

    if (!empty($options['products'])) {
        $addEntry(routeUrl('products', [], true), gmdate('c'), 'daily', '0.8');
        try {
            $productStmt = $pdo->query('SELECT slug, updated_at, created_at FROM products WHERE is_active = 1');
            foreach ($productStmt->fetchAll(PDO::FETCH_ASSOC) as $product) {
                $slug = $product['slug'] ?? '';
                if ($slug === '') {
                    continue;
                }
                $lastmodSource = $product['updated_at'] ?? $product['created_at'] ?? null;
                $lastmod = $lastmodSource ? gmdate('c', strtotime((string)$lastmodSource)) : null;
                $addEntry(routeUrl('product', ['slug' => $slug], true), $lastmod, 'weekly', '0.7');
            }
        } catch (Throwable $th) {
            // Sessizce devam et
        }
    }

    if (!empty($options['categories'])) {
        try {
            $categoryStmt = $pdo->query('SELECT slug, created_at FROM categories WHERE is_active = 1');
            foreach ($categoryStmt->fetchAll(PDO::FETCH_ASSOC) as $category) {
                $slug = $category['slug'] ?? '';
                if ($slug === '') {
                    continue;
                }
                $lastmod = isset($category['created_at']) ? gmdate('c', strtotime((string)$category['created_at'])) : null;
                $addEntry(routeUrl('product_category', ['slug' => $slug], true), $lastmod, 'weekly', '0.6');
            }
        } catch (Throwable $th) {
            // yoksay
        }
    }

    if (!empty($options['pages'])) {
        try {
            $pageStmt = $pdo->query('SELECT slug, updated_at, created_at FROM pages WHERE is_published = 1');
            foreach ($pageStmt->fetchAll(PDO::FETCH_ASSOC) as $page) {
                $slug = $page['slug'] ?? '';
                if ($slug === '') {
                    continue;
                }
                $lastmodSource = $page['updated_at'] ?? $page['created_at'] ?? null;
                $lastmod = $lastmodSource ? gmdate('c', strtotime((string)$lastmodSource)) : null;
                $addEntry(routeUrl('page', ['slug' => $slug], true), $lastmod, 'monthly', '0.5');
            }
        } catch (Throwable $th) {
            // yoksay
        }
    }

    if (!empty($options['blog'])) {
        $addEntry(routeUrl('blog', [], true), gmdate('c'), 'weekly', '0.6');
        try {
            $blogStmt = $pdo->query('SELECT slug, updated_at, created_at FROM blog_posts WHERE is_published = 1');
            foreach ($blogStmt->fetchAll(PDO::FETCH_ASSOC) as $post) {
                $slugRaw = $post['slug'] ?? null;
                if ($slugRaw === null || $slugRaw === '') {
                    continue;
                }
                $slug = trim((string)$slugRaw);
                if ($slug === '') {
                    continue;
                }
                $lastmodSource = $post['updated_at'] ?? $post['created_at'] ?? null;
                $lastmod = $lastmodSource ? gmdate('c', strtotime((string)$lastmodSource)) : null;
                try {
                    $blogUrl = routeUrl('blog_post', ['slug' => $slug], true);
                    if ($blogUrl !== null && $blogUrl !== '') {
                        $addEntry($blogUrl, $lastmod, 'weekly', '0.6');
                    }
                } catch (Throwable $urlError) {
                    // URL oluşturma hatası, atla
                    continue;
                }
            }
        } catch (Throwable $th) {
            // yoksay
        }
    }

    if (!empty($options['marketplace'])) {
        $addEntry(routeUrl('marketplace', [], true), gmdate('c'), 'weekly', '0.7');
        try {
            $structure = getMarketplaceStructure($pdo, true);
            foreach ($structure as $game) {
                $gameSlug = $game['slug'] ?? '';
                if ($gameSlug === '') {
                    continue;
                }
                $lastmodSource = $game['updated_at'] ?? $game['created_at'] ?? null;
                $lastmod = $lastmodSource ? gmdate('c', strtotime((string)$lastmodSource)) : null;
                $addEntry(routeUrl('marketplace_game', ['game' => $gameSlug], true), $lastmod, 'weekly', '0.6');
                if (!empty($game['categories'])) {
                    foreach ($game['categories'] as $category) {
                        $categorySlug = $category['slug'] ?? '';
                        if ($categorySlug === '') {
                            continue;
                        }
                        $catLastmodSource = $category['updated_at'] ?? $category['created_at'] ?? null;
                        $catLastmod = $catLastmodSource ? gmdate('c', strtotime((string)$catLastmodSource)) : null;
                        $addEntry(routeUrl('marketplace_category', ['game' => $gameSlug, 'category' => $categorySlug], true), $catLastmod, 'weekly', '0.6');
                    }
                }
            }
        } catch (Throwable $th) {
            // yoksay
        }
    }

    if (!empty($options['marketplace_items'])) {
        try {
            // Get all active marketplace servers/items with their game and category slugs
            $itemsStmt = $pdo->query('
                SELECT 
                    gms.slug as server_slug,
                    gms.updated_at as server_updated,
                    gms.created_at as server_created,
                    gmc.slug as category_slug,
                    gm.slug as game_slug
                FROM game_marketplace_servers gms
                INNER JOIN game_marketplace_categories gmc ON gms.category_id = gmc.id
                INNER JOIN game_marketplaces gm ON gmc.game_id = gm.id
                WHERE gms.is_active = 1 AND gmc.is_active = 1 AND gm.is_active = 1
            ');
            
            foreach ($itemsStmt->fetchAll(PDO::FETCH_ASSOC) as $item) {
                $gameSlug = $item['game_slug'] ?? '';
                $categorySlug = $item['category_slug'] ?? '';
                $serverSlug = $item['server_slug'] ?? '';
                
                if ($gameSlug === '' || $categorySlug === '' || $serverSlug === '') {
                    continue;
                }
                
                $lastmodSource = $item['server_updated'] ?? $item['server_created'] ?? null;
                $lastmod = $lastmodSource ? gmdate('c', strtotime((string)$lastmodSource)) : null;
                
                $addEntry(
                    routeUrl('marketplace_server', [
                        'game' => $gameSlug, 
                        'category' => $categorySlug,
                        'server' => $serverSlug
                    ], true), 
                    $lastmod, 
                    'weekly', 
                    '0.6'
                );
            }
        } catch (Throwable $th) {
            // yoksay
        }
    }

    if (!empty($options['support'])) {
        $addEntry(routeUrl('support', [], true), null, 'monthly', '0.4');
    }

    $doc = new DOMDocument('1.0', 'UTF-8');
    $doc->formatOutput = true;
    $urlset = $doc->createElement('urlset');
    $urlset->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');

    foreach ($entries as $entry) {
        $urlEl = $doc->createElement('url');

        $locEl = $doc->createElement('loc');
        $locEl->appendChild($doc->createTextNode($entry['loc']));
        $urlEl->appendChild($locEl);

        if (!empty($entry['lastmod'])) {
            $lastmodEl = $doc->createElement('lastmod');
            $lastmodEl->appendChild($doc->createTextNode($entry['lastmod']));
            $urlEl->appendChild($lastmodEl);
        }

        if (!empty($entry['changefreq'])) {
            $freqEl = $doc->createElement('changefreq');
            $freqEl->appendChild($doc->createTextNode($entry['changefreq']));
            $urlEl->appendChild($freqEl);
        }

        if (!empty($entry['priority'])) {
            $priorityEl = $doc->createElement('priority');
            $priorityEl->appendChild($doc->createTextNode($entry['priority']));
            $urlEl->appendChild($priorityEl);
        }

        $urlset->appendChild($urlEl);
    }

    $doc->appendChild($urlset);

    return [
        'xml' => $doc->saveXML(),
        'count' => count($entries),
    ];
}

function getAppSettings(bool $force = false): array {
    return loadSettings($force);
}

function getSetting(string $key, $default = null) {
    $settings = loadSettings();
    if (!array_key_exists($key, $settings)) {
        return $default;
    }
    $value = $settings[$key];
    return $value === null || $value === '' ? $default : $value;
}

function getBoolSetting(string $key, bool $default = false): bool {
    $value = getSetting($key, null);
    if ($value === null) {
        return $default;
    }

    $normalized = strtolower((string)$value);
    return in_array($normalized, ['1', 'true', 'on', 'yes'], true);
}

function saveSetting(string $key, $value, bool $refreshCache = true): void {
    global $pdo;

    $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
    $stmt->execute([$key, $value]);

    if ($refreshCache) {
        loadSettings(true);
        if ($key === 'site_base_url') {
            siteBaseUrl(true);
            siteBasePath(true);
        }
    }
}

function siteName(): string {
    return getSetting('site_name', SITE_NAME);
}

function siteDescription(): string {
    return getSetting('site_description', SITE_DESCRIPTION);
}

function siteKeywords(): string {
    return getSetting('site_keywords', SITE_KEYWORDS);
}

function getCommissionRate(string $method): float
{
    switch (strtolower($method)) {
        case 'paytr':
            $value = getSetting('paytr_commission_rate', PAYTR_COMMISSION_RATE_DEFAULT);
            break;
        case 'shopier':
            $value = getSetting('shopier_commission_rate', SHOPIER_COMMISSION_RATE_DEFAULT);
            break;
        default:
            $value = 0;
            break;
    }

    if ($value === null || $value === '') {
        return 0.0;
    }

    $normalized = str_replace(',', '.', (string)$value);
    $number = (float)$normalized;
    return $number < 0 ? 0.0 : $number;
}

function ensureSchemaUpgrades(PDO $pdo): void
{
    static $ran = false;
    if ($ran) {
        return;
    }
    $ran = true;

    $alterStatements = [
        "ALTER TABLE orders ADD COLUMN payable_amount DECIMAL(10,2) NOT NULL DEFAULT 0 AFTER total_amount",
        "ALTER TABLE orders ADD COLUMN commission_amount DECIMAL(10,2) NOT NULL DEFAULT 0 AFTER payable_amount",
        "ALTER TABLE orders ADD COLUMN commission_rate DECIMAL(5,2) NOT NULL DEFAULT 0 AFTER commission_amount",
        "ALTER TABLE orders ADD COLUMN order_type ENUM('purchase','deposit') NOT NULL DEFAULT 'purchase' AFTER payment_method",
        "ALTER TABLE categories ADD COLUMN parent_id INT NULL DEFAULT NULL AFTER id",
        "ALTER TABLE categories ADD COLUMN display_order INT NOT NULL DEFAULT 0 AFTER parent_id",
        "ALTER TABLE categories ADD COLUMN image VARCHAR(255) NULL AFTER icon",
        "ALTER TABLE categories ADD COLUMN banner_image VARCHAR(255) NULL AFTER image",
        "ALTER TABLE categories ADD COLUMN meta_title VARCHAR(150) NULL AFTER banner_image",
        "ALTER TABLE categories ADD COLUMN meta_description VARCHAR(255) NULL AFTER meta_title",
        "ALTER TABLE categories ADD COLUMN delivery_type ENUM('auto','manual') NOT NULL DEFAULT 'auto' AFTER meta_description",
        "ALTER TABLE categories ADD INDEX idx_categories_parent (parent_id)",
        "ALTER TABLE game_marketplace_servers ADD COLUMN delivery_type ENUM('auto','manual') NOT NULL DEFAULT 'auto' AFTER is_active",
        "ALTER TABLE game_marketplace_servers ADD COLUMN image VARCHAR(255) NULL AFTER description",
        "ALTER TABLE users ADD COLUMN email_verified TINYINT(1) NOT NULL DEFAULT 0 AFTER is_active",
        "ALTER TABLE users ADD COLUMN email_verified_at DATETIME NULL AFTER email_verified",
        "ALTER TABLE users ADD COLUMN email_verification_token VARCHAR(100) NULL AFTER email_verified_at",
        "ALTER TABLE users ADD COLUMN email_verification_sent_at DATETIME NULL AFTER email_verification_token",
        "ALTER TABLE users ADD COLUMN payout_account_name VARCHAR(150) NULL AFTER phone",
        "ALTER TABLE users ADD COLUMN payout_bank_name VARCHAR(150) NULL AFTER payout_account_name",
        "ALTER TABLE users ADD COLUMN payout_iban VARCHAR(34) NULL AFTER payout_bank_name",
        "ALTER TABLE navigation_links MODIFY COLUMN location ENUM('header','footer','footer_bottom') NOT NULL DEFAULT 'header'"
    ];

    $alterStatements[] = "ALTER TABLE transactions MODIFY COLUMN type ENUM('deposit','purchase','refund','withdraw') NOT NULL";

    foreach ($alterStatements as $sql) {
        try {
            $pdo->exec($sql);
        } catch (PDOException $e) {
            $message = strtolower($e->getMessage());
            if (strpos($message, 'duplicate') === false && strpos($message, 'exists') === false) {
                // Unexpected error, rethrow for visibility
                throw $e;
            }
        }
    }

    try {
        $pdo->exec("UPDATE orders SET payable_amount = total_amount WHERE payable_amount = 0");
        $pdo->exec("UPDATE orders SET commission_amount = 0 WHERE commission_amount IS NULL");
        $pdo->exec("UPDATE orders SET commission_rate = 0 WHERE commission_rate IS NULL");
        $pdo->exec("UPDATE orders SET order_type = 'deposit' WHERE order_type = 'purchase' AND payment_method IN ('paytr','shopier')");
    } catch (PDOException $e) {
        // ignore data update errors
    }

    $createStatements = [
        "CREATE TABLE IF NOT EXISTS game_marketplaces (
            id INT(11) NOT NULL AUTO_INCREMENT,
            name VARCHAR(150) NOT NULL,
            slug VARCHAR(150) NOT NULL,
            description TEXT NULL,
            image VARCHAR(255) NULL,
            banner_image VARCHAR(255) NULL,
            product_grid_columns TINYINT(1) NOT NULL DEFAULT 4,
            category_grid_columns TINYINT(1) NOT NULL DEFAULT 3,
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            display_order INT NOT NULL DEFAULT 0,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY uq_game_marketplaces_slug (slug)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS game_marketplace_categories (
            id INT(11) NOT NULL AUTO_INCREMENT,
            marketplace_id INT(11) NOT NULL,
            name VARCHAR(150) NOT NULL,
            slug VARCHAR(150) NOT NULL,
            description TEXT NULL,
            warning_text TEXT NULL,
            image VARCHAR(255) NULL,
            banner_image VARCHAR(255) NULL,
            requires_character_name TINYINT(1) NOT NULL DEFAULT 1,
            auto_approve_orders TINYINT(1) NOT NULL DEFAULT 0,
            custom_product_columns TINYINT(1) NULL,
            custom_category_columns TINYINT(1) NULL,
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            display_order INT NOT NULL DEFAULT 0,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY uq_marketplace_category_slug (marketplace_id, slug),
            KEY idx_gm_categories_marketplace (marketplace_id),
            CONSTRAINT fk_gm_categories_marketplace FOREIGN KEY (marketplace_id) REFERENCES game_marketplaces(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS game_marketplace_servers (
            id INT(11) NOT NULL AUTO_INCREMENT,
            category_id INT(11) NOT NULL,
            name VARCHAR(150) NOT NULL,
            slug VARCHAR(150) NOT NULL,
            description TEXT NULL,
            image VARCHAR(255) NULL,
            buy_price DECIMAL(10,2) NOT NULL DEFAULT 0,
            sell_price DECIMAL(10,2) NOT NULL DEFAULT 0,
            min_sell_quantity DECIMAL(12,2) NOT NULL DEFAULT 1,
            max_sell_quantity DECIMAL(12,2) NULL,
            min_buy_quantity DECIMAL(12,2) NOT NULL DEFAULT 1,
            max_buy_quantity DECIMAL(12,2) NULL,
            sell_stock DECIMAL(14,2) NOT NULL DEFAULT 0,
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            display_order INT NOT NULL DEFAULT 0,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY uq_marketplace_server_slug (category_id, slug),
            KEY idx_gm_servers_category (category_id),
            CONSTRAINT fk_gm_servers_category FOREIGN KEY (category_id) REFERENCES game_marketplace_categories(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS game_marketplace_orders (
            id INT(11) NOT NULL AUTO_INCREMENT,
            user_id INT(11) NOT NULL,
            marketplace_id INT(11) NOT NULL,
            category_id INT(11) NOT NULL,
            server_id INT(11) NOT NULL,
            order_type ENUM('buy','sell') NOT NULL,
            status ENUM('pending','awaiting_payment','processing','approved','completed','cancelled','rejected') NOT NULL DEFAULT 'pending',
            quantity DECIMAL(14,2) NOT NULL,
            price_per_unit DECIMAL(10,2) NOT NULL,
            total_amount DECIMAL(14,2) NOT NULL,
            currency VARCHAR(10) NOT NULL DEFAULT 'TRY',
            character_name VARCHAR(150) NULL,
            contact_note TEXT NULL,
            admin_note TEXT NULL,
            admin_id INT(11) NULL,
            processed_at TIMESTAMP NULL DEFAULT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_gmo_user (user_id),
            KEY idx_gmo_marketplace (marketplace_id),
            KEY idx_gmo_category (category_id),
            KEY idx_gmo_server (server_id),
            KEY idx_gmo_status (status),
            KEY idx_gmo_type (order_type),
            CONSTRAINT fk_gmo_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            CONSTRAINT fk_gmo_marketplace FOREIGN KEY (marketplace_id) REFERENCES game_marketplaces(id) ON DELETE CASCADE,
            CONSTRAINT fk_gmo_category FOREIGN KEY (category_id) REFERENCES game_marketplace_categories(id) ON DELETE CASCADE,
            CONSTRAINT fk_gmo_server FOREIGN KEY (server_id) REFERENCES game_marketplace_servers(id) ON DELETE CASCADE,
            CONSTRAINT fk_gmo_admin FOREIGN KEY (admin_id) REFERENCES users(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS navigation_links (
            id INT(11) NOT NULL AUTO_INCREMENT,
            title VARCHAR(150) NOT NULL,
            url VARCHAR(255) DEFAULT NULL,
            menu_type ENUM('link','marketplace','balance') NOT NULL DEFAULT 'link',
            location ENUM('header','footer') NOT NULL DEFAULT 'header',
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            open_new_tab TINYINT(1) NOT NULL DEFAULT 0,
            sort_order INT(11) NOT NULL DEFAULT 0,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_navigation_location_sort (location, sort_order)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS cart_items (
            id INT(11) NOT NULL AUTO_INCREMENT,
            user_id INT(11) NOT NULL,
            item_type ENUM('product','marketplace') NOT NULL,
            reference_id INT(11) NOT NULL,
            quantity DECIMAL(12,2) NOT NULL DEFAULT 1,
            unit_price DECIMAL(10,2) NOT NULL DEFAULT 0,
            character_name VARCHAR(150) NOT NULL DEFAULT '',
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY uniq_cart_item (user_id, item_type, reference_id, character_name),
            KEY idx_cart_user (user_id),
            CONSTRAINT fk_cart_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS password_reset_tokens (
            id INT(11) NOT NULL AUTO_INCREMENT,
            user_id INT(11) NOT NULL,
            token VARCHAR(100) NOT NULL,
            expires_at DATETIME NOT NULL,
            used_at DATETIME NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY uq_reset_token (token),
            KEY idx_reset_user (user_id),
            KEY idx_reset_expires (expires_at),
            CONSTRAINT fk_reset_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS support_tickets (
            id INT(11) NOT NULL AUTO_INCREMENT,
            user_id INT(11) NOT NULL,
            subject VARCHAR(200) NOT NULL,
            status ENUM('open','answered','customer_reply','resolved') NOT NULL DEFAULT 'open',
            admin_id INT(11) NULL,
            last_status_changed_at DATETIME NULL,
            last_message_at DATETIME NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_support_tickets_user (user_id),
            KEY idx_support_tickets_status (status),
            KEY idx_support_tickets_last_message (last_message_at),
            KEY idx_support_tickets_admin (admin_id),
            CONSTRAINT fk_support_tickets_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            CONSTRAINT fk_support_tickets_admin FOREIGN KEY (admin_id) REFERENCES users(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS support_ticket_messages (
            id INT(11) NOT NULL AUTO_INCREMENT,
            ticket_id INT(11) NOT NULL,
            sender_type ENUM('user','admin') NOT NULL,
            sender_id INT(11) NULL,
            message TEXT NOT NULL,
            attachment_path VARCHAR(255) NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_support_ticket_messages_ticket (ticket_id),
            KEY idx_support_ticket_messages_sender (sender_type, sender_id),
            CONSTRAINT fk_support_ticket_messages_ticket FOREIGN KEY (ticket_id) REFERENCES support_tickets(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS withdraw_requests (
            id INT(11) NOT NULL AUTO_INCREMENT,
            user_id INT(11) NOT NULL,
            amount DECIMAL(10,2) NOT NULL,
            status ENUM('pending','approved','rejected') NOT NULL DEFAULT 'pending',
            admin_id INT(11) NULL,
            admin_note TEXT NULL,
            account_name VARCHAR(150) NOT NULL,
            bank_name VARCHAR(150) NOT NULL,
            iban VARCHAR(34) NOT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            processed_at TIMESTAMP NULL DEFAULT NULL,
            PRIMARY KEY (id),
            KEY idx_withdraw_user (user_id),
            KEY idx_withdraw_status (status),
            CONSTRAINT fk_withdraw_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            CONSTRAINT fk_withdraw_admin FOREIGN KEY (admin_id) REFERENCES users(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS homepage_slider_items (
            id INT(11) NOT NULL AUTO_INCREMENT,
            display_order INT(11) NOT NULL DEFAULT 0,
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            eyebrow VARCHAR(150) NOT NULL DEFAULT '',
            title VARCHAR(200) NOT NULL,
            highlight VARCHAR(150) NULL,
            description TEXT NULL,
            primary_label VARCHAR(120) DEFAULT NULL,
            primary_icon VARCHAR(60) DEFAULT NULL,
            primary_url VARCHAR(255) DEFAULT NULL,
            secondary_label VARCHAR(120) DEFAULT NULL,
            secondary_icon VARCHAR(60) DEFAULT NULL,
            secondary_url VARCHAR(255) DEFAULT NULL,
            image VARCHAR(255) DEFAULT NULL,
            badge_primary_icon VARCHAR(60) DEFAULT NULL,
            badge_primary_text VARCHAR(150) DEFAULT NULL,
            badge_secondary_icon VARCHAR(60) DEFAULT NULL,
            badge_secondary_text VARCHAR(150) DEFAULT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_slider_order (display_order),
            KEY idx_slider_active (is_active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS homepage_side_banners (
            id INT(11) NOT NULL AUTO_INCREMENT,
            position TINYINT(1) NOT NULL DEFAULT 1 COMMENT '1=top, 2=bottom',
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            title VARCHAR(200) DEFAULT NULL,
            link_url VARCHAR(255) DEFAULT NULL,
            image VARCHAR(255) NOT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_position (position),
            KEY idx_active (is_active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",

        "CREATE TABLE IF NOT EXISTS homepage_bottom_banners (
            id INT(11) NOT NULL AUTO_INCREMENT,
            display_order INT(11) NOT NULL DEFAULT 0,
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            title VARCHAR(200) DEFAULT NULL,
            link_url VARCHAR(255) DEFAULT NULL,
            image VARCHAR(255) NOT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_order (display_order),
            KEY idx_active (is_active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci"
    ];

    foreach ($createStatements as $sql) {
        try {
            $pdo->exec($sql);
        } catch (PDOException $e) {
            throw $e;
        }
    }

    // Backward compatibility for legacy table names expected by old admin dashboard
    try {
        $pdo->exec('CREATE OR REPLACE VIEW tickets AS SELECT * FROM support_tickets');
    } catch (PDOException $e) {
        // View oluşturma yetkisi bulunmayabilir; hatayı yoksayarak devam et
    }

    try {
        $pdo->exec('CREATE OR REPLACE VIEW ticket_messages AS SELECT * FROM support_ticket_messages');
    } catch (PDOException $e) {
        // View oluşturma yetkisi bulunmayabilir; hatayı yoksayarak devam et
    }

    $pdo->exec("INSERT IGNORE INTO settings (setting_key, setting_value) VALUES
        ('email_base_url', ''),
        ('email_template_password_reset', ''),
        ('auth_require_email_verification', '0'),
        ('site_base_url', ''),
        ('branding_logo_mode', 'text'),
        ('branding_logo_text', ''),
        ('branding_logo_tagline', ''),
        ('branding_logo_image', ''),
        ('branding_favicon_image', ''),
        ('branding_logo_width', ''),
        ('branding_logo_height', ''),
        ('slider_autoplay_delay', '5000'),
        ('slider_autoplay_enabled', '1'),
        ('footer_brand_description', 'Dijital oyun kodları, marketplace ilanları ve oyun parasında güvenilir çözüm ortağınız.'),
        ('footer_brand_highlight', 'Hızlı teslimat, güvenli ödeme ve deneyimli destek ekibimizle alışveriş deneyiminizi bir üst seviyeye taşıyoruz.'),
        ('footer_brand_title', ''),
        ('footer_brand_tagline', ''),
        ('footer_social_instagram', ''),
        ('footer_social_twitter', ''),
        ('footer_social_whatsapp', ''),
        ('footer_social_youtube', ''),
        ('footer_contact_phone', '+90 (555) 123 45 67'),
        ('footer_contact_email', 'destek@enbmedya.com'),
        ('footer_contact_address', 'İstanbul / Türkiye'),
        ('footer_menu_heading', 'Menü'),
        ('footer_category_heading', 'Popüler kategoriler'),
        ('footer_contact_heading', 'İletişim'),
        ('footer_payment_badges', 'PayTR|Shopier|3D Secure|SSL'),
        ('footer_bottom_text', ''),
        ('footer_category_ids', ''),
        ('header_top_phone', ''),
        ('header_top_email', ''),
        ('sitemap_include_home', '1'),
        ('sitemap_include_products', '1'),
        ('sitemap_include_categories', '1'),
        ('sitemap_include_pages', '1'),
        ('sitemap_include_blog', '1'),
        ('sitemap_include_marketplace', '1'),
        ('sitemap_include_support', '1'),
        ('seo_robots_content', '')");

    try {
        $navCount = (int)$pdo->query("SELECT COUNT(*) FROM navigation_links WHERE location = 'header'")->fetchColumn();
        if ($navCount === 0) {
            $defaults = [
                ['Ana Sayfa', 'index.php', 'link', 10],
                ['Oyunlar', 'products.php', 'link', 20],
                ['Marketplace', 'marketplace.php', 'marketplace', 30],
                ['Pazaryeri', 'products.php?tab=market', 'link', 40],
                ['Oyun Parası', 'products.php?tab=coins', 'link', 50],
                ['Bakiyem', null, 'balance', 60],
                ['Destek', 'support.php', 'link', 70],
                ['Blog', 'blog.php', 'link', 80],
                ['PUBG UC', 'products.php?q=PUBG%20UC', 'link', 90],
            ];
            $stmt = $pdo->prepare("INSERT INTO navigation_links (title, url, menu_type, location, sort_order) VALUES (:title, :url, :type, 'header', :sort)");
            foreach ($defaults as [$title, $url, $type, $order]) {
                $stmt->execute([
                    ':title' => $title,
                    ':url' => $url,
                    ':type' => $type,
                    ':sort' => $order,
                ]);
            }
        }
        $pdo->exec("UPDATE navigation_links SET url = 'support.php' WHERE title = 'Destek' AND (url IS NULL OR url = 'index.php#destek')");

        $footerCount = (int)$pdo->query("SELECT COUNT(*) FROM navigation_links WHERE location = 'footer'")->fetchColumn();
        if ($footerCount === 0) {
            $footerDefaults = [
                ['Hakkımızda', 'page.php?slug=hakkimizda', 'link', 10],
                ['Blog', 'blog.php', 'link', 20],
                ['Destek', 'support.php', 'link', 30],
                ['Kullanıcı Sözleşmesi', 'page.php?slug=kullanici-sozlesmesi', 'link', 40],
            ];
            $stmtFooter = $pdo->prepare("INSERT INTO navigation_links (title, url, menu_type, location, sort_order, is_active, open_new_tab) VALUES (:title, :url, :type, 'footer', :sort, 1, 0)");
            foreach ($footerDefaults as [$title, $url, $type, $order]) {
                $stmtFooter->execute([
                    ':title' => $title,
                    ':url' => $url,
                    ':type' => $type,
                    ':sort' => $order,
                ]);
            }
        }

        $footerBottomCount = (int)$pdo->query("SELECT COUNT(*) FROM navigation_links WHERE location = 'footer_bottom'")->fetchColumn();
        if ($footerBottomCount === 0) {
            $bottomDefaults = [
                ['Kullanıcı Sözleşmesi', 'page.php?slug=kullanici-sozlesmesi', 'link', 10],
                ['Gizlilik Politikası', 'page.php?slug=gizlilik-politikasi', 'link', 20],
                ['İade & İptal', 'page.php?slug=iade-iptal', 'link', 30],
            ];
            $stmtBottom = $pdo->prepare("INSERT INTO navigation_links (title, url, menu_type, location, sort_order, is_active, open_new_tab) VALUES (:title, :url, :type, 'footer_bottom', :sort, 1, 0)");
            foreach ($bottomDefaults as [$title, $url, $type, $order]) {
                $stmtBottom->execute([
                    ':title' => $title,
                    ':url' => $url,
                    ':type' => $type,
                    ':sort' => $order,
                ]);
            }
        }

        $slideCount = (int)$pdo->query("SELECT COUNT(*) FROM homepage_slider_items")->fetchColumn();
        if ($slideCount === 0) {
            $defaultSlides = [
                [
                    'order' => 10,
                    'eyebrow' => 'Yeni sezon kampanyaları',
                    'title' => 'Oyun alışverişinde anında teslimat dönemini başlatın.',
                    'highlight' => 'anında teslimat',
                    'description' => siteName() . ' ile popüler oyun kodları, marketplace ilanları ve oyun parasını güvenle satın alın. Siparişleriniz saniyeler içinde teslim edilir.',
                    'primary_label' => 'Ürünlere göz at',
                    'primary_icon' => 'fa-store',
                    'primary_url' => routeUrl('products'),
                    'secondary_label' => 'İlan Pazarı',
                    'secondary_icon' => 'fa-basket-shopping',
                    'secondary_url' => routeUrl('marketplace'),
                    'image' => 'assets/images/hero-showcase.svg',
                    'badge_primary_icon' => 'fa-shield',
                    'badge_primary_text' => 'Yetkili satıcı',
                    'badge_secondary_icon' => 'fa-clock',
                    'badge_secondary_text' => '7/24 teslimat',
                ],
                [
                    'order' => 20,
                    'eyebrow' => 'Yeni çıkan oyunlar',
                    'title' => 'Popüler oyun kodlarını saniyede teslim alın.',
                    'highlight' => 'saniyede',
                    'description' => 'Geniş dijital ürün kataloğuyla aradığınız oyun kodunu hemen bulun, otomatik teslimatla vakit kaybetmeyin.',
                    'primary_label' => 'Hızlı teslim ürünler',
                    'primary_icon' => 'fa-bolt',
                    'primary_url' => routeUrl('products'),
                    'secondary_label' => 'Marketplace',
                    'secondary_icon' => 'fa-gamepad',
                    'secondary_url' => routeUrl('marketplace'),
                    'image' => 'assets/images/hero-showcase.svg',
                    'badge_primary_icon' => 'fa-badge-check',
                    'badge_primary_text' => 'Güvenli ödeme',
                    'badge_secondary_icon' => 'fa-face-smile',
                    'badge_secondary_text' => 'Mutlu oyuncular',
                ],
                [
                    'order' => 30,
                    'eyebrow' => 'Bakiye fırsatları',
                    'title' => 'Tek panelden pazaryeri ve bakiye yönetimi.',
                    'highlight' => 'Tek panelden',
                    'description' => 'Bakiye yükleme, marketplace ilanı açma ve oyun parası satışı için profesyonel çözümler.',
                    'primary_label' => 'Bakiye Yükle',
                    'primary_icon' => 'fa-wallet',
                    'primary_url' => routeUrl('balance'),
                    'secondary_label' => 'Marketplace',
                    'secondary_icon' => 'fa-store',
                    'secondary_url' => routeUrl('marketplace'),
                    'image' => 'assets/images/hero-showcase.svg',
                    'badge_primary_icon' => 'fa-coins',
                    'badge_primary_text' => 'Komisyon avantajı',
                    'badge_secondary_icon' => 'fa-headset',
                    'badge_secondary_text' => '7/24 destek',
                ],
            ];
            $insertSlide = $pdo->prepare('INSERT INTO homepage_slider_items (display_order, is_active, eyebrow, title, highlight, description, primary_label, primary_icon, primary_url, secondary_label, secondary_icon, secondary_url, image, badge_primary_icon, badge_primary_text, badge_secondary_icon, badge_secondary_text) VALUES (:order, 1, :eyebrow, :title, :highlight, :description, :p_label, :p_icon, :p_url, :s_label, :s_icon, :s_url, :image, :b1_icon, :b1_text, :b2_icon, :b2_text)');
            foreach ($defaultSlides as $slide) {
                $insertSlide->execute([
                    ':order' => $slide['order'],
                    ':eyebrow' => $slide['eyebrow'],
                    ':title' => $slide['title'],
                    ':highlight' => $slide['highlight'],
                    ':description' => $slide['description'],
                    ':p_label' => $slide['primary_label'],
                    ':p_icon' => $slide['primary_icon'],
                    ':p_url' => $slide['primary_url'],
                    ':s_label' => $slide['secondary_label'],
                    ':s_icon' => $slide['secondary_icon'],
                    ':s_url' => $slide['secondary_url'],
                    ':image' => $slide['image'],
                    ':b1_icon' => $slide['badge_primary_icon'],
                    ':b1_text' => $slide['badge_primary_text'],
                    ':b2_icon' => $slide['badge_secondary_icon'],
                    ':b2_text' => $slide['badge_secondary_text'],
                ]);
            }
        }
    } catch (Throwable $th) {
        // ignore seed issues
    }

    $pdo->exec("UPDATE homepage_slider_items SET image = 'assets/images/hero-showcase.svg' WHERE image IS NULL OR image = '' OR image LIKE 'assets/images/hero-slide-%'");

    // Seed default side banners
    try {
        $sideBannerCount = (int)$pdo->query("SELECT COUNT(*) FROM homepage_side_banners")->fetchColumn();
        if ($sideBannerCount === 0) {
            $pdo->exec("INSERT INTO homepage_side_banners (position, is_active, title, link_url, image) VALUES 
                (1, 1, 'Kategoriler', 'products', 'assets/images/home_category/7b6d683e-a3a1-42dc-bed9-d02bb111347964415.jpg'),
                (2, 1, 'Marketplace', 'marketplace', 'assets/images/home_category/b0c7fb53-7fa5-451d-9d80-386b6a62c6c067010.jpg')");
        }
    } catch (Throwable $th) {
        // ignore seed issues
    }

    // Seed default bottom banners
    try {
        $bottomBannerCount = (int)$pdo->query("SELECT COUNT(*) FROM homepage_bottom_banners")->fetchColumn();
        if ($bottomBannerCount === 0) {
            $pdo->exec("INSERT INTO homepage_bottom_banners (display_order, is_active, title, link_url, image) VALUES 
                (1, 1, 'Kategoriler', 'products', 'assets/images/home_category/7b6d683e-a3a1-42dc-bed9-d02bb111347964415.jpg'),
                (2, 1, 'Marketplace', 'marketplace', 'assets/images/home_category/b0c7fb53-7fa5-451d-9d80-386b6a62c6c067010.jpg'),
                (3, 1, 'Ürünler', 'products', 'assets/images/home_category/7b6d683e-a3a1-42dc-bed9-d02bb111347964415.jpg')");
        }
    } catch (Throwable $th) {
        // ignore seed issues
    }

    $alterMarketplaceGames = [
        "ALTER TABLE game_marketplaces ADD COLUMN image VARCHAR(255) NULL AFTER description",
        "ALTER TABLE game_marketplaces ADD COLUMN banner_image VARCHAR(255) NULL AFTER image",
        "ALTER TABLE game_marketplaces ADD COLUMN product_grid_columns TINYINT(1) NOT NULL DEFAULT 4 AFTER banner_image",
        "ALTER TABLE game_marketplaces ADD COLUMN category_grid_columns TINYINT(1) NOT NULL DEFAULT 3 AFTER product_grid_columns"
    ];

    foreach ($alterMarketplaceGames as $sql) {
        try {
            $pdo->exec($sql);
        } catch (PDOException $e) {
            $message = strtolower($e->getMessage());
            if (strpos($message, 'duplicate') === false && strpos($message, 'exists') === false && strpos($message, 'unknown column') === false) {
                throw $e;
            }
        }
    }

    $alterMarketplaceCategories = [
        "ALTER TABLE game_marketplace_categories ADD COLUMN requires_character_name TINYINT(1) NOT NULL DEFAULT 1 AFTER banner_image",
        "ALTER TABLE game_marketplace_categories ADD COLUMN auto_approve_orders TINYINT(1) NOT NULL DEFAULT 0 AFTER requires_character_name",
        "ALTER TABLE game_marketplace_categories ADD COLUMN custom_product_columns TINYINT(1) NULL AFTER auto_approve_orders",
        "ALTER TABLE game_marketplace_categories ADD COLUMN custom_category_columns TINYINT(1) NULL AFTER custom_product_columns"
    ];

    foreach ($alterMarketplaceCategories as $sql) {
        try {
            $pdo->exec($sql);
        } catch (PDOException $e) {
            $message = strtolower($e->getMessage());
            if (strpos($message, 'duplicate') === false && strpos($message, 'exists') === false && strpos($message, 'unknown column') === false) {
                throw $e;
            }
        }
    }

    $createMarketplaceCodesSql = "CREATE TABLE IF NOT EXISTS game_marketplace_codes (
            id INT(11) NOT NULL AUTO_INCREMENT,
            server_id INT(11) NOT NULL,
            code VARCHAR(255) NOT NULL,
            is_sold TINYINT(1) NOT NULL DEFAULT 0,
            sold_to_user_id INT(11) NULL,
            order_id INT(11) NULL,
            sold_at DATETIME NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_gmc_server (server_id),
            KEY idx_gmc_order (order_id),
            KEY idx_gmc_sold (is_sold),
            CONSTRAINT fk_gmc_server FOREIGN KEY (server_id) REFERENCES game_marketplace_servers(id) ON DELETE CASCADE,
            CONSTRAINT fk_gmc_user FOREIGN KEY (sold_to_user_id) REFERENCES users(id) ON DELETE SET NULL,
            CONSTRAINT fk_gmc_order FOREIGN KEY (order_id) REFERENCES game_marketplace_orders(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    try {
        $tableExists = $pdo->query("SHOW TABLES LIKE 'game_marketplace_codes'");
        if (!$tableExists || !$tableExists->fetch()) {
            $pdo->exec($createMarketplaceCodesSql);
        }
    } catch (PDOException $e) {
        throw $e;
    }

    $alterMarketplaceCodes = [
        "ALTER TABLE game_marketplace_codes ADD COLUMN order_id INT(11) NULL AFTER sold_to_user_id",
        "ALTER TABLE game_marketplace_codes ADD COLUMN sold_to_user_id INT(11) NULL AFTER is_sold",
        "ALTER TABLE game_marketplace_codes ADD COLUMN sold_at DATETIME NULL AFTER order_id",
        "ALTER TABLE game_marketplace_codes ADD INDEX idx_gmc_order (order_id)",
        "ALTER TABLE game_marketplace_codes ADD CONSTRAINT fk_gmc_order FOREIGN KEY (order_id) REFERENCES game_marketplace_orders(id) ON DELETE SET NULL",
        "ALTER TABLE game_marketplace_codes ADD CONSTRAINT fk_gmc_user FOREIGN KEY (sold_to_user_id) REFERENCES users(id) ON DELETE SET NULL"
    ];

    $marketplaceCodesExistsStmt = $pdo->query("SHOW TABLES LIKE 'game_marketplace_codes'");
    $marketplaceCodesExists = $marketplaceCodesExistsStmt && $marketplaceCodesExistsStmt->fetch();

    if ($marketplaceCodesExists) {
        foreach ($alterMarketplaceCodes as $sql) {
            try {
                $pdo->exec($sql);
            } catch (PDOException $e) {
                $message = strtolower($e->getMessage());
                if (
                    strpos($message, 'duplicate') === false &&
                    strpos($message, 'exists') === false &&
                    strpos($message, 'base table or view not found') === false
                ) {
                    throw $e;
                }
            }
        }
    }
}

function slugify(string $text): string {
    $text = safe_trim($text);
    if ($text === '') {
        return '';
    }
    $text = iconv('UTF-8', 'ASCII//TRANSLIT', $text);
    $text = preg_replace('~[^\pL\d]+~u', '-', $text);
    $text = preg_replace('~[^-\w]+~', '', $text);
    $text = safe_trim($text, '-');
    $text = preg_replace('~-+~', '-', $text);
    return strtolower($text ?: uniqid());
}

function ensureUploadDirectory(string $subDir): string {
    $path = safe_rtrim(UPLOAD_DIR, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . safe_trim($subDir, DIRECTORY_SEPARATOR);
    if (!is_dir($path)) {
        mkdir($path, 0775, true);
    }
    return $path;
}

function getActivityLogDirectory(): string {
    $path = __DIR__ . '/logs';
    if (!is_dir($path)) {
        @mkdir($path, 0775, true);
    }
    return $path;
}

function writeActivityLogToFile(array $record): void {
    try {
        $logDir = rtrim(getActivityLogDirectory(), DIRECTORY_SEPARATOR);
        if ($logDir === '') {
            return;
        }

        if (!isset($record['timestamp'])) {
            $record['timestamp'] = date('c');
        }

        $json = json_encode($record, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        if ($json === false) {
            return;
        }

        $fileName = 'activity-' . date('Y-m-d') . '.log';
        $logPath = $logDir . DIRECTORY_SEPARATOR . $fileName;
        $line = date('Y-m-d H:i:s') . ' ' . $json . PHP_EOL;
        @file_put_contents($logPath, $line, FILE_APPEND | LOCK_EX);
    } catch (Throwable $th) {
        // sessizlik
    }
}

/**
 * Log dosyalarından tüm log kayıtlarını okur
 * @param string|null $fromDate Başlangıç tarihi (Y-m-d formatında)
 * @param string|null $toDate Bitiş tarihi (Y-m-d formatında)
 * @return array Log kayıtları
 */
function readActivityLogsFromFiles(?string $fromDate = null, ?string $toDate = null): array {
    $logDir = getActivityLogDirectory();
    $allLogs = [];
    
    try {
        // Tarih aralığını belirle
        if ($fromDate === null) {
            $fromDate = date('Y-m-d', strtotime('-30 days')); // Son 30 gün
        }
        if ($toDate === null) {
            $toDate = date('Y-m-d');
        }
        
        $start = new DateTime($fromDate);
        $end = new DateTime($toDate);
        $end->modify('+1 day'); // Bitiş tarihini dahil etmek için
        
        $current = clone $start;
        
        // Her gün için log dosyasını oku
        while ($current < $end) {
            $dateStr = $current->format('Y-m-d');
            $fileName = 'activity-' . $dateStr . '.log';
            $filePath = $logDir . DIRECTORY_SEPARATOR . $fileName;
            
            if (is_file($filePath) && is_readable($filePath)) {
                $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                if ($lines !== false) {
                    foreach ($lines as $line) {
                        $line = trim($line);
                        if ($line === '') {
                            continue;
                        }
                        
                        // Satır formatı: "YYYY-MM-DD HH:MM:SS {json}"
                        // İlk 19 karakter tarih ve saat (YYYY-MM-DD HH:MM:SS)
                        if (strlen($line) < 20) {
                            continue;
                        }
                        
                        $timePart = substr($line, 0, 19); // "YYYY-MM-DD HH:MM:SS"
                        $jsonPart = trim(substr($line, 20)); // Geri kalan JSON kısmı
                        
                        $data = json_decode($jsonPart, true);
                        if ($data !== null && is_array($data)) {
                            // Veritabanı formatına uygun hale getir
                            $logEntry = [
                                'id' => md5($line), // Benzersiz ID için hash
                                'user_id' => $data['user_id'] ?? null,
                                'session_id' => $data['session_id'] ?? null,
                                'device_id' => $data['device_id'] ?? null,
                                'ip_address' => $data['ip'] ?? '',
                                'user_agent' => $data['context']['user_agent'] ?? null,
                                'request_method' => $data['request_method'] ?? null,
                                'request_uri' => $data['request_uri'] ?? '',
                                'referer' => $data['referer'] ?? null,
                                'country' => $data['country'] ?? null,
                                'city' => $data['city'] ?? null,
                                'region' => $data['region'] ?? null,
                                'latitude' => $data['latitude'] ?? null,
                                'longitude' => $data['longitude'] ?? null,
                                'event_type' => $data['event_type'] ?? 'page_view',
                                'context' => json_encode($data['context'] ?? [], JSON_UNESCAPED_UNICODE),
                                'created_at' => $timePart,
                                'username' => null,
                                'email' => null,
                            ];
                            
                            // Kullanıcı bilgilerini al
                            if ($logEntry['user_id'] !== null) {
                                global $pdo;
                                try {
                                    $userStmt = $pdo->prepare('SELECT username, email FROM users WHERE id = ?');
                                    $userStmt->execute([$logEntry['user_id']]);
                                    $user = $userStmt->fetch();
                                    if ($user) {
                                        $logEntry['username'] = $user['username'];
                                        $logEntry['email'] = $user['email'];
                                    }
                                } catch (Throwable $th) {
                                    // Sessiz geç
                                }
                            }
                            
                            $allLogs[] = $logEntry;
                        }
                    }
                }
            }
            
            $current->modify('+1 day');
        }
        
        // Tarihe göre sırala (en yeni önce)
        usort($allLogs, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
    } catch (Throwable $th) {
        // Hata durumunda boş array döndür
    }
    
    return $allLogs;
}

function deleteUploadedFile(?string $relativePath): void {
    if (!$relativePath) {
        return;
    }
    $fullPath = __DIR__ . '/' . ltrim($relativePath, '/\\');
    if (is_file($fullPath)) {
        @unlink($fullPath);
    }
}

function uploadImageFile(array $file, string $subDir, ?string $currentPath = null): ?string {
    // Eğer dosya seçilmemişse, mevcut path'i koru
    if (!isset($file['error']) || $file['error'] === UPLOAD_ERR_NO_FILE || empty($file['name'])) {
        return $currentPath;
    }

    if ($file['error'] !== UPLOAD_ERR_OK) {
        throw new RuntimeException('Dosya yükleme başarısız (hata kodu: ' . $file['error'] . ').');
    }

    if ($file['size'] > 4 * 1024 * 1024) {
        throw new RuntimeException('Dosya boyutu 4MB sınırını aşıyor.');
    }

    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']);
    $allowed = [
        'image/jpeg' => 'jpg',
        'image/png'  => 'png',
        'image/webp' => 'webp',
        'image/gif'  => 'gif'
    ];

    if (!isset($allowed[$mime])) {
        throw new RuntimeException('Desteklenmeyen dosya formatı.');
    }

    $directory = ensureUploadDirectory($subDir);
    $prefixRaw = safe_trim($subDir, '/\\');
    $prefix = $prefixRaw !== '' ? str_replace(['/', '\\'], '-', $prefixRaw) . '_' : '';
    try {
        $randomPart = bin2hex(random_bytes(8));
    } catch (Throwable $th) {
        $randomPart = uniqid('', true);
    }
    $filename = $prefix . $randomPart . '.' . $allowed[$mime];
    $destination = rtrim($directory, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $filename;

    $moved = move_uploaded_file($file['tmp_name'], $destination);
    if (!$moved) {
        $moved = @rename($file['tmp_name'], $destination);
    }
    if (!$moved) {
        $moved = @copy($file['tmp_name'], $destination);
        if ($moved) {
            @unlink($file['tmp_name']);
        }
    }

    if (!$moved) {
        throw new RuntimeException('Dosya sunucuya taşınamadı.');
    }

    @chmod($destination, 0644);

    // Eski dosyayı sil (eğer yeni dosya yüklendiyse)
    if ($currentPath) {
        deleteUploadedFile($currentPath);
    }

    $relativeDir = safe_trim($subDir, '/\\');
    $path = $relativeDir === '' ? UPLOAD_URL . '/' . $filename : UPLOAD_URL . '/' . $relativeDir . '/' . $filename;
    
    return $path;
}

function assetUrl(?string $path): string {
    if ($path === null || $path === '') {
        return '';
    }

    if (preg_match('#^https?://#i', $path)) {
        return $path;
    }

    $normalized = ltrim(str_replace('\\', '/', $path), '/');

    return publicUrl($normalized, true);
}

function publicUrl(string $path, bool $absolute = false): string
{
    $normalized = '/' . ltrim(str_replace('\\', '/', $path), '/');
    $relative = buildRelativeUrl($normalized);
    if ($absolute) {
        $baseUrl = siteBaseUrl();
        $basePath = siteBasePath();
        $tail = $relative;
        if ($basePath !== '' && str_starts_with($relative, $basePath)) {
            $tail = substr($relative, strlen($basePath));
            if ($tail === '' || $tail === false) {
                $tail = '/';
            }
        }
        return rtrim($baseUrl, '/') . ($tail === '/' ? '/' : $tail);
    }
    return $relative;
}

// Helper Fonksiyonlar
function redirect($url) {
    header("Location: " . $url);
    exit();
}

function setFlashMessage($type, $message) {
    if (!isset($_SESSION)) {
        session_start();
    }
    $_SESSION['flash_message'] = [
        'type' => $type,
        'message' => $message
    ];
}

function displayFlashMessage() {
    if (!isset($_SESSION)) {
        session_start();
    }
    if (isset($_SESSION['flash_message'])) {
        $flash = $_SESSION['flash_message'];
        $alertClass = $flash['type'] === 'error' ? 'alert-danger' : 'alert-success';
        echo '<div class="alert ' . $alertClass . ' alert-dismissible fade show" role="alert">';
        echo htmlspecialchars($flash['message'], ENT_QUOTES, 'UTF-8');
        echo '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>';
        echo '</div>';
        unset($_SESSION['flash_message']);
    }
}

function isLoggedIn(): bool {
    return isset($_SESSION['user_id']);
}

function isAdmin(): bool {
    return isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1;
}

function requireLogin() {
    if (!isLoggedIn()) {
        redirect('login.php');
    }
}

function requireAdmin() {
    if (!isLoggedIn()) {
        redirect('../login.php');
    }

    if (!isAdmin()) {
        redirect('../index.php');
    }
}

function sanitize($data): string {
    if ($data === null) {
        return '';
    }
    return htmlspecialchars(trim((string)$data), ENT_QUOTES, 'UTF-8');
}

/**
 * Metni belirtilen uzunlukta kısaltır
 */
function shortenText($text, $maxLength = 100, $suffix = '...'): string {
    $text = strip_tags((string)$text);
    $text = trim($text);
    if (mb_strlen($text) <= $maxLength) {
        return $text;
    }
    return mb_substr($text, 0, $maxLength) . $suffix;
}

function generateOrderNumber(): string {
    try {
        $random = strtoupper(bin2hex(random_bytes(5)));
    } catch (Exception $e) {
        $random = strtoupper(substr(md5(uniqid((string)mt_rand(), true)), 0, 10));
    }

    return 'ORD' . date('Ymd') . $random;
}

function generateSlug(string $text, string $separator = '-'): string {
    $text = safe_trim($text);
    if ($text === '') {
        return 'icerik-' . substr(md5((string)microtime(true)), 0, 6);
    }

    $encoded = iconv('UTF-8', 'ASCII//TRANSLIT', $text);
    if ($encoded !== false) {
        $text = $encoded;
    }

    $text = preg_replace('~[^\pL\d]+~u', $separator, $text ?? '');
    $text = preg_replace('~[^-\w]+~', '', $text ?? '');
    $text = safe_trim($text, $separator);
    $text = preg_replace('~-+~', $separator, (string)$text);
    $text = strtolower((string)$text);

    if ($text === '') {
        $text = 'icerik-' . substr(md5((string)microtime(true)), 0, 6);
    }

    return $text;
}

function buildMetaDefaults(string $title, string $content = ''): array {
    $cleanTitle = safe_trim(strip_tags($title));
    $cleanContent = safe_trim(preg_replace('/\s+/', ' ', strip_tags($content)));

    $metaTitle = $cleanTitle !== '' ? $cleanTitle : 'Epin Platformu';

    $description = $cleanContent !== '' ? mb_substr($cleanContent, 0, 180) : $metaTitle;
    if (mb_strlen($description) === 180 && mb_strlen($cleanContent) > 180) {
        $description = safe_rtrim($description, ',.;:- ') . '...';
    }

    $keywords = [];
    if ($cleanTitle !== '') {
        $keywords[] = $cleanTitle;
    }
    if ($cleanContent !== '') {
        $words = preg_split('/\s+/u', mb_strtolower($cleanContent));
        $freq = [];
        foreach ($words as $word) {
            $word = safe_trim(preg_replace('/[^\p{L}\p{N}]+/u', '', $word));
            if ($word === '' || mb_strlen($word) < 3) {
                continue;
            }
            if (!isset($freq[$word])) {
                $freq[$word] = 0;
            }
            $freq[$word]++;
        }
        arsort($freq);
        $keywords = array_merge($keywords, array_slice(array_keys($freq), 0, 10));
    }
    $keywords = array_unique(array_map('trim', array_filter($keywords)));

    return [
        'meta_title' => $metaTitle,
        'meta_description' => $description,
        'meta_keywords' => implode(', ', $keywords)
    ];
}

function getClientIpAddress(): string {
    $candidates = [
        'HTTP_CF_CONNECTING_IP',
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_REAL_IP',
        'HTTP_CLIENT_IP',
        'REMOTE_ADDR',
    ];

    foreach ($candidates as $key) {
        if (empty($_SERVER[$key])) {
            continue;
        }

        $value = $_SERVER[$key];
        if ($key === 'HTTP_X_FORWARDED_FOR') {
            $parts = explode(',', (string)$value);
            $value = safe_trim($parts[0]);
        }

        $value = safe_trim($value);
        if ($value === '') {
            continue;
        }

        if (filter_var($value, FILTER_VALIDATE_IP)) {
            return $value;
        }
    }

    return '0.0.0.0';
}

function getUserAgentString(): string {
    return isset($_SERVER['HTTP_USER_AGENT']) ? trim((string)$_SERVER['HTTP_USER_AGENT']) : '';
}

function getSessionIdentifier(): ?string {
    $sessionId = session_id();
    return $sessionId !== '' ? $sessionId : null;
}

function ensureDeviceIdentifier(): string {
    static $deviceId = null;
    if ($deviceId !== null) {
        return $deviceId;
    }

    $cookieName = 'epin_device_id';
    $existing = $_COOKIE[$cookieName] ?? '';
    if (is_string($existing) && preg_match('/^[a-f0-9]{32}$/', $existing)) {
        $deviceId = $existing;
    } else {
        try {
            $deviceId = bin2hex(random_bytes(16));
        } catch (Exception $e) {
            $deviceId = md5((string)microtime(true));
        }
    }

    if (!isset($_COOKIE[$cookieName]) || $_COOKIE[$cookieName] !== $deviceId) {
        $cookieOptions = [
            'expires' => time() + (86400 * 365 * 2),
            'path' => '/',
            'secure' => IS_HTTPS_ENVIRONMENT,
            'httponly' => true,
        ];

        if (PHP_VERSION_ID >= 70300) {
            $cookieOptions['samesite'] = IS_HTTPS_ENVIRONMENT ? 'None' : 'Lax';
            setcookie($cookieName, $deviceId, $cookieOptions);
        } else {
            setcookie(
                $cookieName,
                $deviceId,
                $cookieOptions['expires'],
                '/; SameSite=' . (IS_HTTPS_ENVIRONMENT ? 'None' : 'Lax'),
                '',
                $cookieOptions['secure'],
                $cookieOptions['httponly']
            );
        }

        $_COOKIE[$cookieName] = $deviceId;
    }

    return $deviceId;
}

function sanitizeDataForLog($value, array $sensitiveKeys = []): array|string {
    if (is_array($value)) {
        $result = [];
        foreach ($value as $key => $item) {
            $normalizedKey = strtolower((string)$key);
            if (in_array($normalizedKey, $sensitiveKeys, true)) {
                $result[$key] = '[gizli]';
                continue;
            }
            $result[$key] = sanitizeDataForLog($item, $sensitiveKeys);
        }
        return $result;
    }

    if (is_scalar($value)) {
        $stringValue = (string)$value;
        if (mb_strlen($stringValue) > 200) {
            $stringValue = mb_substr($stringValue, 0, 200) . '…';
        }
        return $stringValue;
    }

    if (is_object($value)) {
        return '[object:' . get_class($value) . ']';
    }

    return '[' . gettype($value) . ']';
}

function resolveIpLocation(PDO $pdo, string $ip): array {
    static $cache = [];
    if (isset($cache[$ip])) {
        return $cache[$ip];
    }

    if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
        return $cache[$ip] = [];
    }

    try {
        $stmt = $pdo->prepare('SELECT * FROM ip_location_cache WHERE ip_address = ? LIMIT 1');
        $stmt->execute([$ip]);
        $row = $stmt->fetch();
    } catch (Throwable $th) {
        $row = null;
    }

    $freshThreshold = time() - (86400 * 7);
    if ($row && isset($row['last_lookup_at']) && strtotime((string)$row['last_lookup_at']) > $freshThreshold) {
        return $cache[$ip] = [
            'country' => $row['country'] ?? null,
            'city' => $row['city'] ?? null,
            'region' => $row['region'] ?? null,
            'latitude' => $row['latitude'] ?? null,
            'longitude' => $row['longitude'] ?? null,
            'isp' => $row['isp'] ?? null,
            'timezone' => $row['timezone'] ?? null,
        ];
    }

    $location = [];
    $fields = 'status,message,country,regionName,city,lat,lon,isp,timezone';
    $endpoint = 'http://ip-api.com/json/' . rawurlencode($ip) . '?fields=' . $fields;
    $context = stream_context_create(['http' => ['timeout' => 3]]);
    $response = @file_get_contents($endpoint, false, $context);

    if ($response !== false) {
        $data = json_decode($response, true);
        if (is_array($data) && ($data['status'] ?? '') === 'success') {
            $location = [
                'country' => $data['country'] ?? null,
                'city' => $data['city'] ?? null,
                'region' => $data['regionName'] ?? null,
                'latitude' => isset($data['lat']) ? (float)$data['lat'] : null,
                'longitude' => isset($data['lon']) ? (float)$data['lon'] : null,
                'isp' => $data['isp'] ?? null,
                'timezone' => $data['timezone'] ?? null,
            ];

            try {
                $stmt = $pdo->prepare('INSERT INTO ip_location_cache (ip_address, country, city, region, latitude, longitude, isp, timezone, lookup_source, raw_response) VALUES (:ip, :country, :city, :region, :lat, :lon, :isp, :timezone, :source, :raw)
                    ON DUPLICATE KEY UPDATE country = VALUES(country), city = VALUES(city), region = VALUES(region), latitude = VALUES(latitude), longitude = VALUES(longitude), isp = VALUES(isp), timezone = VALUES(timezone), lookup_source = VALUES(lookup_source), raw_response = VALUES(raw_response)');
                $stmt->execute([
                    ':ip' => $ip,
                    ':country' => $location['country'],
                    ':city' => $location['city'],
                    ':region' => $location['region'],
                    ':lat' => $location['latitude'],
                    ':lon' => $location['longitude'],
                    ':isp' => $location['isp'],
                    ':timezone' => $location['timezone'],
                    ':source' => 'ip-api',
                    ':raw' => $response,
                ]);
            } catch (Throwable $th) {
                // Yoksay
            }
        }
    }

    if (empty($location) && $row) {
        $location = [
            'country' => $row['country'] ?? null,
            'city' => $row['city'] ?? null,
            'region' => $row['region'] ?? null,
            'latitude' => $row['latitude'] ?? null,
            'longitude' => $row['longitude'] ?? null,
            'isp' => $row['isp'] ?? null,
            'timezone' => $row['timezone'] ?? null,
        ];
    }

    return $cache[$ip] = $location;
}

function findActiveIpBan(PDO $pdo, string $ip): ?array {
    if ($ip === '' || $ip === '0.0.0.0') {
        return null;
    }

    try {
        $stmt = $pdo->prepare('SELECT * FROM banned_ips WHERE ip_address = ? LIMIT 1');
        $stmt->execute([$ip]);
        $row = $stmt->fetch();
    } catch (Throwable $th) {
        return null;
    }

    if (!$row) {
        return null;
    }

    if (!empty($row['expires_at']) && strtotime((string)$row['expires_at']) < time()) {
        return null;
    }

    return $row;
}

function findActiveDeviceBan(PDO $pdo, string $deviceId): ?array {
    if ($deviceId === '') {
        return null;
    }

    try {
        $stmt = $pdo->prepare('SELECT * FROM banned_devices WHERE device_id = ? LIMIT 1');
        $stmt->execute([$deviceId]);
        $row = $stmt->fetch();
    } catch (Throwable $th) {
        return null;
    }

    if (!$row) {
        return null;
    }

    if (!empty($row['expires_at']) && strtotime((string)$row['expires_at']) < time()) {
        return null;
    }

    return $row;
}

function enforceAccessRestrictions(PDO $pdo): void {
    $ip = getClientIpAddress();
    $deviceId = ensureDeviceIdentifier();
    $ipBan = findActiveIpBan($pdo, $ip);
    $deviceBan = findActiveDeviceBan($pdo, $deviceId);

    if (!$ipBan && !$deviceBan) {
        return;
    }

    $banContext = [
        'ip_ban' => $ipBan ? ['reason' => $ipBan['reason'] ?? null, 'expires_at' => $ipBan['expires_at'] ?? null] : null,
        'device_ban' => $deviceBan ? ['reason' => $deviceBan['reason'] ?? null, 'expires_at' => $deviceBan['expires_at'] ?? null] : null,
    ];

    logUserActivity($pdo, 'access_denied', $banContext);

    if (session_status() === PHP_SESSION_ACTIVE) {
        session_unset();
    }

    http_response_code(403);
    header('Content-Type: text/html; charset=UTF-8');
    echo '<!DOCTYPE html><html lang="tr"><head><meta charset="UTF-8"><title>Erişim Engellendi</title><style>body{font-family:Arial,Helvetica,sans-serif;background:#0f172a;color:#e2e8f0;display:flex;align-items:center;justify-content:center;height:100vh;margin:0;} .card{background:#111827;padding:2.5rem;border-radius:1rem;box-shadow:0 20px 45px rgba(15,23,42,0.45);max-width:520px;text-align:center;} h1{margin-bottom:1rem;font-size:1.8rem;} p{margin:0.6rem 0;color:#94a3b8;} .reason{margin-top:1.2rem;padding:0.75rem;border-radius:0.75rem;background:rgba(239,68,68,0.15);color:#fca5a5;} .meta{margin-top:1.2rem;font-size:0.9rem;color:#64748b;} </style></head><body><div class="card"><h1>Erişim Engellendi</h1><p>Güvenlik politikalarımız gereği bu cihaz veya IP adresinden erişim engellenmiştir.</p>';

    if ($ipBan && !empty($ipBan['reason'])) {
        echo '<div class="reason">IP yasağı nedeni: ' . htmlspecialchars($ipBan['reason'], ENT_QUOTES, 'UTF-8') . '</div>';
    }
    if ($deviceBan && !empty($deviceBan['reason'])) {
        echo '<div class="reason">Cihaz yasağı nedeni: ' . htmlspecialchars($deviceBan['reason'], ENT_QUOTES, 'UTF-8') . '</div>';
    }

    echo '<div class="meta">IP: ' . htmlspecialchars($ip, ENT_QUOTES, 'UTF-8') . '<br>Cihaz ID: ' . htmlspecialchars($deviceId, ENT_QUOTES, 'UTF-8');
    if (($ipBan['expires_at'] ?? null) || ($deviceBan['expires_at'] ?? null)) {
        $expiry = $ipBan['expires_at'] ?? $deviceBan['expires_at'];
        if ($expiry) {
            echo '<br>Ban bitiş tarihi: ' . htmlspecialchars(date('d.m.Y H:i', strtotime((string)$expiry)), ENT_QUOTES, 'UTF-8');
        }
    }
    echo '</div><p class="meta">Yanlışlık olduğunu düşünüyorsanız lütfen destek ekibimizle iletişime geçin.</p></div></body></html>';
    exit;
}

function shouldSkipActivityLogging(): bool {
    if (PHP_SAPI === 'cli') {
        return true;
    }

    $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
    if (stripos($scriptName, 'install') !== false) {
        return true;
    }

    $uri = $_SERVER['REQUEST_URI'] ?? '';
    $skipPatterns = ['/admin/assets/', '/assets/', '/uploads/'];
    foreach ($skipPatterns as $pattern) {
        if (stripos($uri, $pattern) === 0) {
            return true;
        }
    }

    $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
    if (strpos($scriptName, '/admin/') !== false || strpos($uri, '/admin/') !== false) {
        return true;
    }

    $referer = $_SERVER['HTTP_REFERER'] ?? '';
    if ($referer !== '' && strpos($referer, '/admin/') !== false) {
        return true;
    }

    return false;
}

function logUserActivity(PDO $pdo, string $eventType = 'page_view', array $context = []): void {
    try {
        $ip = getClientIpAddress();
        $userAgent = getUserAgentString();
        $deviceId = ensureDeviceIdentifier();
        $sessionId = getSessionIdentifier();
        $userId = isLoggedIn() ? (int)$_SESSION['user_id'] : null;
        $requestMethod = $_SERVER['REQUEST_METHOD'] ?? null;
        $requestUri = $_SERVER['REQUEST_URI'] ?? '';
        $referer = $_SERVER['HTTP_REFERER'] ?? null;

        $location = resolveIpLocation($pdo, $ip);

        $sensitiveKeys = ['password', 'new_password', 'current_password', 'token', 'csrf_token', 'security_answer'];
        $contextData = array_merge([
            'get' => sanitizeDataForLog($_GET, $sensitiveKeys),
            'post' => sanitizeDataForLog($_POST, $sensitiveKeys),
            'query_string' => $_SERVER['QUERY_STRING'] ?? '',
            'referer' => $referer,
            'user_agent' => $userAgent,
        ], $context);

        // NOT: Loglar artık sadece dosyaya kaydediliyor, veritabanına kaydedilmiyor
        // Veritabanı şişmesini önlemek için bu değişiklik yapıldı
        // Eski logları temizlemek için admin/activity-logs.php sayfasındaki "Log Temizle" butonunu kullanın

        $filePayload = [
            'timestamp' => date('c'),
            'event_type' => $eventType,
            'user_id' => $userId,
            'session_id' => $sessionId,
            'device_id' => $deviceId,
            'ip' => $ip,
            'request_method' => $requestMethod,
            'request_uri' => $requestUri,
            'referer' => $referer,
            'country' => $location['country'] ?? null,
            'city' => $location['city'] ?? null,
            'region' => $location['region'] ?? null,
            'latitude' => $location['latitude'] ?? null,
            'longitude' => $location['longitude'] ?? null,
            'context' => $contextData,
        ];

        writeActivityLogToFile($filePayload);
    } catch (Throwable $th) {
        // Log sisteminin hata vermemesi için sessiz geçiyoruz.
    }
}

function initializeRequestTracking(PDO $pdo): void {
    static $initialized = false;
    if ($initialized) {
        return;
    }
    $initialized = true;

    ensureDeviceIdentifier();
    enforceAccessRestrictions($pdo);

    if (shouldSkipActivityLogging()) {
        return;
    }

    logUserActivity($pdo, 'page_view');
}

function formatPrice($price): string {
    return number_format($price, 2, ',', '.') . ' ' . CURRENCY;
}

function timeAgo($timestamp): string {
    $time = strtotime($timestamp);
    $diff = time() - $time;
    
    if ($diff < 60) {
        return $diff . ' saniye önce';
    } elseif ($diff < 3600) {
        return floor($diff / 60) . ' dakika önce';
    } elseif ($diff < 86400) {
        return floor($diff / 3600) . ' saat önce';
    } elseif ($diff < 2592000) {
        return floor($diff / 86400) . ' gün önce';
    } else {
        return date('d.m.Y H:i', $time);
    }
}

function getCategories(PDO $pdo, bool $onlyActive = true): array {
    $sql = "SELECT * FROM categories";
    if ($onlyActive) {
        $sql .= " WHERE is_active = 1";
    }
    $sql .= " ORDER BY (parent_id IS NOT NULL), display_order ASC, name ASC";
    $stmt = $pdo->query($sql);
    return $stmt->fetchAll();
}

function buildCategoryTree(array $categories): array {
    $indexed = [];
    foreach ($categories as $category) {
        $category['children'] = [];
        $indexed[$category['id']] = $category;
    }

    $tree = [];
    foreach ($indexed as $id => &$category) {
        $parentId = $category['parent_id'] ?? null;
        if ($parentId && isset($indexed[$parentId])) {
            $indexed[$parentId]['children'][] =& $category;
        } else {
            $tree[] =& $category;
        }
    }
    unset($category);

    return $tree;
}

function buildCategoryIndex(array $categories): array {
    $byId = [];
    $children = [];
    foreach ($categories as $category) {
        $byId[$category['id']] = $category;
        $parentId = $category['parent_id'] ?? null;
        if (!isset($children[$parentId])) {
            $children[$parentId] = [];
        }
        $children[$parentId][] = $category['id'];
    }

    return ['byId' => $byId, 'children' => $children];
}

function getCategoryDescendantIds(array $index, int $categoryId): array {
    $ids = [$categoryId];
    $children = $index['children'][$categoryId] ?? [];
    foreach ($children as $childId) {
        $ids = array_merge($ids, getCategoryDescendantIds($index, $childId));
    }
    return $ids;
}

function getMarketplaceGames(PDO $pdo, bool $onlyActive = true): array {
    $sql = "SELECT * FROM game_marketplaces";
    if ($onlyActive) {
        $sql .= " WHERE is_active = 1";
    }
    $sql .= " ORDER BY display_order ASC, name ASC";
    $stmt = $pdo->query($sql);
    return $stmt->fetchAll() ?: [];
}

function getMarketplaceStructure(PDO $pdo, bool $onlyActive = true): array {
    $games = getMarketplaceGames($pdo, $onlyActive);
    if (empty($games)) {
        return [];
    }

    $gameMap = [];
    $gameIds = [];
    foreach ($games as $game) {
        $game['categories'] = [];
        $gameMap[$game['id']] = $game;
        $gameIds[] = (int)$game['id'];
    }

    $categories = [];
    $categoryMap = [];
    if (!empty($gameIds)) {
        $placeholders = implode(',', array_fill(0, count($gameIds), '?'));
        $sql = "SELECT * FROM game_marketplace_categories WHERE marketplace_id IN ($placeholders)";
        if ($onlyActive) {
            $sql .= " AND is_active = 1";
        }
        $sql .= " ORDER BY display_order ASC, name ASC";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($gameIds);
        $categories = $stmt->fetchAll() ?: [];
    }

    $categoryIds = [];
    foreach ($categories as $category) {
        $category['servers'] = [];
        $categoryMap[$category['id']] = $category;
        $categoryIds[] = (int)$category['id'];
        if (isset($gameMap[$category['marketplace_id']])) {
            $gameMap[$category['marketplace_id']]['categories'][] = &$categoryMap[$category['id']];
        }
    }
    unset($category);

    if (!empty($categoryIds)) {
        $placeholders = implode(',', array_fill(0, count($categoryIds), '?'));
        $sql = "SELECT * FROM game_marketplace_servers WHERE category_id IN ($placeholders)";
        if ($onlyActive) {
            $sql .= " AND is_active = 1";
        }
        $sql .= " ORDER BY display_order ASC, name ASC";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($categoryIds);
        $servers = $stmt->fetchAll() ?: [];

        foreach ($servers as $server) {
            if (isset($categoryMap[$server['category_id']])) {
                $categoryMap[$server['category_id']]['servers'][] = $server;
            }
        }
        unset($server);
    }

    return array_values($gameMap);
}

function getNavigationMenu(PDO $pdo, string $location = 'header'): array
{
    static $cache = [];
    $allowedLocations = ['header', 'footer', 'footer_bottom'];
    if (!in_array($location, $allowedLocations, true)) {
        $location = 'header';
    }

    if (isset($cache[$location])) {
        return $cache[$location];
    }

    try {
        $stmt = $pdo->prepare("SELECT title, url, menu_type, open_new_tab FROM navigation_links WHERE location = :location AND is_active = 1 ORDER BY sort_order ASC, id ASC");
        $stmt->execute([':location' => $location]);
        $rows = $stmt->fetchAll() ?: [];
    } catch (Throwable $th) {
        $rows = [];
    }

    if (empty($rows) && $location === 'header') {
        $rows = [
            ['title' => 'Ana Sayfa', 'url' => 'index.php', 'menu_type' => 'link', 'open_new_tab' => 0],
            ['title' => 'Oyunlar', 'url' => 'products.php', 'menu_type' => 'link', 'open_new_tab' => 0],
            ['title' => 'Marketplace', 'url' => 'marketplace.php', 'menu_type' => 'marketplace', 'open_new_tab' => 0],
            ['title' => 'Pazaryeri', 'url' => 'products.php?tab=market', 'menu_type' => 'link', 'open_new_tab' => 0],
            ['title' => 'Oyun Parası', 'url' => 'products.php?tab=coins', 'menu_type' => 'link', 'open_new_tab' => 0],
            ['title' => 'Bakiyem', 'url' => null, 'menu_type' => 'balance', 'open_new_tab' => 0],
            ['title' => 'Destek', 'url' => 'support.php', 'menu_type' => 'link', 'open_new_tab' => 0],
            ['title' => 'Blog', 'url' => 'blog.php', 'menu_type' => 'link', 'open_new_tab' => 0],
            ['title' => 'PUBG UC', 'url' => 'products.php?q=PUBG%20UC', 'menu_type' => 'link', 'open_new_tab' => 0],
        ];
    }

    $menu = array_map(static function ($row) {
        return [
            'title' => $row['title'] ?? '',
            'url' => isset($row['url']) && $row['url'] !== null ? seoFriendlyUrl($row['url']) : null,
            'menu_type' => $row['menu_type'] ?? 'link',
            'open_new_tab' => !empty($row['open_new_tab']),
        ];
    }, $rows);

    return $cache[$location] = $menu;
}

function getMarketplaceCategory(PDO $pdo, int $categoryId, bool $onlyActive = true): ?array {
    $sql = "SELECT * FROM game_marketplace_categories WHERE id = ?";
    if ($onlyActive) {
        $sql .= " AND is_active = 1";
    }
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$categoryId]);
    $category = $stmt->fetch();
    return $category ?: null;
}

function getMarketplaceCategoryBySlug(PDO $pdo, string $slug, ?int $marketplaceId = null, bool $onlyActive = true): ?array {
    $params = [$slug];
    $sql = "SELECT * FROM game_marketplace_categories WHERE slug = ?";
    if ($marketplaceId !== null) {
        $sql .= " AND marketplace_id = ?";
        $params[] = $marketplaceId;
    }
    if ($onlyActive) {
        $sql .= " AND is_active = 1";
    }
    $stmt = $pdo->prepare($sql . " LIMIT 1");
    $stmt->execute($params);
    $category = $stmt->fetch();
    return $category ?: null;
}

function getMarketplaceServer(PDO $pdo, int $serverId, bool $onlyActive = true): ?array {
    $sql = "SELECT s.*, s.delivery_type, c.marketplace_id, c.requires_character_name, c.auto_approve_orders FROM game_marketplace_servers s JOIN game_marketplace_categories c ON c.id = s.category_id WHERE s.id = ?";
    if ($onlyActive) {
        $sql .= " AND s.is_active = 1 AND c.is_active = 1";
    }
    $stmt = $pdo->prepare($sql . " LIMIT 1");
    $stmt->execute([$serverId]);
    $server = $stmt->fetch();
    return $server ?: null;
}

function getMarketplaceServerBySlug(PDO $pdo, string $slug, ?int $categoryId = null, bool $onlyActive = true): ?array {
    $params = [$slug];
    $sql = "SELECT s.*, s.delivery_type, c.marketplace_id, c.requires_character_name, c.auto_approve_orders FROM game_marketplace_servers s JOIN game_marketplace_categories c ON c.id = s.category_id WHERE s.slug = ?";
    if ($categoryId !== null) {
        $sql .= " AND s.category_id = ?";
        $params[] = $categoryId;
    }
    if ($onlyActive) {
        $sql .= " AND s.is_active = 1 AND c.is_active = 1";
    }
    $stmt = $pdo->prepare($sql . " LIMIT 1");
    $stmt->execute($params);
    $server = $stmt->fetch();
    return $server ?: null;
}

function setLastEmailError(?string $message): void
{
    $GLOBALS['__last_email_error'] = $message ?? '';
}

function getLastEmailError(): string
{
    return $GLOBALS['__last_email_error'] ?? '';
}

function sendEmail($to, string $subject, string $body, ?string $altBody = null): bool {
    setLastEmailError('');

    $smtpHost = trim((string)getSetting('smtp_host', SMTP_HOST));
    $smtpPort = (int)getSetting('smtp_port', SMTP_PORT);
    $smtpUser = (string)getSetting('smtp_username', SMTP_USERNAME);
    $smtpPass = (string)getSetting('smtp_password', SMTP_PASSWORD);
    $smtpEnc = strtolower((string)getSetting('smtp_encryption', SMTP_ENCRYPTION));
    $fromEmail = trim((string)getSetting('smtp_from_email', SMTP_FROM_EMAIL));
    $fromName = trim((string)getSetting('smtp_from_name', SMTP_FROM_NAME));
    $testMode = getBoolSetting('smtp_test_mode', SMTP_TEST_MODE);
    $baseUrl = trim((string)getSetting('email_base_url', EMAIL_BASE_URL));

    if ($fromEmail === '') {
        $fromEmail = SMTP_FROM_EMAIL;
    }
    if ($fromName === '') {
        $fromName = SMTP_FROM_NAME;
    }

    if ($smtpHost === '' || $smtpPort <= 0) {
        setLastEmailError('SMTP sunucusu veya port bilgisi eksik.');
        return false;
    }

    $phpMailerAvailable = class_exists('PHPMailer\\PHPMailer\\PHPMailer');
    if ($phpMailerAvailable) {
        require_once __DIR__ . '/vendor/autoload.php';
        $phpMailerAvailable = class_exists('PHPMailer\\PHPMailer\\PHPMailer');
    }

    if ($phpMailerAvailable) {
        $mailer = new PHPMailer\PHPMailer\PHPMailer(true);

        try {
            $mailer->isSMTP();
            $mailer->Host = $smtpHost;
            $mailer->SMTPAuth = $smtpUser !== '' || $smtpPass !== '';
            if ($mailer->SMTPAuth) {
                $mailer->Username = $smtpUser;
                $mailer->Password = $smtpPass;
            }
            if ($smtpEnc === 'ssl') {
                $mailer->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS;
            } elseif ($smtpEnc === 'tls' || $smtpEnc === '') {
                $mailer->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
            } else {
                $mailer->SMTPSecure = false;
            }
            $mailer->SMTPAutoTLS = $mailer->SMTPSecure !== false;
            $mailer->Port = $smtpPort > 0 ? $smtpPort : ($mailer->SMTPSecure === PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_SMTPS ? 465 : 587);
            if ($testMode) {
                $mailer->SMTPOptions = [
                    'ssl' => [
                        'verify_peer' => false,
                        'verify_peer_name' => false,
                        'allow_self_signed' => true,
                    ],
                ];
            }

            $mailer->CharSet = 'UTF-8';
            $mailer->setFrom($fromEmail, $fromName);

            $recipients = is_array($to) ? $to : [$to];
            foreach ($recipients as $recipient) {
                if (is_array($recipient)) {
                    $mailer->addAddress($recipient['email'], $recipient['name'] ?? '');
                } else {
                    $mailer->addAddress($recipient);
                }
            }

            $mailer->isHTML(true);
            $mailer->Subject = $subject;
            $mailer->Body = $body;
            $mailer->AltBody = $altBody ?? strip_tags($body);

            $mailer->send();
            return true;
        } catch (Throwable $th) {
            $message = $th->getMessage();
            setLastEmailError($message);
            if (!function_exists('logError')) {
                error_log('SMTP error: ' . $message);
            } else {
                logError('SMTP error: ' . $message);
            }
            if ($smtpHost === '' || $smtpPort <= 0) {
                return false;
            }
            // continue with fallback below
        }
    }

    if ($smtpHost === '' || $smtpPort <= 0) {
        setLastEmailError('SMTP sunucusu veya port bilgisi eksik.');
        return false;
    }

    $recipients = is_array($to) ? $to : [$to];
    $recipientEmails = [];
    foreach ($recipients as $recipient) {
        $recipientEmails[] = is_array($recipient) ? $recipient['email'] : $recipient;
    }

    $useTls = $smtpEnc === 'tls';
    $useSsl = $smtpEnc === 'ssl';
    $remoteHost = $useSsl ? 'ssl://' . $smtpHost : $smtpHost;

    $contextOptions = [];
    if ($testMode) {
        $contextOptions['ssl'] = [
            'verify_peer' => false,
            'verify_peer_name' => false,
            'allow_self_signed' => true,
        ];
    }

    $socket = @stream_socket_client($remoteHost . ':' . $smtpPort, $errno, $errstr, 30, STREAM_CLIENT_CONNECT, stream_context_create($contextOptions));
    if (!$socket) {
        setLastEmailError('SMTP sunucusuna bağlanılamadı: ' . $errstr . ' (' . $errno . ')');
        return false;
    }

    $read = static function ($sock) {
        $data = '';
        while ($line = fgets($sock, 515)) {
            $data .= $line;
            if (isset($line[3]) && $line[3] === ' ') {
                break;
            }
        }
        return $data;
    };

    $write = static function ($sock, string $command) {
        return fwrite($sock, $command . "\r\n");
    };

    $expect = static function ($sock, array $codes) use ($read) {
        $response = $read($sock);
        foreach ($codes as $code) {
            if (str_starts_with($response, (string)$code)) {
                return $response;
            }
        }
        throw new RuntimeException(safe_trim($response));
    };

    try {
        $expect($socket, [220]);
        $write($socket, 'EHLO ' . ($_SERVER['SERVER_NAME'] ?? 'localhost'));
        $expect($socket, [250]);

        if ($useTls) {
            $write($socket, 'STARTTLS');
            $expect($socket, [220]);
            if (!stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
                throw new RuntimeException('STARTTLS kurulamadı.');
            }
            $write($socket, 'EHLO ' . ($_SERVER['SERVER_NAME'] ?? 'localhost'));
            $expect($socket, [250]);
        }

        if ($smtpUser !== '' || $smtpPass !== '') {
            $write($socket, 'AUTH LOGIN');
            $expect($socket, [334]);
            $write($socket, base64_encode($smtpUser));
            $expect($socket, [334]);
            $write($socket, base64_encode($smtpPass));
            $expect($socket, [235]);
        }

        $write($socket, 'MAIL FROM:<' . $fromEmail . '>');
        $expect($socket, [250]);

        foreach ($recipientEmails as $email) {
            $write($socket, 'RCPT TO:<' . $email . '>');
            $expect($socket, [250, 251]);
        }

        $write($socket, 'DATA');
        $expect($socket, [354]);

        $headers = [];
        $headers[] = 'From: ' . ($fromName !== '' ? '"' . addslashes($fromName) . '" ' : '') . '<' . $fromEmail . '>';
        if (!empty($recipientEmails)) {
            $headers[] = 'To: ' . implode(', ', $recipientEmails);
        }
        $headers[] = 'Subject: ' . $subject;
        $headers[] = 'MIME-Version: 1.0';
        $headers[] = 'Content-Type: text/html; charset=UTF-8';
        $headers[] = 'Content-Transfer-Encoding: 8bit';

        $message = implode("\r\n", $headers) . "\r\n\r\n" . $body . "\r\n";
        $write($socket, $message . '.');
        $expect($socket, [250]);

        $write($socket, 'QUIT');
        $expect($socket, [221]);
        fclose($socket);
        return true;
    } catch (Throwable $th) {
        setLastEmailError($th->getMessage());
        if (is_resource($socket)) {
            fclose($socket);
        }
        if (!function_exists('logError')) {
            error_log('SMTP fallback error: ' . $th->getMessage());
        } else {
            logError('SMTP fallback error: ' . $th->getMessage());
        }
        return false;
    }
}

function sendMarketplaceEmail($to, string $subject, string $body, ?string $altBody = null): bool {
    return sendEmail($to, $subject, $body, $altBody);
}

function recaptchaEnabled(string $context): bool
{
    $settings = loadSettings();
    $siteKey = $settings['recaptcha_site_key'] ?? RECAPTCHA_SITE_KEY;
    $secretKey = $settings['recaptcha_secret_key'] ?? RECAPTCHA_SECRET_KEY;
    if ($siteKey === '' || $secretKey === '') {
        return false;
    }

    switch ($context) {
        case 'login':
            return !empty($settings['recaptcha_enable_login']);
        case 'register':
            return !empty($settings['recaptcha_enable_register']);
        case 'forgot':
            return !empty($settings['recaptcha_enable_forgot']);
        case 'admin_login':
            return !empty($settings['recaptcha_enable_admin_login']);
        default:
            return false;
    }
}

function verifyRecaptcha(string $token): bool
{
    if ($token === '') {
        return false;
    }
    $settings = loadSettings();
    $secretKey = $settings['recaptcha_secret_key'] ?? RECAPTCHA_SECRET_KEY;
    if ($secretKey === '') {
        return false;
    }
    $response = @file_get_contents('https://www.google.com/recaptcha/api/siteverify', false, stream_context_create([
        'http' => [
            'method' => 'POST',
            'header' => 'Content-type: application/x-www-form-urlencoded',
            'content' => http_build_query([
                'secret' => $secretKey,
                'response' => $token,
                'remoteip' => getClientIpAddress(),
            ]),
            'timeout' => 5
        ]
    ]));
    if ($response === false) {
        return false;
    }
    $data = json_decode($response, true);
    return is_array($data) && !empty($data['success']);
}

function refreshCartCount(PDO $pdo, int $userId): void
{
    try {
        $stmt = $pdo->prepare('SELECT COALESCE(SUM(quantity),0) FROM cart_items WHERE user_id = ?');
        $stmt->execute([$userId]);
        $count = (float)($stmt->fetchColumn() ?? 0);
        $_SESSION['cart_count'] = (int)round($count);
    } catch (Throwable $th) {
        $_SESSION['cart_count'] = 0;
    }
}

function addProductToCart(PDO $pdo, int $userId, int $productId, int $quantity): array
{
    $quantity = max(1, $quantity);
    $stmt = $pdo->prepare('SELECT id, name, price, stock, is_active FROM products WHERE id = ?');
    $stmt->execute([$productId]);
    $product = $stmt->fetch();
    if (!$product || (int)$product['is_active'] !== 1) {
        return ['success' => false, 'message' => 'Ürün bulunamadı.'];
    }

    $stock = (int)$product['stock'];
    if ($stock <= 0) {
        return ['success' => false, 'message' => 'Bu ürün stokta yok.'];
    }

    $existingStmt = $pdo->prepare('SELECT id, quantity FROM cart_items WHERE user_id = ? AND item_type = "product" AND reference_id = ? AND character_name = "" LIMIT 1');
    $existingStmt->execute([$userId, $productId]);
    $existing = $existingStmt->fetch();
    $currentQty = $existing ? (int)$existing['quantity'] : 0;

    // Stok kontrolü - fazla miktar girildiğinde hata ver
    if ($currentQty + $quantity > $stock) {
        return ['success' => false, 'message' => 'Bu ürünün stoğu "' . $stock . '" adet. Lütfen stok miktarını aşmayın.'];
    }

    if ($existing) {
        $update = $pdo->prepare('UPDATE cart_items SET quantity = quantity + ?, unit_price = ?, updated_at = NOW() WHERE id = ?');
        $update->execute([$quantity, $product['price'], $existing['id']]);
    } else {
        $insert = $pdo->prepare('INSERT INTO cart_items (user_id, item_type, reference_id, quantity, unit_price, character_name) VALUES (?, "product", ?, ?, ?, "")');
        $insert->execute([$userId, $productId, $quantity, $product['price']]);
    }

    refreshCartCount($pdo, $userId);
    return ['success' => true, 'message' => 'Ürün sepete eklendi.'];
}

function addMarketplaceItemToCart(PDO $pdo, int $userId, int $serverId, float $quantity, ?string $characterName = null): array
{
    $quantity = max(0.01, $quantity);
    $server = getMarketplaceServer($pdo, $serverId, true);
    if (!$server) {
        return ['success' => false, 'message' => 'Sunucu bulunamadı veya pasif durumda.'];
    }

    $price = (float)$server['sell_price'];
    if ($price <= 0) {
        return ['success' => false, 'message' => 'Bu ilan için satış fiyatı tanımlanmamış.'];
    }

    $characterName = trim((string)$characterName);
    if ((int)$server['requires_character_name'] === 1) {
        if ($characterName === '') {
            return ['success' => false, 'message' => 'Karakter adını girmeniz gerekiyor.'];
        }
    }
    if ($characterName !== '') {
        $characterName = mb_substr($characterName, 0, 150);
    }

    $minQuantity = (float)$server['min_buy_quantity'];
    $maxQuantity = $server['max_buy_quantity'] !== null ? (float)$server['max_buy_quantity'] : null;
    if ($quantity < $minQuantity) {
        return ['success' => false, 'message' => 'Minimum miktar: ' . number_format($minQuantity, 2, ',', '.')];
    }
    if ($maxQuantity !== null && $quantity > $maxQuantity) {
        $quantity = $maxQuantity;
    }

    $existingStmt = $pdo->prepare('SELECT id, quantity FROM cart_items WHERE user_id = ? AND item_type = "marketplace" AND reference_id = ? AND character_name = ? LIMIT 1');
    $existingStmt->execute([$userId, $serverId, $characterName]);
    $existing = $existingStmt->fetch();
    $currentQty = $existing ? (float)$existing['quantity'] : 0.0;

    $available = (float)$server['sell_stock'];
    if ($available > 0 && $currentQty + $quantity > $available) {
        $quantity = max(0.0, $available - $currentQty);
        if ($quantity <= 0) {
            return ['success' => false, 'message' => 'Sepette bu ilan için izin verilen maksimum stok mevcut.'];
        }
    }

    if ($existing) {
        $update = $pdo->prepare('UPDATE cart_items SET quantity = quantity + ?, unit_price = ?, updated_at = NOW() WHERE id = ?');
        $update->execute([$quantity, $price, $existing['id']]);
    } else {
        $insert = $pdo->prepare('INSERT INTO cart_items (user_id, item_type, reference_id, quantity, unit_price, character_name) VALUES (?, "marketplace", ?, ?, ?, ?)');
        $insert->execute([$userId, $serverId, $quantity, $price, $characterName]);
    }

    refreshCartCount($pdo, $userId);
    return ['success' => true, 'message' => 'İlan sepete eklendi.'];
}

function createPasswordResetToken(PDO $pdo, int $userId, int $ttlMinutes = 60): ?string
{
    try {
        $pdo->prepare('DELETE FROM password_reset_tokens WHERE user_id = ?')->execute([$userId]);
        $token = bin2hex(random_bytes(32));
        $expiresAt = date('Y-m-d H:i:s', time() + max(5, $ttlMinutes) * 60);
        $stmt = $pdo->prepare('INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (:user_id, :token, :expires_at)');
        $stmt->execute([
            ':user_id' => $userId,
            ':token' => $token,
            ':expires_at' => $expiresAt,
        ]);
        return $token;
    } catch (Throwable $th) {
        if (function_exists('logError')) {
            logError('Password reset token error: ' . $th->getMessage());
        }
        return null;
    }
}

function findPasswordResetToken(PDO $pdo, string $token): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM password_reset_tokens WHERE token = ? AND used_at IS NULL LIMIT 1');
    $stmt->execute([$token]);
    $row = $stmt->fetch();
    if (!$row) {
        return null;
    }
    if (strtotime((string)$row['expires_at']) < time()) {
        return null;
    }
    return $row;
}

function markPasswordResetUsed(PDO $pdo, int $tokenId): void
{
    try {
        $pdo->prepare('UPDATE password_reset_tokens SET used_at = NOW() WHERE id = ?')->execute([$tokenId]);
    } catch (Throwable $th) {
        if (function_exists('logError')) {
            logError('Password reset mark error: ' . $th->getMessage());
        }
    }
}

function emailVerificationRequired(): bool
{
    return getBoolSetting('auth_require_email_verification', false);
}

function prepareEmailVerification(PDO $pdo, int $userId, bool $forceNew = false): ?array
{
    try {
        $stmt = $pdo->prepare('SELECT id, username, email, email_verified, email_verification_token, email_verification_sent_at FROM users WHERE id = ? LIMIT 1');
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        if (!$user) {
            return null;
        }

        if (!$forceNew && !empty($user['email_verification_token']) && !empty($user['email_verification_sent_at'])) {
            $sentAt = strtotime((string)$user['email_verification_sent_at']);
            if ($sentAt && (time() - $sentAt) < 300) {
                return ['user' => $user, 'token' => $user['email_verification_token']];
            }
        }

        try {
            $token = bin2hex(random_bytes(32));
        } catch (Throwable $th) {
            $token = md5((string)microtime(true) . (string)$userId);
        }

        $update = $pdo->prepare('UPDATE users SET email_verification_token = :token, email_verification_sent_at = NOW(), email_verified = CASE WHEN email_verified = 1 THEN 1 ELSE 0 END WHERE id = :id');
        $update->execute([
            ':token' => $token,
            ':id' => $userId,
        ]);

        $user['email_verification_token'] = $token;
        $user['email_verification_sent_at'] = date('Y-m-d H:i:s');

        return ['user' => $user, 'token' => $token];
    } catch (Throwable $th) {
        if (function_exists('logError')) {
            logError('Email verification prepare error: ' . $th->getMessage());
        }
        return null;
    }
}

function getEmailVerificationUrl(string $token): string
{
    $baseUrl = trim((string)getSetting('email_base_url', EMAIL_BASE_URL));
    if ($baseUrl === '') {
        $baseUrl = defined('SITE_URL') ? SITE_URL : '';
    }
    $baseUrl = rtrim($baseUrl, '/');
    if ($baseUrl === '') {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $baseUrl = $scheme . $host;
    }
    return $baseUrl . '/verify-email.php?token=' . urlencode($token);
}

function sendEmailVerificationEmail(array $user, string $verificationUrl): bool
{
    $username = $user['username'] ?? '';
    $email = $user['email'] ?? '';
    if ($email === '') {
        return false;
    }

    $site = siteName();
    $subject = $site . ' - E-posta Doğrulama';
    $safeName = htmlspecialchars($username ?: $email, ENT_QUOTES, 'UTF-8');
    $safeUrl = htmlspecialchars($verificationUrl, ENT_QUOTES, 'UTF-8');

    $body = '<p>Merhaba ' . $safeName . ',</p>'
        . '<p>' . htmlspecialchars($site, ENT_QUOTES, 'UTF-8') . ' hesabınızı aktifleştirmek için e-posta adresinizi doğrulamanız gerekiyor.</p>'
        . '<p><a href="' . $safeUrl . '" style="display:inline-block;padding:10px 18px;background-color:#0d6efd;color:#ffffff;text-decoration:none;border-radius:6px;">E-postamı Doğrula</a></p>'
        . '<p>Buton çalışmazsa aşağıdaki bağlantıyı tarayıcınıza yapıştırabilirsiniz:</p>'
        . '<p><a href="' . $safeUrl . '">' . $safeUrl . '</a></p>'
        . '<p>Bu isteği siz yapmadıysanız bu e-postayı yok sayabilirsiniz.</p>'
        . '<p>Teşekkürler,<br>' . htmlspecialchars($site, ENT_QUOTES, 'UTF-8') . ' Ekibi</p>';

    $altBody = "Merhaba $safeName,\n\nHesabınızı aktifleştirmek için aşağıdaki bağlantıyı ziyaret edin:\n$verificationUrl\n\nEğer bu isteği siz yapmadıysanız bu e-postayı yok sayabilirsiniz.";

    return sendEmail($email, $subject, $body, $altBody);
}

function findUserByVerificationToken(PDO $pdo, string $token): ?array
{
    try {
        $stmt = $pdo->prepare('SELECT * FROM users WHERE email_verification_token = ? LIMIT 1');
        $stmt->execute([$token]);
        $user = $stmt->fetch();
        return $user ?: null;
    } catch (Throwable $th) {
        if (function_exists('logError')) {
            logError('Email verification lookup error: ' . $th->getMessage());
        }
        return null;
    }
}

function markEmailAsVerified(PDO $pdo, int $userId): void
{
    try {
        $stmt = $pdo->prepare('UPDATE users SET email_verified = 1, email_verified_at = NOW(), email_verification_token = NULL, email_verification_sent_at = NULL WHERE id = ?');
        $stmt->execute([$userId]);
    } catch (Throwable $th) {
        if (function_exists('logError')) {
            logError('Email verification mark error: ' . $th->getMessage());
        }
    }
}

function getSupportTicketStatusMap(): array
{
    return [
        'open' => 'Aktif',
        'answered' => 'Yanıtlandı',
        'customer_reply' => 'Müşteri Yanıtı',
        'resolved' => 'Çözüldü',
    ];
}

function getSupportTicketStatusLabel(string $status): string
{
    $map = getSupportTicketStatusMap();
    return $map[$status] ?? ucfirst(str_replace('_', ' ', $status));
}

function isValidSupportTicketStatus(string $status): bool
{
    return array_key_exists($status, getSupportTicketStatusMap());
}

function getUserInfo($pdo, $userId): ?array {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    return $stmt->fetch() ?: null;
}

function flashMessage($type, $message) {
    $_SESSION['flash_message'] = ['type' => $type, 'message' => $message];
}

function getFlashMessage(): ?array {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        unset($_SESSION['flash_message']);
        return $message;
    }
    return null;
}

// CSRF Token Fonksiyonları
function generateCSRFToken(): string {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function verifyCSRFToken($token): bool {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Pagination helper fonksiyonu
 * @param int $totalRecords Toplam kayıt sayısı
 * @param int $perPage Sayfa başına kayıt sayısı (varsayılan: 10)
 * @param int|null $currentPage Mevcut sayfa (null ise $_GET['page']'den alır)
 * @return array ['offset' => int, 'limit' => int, 'current_page' => int, 'total_pages' => int, 'start' => int, 'end' => int]
 */
function getPaginationData(int $totalRecords, int $perPage = 10, ?int $currentPage = null): array {
    $currentPage = $currentPage ?? (int)($_GET['page'] ?? 1);
    $currentPage = max(1, $currentPage);
    $totalPages = max(1, (int)ceil($totalRecords / $perPage));
    $currentPage = min($currentPage, $totalPages);
    $offset = ($currentPage - 1) * $perPage;
    $start = min($totalRecords, $offset + 1);
    $end = min($totalRecords, $offset + $perPage);
    
    return [
        'offset' => $offset,
        'limit' => $perPage,
        'current_page' => $currentPage,
        'total_pages' => $totalPages,
        'start' => $start,
        'end' => $end,
        'total' => $totalRecords,
    ];
}

/**
 * Pagination HTML'i oluşturur
 * @param array $paginationData getPaginationData() fonksiyonundan dönen dizi
 * @param string|null $baseUrl URL (null ise mevcut sayfa kullanılır)
 * @return string HTML
 */
function renderPagination(array $paginationData, ?string $baseUrl = null): string {
    if ($paginationData['total_pages'] <= 1) {
        return '';
    }
    
    $currentPage = $paginationData['current_page'];
    $totalPages = $paginationData['total_pages'];
    $start = $paginationData['start'];
    $end = $paginationData['end'];
    $total = $paginationData['total'];
    
    // Base URL oluştur
    if ($baseUrl === null) {
        $baseUrl = $_SERVER['PHP_SELF'];
        $queryParams = $_GET;
        unset($queryParams['page']);
        if (!empty($queryParams)) {
            $baseUrl .= '?' . http_build_query($queryParams) . '&';
        } else {
            $baseUrl .= '?';
        }
    }
    
    $html = '<div class="d-flex justify-content-between align-items-center mt-4">';
    $html .= '<div class="text-muted"><small>Gösterilen: ' . $start . '-' . $end . ' / ' . $total . ' kayıt</small></div>';
    $html .= '<nav><ul class="pagination pagination-sm mb-0">';
    
    // Previous button
    if ($currentPage > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=' . ($currentPage - 1) . '"><i class="fas fa-chevron-left"></i></a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link"><i class="fas fa-chevron-left"></i></span></li>';
    }
    
    // Page numbers
    $startPage = max(1, $currentPage - 2);
    $endPage = min($totalPages, $currentPage + 2);
    
    if ($startPage > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=1">1</a></li>';
        if ($startPage > 2) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
    }
    
    for ($i = $startPage; $i <= $endPage; $i++) {
        if ($i === $currentPage) {
            $html .= '<li class="page-item active"><span class="page-link">' . $i . '</span></li>';
        } else {
            $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=' . $i . '">' . $i . '</a></li>';
        }
    }
    
    if ($endPage < $totalPages) {
        if ($endPage < $totalPages - 1) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=' . $totalPages . '">' . $totalPages . '</a></li>';
    }
    
    // Next button
    if ($currentPage < $totalPages) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=' . ($currentPage + 1) . '"><i class="fas fa-chevron-right"></i></a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link"><i class="fas fa-chevron-right"></i></span></li>';
    }
    
    $html .= '</ul></nav></div>';
    
    return $html;
}

initializeRequestTracking($pdo);

/**
 * Get active advertisements for a specific position
 * @param string $position Advertisement position (slider_top, slider_bottom, etc.)
 * @return array Array of active advertisements
 */
function getAdvertisements(string $position): array
{
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM advertisements 
            WHERE position = ? 
            AND is_active = 1
            AND (start_date IS NULL OR start_date <= NOW())
            AND (end_date IS NULL OR end_date >= NOW())
            ORDER BY display_order ASC, id ASC
        ");
        $stmt->execute([$position]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    } catch (Throwable $e) {
        return [];
    }
}

/**
 * Render advertisement HTML
 * @param array $ad Advertisement data
 * @param bool $trackViews Whether to track views
 * @return string HTML output
 */
function renderAdvertisement(array $ad, bool $trackViews = true): string
{
    if ($trackViews && !empty($ad["id"])) {
        global $pdo;
        try {
            $stmt = $pdo->prepare("UPDATE advertisements SET views_count = views_count + 1 WHERE id = ?");
            $stmt->execute([$ad["id"]]);
        } catch (Throwable $e) {
            // Silent fail
        }
    }
    
    $output = "";
    
    switch ($ad["type"]) {
        case "image":
            $imageUrl = htmlspecialchars($ad["content"], ENT_QUOTES, "UTF-8");
            $linkUrl = !empty($ad["link"]) ? htmlspecialchars($ad["link"], ENT_QUOTES, "UTF-8") : "#";
            $target = $ad["target"] === "_blank" ? " target=\"_blank\" rel=\"noopener\"" : "";
            
            if (!empty($ad["link"])) {
                $output = sprintf(
                    "<a href=\"%s\"%s class=\"advertisement-link\" data-ad-id=\"%d\"><img src=\"%s\" alt=\"%s\" class=\"advertisement-image img-fluid\" loading=\"lazy\"></a>",
                    $linkUrl,
                    $target,
                    $ad["id"],
                    $imageUrl,
                    htmlspecialchars($ad["title"], ENT_QUOTES, "UTF-8")
                );
            } else {
                $output = sprintf(
                    "<img src=\"%s\" alt=\"%s\" class=\"advertisement-image img-fluid\" loading=\"lazy\">",
                    $imageUrl,
                    htmlspecialchars($ad["title"], ENT_QUOTES, "UTF-8")
                );
            }
            break;
            
        case "html":
            $output = $ad["content"];
            break;
            
        case "script":
            $output = $ad["content"];
            break;
    }
    
    return $output;
}
