<?php
require_once 'config.php';
requireLogin();

$user = getUserInfo($pdo, $_SESSION['user_id']);

// Deposit summary
$depositSummaryStmt = $pdo->prepare('
    SELECT
        COUNT(*) AS total_count,
        SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'completed\' THEN 1 ELSE 0 END) AS completed_count,
        SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'pending\' THEN 1 ELSE 0 END) AS pending_count,
        SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'cancelled\' THEN 1 ELSE 0 END) AS cancelled_count,
        COALESCE(SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'completed\' THEN t.amount ELSE 0 END), 0) AS completed_total,
        COALESCE(SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'completed\'
            THEN CASE
                WHEN o.payable_amount IS NOT NULL THEN o.payable_amount
                ELSE t.amount + COALESCE(o.commission_amount, 0)
            END
            ELSE 0
        END), 0) AS completed_payable,
        COALESCE(SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'completed\'
            THEN CASE
                WHEN o.commission_amount IS NOT NULL THEN o.commission_amount
                WHEN o.payable_amount IS NOT NULL THEN o.payable_amount - t.amount
                ELSE 0
            END
            ELSE 0
        END), 0) AS completed_commission,
        COALESCE(SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'pending\'
            THEN CASE
                WHEN o.payable_amount IS NOT NULL THEN o.payable_amount
                ELSE t.amount + COALESCE(o.commission_amount, 0)
            END
            ELSE 0
        END), 0) AS pending_total
    FROM transactions t
    LEFT JOIN orders o ON o.id = t.order_id
    WHERE t.user_id = :user_id AND t.type = \'deposit\'
');
$depositSummaryStmt->execute([':user_id' => $_SESSION['user_id']]);
$depositSummaryRow = $depositSummaryStmt->fetch(PDO::FETCH_ASSOC) ?: [];
$depositSummary = [
    'total_count' => (int)($depositSummaryRow['total_count'] ?? 0),
    'completed_count' => (int)($depositSummaryRow['completed_count'] ?? 0),
    'pending_count' => (int)($depositSummaryRow['pending_count'] ?? 0),
    'cancelled_count' => (int)($depositSummaryRow['cancelled_count'] ?? 0),
    'completed_total' => (float)($depositSummaryRow['completed_total'] ?? 0),
    'completed_payable' => (float)($depositSummaryRow['completed_payable'] ?? 0),
    'completed_commission' => (float)($depositSummaryRow['completed_commission'] ?? 0),
    'pending_total' => (float)($depositSummaryRow['pending_total'] ?? 0)
];

// Pagination
$depositPerPage = 10;
$depositPage = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$depositCountStmt = $pdo->prepare('SELECT COUNT(*) FROM transactions WHERE user_id = ? AND type = \'deposit\'');
$depositCountStmt->execute([$_SESSION['user_id']]);
$depositTotalOrders = (int)$depositCountStmt->fetchColumn();
$depositTotalPages = max(1, (int)ceil($depositTotalOrders / $depositPerPage));
if ($depositPage > $depositTotalPages) {
    $depositPage = $depositTotalPages;
}
$depositOffset = ($depositPage - 1) * $depositPerPage;

// Get deposits
$depositOrdersStmt = $pdo->prepare('
    SELECT
        t.*,
        o.id AS order_id,
        o.order_number,
        o.payment_method,
        o.status AS order_status,
        o.payable_amount,
        o.commission_amount,
        o.total_amount AS order_total,
        o.created_at AS order_created_at
    FROM transactions t
    LEFT JOIN orders o ON o.id = t.order_id
    WHERE t.user_id = :user_id AND t.type = \'deposit\'
    ORDER BY t.created_at DESC
    LIMIT :offset, :limit
');
$depositOrdersStmt->bindValue(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);
$depositOrdersStmt->bindValue(':offset', $depositOffset, PDO::PARAM_INT);
$depositOrdersStmt->bindValue(':limit', $depositPerPage, PDO::PARAM_INT);
$depositOrdersStmt->execute();
$depositOrders = $depositOrdersStmt->fetchAll();

$pageTitle = 'Bakiye Yüklemelerim';
include 'includes/header.php';
?>

<section class="page-section">
    <div class="container-wide">
        <div class="page-header">
            <div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="<?= routeUrl('profile') ?>"><i class="fa-solid fa-user me-1"></i>Profilim</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Bakiye Yüklemelerim</li>
                    </ol>
                </nav>
                <h1 class="page-title"><i class="fa-solid fa-wallet me-2"></i>Bakiye Yüklemelerim</h1>
                <p class="page-description">Tüm bakiye yükleme işlemlerinizi ve ödeme detaylarınızı buradan görüntüleyebilirsiniz.</p>
            </div>
            <div>
                <a href="<?= routeUrl('balance') ?>" class="btn btn-primary">
                    <i class="fa-solid fa-plus me-2"></i>Bakiye Yükle
                </a>
            </div>
        </div>

        <div class="profile-metric-strip">
            <div class="profile-metric-card">
                <span>Toplam İşlem</span>
                <strong><?= number_format($depositSummary['total_count'], 0, ',', '.') ?></strong>
                <small>Tüm bakiye yüklemeleriniz</small>
            </div>
            <div class="profile-metric-card">
                <span>Tamamlanan Tutar</span>
                <strong><?= formatPrice($depositSummary['completed_total']) ?></strong>
                <small><?= $depositSummary['completed_count'] ?> onaylı işlem</small>
            </div>
            <div class="profile-metric-card">
                <span>Ödenen Tutar</span>
                <strong class="text-info"><?= formatPrice($depositSummary['completed_payable']) ?></strong>
                <small>Komisyon dahil toplam ödeme</small>
            </div>
            <div class="profile-metric-card">
                <span>Bekleyen</span>
                <strong class="text-warning"><?= formatPrice($depositSummary['pending_total']) ?></strong>
                <small><?= $depositSummary['pending_count'] ?> işlem beklemede</small>
            </div>
        </div>

        <?php if (empty($depositOrders)): ?>
            <div class="empty-state">
                <i class="fa-solid fa-wallet"></i>
                <p>Henüz bakiye yükleme işleminiz bulunmuyor.</p>
                <a href="<?= routeUrl('balance') ?>" class="btn btn-primary">
                    <i class="fa-solid fa-plus me-2"></i>Hemen Yükle
                </a>
            </div>
        <?php else: ?>
            <div class="card">
                <div class="table-responsive">
                    <table class="profile-table">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Tarih</th>
                                <th>Ödeme</th>
                                <th>Sipariş No</th>
                                <th>Yüklenen</th>
                                <th>Komisyon</th>
                                <th>Ödenen</th>
                                <th>Durum</th>
                                <th>Açıklama</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $depositStatusMap = [
                                'pending' => ['class' => 'status-badge status-pending', 'label' => 'Beklemede'],
                                'completed' => ['class' => 'status-badge status-completed', 'label' => 'Tamamlandı'],
                                'cancelled' => ['class' => 'status-badge status-cancelled', 'label' => 'İptal'],
                            ];
                            ?>
                            <?php foreach ($depositOrders as $deposit): ?>
                            <?php
                                $orderStatusKey = strtolower((string)($deposit['order_status'] ?? 'completed'));
                                $statusInfo = $depositStatusMap[$orderStatusKey] ?? ['class' => 'status-badge status-completed', 'label' => $deposit['order_status'] ?? 'Tamamlandı'];
                                $depositDate = $deposit['created_at'] ?? $deposit['order_created_at'] ?? null;
                                $paymentMethodRaw = trim((string)($deposit['payment_method'] ?? ''));
                                $paymentMethod = $paymentMethodRaw !== '' ? strtoupper($paymentMethodRaw) : ($deposit['description'] ? 'MANUEL' : '-');
                                $netAmount = (float)($deposit['amount'] ?? 0);
                                $orderPayable = isset($deposit['payable_amount']) ? (float)$deposit['payable_amount'] : null;
                                $orderCommission = isset($deposit['commission_amount']) ? (float)$deposit['commission_amount'] : null;
                                if ($orderPayable === null && $orderCommission !== null) {
                                    $orderPayable = $netAmount + $orderCommission;
                                }
                                if ($orderCommission === null && $orderPayable !== null) {
                                    $orderCommission = max($orderPayable - $netAmount, 0);
                                }
                                $grossAmount = $orderPayable ?? ($netAmount + ($orderCommission ?? 0));
                                $commissionAmount = $orderCommission ?? max($grossAmount - $netAmount, 0);
                                $description = $deposit['description'] ?? '';
                                $orderNumber = $deposit['order_number'] ?? '';
                                $identifier = $deposit['order_id'] ?? $deposit['id'];
                            ?>
                            <tr>
                                <td><?= (int)$identifier ?></td>
                                <td><?= $depositDate ? date('d.m.Y H:i', strtotime($depositDate)) : '-' ?></td>
                                <td>
                                    <span class="profile-tag profile-tag--deposit"><i class="fa-solid fa-credit-card"></i><?= sanitize($paymentMethod) ?></span>
                                </td>
                                <td>
                                    <?php if ($orderNumber !== ''): ?>
                                        <?= sanitize($orderNumber) ?>
                                    <?php else: ?>
                                        <span class="profile-muted">—</span>
                                    <?php endif; ?>
                                </td>
                                <td><strong class="text-success">+<?= formatPrice($netAmount) ?></strong></td>
                                <td class="text-warning">+<?= formatPrice($commissionAmount) ?></td>
                                <td><strong class="text-info"><?= formatPrice($grossAmount) ?></strong></td>
                                <td><span class="<?= $statusInfo['class'] ?>"><?= sanitize($statusInfo['label']) ?></span></td>
                                <td>
                                    <?php if ($description !== ''): ?>
                                        <?= sanitize($description) ?>
                                    <?php else: ?>
                                        <span class="profile-muted">-</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <?php if ($depositTotalPages > 1): ?>
            <nav class="pagination-nav" aria-label="Bakiye yüklemeleri sayfalama">
                <ul class="pagination justify-content-center">
                    <li class="page-item<?= $depositPage <= 1 ? ' disabled' : '' ?>">
                        <a class="page-link" href="<?= $depositPage <= 1 ? '#' : routeUrl('deposits') . '?page=' . ($depositPage - 1) ?>">
                            <i class="fa-solid fa-chevron-left"></i>
                        </a>
                    </li>
                    <?php for ($i = 1; $i <= $depositTotalPages; $i++): ?>
                        <li class="page-item<?= $i === $depositPage ? ' active' : '' ?>">
                            <a class="page-link" href="<?= routeUrl('deposits') ?>?page=<?= $i ?>"><?= $i ?></a>
                        </li>
                    <?php endfor; ?>
                    <li class="page-item<?= $depositPage >= $depositTotalPages ? ' disabled' : '' ?>">
                        <a class="page-link" href="<?= $depositPage >= $depositTotalPages ? '#' : routeUrl('deposits') . '?page=' . ($depositPage + 1) ?>">
                            <i class="fa-solid fa-chevron-right"></i>
                        </a>
                    </li>
                </ul>
            </nav>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</section>

<?php include 'includes/footer.php'; ?>

