<?php
require_once 'config.php';

if (!function_exists('shortenText')) {
    function shortenText(?string $text, int $limit = 110): string
    {
        $text = trim((string)$text);
        if ($text === '') {
            return '';
        }
        if (mb_strlen($text) <= $limit) {
            return $text;
        }
        return rtrim(mb_substr($text, 0, $limit - 3)) . '...';
    }
}

if (!function_exists('formatMetric')) {
    function formatMetric(int $number): string
    {
        if ($number >= 1000000) {
            return number_format($number / 1000000, 1) . 'M';
        }
        if ($number >= 1000) {
            return number_format($number / 1000, 1) . 'K';
        }
        return (string)number_format($number);
    }
}

if (!function_exists('resolveCategoryAsset')) {
    function resolveCategoryAsset(array $entity, array $map): string
    {
        if (!empty($entity['image'])) {
            $relative = ltrim((string)$entity['image'], '/\\');
            $fullPath = __DIR__ . '/' . $relative;
            if (is_file($fullPath)) {
                return $relative;
            }
        }

        $slug = $entity['category_slug'] ?? ($entity['slug'] ?? '');
        if (isset($map[$slug])) {
            return $map[$slug];
        }
        foreach ($map as $key => $path) {
            if ($slug !== '' && str_contains($slug, $key)) {
                return $path;
            }
        }
        return 'assets/images/category-oyun.svg';
    }
}

$categoryMediaMap = [
    'oyun-pinleri' => 'assets/images/category-oyun.svg',
    'oyunlar' => 'assets/images/category-oyun.svg',
    'oyun' => 'assets/images/category-oyun.svg',
    'pazaryeri' => 'assets/images/category-market.svg',
    'market' => 'assets/images/category-market.svg',
    'oyun-parasi' => 'assets/images/category-bakiye.svg',
    'bakiye' => 'assets/images/category-bakiye.svg',
    'destek' => 'assets/images/category-destek.svg',
];

// Get homepage settings
$showSlider = getSetting('homepage_show_slider', '1') === '1';
$showCategories = getSetting('homepage_show_categories', '1') === '1';
$showPopular = getSetting('homepage_show_popular', '1') === '1';
$showFeatured = getSetting('homepage_show_featured', '1') === '1';
$showNew = getSetting('homepage_show_new', '1') === '1';

// Get selected categories for popular and featured sections
$popularCategoriesJson = getSetting('homepage_popular_categories', '[]');
$popularCategoryIds = json_decode($popularCategoriesJson, true) ?: [];

$featuredCategoriesJson = getSetting('homepage_featured_categories', '[]');
$featuredCategoryIds = json_decode($featuredCategoriesJson, true) ?: [];

$newProductsCategory = (int)getSetting('homepage_new_products_category', 0);

// Get product sections settings
$showFeaturedProducts = getSetting('homepage_show_featured_products', '1') === '1';
$featuredProductsCount = max(1, min(50, (int)getSetting('homepage_featured_products_count', 6)));

$showBestSellers = getSetting('homepage_show_best_sellers', '1') === '1';
$bestSellersCount = max(1, min(50, (int)getSetting('homepage_best_sellers_count', 4)));

$newProductsCount = max(1, min(50, (int)getSetting('homepage_new_products_count', 8)));

// Başlık İkonları
$iconPopularCategories = getSetting('homepage_icon_popular_categories', 'fa-solid fa-layer-group');
$iconFeaturedCategories = getSetting('homepage_icon_featured_categories', 'fa-solid fa-fire');
$iconFeaturedProducts = getSetting('homepage_icon_featured_products', 'fa-solid fa-star');
$iconNewProducts = getSetting('homepage_icon_new_products', 'fa-solid fa-sparkles');
$iconBestSellers = getSetting('homepage_icon_best_sellers', 'fa-solid fa-trophy');

// Başlık İsimleri
$titlePopularCategories = getSetting('homepage_title_popular_categories', 'Popüler kategoriler');
$titleFeaturedCategories = getSetting('homepage_title_featured_categories', 'Öne Çıkan Kategoriler');
$titleFeaturedProducts = getSetting('homepage_title_featured_products', 'Öne çıkan ürünler');
$titleNewProducts = getSetting('homepage_title_new_products', 'Yeni eklenenler');
$titleBestSellers = getSetting('homepage_title_best_sellers', 'En çok satanlar');

// Neden Biz? Bölümü ayarları
$showWhyUs = getSetting('homepage_show_why_us', '1') === '1';
$whyUsTitle = getSetting('homepage_why_us_title', 'Neden ' . siteName() . '?');
$whyUsSubtitle = getSetting('homepage_why_us_subtitle', 'Sizin için en iyi deneyimi sunmak için her detayı düşündük.');
$whyUsFeaturesJson = getSetting('homepage_why_us_features', '[]');
$whyUsFeatures = json_decode($whyUsFeaturesJson, true) ?: [];

// Varsayılan özellikler
if (empty($whyUsFeatures)) {
    $whyUsFeatures = [
        ['icon' => 'fa-solid fa-bolt', 'title' => '15 Dakikada Teslim', 'description' => 'Ödeme tamamlandığında kodlarınız panelinize otomatik tanımlanır.'],
        ['icon' => 'fa-solid fa-shield', 'title' => 'Güvenli Ödeme Altyapısı', 'description' => 'PayTR ve Shopier güvencesiyle kart bilgileriniz saklanmadan işlem yapın.'],
        ['icon' => 'fa-solid fa-headset', 'title' => '7/24 Destek', 'description' => 'Canlı destek ekibimiz, tüm sorularınızda bir mesaj uzağınızda.'],
        ['icon' => 'fa-solid fa-gift', 'title' => 'Bonus ve Kampanyalar', 'description' => 'Düzenli kampanyalar ve bonus bakiyelerle kazancınızı artırın.']
    ];
}

// Get featured products
$featuredProducts = [];
if ($showFeaturedProducts) {
    $stmt = $pdo->prepare('
        SELECT p.*, c.name AS category_name, c.slug AS category_slug
        FROM products p
        JOIN categories c ON p.category_id = c.id
        WHERE p.is_active = 1 AND p.featured = 1
        ORDER BY p.sales_count DESC
        LIMIT ?
    ');
    $stmt->execute([$featuredProductsCount]);
    $featuredProducts = $stmt->fetchAll();
}

$categories = getCategories($pdo);

// Populate categories for popular section
$topCategories = [];
if ($showPopular && !empty($popularCategoryIds)) {
    foreach ($categories as $category) {
        if (in_array((int)$category['id'], $popularCategoryIds) && !empty($category['is_active'])) {
            // Add subcategories if needed
            if ($showSubcategories) {
                $stmt = $pdo->prepare('SELECT * FROM categories WHERE parent_id = ? AND is_active = 1 ORDER BY display_order ASC LIMIT 3');
                $stmt->execute([$category['id']]);
                $category['subcategories'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            }
            $topCategories[] = $category;
        }
    }
} elseif ($showPopular) {
    // If no specific categories selected, show first 12 active categories
    $filteredCats = array_filter($categories, function($cat) {
        return !empty($cat['is_active']);
    });
    $topCategories = array_slice($filteredCats, 0, 12);
    
    // Add subcategories if needed
    if ($showSubcategories) {
        foreach ($topCategories as &$category) {
            $stmt = $pdo->prepare('SELECT * FROM categories WHERE parent_id = ? AND is_active = 1 ORDER BY display_order ASC LIMIT 3');
            $stmt->execute([$category['id']]);
            $category['subcategories'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
    }
}

// Get marketplace games for featured section
$featuredMarketplace = [];
if ($showFeatured) {
    if (!empty($featuredCategoryIds)) {
        // Get selected marketplace games
        $placeholders = str_repeat('?,', count($featuredCategoryIds) - 1) . '?';
        $stmt = $pdo->prepare("
            SELECT gm.*, 
                   (SELECT COUNT(*) FROM game_marketplace_categories gmc WHERE gmc.game_id = gm.id AND gmc.is_active = 1) as category_count
            FROM game_marketplaces gm
            WHERE gm.id IN ($placeholders) AND gm.is_active = 1
            ORDER BY FIELD(gm.id, $placeholders)
        ");
        $stmt->execute(array_merge($featuredCategoryIds, $featuredCategoryIds));
        $games = $stmt->fetchAll();
        
        // Get categories for each game
        foreach ($games as $game) {
            $catStmt = $pdo->prepare('
                SELECT gmc.*, gms.name as server_name 
                FROM game_marketplace_categories gmc
                LEFT JOIN game_marketplace_servers gms ON gmc.server_id = gms.id
                WHERE gmc.game_id = ? AND gmc.is_active = 1
                ORDER BY gmc.display_order ASC, gmc.name ASC
            ');
            $catStmt->execute([$game['id']]);
            $game['categories'] = $catStmt->fetchAll();
            $featuredMarketplace[] = $game;
        }
    } else {
        // If no games selected, show all active marketplace games
        $marketplaceStructure = getMarketplaceStructure($pdo, true);
        $featuredMarketplace = $marketplaceStructure;
    }
}

// Get latest products based on category selection
$latestProducts = [];
if ($showNew) {
    if ($newProductsCategory > 0) {
        $stmt = $pdo->prepare('
            SELECT p.*, c.name AS category_name, c.slug AS category_slug
            FROM products p
            JOIN categories c ON p.category_id = c.id
            WHERE p.is_active = 1 AND p.category_id = ?
            ORDER BY p.created_at DESC
            LIMIT ?
        ');
        $stmt->execute([$newProductsCategory, $newProductsCount]);
        $latestProducts = $stmt->fetchAll();
    } else {
        $stmt = $pdo->prepare('
            SELECT p.*, c.name AS category_name, c.slug AS category_slug
            FROM products p
            JOIN categories c ON p.category_id = c.id
            WHERE p.is_active = 1
            ORDER BY p.created_at DESC
            LIMIT ?
        ');
        $stmt->execute([$newProductsCount]);
        $latestProducts = $stmt->fetchAll();
    }
}

// Get best sellers
$bestSellers = [];
if ($showBestSellers) {
    $stmt = $pdo->prepare('
        SELECT p.*, c.name AS category_name, c.slug AS category_slug
        FROM products p
        JOIN categories c ON p.category_id = c.id
        WHERE p.is_active = 1
        ORDER BY p.sales_count DESC
        LIMIT ?
    ');
    $stmt->execute([$bestSellersCount]);
    $bestSellers = $stmt->fetchAll();
}

$statsData = [
    'customers' => (int)$pdo->query('SELECT COUNT(*) FROM users WHERE is_admin = 0')->fetchColumn(),
    'products' => (int)$pdo->query('SELECT COUNT(*) FROM products WHERE is_active = 1')->fetchColumn(),
    'orders' => (int)$pdo->query('SELECT COUNT(*) FROM orders')->fetchColumn(),
    'stocks' => (int)$pdo->query('SELECT COALESCE(SUM(stock), 0) FROM products WHERE is_active = 1')->fetchColumn(),
];

// Fetch slider items
$sliderItems = [];
try {
    $stmt = $pdo->query('SELECT * FROM homepage_slider_items WHERE is_active = 1 ORDER BY display_order ASC, id ASC LIMIT 5');
    $sliderItems = $stmt ? ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: []) : [];
} catch (Throwable $th) {
    $sliderItems = [];
}

// Fetch side banners
$sideBanners = [];
try {
    $stmt = $pdo->query('SELECT * FROM homepage_side_banners WHERE is_active = 1 ORDER BY position ASC LIMIT 2');
    $sideBanners = $stmt ? ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: []) : [];
} catch (Throwable $th) {
    $sideBanners = [];
}

// Default side banners if empty
if (empty($sideBanners)) {
    $sideBanners = [
        [
            'id' => 1,
            'position' => 1,
            'title' => 'Kategoriler',
            'link_url' => 'products',
            'image' => 'assets/images/home_category/7b6d683e-a3a1-42dc-bed9-d02bb111347964415.jpg',
        ],
        [
            'id' => 2,
            'position' => 2,
            'title' => 'Marketplace',
            'link_url' => 'marketplace',
            'image' => 'assets/images/home_category/b0c7fb53-7fa5-451d-9d80-386b6a62c6c067010.jpg',
        ],
    ];
}

// Fetch bottom banners
$bottomBanners = [];
try {
    $stmt = $pdo->query('SELECT * FROM homepage_bottom_banners WHERE is_active = 1 ORDER BY display_order ASC, id ASC LIMIT 3');
    $bottomBanners = $stmt ? ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: []) : [];
} catch (Throwable $th) {
    $bottomBanners = [];
}

// Default bottom banners if empty
if (empty($bottomBanners)) {
    $bottomBanners = [
        [
            'id' => 1,
            'display_order' => 1,
            'title' => 'Kategoriler',
            'link_url' => 'products',
            'image' => 'assets/images/home_category/7b6d683e-a3a1-42dc-bed9-d02bb111347964415.jpg',
        ],
        [
            'id' => 2,
            'display_order' => 2,
            'title' => 'Marketplace',
            'link_url' => 'marketplace',
            'image' => 'assets/images/home_category/b0c7fb53-7fa5-451d-9d80-386b6a62c6c067010.jpg',
        ],
        [
            'id' => 3,
            'display_order' => 3,
            'title' => 'Ürünler',
            'link_url' => 'products',
            'image' => 'assets/images/home_category/7b6d683e-a3a1-42dc-bed9-d02bb111347964415.jpg',
        ],
    ];
}

if (empty($sliderItems)) {
    $sliderItems = [
        [
            'id' => 1,
            'title' => 'Valorant Random Hesap',
            'description' => 'En uygun fiyatlarla güvenli alışveriş',
            'image' => 'assets/images/sliders/59641790-c077-4796-a4c0-634180af4b069727.jpg',
            'primary_url' => routeUrl('products'),
        ],
        [
            'id' => 2,
            'title' => 'PUBG Mobile Random Hesap',
            'description' => 'Anında teslimat garantisi',
            'image' => 'assets/images/sliders/07669aca-0446-44a4-80e9-86ec3ab75f8157458.jpg',
            'primary_url' => routeUrl('products'),
        ],
        [
            'id' => 3,
            'title' => 'E-Posta Garantili Hesaplar',
            'description' => '7/24 canlı destek',
            'image' => 'assets/images/sliders/48718b11-5baa-4995-b6c1-5db8d0e0ae6f66601.jpg',
            'primary_url' => routeUrl('products'),
        ],
    ];
}

$pageTitle = 'Ana Sayfa';

// Slider settings
$sliderAutoplayDelay = (int)(getSetting('slider_autoplay_delay', '5000') ?: 5000);
$sliderAutoplayEnabledSetting = getSetting('slider_autoplay_enabled', '1');
// Enable autoplay by default (true if setting is '1', 1, true, or not set)
$sliderAutoplayEnabled = ($sliderAutoplayEnabledSetting === '1' || $sliderAutoplayEnabledSetting === 1 || $sliderAutoplayEnabledSetting === true || $sliderAutoplayEnabledSetting === null);

// Get advertisements
$adsSliderTop = getAdvertisements('slider_top');
$adsSliderBottom = getAdvertisements('slider_bottom');
$adsCategoryTop = getAdvertisements('category_top');
$adsCategoryBottom = getAdvertisements('category_bottom');
$adsFooterTop = getAdvertisements('footer_top');

// Get subcategory setting
$showSubcategories = getBoolSetting('show_subcategories_on_home', true);

include 'includes/header.php';
?>

<!-- Slider Üstü Reklamlar -->
<?php if (!empty($adsSliderTop)): ?>
<section class="advertisement-section slider-top-ads">
    <div class="container">
        <div class="row g-3">
            <?php foreach ($adsSliderTop as $ad): ?>
                <div class="col-12">
                    <?= renderAdvertisement($ad) ?>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php if ($showSlider): ?>
<section class="fp-slider-area" data-autoplay="<?= $sliderAutoplayEnabled ? 'true' : 'false' ?>" data-delay="<?= $sliderAutoplayDelay ?>">
    <div class="container">
        <div class="grid">
            <div class="grid-left">
                <div class="fp-swiper-home swiper">
                    <div class="swiper-wrapper">
                        <?php foreach ($sliderItems as $slide): ?>
                            <div class="swiper-slide">
                                <a href="<?= htmlspecialchars($slide['primary_url'] ?? routeUrl('products'), ENT_QUOTES, 'UTF-8') ?>">
                                    <div class="fp-swiper-home-item">
                                        <img src="<?= htmlspecialchars(publicUrl($slide['image'] ?? 'assets/images/hero-showcase.svg'), ENT_QUOTES, 'UTF-8') ?>" alt="<?= sanitize($slide['title'] ?? '') ?>" class="img-cover">
                                        <?php if (!empty($slide['title']) || !empty($slide['description'])): ?>
                                            <div class="content">
                                                <?php if (!empty($slide['title'])): ?>
                                                    <h3 class="title"><?= sanitize($slide['title']) ?></h3>
                                                <?php endif; ?>
                                                <?php if (!empty($slide['description'])): ?>
                                                    <p><?= sanitize($slide['description']) ?></p>
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </a>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <!-- Navigation buttons -->
                    <div class="swiper-button-next"></div>
                    <div class="swiper-button-prev"></div>
                </div>
            </div>
            <script>
                // SLIDER - DIREKT ÇALIŞIR
                (function() {
                    function initSlider() {
                        if (typeof Swiper === 'undefined') {
                            setTimeout(initSlider, 100);
                            return;
                        }
                        const sliderArea = document.querySelector('.fp-slider-area');
                        const autoplay = sliderArea?.dataset.autoplay === 'true';
                        const delay = parseInt(sliderArea?.dataset.delay) || 5000;
                        
                        new Swiper('.fp-swiper-home', {
                            spaceBetween: 24,
                            loop: true,
                            autoplay: autoplay ? { delay: delay, disableOnInteraction: false } : false,
                            navigation: {
                                nextEl: '.fp-swiper-home .swiper-button-next',
                                prevEl: '.fp-swiper-home .swiper-button-prev',
                            }
                        });
                    }
                    if (document.readyState === 'loading') {
                        document.addEventListener('DOMContentLoaded', initSlider);
                    } else {
                        initSlider();
                    }
                })();
            </script>
            <div class="grid-right">
                <div class="fp-home-slider-mini-grid">
                    <?php foreach ($sideBanners as $banner): ?>
                        <a href="<?= htmlspecialchars(routeUrl($banner['link_url'] ?? 'products'), ENT_QUOTES, 'UTF-8') ?>">
                            <div class="fp-home-slider-mini">
                                <img src="<?= htmlspecialchars(publicUrl($banner['image']), ENT_QUOTES, 'UTF-8') ?>" alt="<?= htmlspecialchars($banner['title'] ?? '', ENT_QUOTES, 'UTF-8') ?>" class="img-cover">
                            </div>
                        </a>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <div class="fp-slider-bottom-grid">
            <?php foreach ($bottomBanners as $banner): ?>
                <a href="<?= htmlspecialchars(routeUrl($banner['link_url'] ?? 'products'), ENT_QUOTES, 'UTF-8') ?>" class="fp-slider-bottom-item">
                    <img src="<?= htmlspecialchars(publicUrl($banner['image']), ENT_QUOTES, 'UTF-8') ?>" alt="<?= htmlspecialchars($banner['title'] ?? '', ENT_QUOTES, 'UTF-8') ?>" class="img-cover">
                </a>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php if ($showFeatured && !empty($featuredMarketplace)): ?>
<section class="category-showcase section-padding">
    <div class="container-wide">
        <div class="section-heading">
            <h2><i class="<?= sanitize($iconFeaturedCategories) ?>"></i> <?= sanitize($titleFeaturedCategories) ?></h2>
            <div class="section-heading__actions">
                <button class="carousel-nav carousel-nav--prev" data-carousel="featured-categories" disabled>
                    <i class="fa-solid fa-chevron-left"></i>
                </button>
                <button class="carousel-nav carousel-nav--next" data-carousel="featured-categories">
                    <i class="fa-solid fa-chevron-right"></i>
                </button>
            </div>
        </div>
        <div class="carousel-wrapper">
            <div class="popular-categories-carousel" id="featured-categories">
                <?php 
                foreach (array_slice($featuredMarketplace, 0, 12) as $game): 
                    $gameImageRaw = $game['image'] ?? null;
                    // Ensure image is a string, not an array
                    if (is_array($gameImageRaw)) {
                        $gameImageRaw = null;
                    }
                    $gameImage = !empty($gameImageRaw) ? assetUrl($gameImageRaw) : null;
                    $gameIcon = $game['icon'] ?? 'fa-gamepad';
                ?>
                    <a href="<?= htmlspecialchars(routeUrl('marketplace_game', ['game' => $game['slug']]), ENT_QUOTES, 'UTF-8') ?>" 
                       class="popular-category-card">
                        <div class="popular-category-card__image-wrapper">
                            <?php if ($gameImage): ?>
                                <img src="<?= htmlspecialchars($gameImage, ENT_QUOTES, 'UTF-8') ?>" 
                                     alt="<?= sanitize($game['name']) ?>" 
                                     class="popular-category-card__image">
                            <?php else: ?>
                                <div class="popular-category-card__icon">
                                    <i class="fa-solid <?= sanitize($gameIcon) ?>"></i>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="popular-category-card__name"><?= sanitize($game['name']) ?></div>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- Slider Altı Reklamlar -->
<?php if (!empty($adsSliderBottom)): ?>
<section class="advertisement-section slider-bottom-ads py-3">
    <div class="container-wide">
        <div class="row g-3">
            <?php foreach ($adsSliderBottom as $ad): ?>
                <div class="col-12">
                    <?= renderAdvertisement($ad) ?>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- Kategoriler Üstü Reklamlar -->
<?php if (!empty($adsCategoryTop)): ?>
<section class="advertisement-section category-top-ads py-3">
    <div class="container-wide">
        <div class="row g-3">
            <?php foreach ($adsCategoryTop as $ad): ?>
                <div class="col-12">
                    <?= renderAdvertisement($ad) ?>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php if ($showPopular && !empty($topCategories)): ?>
<section class="category-showcase section-padding">
    <div class="container-wide">
        <div class="section-heading">
            <h2><i class="<?= sanitize($iconPopularCategories) ?>"></i> <?= sanitize($titlePopularCategories) ?></h2>
            <div class="section-heading__actions">
                <button class="carousel-nav carousel-nav--prev" data-carousel="popular-categories" disabled>
                    <i class="fa-solid fa-chevron-left"></i>
                </button>
                <button class="carousel-nav carousel-nav--next" data-carousel="popular-categories">
                    <i class="fa-solid fa-chevron-right"></i>
                </button>
            </div>
        </div>
        <div class="carousel-wrapper">
            <div class="popular-categories-carousel" id="popular-categories">
                <?php 
                foreach ($topCategories as $category): 
                    $categoryImageRaw = $category['image'] ?? null;
                    // Ensure image is a string, not an array
                    if (is_array($categoryImageRaw)) {
                        $categoryImageRaw = null;
                    }
                    $categoryImage = !empty($categoryImageRaw) ? assetUrl($categoryImageRaw) : null;
                    $categoryIcon = $category['icon'] ?? 'fa-tag';
                ?>
                    <a href="<?= htmlspecialchars(routeUrl('product_category', ['slug' => $category['slug']]), ENT_QUOTES, 'UTF-8') ?>" 
                       class="popular-category-card">
                        <div class="popular-category-card__image-wrapper">
                            <?php if ($categoryImage): ?>
                                <img src="<?= htmlspecialchars($categoryImage, ENT_QUOTES, 'UTF-8') ?>" 
                                     alt="<?= sanitize($category['name']) ?>" 
                                     class="popular-category-card__image">
                            <?php else: ?>
                                <div class="popular-category-card__icon">
                                    <i class="fa-solid <?= sanitize($categoryIcon) ?>"></i>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="popular-category-card__name"><?= sanitize($category['name']) ?></div>
                        <?php if ($showSubcategories && !empty($category['subcategories'])): ?>
                            <div class="popular-category-card__subcategories">
                                <?php foreach ($category['subcategories'] as $subcat): ?>
                                    <span class="subcategory-badge"><?= sanitize($subcat['name']) ?></span>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- Kategoriler Altı Reklamlar -->
<?php if (!empty($adsCategoryBottom)): ?>
<section class="advertisement-section category-bottom-ads py-3">
    <div class="container-wide">
        <div class="row g-3">
            <?php foreach ($adsCategoryBottom as $ad): ?>
                <div class="col-12">
                    <?= renderAdvertisement($ad) ?>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php if ($showFeaturedProducts && !empty($featuredProducts)): ?>
<section class="section-padding pt-0">
    <div class="container-wide">
        <div class="section-heading">
            <h2><i class="<?= sanitize($iconFeaturedProducts) ?>"></i> <?= sanitize($titleFeaturedProducts) ?></h2>
            <a href="<?= routeUrl('products') ?>?featured=1"><span>Hepsini gör</span><i class="fa-solid fa-arrow-right"></i></a>
        </div>
        <div class="home-products-grid">
            <?php foreach ($featuredProducts as $product): ?>
            <div class="home-product-card">
                <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__image">
                    <?php $productImage = htmlspecialchars(assetUrl(resolveCategoryAsset($product, $categoryMediaMap)), ENT_QUOTES, 'UTF-8'); ?>
                    <img src="<?= $productImage ?>" alt="<?= sanitize($product['name']) ?>">
                    <?php if (!empty($product['is_featured'])): ?>
                        <span class="home-product-card__badge">Öne Çıkan</span>
                    <?php endif; ?>
                </a>
                <div class="home-product-card__content">
                    <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__title">
                        <?= sanitize($product['name']) ?>
                    </a>
                    <div class="home-product-card__price">
                        <?= formatPrice($product['price']) ?>
                    </div>
                </div>
                <div class="home-product-card__actions">
                    <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__btn home-product-card__btn--primary">
                        <i class="fa-solid fa-shopping-bag"></i>
                        <span>Satın Al</span>
                    </a>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php if ($showNew && !empty($latestProducts)): ?>
<section class="section-padding pt-0">
    <div class="container-wide">
        <div class="section-heading">
            <h2><i class="<?= sanitize($iconNewProducts) ?>"></i> <?= sanitize($titleNewProducts) ?></h2>
            <a href="<?= routeUrl('products') ?>?sort=new"><span>Tümü</span><i class="fa-solid fa-arrow-right"></i></a>
        </div>
        <div class="home-products-grid">
            <?php foreach ($latestProducts as $product): ?>
            <div class="home-product-card">
                <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__image">
                    <?php $latestImage = htmlspecialchars(assetUrl(resolveCategoryAsset($product, $categoryMediaMap)), ENT_QUOTES, 'UTF-8'); ?>
                    <img src="<?= $latestImage ?>" alt="<?= sanitize($product['name']) ?>">
                    <span class="home-product-card__badge home-product-card__badge--new">Yeni</span>
                </a>
                <div class="home-product-card__content">
                    <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__title">
                        <?= sanitize($product['name']) ?>
                    </a>
                    <div class="home-product-card__price">
                        <?= formatPrice($product['price']) ?>
                    </div>
                </div>
                <div class="home-product-card__actions">
                    <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__btn home-product-card__btn--primary">
                        <i class="fa-solid fa-shopping-bag"></i>
                        <span>Satın Al</span>
                    </a>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php if ($showBestSellers && !empty($bestSellers)): ?>
<section class="section-padding pt-0">
    <div class="container-wide">
        <div class="section-heading">
            <h2><i class="<?= sanitize($iconBestSellers) ?>"></i> <?= sanitize($titleBestSellers) ?></h2>
            <a href="<?= routeUrl('products') ?>?sort=top"><span>Popülerleri gör</span><i class="fa-solid fa-arrow-right"></i></a>
        </div>
        <div class="home-products-grid">
            <?php foreach ($bestSellers as $product): ?>
            <div class="home-product-card">
                <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__image">
                    <?php $bestImage = htmlspecialchars(assetUrl(resolveCategoryAsset($product, $categoryMediaMap)), ENT_QUOTES, 'UTF-8'); ?>
                    <img src="<?= $bestImage ?>" alt="<?= sanitize($product['name']) ?>">
                    <span class="home-product-card__badge home-product-card__badge--best">Çok Satan</span>
                </a>
                <div class="home-product-card__content">
                    <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__title">
                        <?= sanitize($product['name']) ?>
                    </a>
                    <div class="home-product-card__price">
                        <?= formatPrice($product['price']) ?>
                    </div>
                </div>
                <div class="home-product-card__actions">
                    <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__btn home-product-card__btn--primary">
                        <i class="fa-solid fa-shopping-bag"></i>
                        <span>Satın Al</span>
                    </a>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php if ($showWhyUs): ?>
<section class="section-padding" id="destek">
    <div class="container-wide">
        <div class="section-heading">
            <h2><?= sanitize($whyUsTitle) ?></h2>
            <?php if (!empty($whyUsSubtitle)): ?>
                <p><?= sanitize($whyUsSubtitle) ?></p>
            <?php endif; ?>
        </div>
        <div class="feature-strip">
            <?php foreach ($whyUsFeatures as $feature): ?>
                <?php if (!empty($feature['title'])): ?>
                <div class="feature-item">
                    <i class="<?= sanitize($feature['icon'] ?? 'fa-solid fa-star') ?>"></i>
                    <div>
                        <h5><?= sanitize($feature['title']) ?></h5>
                        <?php if (!empty($feature['description'])): ?>
                            <p><?= sanitize($feature['description']) ?></p>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<script>
// CAROUSEL BUTONLARI - DIREKT ÇALIŞIR
(function() {
    function initCarousels() {
        document.querySelectorAll('[id$="-categories"]').forEach(function(carousel) {
            const id = carousel.id;
            const prevBtn = document.querySelector('.carousel-nav--prev[data-carousel="' + id + '"]');
            const nextBtn = document.querySelector('.carousel-nav--next[data-carousel="' + id + '"]');
            
            if (!prevBtn || !nextBtn) return;
            
            function updateBtns() {
                prevBtn.disabled = carousel.scrollLeft <= 0;
                nextBtn.disabled = carousel.scrollLeft >= carousel.scrollWidth - carousel.clientWidth - 5;
            }
            
            prevBtn.onclick = function(e) {
                e.preventDefault();
                carousel.scrollBy({ left: -carousel.clientWidth * 0.8, behavior: 'smooth' });
                setTimeout(updateBtns, 300);
            };
            
            nextBtn.onclick = function(e) {
                e.preventDefault();
                carousel.scrollBy({ left: carousel.clientWidth * 0.8, behavior: 'smooth' });
                setTimeout(updateBtns, 300);
            };
            
            carousel.addEventListener('scroll', updateBtns);
            updateBtns();
        });
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initCarousels);
    } else {
        initCarousels();
    }
})();
</script>

<!-- Footer Üstü Reklamlar -->
<?php if (!empty($adsFooterTop)): ?>
<section class="advertisement-section footer-top-ads py-4">
    <div class="container-wide">
        <div class="row g-3">
            <?php foreach ($adsFooterTop as $ad): ?>
                <div class="col-12">
                    <?= renderAdvertisement($ad) ?>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<?php include 'includes/footer.php'; ?>
