<?php
require_once 'config.php';
requireLogin();

$user = getUserInfo($pdo, $_SESSION['user_id']);

// Marketplace totals
$marketplaceTotals = [
    'buy_volume' => 0.0,
    'sell_volume' => 0.0,
    'buy_count' => 0,
    'sell_count' => 0
];
$marketplaceTotalsStmt = $pdo->prepare("SELECT order_type, COUNT(*) AS cnt, COALESCE(SUM(total_amount),0) AS total FROM game_marketplace_orders WHERE user_id = ? AND status IN ('completed','approved') GROUP BY order_type");
$marketplaceTotalsStmt->execute([$_SESSION['user_id']]);
foreach ($marketplaceTotalsStmt->fetchAll() as $row) {
    if ($row['order_type'] === 'buy') {
        $marketplaceTotals['buy_volume'] = (float)$row['total'];
        $marketplaceTotals['buy_count'] = (int)$row['cnt'];
    } elseif ($row['order_type'] === 'sell') {
        $marketplaceTotals['sell_volume'] = (float)$row['total'];
        $marketplaceTotals['sell_count'] = (int)$row['cnt'];
    }
}
$marketplaceNet = $marketplaceTotals['sell_volume'] - $marketplaceTotals['buy_volume'];

// Pagination
$mpPerPage = 10;
$mpPage = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;

$mpCountStmt = $pdo->prepare('SELECT COUNT(*) FROM game_marketplace_orders WHERE user_id = ?');
$mpCountStmt->execute([$_SESSION['user_id']]);
$mpTotalOrders = (int)$mpCountStmt->fetchColumn();
$mpTotalPages = max(1, (int)ceil($mpTotalOrders / $mpPerPage));
if ($mpPage > $mpTotalPages) {
    $mpPage = $mpTotalPages;
}
$mpOffset = ($mpPage - 1) * $mpPerPage;

// Get marketplace orders
$recentMarketplaceStmt = $pdo->prepare('SELECT o.*, s.name AS server_name, c.name AS category_name, m.name AS marketplace_name,
    (SELECT GROUP_CONCAT(mc.code ORDER BY mc.id SEPARATOR ", ") FROM game_marketplace_codes mc WHERE mc.order_id = o.id) AS delivered_codes
    FROM game_marketplace_orders o
    LEFT JOIN game_marketplace_servers s ON s.id = o.server_id
    LEFT JOIN game_marketplace_categories c ON c.id = o.category_id
    LEFT JOIN game_marketplaces m ON m.id = o.marketplace_id
    WHERE o.user_id = :user_id
    ORDER BY o.created_at DESC
    LIMIT :offset, :limit');
$recentMarketplaceStmt->bindValue(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);
$recentMarketplaceStmt->bindValue(':offset', $mpOffset, PDO::PARAM_INT);
$recentMarketplaceStmt->bindValue(':limit', $mpPerPage, PDO::PARAM_INT);
$recentMarketplaceStmt->execute();
$recentMarketplaceOrders = $recentMarketplaceStmt->fetchAll();

$pageTitle = 'Marketplace İşlemlerim';
include 'includes/header.php';
?>

<section class="page-section">
    <div class="container-wide">
        <div class="page-header">
            <div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="<?= routeUrl('profile') ?>"><i class="fa-solid fa-user me-1"></i>Profilim</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Marketplace İşlemlerim</li>
                    </ol>
                </nav>
                <h1 class="page-title"><i class="fa-solid fa-cart-shopping me-2"></i>Marketplace İşlemlerim</h1>
                <p class="page-description">Pazaryerinde gerçekleştirdiğiniz alım ve satım işlemlerini buradan görüntüleyebilirsiniz.</p>
            </div>
            <div>
                <a href="<?= routeUrl('market') ?>" class="btn btn-primary">
                    <i class="fa-solid fa-store me-2"></i>Marketplace'e Git
                </a>
            </div>
        </div>

        <div class="profile-metric-strip">
            <div class="profile-metric-card">
                <span>Alım Sayısı</span>
                <strong><?= $marketplaceTotals['buy_count'] ?></strong>
                <small><?= formatPrice($marketplaceTotals['buy_volume']) ?></small>
            </div>
            <div class="profile-metric-card">
                <span>Satış Sayısı</span>
                <strong><?= $marketplaceTotals['sell_count'] ?></strong>
                <small><?= formatPrice($marketplaceTotals['sell_volume']) ?></small>
            </div>
            <div class="profile-metric-card">
                <span>Marketplace Net</span>
                <strong class="<?= $marketplaceNet >= 0 ? 'text-success' : 'text-warning' ?>"><?= formatPrice($marketplaceNet) ?></strong>
                <small>Satış - Alım</small>
            </div>
            <div class="profile-metric-card">
                <span>Toplam İşlem</span>
                <strong><?= $marketplaceTotals['buy_count'] + $marketplaceTotals['sell_count'] ?></strong>
                <small>Tüm işlemler</small>
            </div>
        </div>

        <?php if (empty($recentMarketplaceOrders)): ?>
            <div class="empty-state">
                <i class="fa-solid fa-inbox"></i>
                <p>Henüz marketplace işleminiz bulunmuyor.</p>
                <a href="<?= routeUrl('marketplace') ?>" class="btn btn-primary">
                    <i class="fa-solid fa-store me-2"></i>Keşfet
                </a>
            </div>
        <?php else: ?>
            <div class="card">
                <div class="table-responsive">
                    <table class="profile-table">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Tarih</th>
                                <th>Oyun / Kategori</th>
                                <th>Server</th>
                                <th>Tür</th>
                                <th>Miktar</th>
                                <th>Birim</th>
                                <th>Toplam</th>
                                <th>Durum</th>
                                <th></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $marketplaceStatusMap = [
                                'pending' => ['class' => 'status-badge status-pending', 'label' => 'Beklemede'],
                                'completed' => ['class' => 'status-badge status-completed', 'label' => 'Tamamlandı'],
                                'cancelled' => ['class' => 'status-badge status-cancelled', 'label' => 'İptal'],
                                'approved' => ['class' => 'status-badge status-completed', 'label' => 'Onaylandı'],
                            ];
                            ?>
                            <?php foreach ($recentMarketplaceOrders as $mpOrder): ?>
                            <?php
                                $statusKey = strtolower((string)$mpOrder['status']);
                                $statusInfo = $marketplaceStatusMap[$statusKey] ?? ['class' => 'status-badge status-pending', 'label' => $mpOrder['status']];
                            ?>
                            <tr>
                                <td><?= $mpOrder['id'] ?></td>
                                <td><?= date('d.m.Y H:i', strtotime($mpOrder['created_at'])) ?></td>
                                <td>
                                    <strong><?= sanitize($mpOrder['marketplace_name'] ?? 'Silindi') ?></strong>
                                    <div class="profile-muted"><?= sanitize($mpOrder['category_name'] ?? '-') ?></div>
                                </td>
                                <td><?= sanitize($mpOrder['server_name'] ?? '-') ?></td>
                                <td>
                                    <?php if ($mpOrder['order_type'] === 'sell'): ?>
                                        <span class="profile-tag profile-tag--sell"><i class="fa-solid fa-upload me-1"></i>Bize Sat</span>
                                    <?php else: ?>
                                        <span class="profile-tag profile-tag--buy"><i class="fa-solid fa-download me-1"></i>Satın Al</span>
                                    <?php endif; ?>
                                </td>
                                <td><?= number_format((float)$mpOrder['quantity'], 2, ',', '.') ?></td>
                                <td><?= formatPrice($mpOrder['price_per_unit']) ?></td>
                                <td><strong><?= formatPrice($mpOrder['total_amount']) ?></strong></td>
                                <td><span class="<?= $statusInfo['class'] ?>"><?= sanitize($statusInfo['label']) ?></span></td>
                                <td class="text-end">
                                    <button type="button" class="btn btn-sm btn-outline-primary" data-order-detail
                                        data-order-id="<?= htmlspecialchars((string)$mpOrder['id'], ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-date="<?= htmlspecialchars(date('d.m.Y H:i', strtotime($mpOrder['created_at'])), ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-game="<?= htmlspecialchars($mpOrder['marketplace_name'] ?? '-', ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-category="<?= htmlspecialchars($mpOrder['category_name'] ?? '-', ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-server="<?= htmlspecialchars($mpOrder['server_name'] ?? '-', ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-type="<?= htmlspecialchars($mpOrder['order_type'], ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-status="<?= htmlspecialchars($mpOrder['status'], ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-quantity="<?= htmlspecialchars(number_format((float)$mpOrder['quantity'], 2, ',', '.'), ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-price="<?= htmlspecialchars(formatPrice($mpOrder['price_per_unit']), ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-total="<?= htmlspecialchars(formatPrice($mpOrder['total_amount']), ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-character="<?= htmlspecialchars($mpOrder['character_name'] ?? '-', ENT_QUOTES, 'UTF-8') ?>"
                                        data-order-codes="<?= htmlspecialchars($mpOrder['delivered_codes'] ?? '-', ENT_QUOTES, 'UTF-8') ?>">
                                        <i class="fa-solid fa-eye me-1"></i>Detay
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <?php if ($mpTotalPages > 1): ?>
            <nav class="pagination-nav" aria-label="Marketplace sayfalama">
                <ul class="pagination justify-content-center">
                    <li class="page-item<?= $mpPage <= 1 ? ' disabled' : '' ?>">
                        <a class="page-link" href="<?= $mpPage <= 1 ? '#' : routeUrl('marketplace_orders') . '?page=' . ($mpPage - 1) ?>">
                            <i class="fa-solid fa-chevron-left"></i>
                        </a>
                    </li>
                    <?php for ($i = 1; $i <= $mpTotalPages; $i++): ?>
                        <li class="page-item<?= $i === $mpPage ? ' active' : '' ?>">
                            <a class="page-link" href="<?= routeUrl('marketplace_orders') ?>?page=<?= $i ?>"><?= $i ?></a>
                        </li>
                    <?php endfor; ?>
                    <li class="page-item<?= $mpPage >= $mpTotalPages ? ' disabled' : '' ?>">
                        <a class="page-link" href="<?= $mpPage >= $mpTotalPages ? '#' : routeUrl('marketplace_orders') . '?page=' . ($mpPage + 1) ?>">
                            <i class="fa-solid fa-chevron-right"></i>
                        </a>
                    </li>
                </ul>
            </nav>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</section>

<!-- Marketplace Order Modal -->
<div class="modal fade" id="marketplaceOrderModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered modal-lg">
        <div class="modal-content">
            <div class="modal-header border-0">
                <h5 class="modal-title"><i class="fa-solid fa-receipt me-2"></i>İşlem Detayı</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Kapat"></button>
            </div>
            <div class="modal-body">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="profile-muted d-block">İşlem No</label>
                        <strong data-field="id">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Tarih</label>
                        <strong data-field="date">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Oyun</label>
                        <strong data-field="game">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Kategori</label>
                        <strong data-field="category">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Server</label>
                        <strong data-field="server">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Karakter Adı</label>
                        <strong data-field="character">—</strong>
                    </div>
                </div>
                <hr>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="profile-muted d-block">Tür</label>
                        <span class="badge" data-field="type">—</span>
                    </div>
                    <div class="col-md-4">
                        <label class="profile-muted d-block">Durum</label>
                        <span class="badge bg-secondary" data-field="status">—</span>
                    </div>
                    <div class="col-md-4">
                        <label class="profile-muted d-block">Miktar</label>
                        <strong data-field="quantity">—</strong>
                    </div>
                </div>
                <div class="row g-3 mt-2">
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Birim Fiyat</label>
                        <strong data-field="price">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Toplam Tutar</label>
                        <strong data-field="total">—</strong>
                    </div>
                </div>
                <div class="mt-3">
                    <label class="profile-muted d-block">Teslim Edilen Kodlar</label>
                    <div class="badge bg-secondary" data-field="codes">—</div>
                </div>
            </div>
            <div class="modal-footer border-0">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Kapat</button>
            </div>
        </div>
    </div>
</div>

<?php
$scriptVersionMarketplaceModal = file_exists(__DIR__ . '/assets/js/marketplace-modal.js') ? filemtime(__DIR__ . '/assets/js/marketplace-modal.js') : time();
?>
<script>window.__MARKETPLACE_MODAL_OVERRIDE = true;</script>
<script src="<?= htmlspecialchars(publicUrl('assets/js/marketplace-modal.js'), ENT_QUOTES, 'UTF-8') ?>?v=<?= $scriptVersionMarketplaceModal ?>"></script>

<?php include 'includes/footer.php'; ?>

