<?php
require_once 'config.php';

if (!function_exists('shortenText')) {
    function shortenText(?string $text, int $limit = 200): string
    {
        $text = trim((string)$text);
        if ($text === '') {
            return '';
        }
        if (mb_strlen($text) <= $limit) {
            return $text;
        }
        return rtrim(mb_substr($text, 0, $limit - 3)) . '...';
    }
}

$formatDecimal = static function ($value): string {
    if ($value === null || $value === '') {
        return '';
    }
    $number = (float)$value;
    return rtrim(rtrim(number_format($number, 8, '.', ''), '0'), '.');
};

$structure = getMarketplaceStructure($pdo, true);
if (empty($structure)) {
    $pageTitle = 'Marketplace';
    include 'includes/header.php';
    echo '<section class="section-padding"><div class="container-wide text-center text-white-50"><h3>Marketplace henüz yapılandırılmadı.</h3></div></section>';
    include 'includes/footer.php';
    exit;
}

$gameSlug = $_GET['game'] ?? $structure[0]['slug'];
$selectedGame = null;
foreach ($structure as $game) {
    if ($game['slug'] === $gameSlug) {
        $selectedGame = $game;
        break;
    }
}

if (!$selectedGame) {
    $selectedGame = $structure[0];
    $gameSlug = $selectedGame['slug'];
}

$categorySlug = $_GET['category'] ?? ($selectedGame['categories'][0]['slug'] ?? null);
$selectedCategory = null;
if (!empty($selectedGame['categories'])) {
    foreach ($selectedGame['categories'] as $category) {
        if ($category['slug'] === $categorySlug) {
            $selectedCategory = $category;
            break;
        }
    }
    if (!$selectedCategory) {
        $selectedCategory = $selectedGame['categories'][0];
        $categorySlug = $selectedCategory['slug'];
    }
}

$search = trim($_GET['search'] ?? '');

$pageTitle = ($selectedGame['name'] ?? 'Marketplace') . ' Marketplace';
include 'includes/header.php';

// Game color mapping with RGB values for glow effects
$gameColors = [
    'valorant' => ['#ff4655', '#ff70a6', '255, 70, 85'],
    'league-of-legends' => ['#c89b3c', '#f0e6d2', '200, 155, 60'],
    'gta' => ['#75c949', '#a7eb7d', '117, 201, 73'],
    'pubg' => ['#f5a623', '#f8d676', '245, 166, 35'],
    'brawl-stars' => ['#4ade80', '#86efac', '74, 222, 128'],
    'mobile-legends' => ['#a855f7', '#c084fc', '168, 85, 247'],
];

$selectedCategoryData = $selectedCategory ?? [];
$heroImagePath = $selectedCategoryData['banner_image'] ?? $selectedCategoryData['image'] ?? ($selectedGame['banner_image'] ?? $selectedGame['image'] ?? null);
// Ensure heroImagePath is a string, not an array
if (is_array($heroImagePath)) {
    $heroImagePath = null;
}
$heroTitle = $selectedCategoryData['name'] ?? ($selectedGame['name'] ?? 'Marketplace');
$warningText = $selectedCategoryData['warning_text'] ?? '';
$heroDescription = $selectedCategoryData['description'] ?? ($selectedGame['description'] ?? '');
$categoryServers = $selectedCategoryData['servers'] ?? [];
$categoryServerCount = count($categoryServers);
$categoryTotalStock = 0;
$categoryMinSellPrice = null;
foreach ($categoryServers as $serverStatsRow) {
    $categoryTotalStock += (float)($serverStatsRow['sell_stock'] ?? 0);
    $serverSellPrice = isset($serverStatsRow['sell_price']) ? (float)$serverStatsRow['sell_price'] : null;
    if ($serverSellPrice && ($categoryMinSellPrice === null || $serverSellPrice < $categoryMinSellPrice)) {
        $categoryMinSellPrice = $serverSellPrice;
    }
}
$autoApproveEnabled = !empty($selectedCategoryData['auto_approve_orders']);
$requiresCharacter = !empty($selectedCategoryData['requires_character_name']);
$gameProductColumns = max(1, (int)($selectedGame['product_grid_columns'] ?? 4));
$gameCategoryColumns = max(1, (int)($selectedGame['category_grid_columns'] ?? 3));
$categoryProductColumns = $selectedCategoryData['custom_product_columns'] !== null ? max(1, (int)$selectedCategoryData['custom_product_columns']) : $gameProductColumns;
$categoryGridColumns = $selectedCategoryData['custom_category_columns'] !== null ? max(1, (int)$selectedCategoryData['custom_category_columns']) : $gameCategoryColumns;
$catColsDesktop = $categoryGridColumns;
$catColsTablet = max(1, min($catColsDesktop, 2));
$catColsMobile = 1;
$productColsDesktop = $categoryProductColumns;
$productColsTablet = max(1, min($productColsDesktop, 2));
$productColsMobile = 1;
$heroStyle = $heroImagePath ? ' style="--category-hero-image: url(' . htmlspecialchars(assetUrl($heroImagePath), ENT_QUOTES, 'UTF-8') . ')"' : '';
?>

<section class="category-hero<?= $heroImagePath ? ' category-hero--has-image' : '' ?>"<?= $heroStyle ?>>
    <div class="category-hero__overlay"></div>
    <div class="container-wide category-hero__inner">
        <div class="category-hero__body">
            <span class="category-hero__eyebrow"><i class="fa-solid fa-gamepad"></i><?= sanitize($selectedGame['name']) ?> Marketplace</span>
            <h1><?= sanitize($heroTitle) ?></h1>
            <?php if (!empty($heroDescription)): ?>
                <p><?= sanitize(shortenText($heroDescription, 220)) ?></p>
            <?php endif; ?>
            <div class="category-hero__tags">
                <span class="category-hero__tag"><i class="fa-solid fa-layer-group"></i><?= number_format($categoryServerCount, 0, ',', '.') ?> Sunucu</span>
                <?php if ($categoryTotalStock > 0): ?>
                    <span class="category-hero__tag"><i class="fa-solid fa-box"></i>Toplam Stok: <?= number_format($categoryTotalStock, 0, ',', '.') ?></span>
                <?php endif; ?>
            </div>
        </div>
        <div class="category-hero__panel">
            <div class="category-hero__stats">
                <div class="category-hero__stat">
                    <span>Başlangıç Fiyatı</span>
                    <strong><?= $categoryMinSellPrice !== null ? formatPrice($categoryMinSellPrice) : '—' ?></strong>
                </div>
                <div class="category-hero__stat">
                    <span>Sunucu Sayısı</span>
                    <strong><?= number_format($categoryServerCount, 0, ',', '.') ?></strong>
                </div>
                <div class="category-hero__stat">
                    <span>Stok Durumu</span>
                    <strong><?= $categoryTotalStock > 0 ? number_format($categoryTotalStock, 0, ',', '.') : 'Bilgi Yok' ?></strong>
                </div>
            </div>
        </div>
    </div>
</section>

<?php if (!empty($selectedGame['categories'])): ?>
<section class="subcategory-section">
    <div class="container-wide">
        <div class="subcategory-section__head">
            <div>
                <h3><?= sanitize($selectedGame['name']) ?> Kategorileri</h3>
                <p>Listelediğiniz kategoriyi seçerek ürünleri görüntüleyin.</p>
            </div>
        </div>
        <div class="subcategory-grid" style="--cat-cols-desktop: <?= $catColsDesktop ?>; --cat-cols-tablet: <?= $catColsTablet ?>; --cat-cols-mobile: <?= $catColsMobile ?>;">
            <?php foreach ($selectedGame['categories'] as $category): ?>
                <?php
                    $catImage = $category['image'] ?? $category['banner_image'] ?? null;
                    // Ensure catImage is a string, not an array
                    if (is_array($catImage)) {
                        $catImage = null;
                    }
                    $style = $catImage ? ' style="--subcategory-image: url(' . htmlspecialchars(assetUrl($catImage), ENT_QUOTES, 'UTF-8') . ')"' : '';
                    $isActive = $category['slug'] === $categorySlug;
                ?>
                <a href="<?= htmlspecialchars(routeUrl('marketplace_category', ['game' => $selectedGame['slug'], 'category' => $category['slug']]), ENT_QUOTES, 'UTF-8') ?>" class="subcategory-card<?= $isActive ? ' active' : '' ?>"<?= $style ?>>
                    <div class="subcategory-card__body">
                        <h5><?= sanitize($category['name']) ?></h5>
                        <?php if (!empty($category['description'])): ?>
                            <p><?= sanitize(shortenText($category['description'], 100)) ?></p>
                        <?php endif; ?>
                        <span class="subcategory-card__cta"><i class="fa-solid fa-arrow-right"></i>İlanları Gör</span>
                    </div>
                </a>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<section class="marketplace-section">
    <div class="container-wide">
        <div class="marketplace-toolbar marketplace-toolbar--search-only">
            <form class="product-search product-search--full" method="get">
                <input type="hidden" name="game" value="<?= urlencode($gameSlug) ?>">
                <?php if ($categorySlug): ?><input type="hidden" name="category" value="<?= urlencode($categorySlug) ?>"><?php endif; ?>
                <div class="product-search-field">
                    <i class="fa-solid fa-search"></i>
                    <input type="search" name="search" value="<?= sanitize($search ?? '') ?>" placeholder="Ürün arayın...">
                    <?php if (!empty($search)): ?>
                        <button class="clear-btn" type="button" onclick="this.form.search.value='';this.form.submit();"><i class="fa-solid fa-xmark"></i></button>
                    <?php endif; ?>
                </div>
                <button type="submit" class="search-submit-btn">
                    <i class="fa-solid fa-search"></i>
                    <span>Arama Yap</span>
                </button>
            </form>
        </div>

        <?php
        $servers = $selectedCategoryData['servers'] ?? [];
        if ($search !== '') {
            $servers = array_filter($servers, function ($server) use ($search) {
                return stripos($server['name'], $search) !== false;
            });
        }
        ?>

        <?php if (empty($servers)): ?>
            <div class="marketplace-empty mt-4">
                <i class="fa-solid fa-inbox"></i>
                <h4>Ürün bulunamadı</h4>
                <p>Seçtiğiniz kategoride listelenmiş ürün bulunmuyor. Başka bir kategori seçmeyi deneyin.</p>
            </div>
        <?php else: ?>
            <?php if (!empty($warningText)): ?>
                <div class="alert alert-warning mt-4">
                    <i class="fa-solid fa-circle-exclamation me-2"></i><?= sanitize($warningText) ?>
                </div>
            <?php endif; ?>

            <div class="product-catalog" style="--product-cols-desktop: <?= $productColsDesktop ?>; --product-cols-tablet: <?= $productColsTablet ?>; --product-cols-mobile: <?= $productColsMobile ?>;">
                <?php foreach ($servers as $server): ?>
                <?php
                    $serverImage = !empty($server['image']) ? assetUrl($server['image']) : null;
                    $fallbackImage = $selectedCategoryData['image'] ?? $selectedCategoryData['banner_image'] ?? 'assets/images/category-market.svg';
                    // Ensure fallbackImage is a string, not an array
                    if (is_array($fallbackImage)) {
                        $fallbackImage = 'assets/images/category-market.svg';
                    }
                    $displayImage = $serverImage ?: assetUrl($fallbackImage);
                    $hasBuyPrice = isset($server['buy_price']) && (float)$server['buy_price'] > 0;
                    $requiresCharacter = !empty($selectedCategoryData['requires_character_name']);
                    $autoApproveOrders = !empty($selectedCategoryData['auto_approve_orders']);

                    $specs = [];
                    $specs[] = [
                        'icon' => 'fa-bolt',
                        'label' => 'Satış',
                        'value' => formatPrice($server['sell_price']),
                        'variant' => 'sell'
                    ];
                    if ($hasBuyPrice) {
                        $specs[] = [
                            'icon' => 'fa-hand-holding-dollar',
                            'label' => 'Bize Sat',
                            'value' => formatPrice($server['buy_price']),
                            'variant' => 'buy'
                        ];
                    }
                    $specs[] = [
                        'icon' => 'fa-box',
                        'label' => 'Stok',
                        'value' => number_format((float)$server['sell_stock'], 0, ',', '.'),
                        'variant' => 'stock'
                    ];
                ?>
                <article class="product-card marketplace-card">
                    <div class="product-card__media">
                        <img src="<?= htmlspecialchars($displayImage, ENT_QUOTES, 'UTF-8') ?>" alt="<?= sanitize($server['name']) ?>">
                        <span class="product-card__badge"><i class="fa-solid fa-server"></i><?= sanitize($server['name']) ?></span>
                    </div>
                    <div class="product-card__body">
                        <span class="product-card__category"><?= sanitize($selectedCategoryData['name'] ?? $selectedGame['name']) ?></span>
                        <h5 class="product-card__title"><?= sanitize($server['name']) ?></h5>
                        <?php if (!empty($server['description'])): ?>
                            <p class="product-card__summary"><?= sanitize(shortenText($server['description'], 140)) ?></p>
                        <?php endif; ?>
                        <?php if (!empty($specs)): ?>
                        <div class="marketplace-specs">
                            <?php foreach ($specs as $spec): ?>
                                <?php $variantClass = !empty($spec['variant']) ? ' marketplace-spec--' . htmlspecialchars($spec['variant'], ENT_QUOTES, 'UTF-8') : ''; ?>
                                <div class="marketplace-spec<?= $variantClass ?>">
                                    <span class="marketplace-spec__label"><i class="fa-solid <?= $spec['icon'] ?>"></i><?= sanitize($spec['label']) ?></span>
                                    <strong><?= sanitize($spec['value']) ?></strong>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="product-card__actions">
                        <?php if ($hasBuyPrice): ?>
                        <button class="btn-outline" data-marketplace-action="sell"
                                data-marketplace-server-id="<?= (int)$server['id'] ?>"
                                data-marketplace-name="<?= sanitize($server['name']) ?>"
                                data-marketplace-warning="<?= htmlspecialchars($warningText, ENT_QUOTES, 'UTF-8') ?>"
                                data-marketplace-buy-price="<?= $formatDecimal($server['buy_price']) ?>"
                                data-marketplace-sell-price="<?= $formatDecimal($server['sell_price']) ?>"
                                data-marketplace-min-sell="<?= $formatDecimal($server['min_sell_quantity']) ?>"
                                data-marketplace-max-sell="<?= $formatDecimal($server['max_sell_quantity']) ?>"
                                data-marketplace-min-buy="<?= $formatDecimal($server['min_buy_quantity']) ?>"
                                data-marketplace-max-buy="<?= $formatDecimal($server['max_buy_quantity']) ?>"
                                data-marketplace-requires-character="<?= $requiresCharacter ? '1' : '0' ?>"
                                data-marketplace-auto-approve="<?= $autoApproveOrders ? '1' : '0' ?>">
                                <i class="fa-solid fa-hand-holding-dollar"></i>Bize sat
                            </button>
                        <?php endif; ?>
                        <button class="btn-primary" data-marketplace-action="buy"
                            data-marketplace-server-id="<?= (int)$server['id'] ?>"
                            data-marketplace-name="<?= sanitize($server['name']) ?>"
                            data-marketplace-warning="<?= htmlspecialchars($warningText, ENT_QUOTES, 'UTF-8') ?>"
                            data-marketplace-buy-price="<?= $formatDecimal($server['buy_price']) ?>"
                            data-marketplace-sell-price="<?= $formatDecimal($server['sell_price']) ?>"
                            data-marketplace-min-sell="<?= $formatDecimal($server['min_sell_quantity']) ?>"
                            data-marketplace-max-sell="<?= $formatDecimal($server['max_sell_quantity']) ?>"
                            data-marketplace-min-buy="<?= $formatDecimal($server['min_buy_quantity']) ?>"
                            data-marketplace-max-buy="<?= $formatDecimal($server['max_buy_quantity']) ?>"
                            data-marketplace-requires-character="<?= $requiresCharacter ? '1' : '0' ?>"
                            data-marketplace-auto-approve="<?= $autoApproveOrders ? '1' : '0' ?>">
                            <i class="fa-solid fa-cart-shopping"></i>Sepete ekle
                        </button>
                    </div>
                </article>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</section>

<div class="modal fade" id="marketplaceModal" tabindex="-1" aria-hidden="true" 
     data-cart-url="<?= htmlspecialchars(routeUrl('cart'), ENT_QUOTES, 'UTF-8') ?>"
     data-api-url="<?= htmlspecialchars(publicUrl('marketplace/api/create-order.php'), ENT_QUOTES, 'UTF-8') ?>">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content bg-dark text-white">
            <form method="post" action="<?= htmlspecialchars(routeUrl('cart'), ENT_QUOTES, 'UTF-8') ?>">
                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                <input type="hidden" name="action" value="add_marketplace">
                <input type="hidden" name="server_id" value="">
                <div class="modal-header border-0">
                    <h5 class="modal-title" data-modal-title>Marketplace İşlemi</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Kapat"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-success d-none" data-modal-alert></div>
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <span class="badge bg-primary" data-modal-type>Satın Al</span>
                        <div>
                            <span class="text-white-50">Birim Fiyat:</span>
                            <strong data-modal-price>0 TL</strong>
                        </div>
                    </div>
                    <div class="mb-3" data-modal-character-group>
                        <label class="form-label" data-modal-character-label>Karakter Adı</label>
                        <input type="text" class="form-control" name="character_name">
                        <small class="text-muted d-block mt-1" data-modal-character-hint>Karakter adınızı eksiksiz giriniz.</small>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Miktar</label>
                        <input type="number" class="form-control" name="quantity" data-modal-quantity min="1" step="0.01" value="1" required>
                        <small class="text-muted">Minimum: <span data-modal-min>1</span> • Maksimum: <span data-modal-max>Sınırsız</span></small>
                    </div>
                    <div class="alert alert-warning" data-modal-warning></div>
                    <div class="d-flex justify-content-between">
                        <span class="text-white-50">Ödenecek Tutar:</span>
                        <strong data-modal-total>0 TL</strong>
                    </div>
                </div>
                <div class="modal-footer border-0">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Vazgeç</button>
                    <button type="submit" class="btn btn-primary">Onayla</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
$scriptVersionMarketplace = file_exists(__DIR__ . '/assets/js/marketplace-modal.js') ? filemtime(__DIR__ . '/assets/js/marketplace-modal.js') : time();
?>
<script>window.__MARKETPLACE_MODAL_OVERRIDE = true;</script>
<script src="<?= htmlspecialchars(publicUrl('assets/js/marketplace-modal.js'), ENT_QUOTES, 'UTF-8') ?>?v=<?= $scriptVersionMarketplace ?>"></script>
<?php include 'includes/footer.php'; ?>
