<?php
require_once '../../config.php';

header('Content-Type: application/json');

function marketplaceRespond(bool $success, string $message, array $extra = []): void {
    echo json_encode(array_merge([
        'success' => $success,
        'message' => $message
    ], $extra));
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    marketplaceRespond(false, 'Geçersiz istek.');
}

if (!isLoggedIn()) {
    marketplaceRespond(false, 'İşlem için giriş yapmalısınız.');
}

$input = $_POST;
if (empty($input) && ($json = file_get_contents('php://input'))) {
    $decoded = json_decode($json, true);
    if (is_array($decoded)) {
        $input = $decoded;
    }
}

$csrfToken = $input['csrf_token'] ?? '';
if (!verifyCSRFToken($csrfToken)) {
    marketplaceRespond(false, 'Güvenlik doğrulaması başarısız. Lütfen sayfayı yenileyip tekrar deneyin.');
}

$serverId = isset($input['server_id']) ? (int)$input['server_id'] : 0;
$orderType = $input['order_type'] ?? '';
$quantity = isset($input['quantity']) ? (float)$input['quantity'] : 0;
$characterName = trim($input['character_name'] ?? '');

if ($serverId <= 0 || !in_array($orderType, ['buy', 'sell'], true)) {
    error_log('[Marketplace] Invalid parameters for create-order: server=' . $serverId . ' type=' . $orderType);
    marketplaceRespond(false, 'İşlem başlatılamadı. Lütfen sayfayı yenileyip tekrar deneyin.');
}

$server = getMarketplaceServer($pdo, $serverId, true);
if (!$server) {
    marketplaceRespond(false, 'Seçilen sunucu bulunamadı veya pasif durumda.');
}

$requiresCharacter = (int)($server['requires_character_name'] ?? 1) === 1;
$autoApproveOrders = (int)($server['auto_approve_orders'] ?? 0) === 1;

if ($requiresCharacter) {
    if ($characterName === '') {
        marketplaceRespond(false, 'Lütfen karakter adınızı girin.');
    }
    if (mb_strlen($characterName) > 150) {
        marketplaceRespond(false, 'Karakter adı 150 karakterden uzun olamaz.');
    }
    $characterName = mb_substr($characterName, 0, 150);
} else {
    if ($characterName !== '') {
        if (mb_strlen($characterName) > 150) {
            marketplaceRespond(false, 'Karakter adı 150 karakterden uzun olamaz.');
        }
        $characterName = mb_substr($characterName, 0, 150);
    } else {
        $characterName = null;
    }
}

$pricePerUnit = $orderType === 'buy' ? (float)$server['sell_price'] : (float)$server['buy_price'];
if ($pricePerUnit <= 0) {
    marketplaceRespond(false, 'Bu işlem için fiyat tanımlanmamış.');
}

if ($quantity <= 0) {
    marketplaceRespond(false, 'Miktar sıfırdan büyük olmalıdır.');
}

$minQuantity = $orderType === 'buy' ? (float)$server['min_buy_quantity'] : (float)$server['min_sell_quantity'];
$maxQuantityRaw = $orderType === 'buy' ? $server['max_buy_quantity'] : $server['max_sell_quantity'];
$maxQuantity = $maxQuantityRaw !== null ? (float)$maxQuantityRaw : null;

if ($quantity < $minQuantity) {
    marketplaceRespond(false, 'Minimum miktar: ' . number_format($minQuantity, 2, ',', '.'));
}

if ($maxQuantity !== null && $quantity > $maxQuantity) {
    marketplaceRespond(false, 'Maksimum miktar: ' . number_format($maxQuantity, 2, ',', '.'));
}

// Stok kontrolü - ZORUNLU
if ($orderType === 'buy') {
    $deliveryType = $server['delivery_type'] ?? 'auto';
    
    error_log('[MARKETPLACE STOK] Server ID: ' . $serverId . ' | Delivery Type: ' . $deliveryType . ' | Quantity: ' . $quantity);
    
    // Otomatik teslimat için gerçek kod stoğunu kontrol et
    if ($deliveryType === 'auto') {
        $quantityRequired = (int)max(1, ceil($quantity));
        $codeCountStmt = $pdo->prepare('SELECT COUNT(*) FROM game_marketplace_codes WHERE server_id = ? AND is_sold = 0');
        $codeCountStmt->execute([$serverId]);
        $availableCodes = (int)$codeCountStmt->fetchColumn();
        
        error_log('[MARKETPLACE STOK] Available Codes: ' . $availableCodes . ' | Required: ' . $quantityRequired);
        
        if ($availableCodes < $quantityRequired) {
            error_log('[MARKETPLACE STOK] BLOCKING ORDER - Insufficient codes');
            marketplaceRespond(false, 'Yeterli kod stoğu bulunmuyor. Mevcut stok: ' . $availableCodes . ' adet, talep edilen: ' . $quantityRequired . ' adet.');
        }
        
        error_log('[MARKETPLACE STOK] Stock check passed - Order will proceed');
    } else {
        error_log('[MARKETPLACE STOK] Manual delivery mode - no code check');
        // Manuel teslimat için sell_stock kontrolü
        $available = (float)$server['sell_stock'];
        if ($available > 0 && $quantity > $available) {
            marketplaceRespond(false, 'Yeterli stok bulunmuyor. Mevcut stok: ' . number_format($available, 2, ',', '.'));
        }
    }
}

$totalAmount = round($pricePerUnit * $quantity, 2);

$status = 'pending';
$deliveredCodes = [];
$autoApprovalApplied = false;
$orderId = null;
$userBalanceBefore = null;

try {
    $pdo->beginTransaction();

    if ($orderType === 'buy') {
        $balanceStmt = $pdo->prepare('SELECT balance FROM users WHERE id = ? FOR UPDATE');
        $balanceStmt->execute([$_SESSION['user_id']]);
        $userBalanceBefore = (float)($balanceStmt->fetchColumn() ?? 0);
        if ($totalAmount > $userBalanceBefore + 0.0001) {
            $pdo->rollBack();
            marketplaceRespond(false, 'Bakiyeniz yetersiz. Gerekli tutar: ' . formatPrice($totalAmount));
        }
    }

    $stmt = $pdo->prepare('INSERT INTO game_marketplace_orders (user_id, marketplace_id, category_id, server_id, order_type, status, quantity, price_per_unit, total_amount, currency, character_name, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())');
    $stmt->execute([
        $_SESSION['user_id'],
        $server['marketplace_id'],
        $server['category_id'],
        $serverId,
        $orderType,
        $status,
        $quantity,
        $pricePerUnit,
        $totalAmount,
        'TRY',
        $characterName
    ]);

    $orderId = (int)$pdo->lastInsertId();

    if ($orderType === 'buy') {
        $balanceUpdate = $pdo->prepare('UPDATE users SET balance = balance - ? WHERE id = ?');
        $balanceUpdate->execute([$totalAmount, $_SESSION['user_id']]);

        $insertTransaction = $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description, order_id, created_at) VALUES (?, ?, ?, ?, ?, NOW())');
        $insertTransaction->execute([
            $_SESSION['user_id'],
            'purchase',
            -$totalAmount,
            'Marketplace satın alma #' . $orderId,
            null
        ]);

        if ((float)$server['sell_stock'] > 0) {
            $updateStock = $pdo->prepare('UPDATE game_marketplace_servers SET sell_stock = GREATEST(sell_stock - ?, 0) WHERE id = ?');
            $updateStock->execute([$quantity, $serverId]);
        }
    }

    // Otomatik teslimat kontrolü (sadece buy siparişleri için)
    $deliveryType = $server['delivery_type'] ?? 'auto';
    if ($orderType === 'buy' && $deliveryType === 'auto') {
        $quantityRequired = (int)max(1, ceil($quantity));
        
        // Marketplace ürün kodlarından mevcut kodları çek
        $codeLookupSql = 'SELECT id, code FROM game_marketplace_codes WHERE server_id = ? AND is_sold = 0 ORDER BY id ASC LIMIT ' . $quantityRequired . ' FOR UPDATE';
        $codeLookup = $pdo->prepare($codeLookupSql);
        $codeLookup->execute([$serverId]);
        $codesToAssign = $codeLookup->fetchAll();

        // Double check: Eğer yeterli kod yoksa (bu olmamalı çünkü önceden kontrol ettik)
        if (count($codesToAssign) < $quantityRequired) {
            $pdo->rollBack();
            marketplaceRespond(false, 'Yeterli kod stoğu bulunamadı. Mevcut stok: ' . count($codesToAssign) . ' adet, talep edilen: ' . $quantityRequired . ' adet.');
        }

        // Kodları kullanıcıya ata
        $assignStmt = $pdo->prepare('UPDATE game_marketplace_codes SET is_sold = 1, sold_to_user_id = ?, sold_at = NOW(), order_id = ? WHERE id = ?');
        foreach ($codesToAssign as $codeRow) {
            $assignStmt->execute([$_SESSION['user_id'], $orderId, $codeRow['id']]);
            $deliveredCodes[] = $codeRow['code'];
        }

        // Siparişi tamamla
        $updateOrder = $pdo->prepare('UPDATE game_marketplace_orders SET status = ?, processed_at = NOW(), updated_at = NOW() WHERE id = ?');
        $updateOrder->execute(['completed', $orderId]);

        $status = 'completed';
        $autoApprovalApplied = true;
    }
    // Manuel teslimat - sipariş pending kalacak, admin panelden elle kod eklenecek

    $pdo->commit();

    $balanceAfterStmt = $pdo->prepare('SELECT balance FROM users WHERE id = ?');
    $balanceAfterStmt->execute([$_SESSION['user_id']]);
    $balanceAfter = (float)($balanceAfterStmt->fetchColumn() ?? 0);

    $marketplaceTotals = [
        'buy_volume' => 0.0,
        'sell_volume' => 0.0
    ];
    $marketplaceTotalsStmt = $pdo->prepare("SELECT order_type, COALESCE(SUM(total_amount),0) AS total FROM game_marketplace_orders WHERE user_id = ? AND status IN ('completed','approved') GROUP BY order_type");
    $marketplaceTotalsStmt->execute([$_SESSION['user_id']]);
    foreach ($marketplaceTotalsStmt->fetchAll() as $row) {
        if ($row['order_type'] === 'buy') {
            $marketplaceTotals['buy_volume'] = (float)$row['total'];
        } elseif ($row['order_type'] === 'sell') {
            $marketplaceTotals['sell_volume'] = (float)$row['total'];
        }
    }

    if ($orderType === 'sell') {
        $adminEmail = getSetting('smtp_admin_email', SMTP_FROM_EMAIL);
        $subject = 'Yeni Bize Sat Talebi';
        $body = '<p><strong>' . sanitize($_SESSION['username']) . '</strong> adlı kullanıcı <strong>' . sanitize($server['name']) . '</strong> için <strong>' . number_format($quantity, 2, ',', '.') . '</strong> adet bize sat talebi oluşturdu.</p>';
        $body .= '<p>Toplam Tutar: <strong>' . formatPrice($totalAmount) . '</strong></p>';
        sendMarketplaceEmail($adminEmail, $subject, $body);
    }

    if ($autoApprovalApplied && $orderId) {
        $userStmt = $pdo->prepare('SELECT email, username FROM users WHERE id = ?');
        $userStmt->execute([$_SESSION['user_id']]);
        if ($user = $userStmt->fetch()) {
            $subject = 'Marketplace siparişiniz tamamlandı';
            $body = '<p>Merhaba <strong>' . sanitize($user['username']) . '</strong>,</p>';
            $body .= '<p>' . sanitize($server['name']) . ' için verdiğiniz ' . number_format($quantity, 2, ',', '.') . ' adet sipariş tamamlandı. Toplam tutar: ' . formatPrice($totalAmount) . '.</p>';
            if (!empty($deliveredCodes)) {
                $body .= '<p><strong>Teslim edilen kodlar:</strong><br>' . implode('<br>', array_map('sanitize', $deliveredCodes)) . '</p>';
            }
            sendMarketplaceEmail($user['email'], $subject, $body);
        }
    }

    $summary = [
        'balance' => $balanceAfter,
        'balance_before' => $userBalanceBefore,
        'marketplace_buy_volume' => $marketplaceTotals['buy_volume'],
        'marketplace_sell_volume' => $marketplaceTotals['sell_volume']
    ];

    marketplaceRespond(true, $orderType === 'sell'
        ? 'Talebiniz alınmıştır. Onaylandığında bakiyenize yansıyacak.'
        : ($autoApprovalApplied ? 'Siparişiniz onaylandı ve kodlarınız teslim edildi.' : 'Siparişiniz oluşturuldu. En kısa sürede işleme alınacaktır.'), [
        'order_id' => $orderId,
        'status' => $status,
        'quantity' => $quantity,
        'unit_price' => $pricePerUnit,
        'total_amount' => $totalAmount,
        'order_type' => $orderType,
        'delivered_codes' => $deliveredCodes,
        'summary' => $summary
    ]);
} catch (Throwable $th) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log('[Marketplace] create-order error: ' . $th->getMessage());
    marketplaceRespond(false, 'İşlem sırasında bir hata oluştu. Lütfen tekrar deneyin.');
}
