<?php
require_once 'config.php';
requireLogin();

// Pagination
$perPage = 10;
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;

// Get total count (normal orders + marketplace orders)
$countStmt = $pdo->prepare('SELECT COUNT(DISTINCT o.id) FROM orders o WHERE o.user_id = ?');
$countStmt->execute([$_SESSION['user_id']]);
$totalOrders = (int)$countStmt->fetchColumn();

$mpCountStmt = $pdo->prepare('SELECT COUNT(*) FROM game_marketplace_orders WHERE user_id = ?');
$mpCountStmt->execute([$_SESSION['user_id']]);
$totalMarketplaceOrders = (int)$mpCountStmt->fetchColumn();

$totalAllOrders = $totalOrders + $totalMarketplaceOrders;
$totalPages = max(1, (int)ceil($totalAllOrders / $perPage));
if ($page > $totalPages) {
    $page = $totalPages;
}
$offset = ($page - 1) * $perPage;

// Get normal orders
$orderStmt = $pdo->prepare('
    SELECT o.*, COUNT(oi.id) AS item_count, "normal" AS order_source
    FROM orders o
    LEFT JOIN order_items oi ON o.id = oi.order_id
    WHERE o.user_id = ?
    GROUP BY o.id
    ORDER BY o.created_at DESC
');
$orderStmt->execute([$_SESSION['user_id']]);
$orders = $orderStmt->fetchAll();

// Get marketplace orders
$mpOrderStmt = $pdo->prepare('
    SELECT mo.*, 
           s.name AS server_name, 
           c.name AS category_name, 
           m.name AS marketplace_name,
           "marketplace" AS order_source,
           (SELECT GROUP_CONCAT(mc.code ORDER BY mc.id SEPARATOR ", ") FROM game_marketplace_codes mc WHERE mc.order_id = mo.id) AS delivered_codes
    FROM game_marketplace_orders mo
    LEFT JOIN game_marketplace_servers s ON s.id = mo.server_id
    LEFT JOIN game_marketplace_categories c ON c.id = mo.category_id
    LEFT JOIN game_marketplaces m ON m.id = mo.marketplace_id
    WHERE mo.user_id = ?
    ORDER BY mo.created_at DESC
');
$mpOrderStmt->execute([$_SESSION['user_id']]);
$marketplaceOrders = $mpOrderStmt->fetchAll();

// Merge and sort all orders by created_at
$allOrders = array_merge($orders, $marketplaceOrders);
usort($allOrders, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

// Paginate the merged array
$paginatedOrders = array_slice($allOrders, $offset, $perPage);

$pageTitle = 'Siparişlerim';
include 'includes/header.php';
?>

<section class="orders-section">
    <div class="container-wide">
        <div class="section-heading">
            <h2 class="section-title">Siparişlerim</h2>
            <p class="section-subtitle">Tüm siparişlerinizi ve teslim edilen kodları buradan görüntüleyebilirsiniz.</p>
        </div>

        <?php if (empty($paginatedOrders)): ?>
            <div class="orders-empty">
                <i class="fa-solid fa-bag-shopping fa-3x mb-3"></i>
                <h4 class="text-white mb-2">Henüz siparişiniz yok</h4>
                <p class="mb-4">Hemen ürünlerimize göz atın, avantajlı fiyatlarla dijital kodlara ulaşın.</p>
                <a href="products.php" class="btn btn-primary"><i class="fa-solid fa-cart-shopping me-2"></i>Alışverişe Başla</a>
            </div>
        <?php else: ?>
            <?php foreach ($paginatedOrders as $order): ?>
                <?php
                $isMarketplace = ($order['order_source'] ?? 'normal') === 'marketplace';
                
                if ($isMarketplace) {
                    // Marketplace order
                    $orderItems = [];
                } else {
                    // Normal order
                    $itemsStmt = $pdo->prepare('
                        SELECT oi.*, pc.code
                        FROM order_items oi
                        LEFT JOIN product_codes pc ON oi.product_code_id = pc.id
                        WHERE oi.order_id = ?
                    ');
                    $itemsStmt->execute([$order['id']]);
                    $orderItems = $itemsStmt->fetchAll();
                }

                $statusClass = 'status-pending';
                if ($order['status'] === 'completed') {
                    $statusClass = 'status-completed';
                } elseif ($order['status'] === 'cancelled') {
                    $statusClass = 'status-cancelled';
                }
                ?>
                <article class="order-card">
                    <div class="order-head">
                        <div>
                            <?php if ($isMarketplace): ?>
                                <h5>
                                    <span class="badge bg-info me-2"><i class="fa-solid fa-store"></i></span>
                                    <?= sanitize($order['marketplace_name'] ?? 'Marketplace') ?>
                                </h5>
                                <div class="order-meta">
                                    <span><i class="fa-solid fa-calendar me-1"></i><?= date('d.m.Y H:i', strtotime($order['created_at'])) ?></span>
                                    <span><i class="fa-solid fa-server me-1"></i><?= sanitize($order['server_name'] ?? '-') ?></span>
                                    <span>
                                        <?php if ($order['order_type'] === 'sell'): ?>
                                            <span class="badge bg-warning">Bize Sat</span>
                                        <?php else: ?>
                                            <span class="badge bg-success">Satın Al</span>
                                        <?php endif; ?>
                                    </span>
                                </div>
                            <?php else: ?>
                                <h5><i class="fa-solid fa-hashtag me-2"></i><?= sanitize($order['order_number']) ?></h5>
                                <div class="order-meta">
                                    <span><i class="fa-solid fa-calendar me-1"></i><?= date('d.m.Y H:i', strtotime($order['created_at'])) ?></span>
                                    <span><i class="fa-solid fa-box me-1"></i><?= (int)($order['item_count'] ?? 0) ?> ürün</span>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="text-end">
                            <span class="status-badge <?= $statusClass ?>">
                                <?php if ($order['status'] === 'completed'): ?>
                                    <i class="fa-solid fa-check-circle"></i>Tamamlandı
                                <?php elseif ($order['status'] === 'pending'): ?>
                                    <i class="fa-solid fa-clock"></i>Beklemede
                                <?php else: ?>
                                    <i class="fa-solid fa-xmark"></i>İptal
                                <?php endif; ?>
                            </span>
                            <div class="order-total"><?= formatPrice($order['total_amount']) ?></div>
                        </div>
                    </div>
                    <div class="order-body">
                        <?php if ($isMarketplace): ?>
                            <!-- Marketplace Order Details -->
                            <div class="marketplace-order-details">
                                <div class="row g-3">
                                    <div class="col-md-3">
                                        <div class="detail-item">
                                            <span class="detail-label">Oyun</span>
                                            <span class="detail-value"><?= sanitize($order['marketplace_name'] ?? '-') ?></span>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="detail-item">
                                            <span class="detail-label">Kategori</span>
                                            <span class="detail-value"><?= sanitize($order['category_name'] ?? '-') ?></span>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="detail-item">
                                            <span class="detail-label">Miktar</span>
                                            <span class="detail-value"><?= number_format((float)($order['quantity'] ?? 0), 2, ',', '.') ?></span>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="detail-item">
                                            <span class="detail-label">Birim Fiyat</span>
                                            <span class="detail-value"><?= formatPrice($order['price_per_unit'] ?? 0) ?></span>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="detail-item">
                                            <span class="detail-label">Toplam</span>
                                            <span class="detail-value"><strong><?= formatPrice($order['total_amount'] ?? 0) ?></strong></span>
                                        </div>
                                    </div>
                                </div>
                                <?php if (!empty($order['character_name'])): ?>
                                <div class="mt-3">
                                    <span class="detail-label"><i class="fa-solid fa-user me-1"></i>Karakter Adı:</span>
                                    <span class="detail-value"><strong><?= sanitize($order['character_name']) ?></strong></span>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($order['delivered_codes'])): ?>
                                <div class="mt-3">
                                    <span class="detail-label"><i class="fa-solid fa-key me-1"></i>Teslim Edilen Kodlar:</span>
                                    <div class="mt-2">
                                        <?php 
                                        $codes = explode(', ', $order['delivered_codes']);
                                        foreach ($codes as $code): 
                                        ?>
                                            <span class="code-badge me-2 mb-2"><?= sanitize($code) ?></span>
                                            <button type="button" class="btn btn-sm btn-outline-primary copy-action" data-copy-code="<?= sanitize($code) ?>">
                                                <i class="fa-solid fa-copy"></i>
                                            </button>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                            </div>
                        <?php else: ?>
                            <!-- Normal Order Details -->
                            <div class="table-responsive">
                                <table class="order-items-table">
                                    <thead>
                                        <tr>
                                            <th>Ürün</th>
                                            <th class="text-center">Adet</th>
                                            <th class="text-center">Fiyat</th>
                                            <th>Kod</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($orderItems as $item): ?>
                                        <tr>
                                            <td><strong><?= sanitize($item['product_name']) ?></strong></td>
                                            <td class="text-center"><?= (int)$item['quantity'] ?></td>
                                            <td class="text-center"><?= formatPrice($item['price']) ?></td>
                                            <td>
                                                <?php if (!empty($item['code'])): ?>
                                                    <span class="code-badge">
                                                        <?= sanitize($item['code']) ?>
                                                    </span>
                                                    <button type="button" class="btn btn-sm btn-outline-primary copy-action" data-copy-code="<?= sanitize($item['code']) ?>">
                                                        <i class="fa-solid fa-copy"></i>
                                                    </button>
                                                <?php else: ?>
                                                    <span class="text-white-50">Kod henüz tanımlanmadı</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </article>
            <?php endforeach; ?>
            
            <?php if ($totalPages > 1): ?>
                <nav class="pagination-nav mt-4">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="<?= routeUrl('orders') ?>?page=<?= $page - 1 ?>">
                                    <i class="fa-solid fa-chevron-left"></i>
                                </a>
                            </li>
                        <?php endif; ?>

                        <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                            <li class="page-item<?= $i === $page ? ' active' : '' ?>">
                                <a class="page-link" href="<?= routeUrl('orders') ?>?page=<?= $i ?>"><?= $i ?></a>
                            </li>
                        <?php endfor; ?>

                        <?php if ($page < $totalPages): ?>
                            <li class="page-item">
                                <a class="page-link" href="<?= routeUrl('orders') ?>?page=<?= $page + 1 ?>">
                                    <i class="fa-solid fa-chevron-right"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</section>

<style>
.marketplace-order-details {
    padding: 1.5rem;
    background: rgba(0, 0, 0, 0.05);
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 12px;
}

body[data-theme="dark"] .marketplace-order-details {
    background: rgba(255, 255, 255, 0.03);
    border-color: rgba(255, 255, 255, 0.1);
}

.detail-item {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.detail-label {
    font-size: 0.85rem;
    color: rgba(0, 0, 0, 0.6);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

body[data-theme="dark"] .detail-label {
    color: rgba(255, 255, 255, 0.6);
}

.detail-value {
    font-size: 1rem;
    color: #1a202c;
    font-weight: 500;
}

body[data-theme="dark"] .detail-value {
    color: #fff;
}

.code-badge {
    display: inline-flex;
    align-items: center;
    padding: 0.5rem 1rem;
    background: rgba(99, 102, 241, 0.1);
    border: 1px solid rgba(99, 102, 241, 0.3);
    border-radius: 8px;
    color: #5b21b6;
    font-family: 'Courier New', monospace;
    font-size: 0.9rem;
    font-weight: 600;
}

body[data-theme="dark"] .code-badge {
    color: #818cf8;
}
</style>

<?php include 'includes/footer.php'; ?>
