<?php
/**
 * PayTR Payment Gateway Integration
 * PHP 8.2+ Compatible
 */

class PayTR {
    private $merchant_id;
    private $merchant_key;
    private $merchant_salt;
    private $test_mode;
    private $ok_url;
    private $fail_url;
    private $callback_url;
    
    public function __construct() {
        $this->merchant_id = getSetting('paytr_merchant_id', PAYTR_MERCHANT_ID);
        $this->merchant_key = getSetting('paytr_merchant_key', PAYTR_MERCHANT_KEY);
        $this->merchant_salt = getSetting('paytr_merchant_salt', PAYTR_MERCHANT_SALT);
        $this->test_mode = getBoolSetting('paytr_test_mode', PAYTR_TEST_MODE);
        
        // Admin paneldeki site URL'sini al
        $siteBaseUrl = rtrim(getSetting('site_base_url', SITE_URL), '/');
        
        $this->ok_url = $this->resolveUrl(getSetting('paytr_ok_url', ''), $siteBaseUrl . '/balance.php?payment=success');
        $this->fail_url = $this->resolveUrl(getSetting('paytr_fail_url', ''), $siteBaseUrl . '/balance.php?payment=failed');
        $this->callback_url = $this->resolveUrl(getSetting('paytr_callback_url', ''), $siteBaseUrl . '/payment/paytr-callback.php');
    }
    
    public function isConfigured(): bool {
        return !empty($this->merchant_id) && !empty($this->merchant_key) && !empty($this->merchant_salt);
    }
    
    /**
     * Ödeme formu oluştur
     */
    public function createPaymentForm($orderData) {
        if (!$this->isConfigured()) {
            throw new Exception('PayTR API bilgileri eksik. Lütfen yönetici panelinden ayarları güncelleyin.');
        }
        
        $merchant_id = $this->merchant_id;
        $merchant_key = $this->merchant_key;
        $merchant_salt = $this->merchant_salt;
        
        // Kullanıcı sepet bilgileri
        $user_basket = base64_encode(json_encode($orderData['basket']));
        
        // Ödeme bilgileri
        $merchant_oid = $orderData['order_id'];
        
        // Merchant OID validasyonu (max 64 karakter)
        if (strlen($merchant_oid) > 64) {
            throw new Exception('Sipariş numarası çok uzun (max 64 karakter).');
        }
        
        $email = $orderData['email'];
        
        // Email validasyonu
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Geçersiz e-posta adresi.');
        }
        
        $payment_amount = $orderData['amount'] * 100; // Kuruş cinsine çevir
        
        // Tutar validasyonu
        if ($payment_amount < 100) { // Minimum 1 TL
            throw new Exception('Minimum ödeme tutarı 1 TL olmalıdır.');
        }
        $user_name = $orderData['name'];
        $user_address = $orderData['address'] ?? 'Adres bilgisi yok';
        
        // Telefon numarası validasyonu ve temizleme
        $user_phone = preg_replace('/[^0-9]/', '', $orderData['phone'] ?? '');
        
        // Eğer telefon boş veya geçersizse varsayılan değer kullan
        if (empty($user_phone) || strlen($user_phone) < 10) {
            $user_phone = '5555555555'; // PayTR geçerli bir numara bekler
        }
        
        // Başında 0 varsa kaldır
        $user_phone = ltrim($user_phone, '0');
        
        // 10 haneli Türkiye telefon numarası formatı
        if (strlen($user_phone) === 10 && substr($user_phone, 0, 1) === '5') {
            // Geçerli format (5XXXXXXXXX)
        } elseif (strlen($user_phone) === 11 && substr($user_phone, 0, 2) === '90') {
            // 90 ile başlıyorsa kaldır (905XXXXXXXXX -> 5XXXXXXXXX)
            $user_phone = substr($user_phone, 2);
        } else {
            // Geçersiz format, varsayılan kullan
            $user_phone = '5555555555';
        }
        $merchant_ok_url = $this->buildReturnHandlerUrl('ok');
        $merchant_fail_url = $this->buildReturnHandlerUrl('fail');
        $user_ip = $_SERVER['REMOTE_ADDR'];
        $timeout_limit = "30";
        $debug_on = $this->test_mode ? 1 : 0;
        $test_mode = $this->test_mode ? 1 : 0;
        $no_installment = 0; // Taksit aktif
        $max_installment = 12;
        $currency = "TL";
        $lang = "tr";
        
        // Hash oluştur
        $hash_str = $merchant_id . $user_ip . $merchant_oid . $email . $payment_amount . $user_basket . $no_installment . $max_installment . $currency . $test_mode;
        $paytr_token = base64_encode(hash_hmac('sha256', $hash_str . $merchant_salt, $merchant_key, true));
        
        // PayTR iframe token al
        $post_vals = [
            'merchant_id' => $merchant_id,
            'user_ip' => $user_ip,
            'merchant_oid' => $merchant_oid,
            'email' => $email,
            'payment_amount' => $payment_amount,
            'paytr_token' => $paytr_token,
            'user_basket' => $user_basket,
            'debug_on' => $debug_on,
            'no_installment' => $no_installment,
            'max_installment' => $max_installment,
            'user_name' => $user_name,
            'user_address' => $user_address,
            'user_phone' => $user_phone,
            'merchant_ok_url' => $merchant_ok_url,
            'merchant_fail_url' => $merchant_fail_url,
            'timeout_limit' => $timeout_limit,
            'currency' => $currency,
            'test_mode' => $test_mode,
            'lang' => $lang
        ];
        
        // Parametreleri logla (güvenlik için token'ı maskeliyoruz)
        $logFile = __DIR__ . '/../logs/paytr_' . date('Y-m-d') . '_request.log';
        $logData = $post_vals;
        $logData['paytr_token'] = substr($paytr_token, 0, 20) . '...';
        $logData['merchant_key'] = substr($merchant_key, 0, 10) . '...';
        $logData['merchant_salt'] = substr($merchant_salt, 0, 10) . '...';
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - REQUEST: " . json_encode($logData, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) . "\n", FILE_APPEND);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://www.paytr.com/odeme/api/get-token");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post_vals);
        curl_setopt($ch, CURLOPT_FRESH_CONNECT, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        
        $result = curl_exec($ch);
        
        if (curl_errno($ch)) {
            $curlError = curl_error($ch);
            curl_close($ch);
            throw new Exception('PayTR bağlantı hatası: ' . $curlError);
        }
        
        curl_close($ch);
        
        // Raw response'u logla
        $logFile = __DIR__ . '/../logs/paytr_' . date('Y-m-d') . '_raw.log';
        @file_put_contents($logFile, date('Y-m-d H:i:s') . " - RAW_RESPONSE: " . $result . "\n", FILE_APPEND);
        
        $result = json_decode($result, true);
        
        if (!is_array($result)) {
            return [
                'success' => false,
                'error' => 'PayTR\'den geçersiz yanıt alındı. Lütfen API bilgilerinizi kontrol edin.'
            ];
        }
        
        if ($result['status'] == 'success') {
            return [
                'success' => true,
                'token' => $result['token']
            ];
        } else {
            return [
                'success' => false,
                'error' => $result['reason'] ?? 'Bilinmeyen PayTR hatası'
            ];
        }
    }

    private function resolveUrl(?string $value, string $fallback): string {
        $value = trim((string)$value);
        if ($value === '' || !filter_var($value, FILTER_VALIDATE_URL)) {
            return $fallback;
        }
        return $value;
    }
    
    private function buildReturnHandlerUrl(string $type): string {
        $target = $type === 'ok' ? $this->ok_url : $this->fail_url;
        
        // Admin paneldeki site_base_url'i al
        $siteBaseUrl = rtrim(getSetting('site_base_url', SITE_URL), '/');
        $siteBaseParts = parse_url($siteBaseUrl);
        
        $targetParts = parse_url($target);
        $fallbackParts = $siteBaseParts ?: parse_url($this->callback_url) ?: parse_url($this->ok_url);

        $scheme = $targetParts['scheme'] ?? $fallbackParts['scheme'] ?? ((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http');
        $host = $targetParts['host'] ?? $fallbackParts['host'] ?? ($_SERVER['HTTP_HOST'] ?? 'localhost');
        $port = isset($targetParts['port']) ? ':' . $targetParts['port'] : (isset($fallbackParts['port']) ? ':' . $fallbackParts['port'] : '');

        // Base path'i site_base_url'den al
        $basePath = isset($siteBaseParts['path']) ? rtrim($siteBaseParts['path'], '/') : '';

        $handlerPath = $basePath . '/payment/paytr-return.php';

        $redirectParam = rawurlencode(base64_encode($target));

        return $scheme . '://' . $host . $port . $handlerPath . '?type=' . urlencode($type) . '&redirect=' . $redirectParam;
    }

    /**
     * Callback doğrulama
     */
    public function verifyCallback($post_data) {
        if (!$this->isConfigured()) {
            return [
                'success' => false,
                'error' => 'PayTR API bilgileri eksik'
            ];
        }
        
        $merchant_key = $this->merchant_key;
        $merchant_salt = $this->merchant_salt;
        
        $hash = base64_encode(hash_hmac('sha256', 
            $post_data['merchant_oid'] . $merchant_salt . $post_data['status'] . $post_data['total_amount'], 
            $merchant_key, true));
        
        if ($hash != $post_data['hash']) {
            return [
                'success' => false,
                'error' => 'Hash doğrulama hatası'
            ];
        }
        
        return [
            'success' => true,
            'merchant_oid' => $post_data['merchant_oid'],
            'status' => $post_data['status'],
            'total_amount' => $post_data['total_amount'],
            'payment_type' => $post_data['payment_type'] ?? 'card',
            'installment_count' => $post_data['installment_count'] ?? 1
        ];
    }
}
