<?php
require_once '../config.php';
requireLogin();

header('Content-Type: application/json; charset=UTF-8');

try {
    // Havale/EFT özelliği aktif mi kontrol et
    $bankTransferEnabled = getBoolSetting('bank_transfer_enabled', false);
    if (!$bankTransferEnabled) {
        throw new Exception('Havale/EFT ödeme yöntemi şu anda aktif değil.');
    }

    // POST verilerini kontrol et
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Geçersiz istek metodu.');
    }

    // Güvenlik kontrolü
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        throw new Exception('Güvenlik doğrulaması başarısız. Lütfen sayfayı yenileyip tekrar deneyin.');
    }

    // Form verilerini al
    $amount = filter_var($_POST['amount'] ?? 0, FILTER_VALIDATE_FLOAT);
    $senderName = trim($_POST['sender_name'] ?? '');
    $transferNote = trim($_POST['transfer_note'] ?? '');

    // Validasyon
    if (!$amount || $amount < 10) {
        throw new Exception('Minimum yükleme tutarı 10 TL olmalıdır.');
    }

    if (empty($senderName)) {
        throw new Exception('Gönderen ad soyad bilgisi gereklidir.');
    }

    // Kullanıcı bilgisini al
    $userId = $_SESSION['user_id'];
    $user = getUserInfo($pdo, $userId);

    // Komisyon hesapla
    $commissionRate = (float)getCommissionRate('bank_transfer');
    $commissionAmount = ($amount * $commissionRate) / 100;
    $totalAmount = $amount + $commissionAmount;

    // Benzersiz sipariş numarası oluştur
    $orderNumber = 'BT-' . strtoupper(uniqid());

    // Site URL'sini önceden hazırla
    $siteUrl = rtrim(getSetting('site_base_url', SITE_URL), '/');
    
    // Transaction başlat
    $pdo->beginTransaction();

    // Order kaydı oluştur
    $orderStmt = $pdo->prepare('
        INSERT INTO orders (
            user_id, order_number, order_type, payment_method, 
            total_amount, payable_amount, commission_amount,
            status, created_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ');
    $orderStmt->execute([
        $userId,
        $orderNumber,
        'deposit',
        'bank_transfer',
        $amount,
        $totalAmount,
        $commissionAmount,
        'pending' // Admin onayı bekleyecek
    ]);

    $orderId = $pdo->lastInsertId();

    // Transaction kaydı oluştur (pending olarak)
    $description = 'Havale/EFT ile bakiye yükleme - Gönderen: ' . $senderName;
    if (!empty($transferNote)) {
        $description .= ' - Not: ' . $transferNote;
    }

    $txnStmt = $pdo->prepare('
        INSERT INTO transactions (
            user_id, order_id, type, amount, 
            description, created_at
        ) VALUES (?, ?, ?, ?, ?, NOW())
    ');
    $txnStmt->execute([
        $userId,
        $orderId,
        'deposit',
        $amount,
        $description
    ]);

    $pdo->commit();
    
    // Başarılı yanıt
    echo json_encode([
        'success' => true,
        'message' => 'Havale/EFT talebi başarıyla oluşturuldu. Admin onayından sonra bakiyenize eklenecektir.',
        'order_id' => $orderId,
        'order_number' => $orderNumber,
        'redirect' => $siteUrl . '/balance.php?payment=pending'
    ]);

} catch (Throwable $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

