<?php
require_once '../config.php';
requireLogin();

$method = $_GET['method'] ?? '';
$amount = floatval($_GET['amount'] ?? 0);
$chargeParam = isset($_GET['charge']) ? floatval($_GET['charge']) : null;

if ($amount < 1) {
    flashMessage('danger', 'Minimum yükleme tutarı 1 TL\'dir.');
    redirect('../balance.php');
}

if (!in_array($method, ['paytr', 'shopier'])) {
    flashMessage('danger', 'Geçersiz ödeme yöntemi.');
    redirect('../balance.php');
}

$user = getUserInfo($pdo, $_SESSION['user_id']);

$commissionRate = 0.0;
if ($method === 'paytr' || $method === 'shopier') {
    $commissionRate = max(0.0, (float)getCommissionRate($method));
}

$calculatedCharge = round($amount + ($amount * $commissionRate / 100), 2);
$chargeAmount = $commissionRate > 0 ? $calculatedCharge : $amount;
if ($chargeParam !== null && $chargeParam > 0) {
    $chargeAmount = max($chargeAmount, round($chargeParam, 2));
}
$commissionAmount = max(0, $chargeAmount - $amount);

// Ödeme kaydı oluştur (pending)
try {
    $order_number = generateOrderNumber();
    
    $stmt = $pdo->prepare("
        INSERT INTO orders (user_id, order_number, total_amount, payable_amount, commission_amount, commission_rate, status, payment_method, order_type) 
        VALUES (?, ?, ?, ?, ?, ?, 'pending', ?, ?)
    ");
    $stmt->execute([
        $_SESSION['user_id'],
        $order_number,
        $amount,
        $chargeAmount,
        $commissionAmount,
        $commissionRate,
        $method,
        'deposit'
    ]);
    $order_id = $pdo->lastInsertId();
    
    // Transaction kaydı
    $stmt = $pdo->prepare("
        INSERT INTO transactions (user_id, type, amount, description, order_id) 
        VALUES (?, 'deposit', ?, ?, ?)
    ");
    $stmt->execute([
        $_SESSION['user_id'], 
        $amount, 
        'Bakiye yükleme - ' . strtoupper($method),
        $order_id
    ]);
    
} catch (Exception $e) {
    flashMessage('danger', 'Sipariş oluşturulamadı.');
    redirect('../balance.php');
}

$pageTitle = 'Ödeme İşlemi';
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?> - <?= siteName() ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }
        .payment-card {
            background: white;
            border-radius: 20px;
            padding: 3rem;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 600px;
            width: 100%;
        }
        .loading {
            text-align: center;
            padding: 2rem;
        }
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #6366f1;
            border-radius: 50%;
            width: 50px;
            height: 50px;
            animation: spin 1s linear infinite;
            margin: 0 auto 1rem;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        #paytr-iframe {
            width: 100%;
            height: 600px;
            border: none;
            border-radius: 10px;
        }
    </style>
</head>
<body>
    <div class="payment-card">
        <div class="text-center mb-4">
            <h3>
                <i class="fas fa-credit-card me-2"></i>
                <?= $method === 'paytr' ? 'PayTR' : 'Shopier' ?> Ödeme
            </h3>
            <p class="text-muted mb-1">Yüklenecek Tutar: <strong><?= formatPrice($amount) ?></strong></p>
            <?php if ($commissionAmount > 0): ?>
            <p class="text-muted mb-1">Komisyon (%<?= number_format($commissionRate, 2, ',', '.') ?>): <strong><?= formatPrice($commissionAmount) ?></strong></p>
            <?php endif; ?>
            <p class="text-muted">Ödenecek Tutar: <strong><?= formatPrice($chargeAmount) ?></strong></p>
            <p class="text-muted">Sipariş No: <strong><?= $order_number ?></strong></p>
        </div>
        
        <div class="loading" id="loading">
            <div class="spinner"></div>
            <p>Ödeme sayfası yükleniyor...</p>
        </div>

        <div id="payment-error" class="alert alert-danger" style="display: none;"></div>
        <div id="payment-container" style="display: none;"></div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            <?php if ($method === 'paytr'): ?>
                initPayTR();
            <?php else: ?>
                initShopier();
            <?php endif; ?>
        });
        
        function initPayTR() {
            fetch('paytr-init.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                credentials: 'include',
                body: JSON.stringify({
                    order_id: '<?= $order_id ?>',
                    order_number: '<?= $order_number ?>',
                    amount: <?= $amount ?>,
                    pay_amount: <?= $chargeAmount ?>,
                    email: '<?= sanitize($user['email']) ?>',
                    name: '<?= sanitize(!empty($user['full_name']) ? $user['full_name'] : $user['username']) ?>',
                    phone: '<?= sanitize(!empty($user['phone']) ? $user['phone'] : '5555555555') ?>',
                    address: '<?= sanitize($user['address'] ?? 'Online Satış') ?>',
                    city: '<?= sanitize($user['city'] ?? 'İstanbul') ?>',
                    zipcode: '<?= sanitize($user['zipcode'] ?? '34000') ?>'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('loading').style.display = 'none';
                    document.getElementById('payment-container').style.display = 'block';
                    document.getElementById('payment-container').innerHTML = 
                        '<iframe id="paytr-iframe" src="https://www.paytr.com/odeme/guvenli/' + data.token + '"></iframe>';
                } else {
                    showPaymentError(data.error || 'Ödeme başlatılamadı.');
                }
            })
            .catch(error => {
                showPaymentError(error);
            });
        }
        
        function initShopier() {
            fetch('shopier-init.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                credentials: 'include',
                body: JSON.stringify({
                    order_id: '<?= $order_id ?>',
                    order_number: '<?= $order_number ?>',
                    amount: <?= $amount ?>,
                    pay_amount: <?= $chargeAmount ?>,
                    email: '<?= sanitize($user['email']) ?>',
                    name: '<?= sanitize(!empty($user['full_name']) ? $user['full_name'] : $user['username']) ?>',
                    phone: '<?= sanitize(!empty($user['phone']) ? $user['phone'] : '5555555555') ?>',
                    address: '<?= sanitize($user['address'] ?? 'Online Satış') ?>',
                    city: '<?= sanitize($user['city'] ?? 'İstanbul') ?>',
                    zipcode: '<?= sanitize($user['zipcode'] ?? '34000') ?>'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('loading').style.display = 'none';
                    const container = document.getElementById('payment-container');
                    container.style.display = 'block';
                    container.innerHTML = '<div class="alert alert-info">Shopier ödeme sayfasına yönlendiriliyorsunuz...<br><small>Pencere açılmazsa <a href="#" id="shopier-manual-link">buraya tıklayın</a>.</small></div>';
                    submitShopierForm(data.url, data.fields);
                } else {
                    showPaymentError(data.error || 'Ödeme başlatılamadı.');
                }
            })
            .catch(error => {
                showPaymentError(error);
            });
        }

        function submitShopierForm(url, fields) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = url;
            form.style.display = 'none';

            Object.entries(fields || {}).forEach(([key, value]) => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = key;
                input.value = value;
                form.appendChild(input);
            });

            document.body.appendChild(form);

            const manualLink = document.getElementById('shopier-manual-link');
            if (manualLink) {
                manualLink.addEventListener('click', function(event) {
                    event.preventDefault();
                    form.submit();
                });
            }

            form.submit();
        }

        function showPaymentError(message) {
            const loading = document.getElementById('loading');
            const errorBox = document.getElementById('payment-error');
            if (loading) {
                loading.style.display = 'none';
            }
            if (errorBox) {
                errorBox.textContent = (typeof message === 'string' && message.trim() !== '') ? message : 'Ödeme işlemi başlatılamadı.';
                errorBox.style.display = 'block';
            }
        }
    </script>
</body>
</html>
