<?php
require_once __DIR__ . '/../config.php';

if (!function_exists('removeTurkishChars')) {
    function removeTurkishChars(string $text): string
    {
        $text = (string)$text;
        $tr = ['ı', 'İ', 'ğ', 'Ğ', 'ü', 'Ü', 'ş', 'Ş', 'ö', 'Ö', 'ç', 'Ç'];
        $en = ['i', 'I', 'g', 'G', 'u', 'U', 's', 'S', 'o', 'O', 'c', 'C'];
        return str_replace($tr, $en, $text);
    }
}

if (!function_exists('getShopierSettings')) {
    function getShopierSettings(): array
    {
        return [
            'api_key' => getSetting('shopier_api_key', SHOPIER_API_KEY),
            'api_secret' => getSetting('shopier_api_secret', SHOPIER_API_SECRET),
            'website_index' => getSetting('shopier_website_index', '0'),
            'test_mode' => getBoolSetting('shopier_test_mode', SHOPIER_TEST_MODE)
        ];
    }
}

if (!function_exists('createShopierPaymentForm')) {
    function createShopierPaymentForm(array $orderData): array
    {
        $settings = getShopierSettings();

        $apiKey = $settings['api_key'];
        $apiSecret = $settings['api_secret'];

        if (empty($apiKey) || empty($apiSecret)) {
            throw new RuntimeException('Shopier API bilgileri eksik.');
        }

        $websiteIndex = $settings['website_index'] !== '' ? $settings['website_index'] : '0';

        $orderId = removeTurkishChars((string)$orderData['order_id']);
        $productName = removeTurkishChars((string)$orderData['product_name']);
        $productType = (string)($orderData['product_type'] ?? '0');

        $buyerName = removeTurkishChars((string)($orderData['buyer_name'] ?? 'Musteri'));
        $buyerSurname = removeTurkishChars((string)($orderData['buyer_surname'] ?? '')); 
        $buyerEmail = (string)($orderData['buyer_email'] ?? 'destek@enbmedya.com');

        $rawPhone = preg_replace('/\D+/', '', (string)($orderData['buyer_phone'] ?? ''));
        if (strlen($rawPhone) < 10) {
            $rawPhone = '5555555555';
        }

        $buyerAddress = removeTurkishChars((string)($orderData['buyer_address'] ?? 'Online Satis'));
        $buyerCity = removeTurkishChars((string)($orderData['buyer_city'] ?? 'Istanbul'));
        $buyerCountry = removeTurkishChars((string)($orderData['buyer_country'] ?? 'Turkey'));
        $buyerPostcode = (string)($orderData['buyer_postcode'] ?? '34000');

        $shippingAddress = removeTurkishChars((string)($orderData['shipping_address'] ?? $buyerAddress));
        $shippingCity = removeTurkishChars((string)($orderData['shipping_city'] ?? $buyerCity));
        $shippingCountry = removeTurkishChars((string)($orderData['shipping_country'] ?? $buyerCountry));
        $shippingPostcode = (string)($orderData['shipping_postcode'] ?? $buyerPostcode);

        $amount = number_format((float)($orderData['amount'] ?? 0), 2, '.', '');
        if ((float)$amount <= 0) {
            throw new RuntimeException('Geçersiz ödeme tutarı.');
        }

        $currencyInput = strtoupper((string)($orderData['currency'] ?? 'TL'));
        $currency = '0';
        if ($currencyInput === 'USD') {
            $currency = '1';
        } elseif ($currencyInput === 'EUR') {
            $currency = '2';
        }

        $randomNr = (string)mt_rand(100000, 999999);

        $productInfo = json_encode([
            [
                'name' => $productName,
                'product_id' => $orderId,
                'product_type' => $productType,
                'quantity' => 1,
                'price' => $amount,
                'total_price' => $amount
            ]
        ], JSON_UNESCAPED_UNICODE);

        $generalInfo = json_encode([
            'order_id' => $orderId,
            'order_date' => date('Y-m-d H:i:s'),
            'product_count' => 1
        ], JSON_UNESCAPED_UNICODE);

        $signatureText = $randomNr . $orderId . $amount . $currency;
        $signature = base64_encode(hash_hmac('sha256', $signatureText, $apiSecret, true));

        $fields = [
            'API_key' => $apiKey,
            'website_index' => $websiteIndex,
            'use_adress' => '0',
            'platform_order_id' => $orderId,
            'product_info' => $productInfo,
            'general_info' => $generalInfo,
            'product_name' => $productName,
            'product_type' => $productType,
            'buyer_name' => $buyerName,
            'buyer_surname' => $buyerSurname,
            'buyer_email' => $buyerEmail,
            'buyer_phone' => $rawPhone,
            'buyer_account_age' => (string)($orderData['buyer_account_age'] ?? '0'),
            'buyer_id_nr' => (string)($orderData['buyer_id_nr'] ?? $randomNr),
            'billing_address' => $buyerAddress,
            'billing_city' => $buyerCity,
            'billing_country' => $buyerCountry,
            'billing_postcode' => $buyerPostcode,
            'shipping_address' => $shippingAddress,
            'shipping_city' => $shippingCity,
            'shipping_country' => $shippingCountry,
            'shipping_postcode' => $shippingPostcode,
            'total_order_value' => $amount,
            'currency' => $currency,
            'platform' => '0',
            'is_in_frame' => '0',
            'current_language' => '0',
            'modul_version' => '2.0.0',
            'random_nr' => $randomNr,
            'signature' => $signature,
            'callback_url' => rtrim(SITE_URL, '/') . '/payment/shopier-callback.php',
            'success_url' => rtrim(SITE_URL, '/') . '/balance.php?payment=success&shopier_redirect=1',
            'cancel_url' => rtrim(SITE_URL, '/') . '/balance.php?payment=cancel'
        ];

        if ($settings['test_mode']) {
            $fields['is_test_mode'] = '1';
        }

        return [
            'url' => 'https://www.shopier.com/ShowProduct/api_pay4.php',
            'fields' => $fields
        ];
    }
}

if (!function_exists('verifyShopierSignature')) {
    function verifyShopierSignature(array $data, string $receivedSignature): bool
    {
        $settings = getShopierSettings();
        $apiSecret = $settings['api_secret'];

        if (empty($apiSecret)) {
            return false;
        }

        $randomNr = (string)($data['random_nr'] ?? '');
        $orderId = (string)($data['platform_order_id'] ?? '');

        if ($randomNr === '' || $orderId === '' || $receivedSignature === '') {
            return false;
        }

        $expectedSignature = hash_hmac('sha256', $randomNr . $orderId, $apiSecret, true);
        $decodedSignature = base64_decode($receivedSignature, true);

        if ($decodedSignature === false) {
            return false;
        }

        return hash_equals($expectedSignature, $decodedSignature);
    }
}

if (!function_exists('processShopierPayment')) {
    function processShopierPayment(array $paymentData): array
    {
        global $pdo;

        $orderNumber = $paymentData['platform_order_id'] ?? '';
        if ($orderNumber === '') {
            return ['success' => false, 'message' => 'Sipariş numarası eksik'];
        }

        $stmt = $pdo->prepare('SELECT * FROM orders WHERE order_number = ? LIMIT 1');
        $stmt->execute([$orderNumber]);
        $order = $stmt->fetch();

        if (!$order) {
            return ['success' => false, 'message' => 'Sipariş bulunamadı'];
        }

        // Status kontrolü - hem integer hem string hem de farklı değerler kabul et
        $statusRaw = $paymentData['status'] ?? '';
        $statusInt = (int)$statusRaw;
        $statusStr = strtolower(trim((string)$statusRaw));
        
        // Başarılı ödeme kontrolü - 1, "1", "success", "completed" gibi değerler
        $isSuccess = (
            $statusInt === 1 || 
            $statusStr === '1' || 
            $statusStr === 'success' || 
            $statusStr === 'completed' ||
            $statusStr === 'approved'
        );

        if ($isSuccess) {
            if ($order['status'] === 'completed') {
                return ['success' => true, 'message' => 'Sipariş zaten tamamlandı'];
            }

            try {
                $pdo->beginTransaction();

                $updateOrder = $pdo->prepare('UPDATE orders SET status = ? WHERE id = ?');
                $updateOrder->execute(['completed', $order['id']]);

                $amount = (float)$order['total_amount'];
                $updateBalance = $pdo->prepare('UPDATE users SET balance = balance + ? WHERE id = ?');
                $updateBalance->execute([$amount, $order['user_id']]);

                $updateTxn = $pdo->prepare("UPDATE transactions SET description = CONCAT(description, ' - Shopier Onaylandı') WHERE order_id = ?");
                $updateTxn->execute([$order['id']]);

                $pdo->commit();
                return ['success' => true, 'message' => 'Ödeme onaylandı'];
            } catch (Throwable $th) {
                $pdo->rollBack();
                return ['success' => false, 'message' => $th->getMessage()];
            }
        }

        // İptal kontrolü - 0, "0", "cancelled", "failed" gibi değerler
        $isCancelled = (
            $statusInt === 0 || 
            $statusStr === '0' || 
            $statusStr === 'cancelled' || 
            $statusStr === 'failed' ||
            $statusStr === 'rejected'
        );

        if ($isCancelled) {
            try {
                $pdo->beginTransaction();

                $updateOrder = $pdo->prepare('UPDATE orders SET status = ? WHERE id = ?');
                $updateOrder->execute(['cancelled', $order['id']]);

                $updateTxn = $pdo->prepare("UPDATE transactions SET description = CONCAT(description, ' - Shopier İptal') WHERE order_id = ?");
                $updateTxn->execute([$order['id']]);

                $pdo->commit();
            } catch (Throwable $th) {
                $pdo->rollBack();
                return ['success' => false, 'message' => $th->getMessage()];
            }
            return ['success' => false, 'message' => 'Ödeme iptal edildi'];
        }

        // Bilinmeyen status
        return ['success' => false, 'message' => 'Bilinmeyen ödeme durumu: ' . $statusRaw];
    }
}

