<?php
require_once 'config.php';

$productId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!function_exists('resolveCategoryAsset')) {
    function resolveCategoryAsset(array $entity, array $map): string
    {
        if (!empty($entity['image'])) {
            $relative = ltrim((string)$entity['image'], '/\\');
            $fullPath = __DIR__ . '/' . $relative;
            if (is_file($fullPath)) {
                return $relative;
            }
        }

        $slug = $entity['category_slug'] ?? ($entity['slug'] ?? '');
        if (isset($map[$slug])) {
            return $map[$slug];
        }
        foreach ($map as $key => $path) {
            if ($slug !== '' && str_contains($slug, $key)) {
                return $path;
            }
        }
        return 'assets/images/category-oyun.svg';
    }
}

$categoryMediaMap = [
    'oyun-pinleri' => 'assets/images/category-oyun.svg',
    'oyunlar' => 'assets/images/category-oyun.svg',
    'oyun' => 'assets/images/category-oyun.svg',
    'pazaryeri' => 'assets/images/category-market.svg',
    'market' => 'assets/images/category-market.svg',
    'oyun-parasi' => 'assets/images/category-bakiye.svg',
    'bakiye' => 'assets/images/category-bakiye.svg',
];

$productSlug = safe_trim($_GET['slug'] ?? '');

if ($productSlug !== '') {
    $stmt = $pdo->prepare('
        SELECT p.*, c.name AS category_name, c.slug AS category_slug
        FROM products p
        JOIN categories c ON p.category_id = c.id
        WHERE p.slug = ? AND p.is_active = 1
    ');
    $stmt->execute([$productSlug]);
} else {
    if ($productId <= 0) {
        flashMessage('danger', 'Ürün bulunamadı.');
        redirect(routeUrl('products'));
    }
    $stmt = $pdo->prepare('
        SELECT p.*, c.name AS category_name, c.slug AS category_slug
        FROM products p
        JOIN categories c ON p.category_id = c.id
        WHERE p.id = ? AND p.is_active = 1
    ');
    $stmt->execute([$productId]);
}
$product = $stmt->fetch();

if (!$product) {
    flashMessage('danger', 'Ürün bulunamadı.');
    redirect(routeUrl('products'));
}

$productId = (int)($product['id'] ?? 0);

$pStmt = $pdo->prepare('UPDATE products SET views = views + 1 WHERE id = ?');
$pStmt->execute([$productId]);

// Benzer ürünler
$similarStmt = $pdo->prepare('
    SELECT p.*, c.slug AS category_slug, c.name AS category_name
    FROM products p
    JOIN categories c ON p.category_id = c.id
    WHERE p.category_id = ? AND p.id != ? AND p.is_active = 1
    ORDER BY RAND()
    LIMIT 4
');
$similarStmt->execute([$product['category_id'], $productId]);
$similarProducts = $similarStmt->fetchAll();

$pageTitle = $product['name'];
include 'includes/header.php';
?>

<section class="detail-page">
    <div class="container-wide">
        <nav class="breadcrumb-trail" aria-label="breadcrumb">
            <ol>
                <li><a href="<?= routeUrl('home') ?>">Ana Sayfa</a></li>
                <li><a href="<?= routeUrl('products') ?>">Ürünler</a></li>
                <li><a href="<?= htmlspecialchars(routeUrl('product_category', ['slug' => $product['category_slug']]), ENT_QUOTES, 'UTF-8') ?>"><?= sanitize($product['category_name']) ?></a></li>
                <li><?= sanitize($product['name']) ?></li>
            </ol>
        </nav>

        <div class="detail-grid">
            <div class="detail-media">
                <div class="detail-card">
                    <span class="tag-pill"><i class="fa-solid fa-tag"></i><?= sanitize($product['category_name']) ?></span>
                    <div class="detail-art mt-4">
                        <?php $productImage = htmlspecialchars(assetUrl(resolveCategoryAsset($product, $categoryMediaMap)), ENT_QUOTES, 'UTF-8'); ?>
                        <img src="<?= $productImage ?>" alt="<?= sanitize($product['name']) ?>">
                    </div>
                </div>
                
                <!-- Features under image -->
                <div class="detail-features-compact">
                    <div class="feature-compact-item">
                        <i class="fa-solid fa-bolt"></i>
                        <span>Anında Teslimat</span>
                    </div>
                    <div class="feature-compact-item">
                        <i class="fa-solid fa-shield-halved"></i>
                        <span>Güvenli Ödeme</span>
                    </div>
                    <div class="feature-compact-item">
                        <i class="fa-solid fa-headset"></i>
                        <span>7/24 Destek</span>
                    </div>
                </div>
            </div>

            <div class="detail-info">
                <h1 class="detail-title"><?= sanitize($product['name']) ?></h1>

                <div class="detail-meta">
                    <span class="tag-pill"><i class="fa-solid fa-eye"></i><?= (int)$product['views'] ?> görüntülenme</span>
                    <span class="tag-pill"><i class="fa-solid fa-fire"></i><?= (int)$product['sales_count'] ?> satış</span>
                    <span class="tag-pill"><i class="fa-solid fa-box"></i>Stok: <?= max(0, (int)$product['stock']) ?></span>
                </div>

                <?php if (!empty($product['description'])): ?>
                <div class="detail-description">
                    <?= nl2br(sanitize($product['description'])) ?>
                </div>
                <?php endif; ?>

                <?php $inStock = (int)$product['stock'] > 0; ?>
                
                <div class="product-purchase-card">
                    <div class="price-section">
                        <div class="price-label">Satış Fiyatı</div>
                        <div class="price-amount"><?= formatPrice($product['price']) ?></div>
                    </div>

                    <?php if (isLoggedIn()): ?>
                        <?php if ($inStock): ?>
                            <form method="post" action="<?= htmlspecialchars(routeUrl('cart'), ENT_QUOTES, 'UTF-8') ?>" class="purchase-form">
                                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                <input type="hidden" name="action" value="add_product">
                                <input type="hidden" name="product_id" value="<?= (int)$product['id'] ?>">
                                
                                <div class="quantity-selector">
                                    <label for="quantity" class="quantity-label">Adet</label>
                                    <div class="quantity-input-group">
                                        <button type="button" class="qty-btn qty-minus" onclick="changeQuantity(-1)">
                                            <i class="fa-solid fa-minus"></i>
                                        </button>
                                        <input type="number" id="quantity" name="quantity" class="qty-input" min="1" max="<?= max(1, (int)$product['stock']) ?>" value="1" readonly>
                                        <button type="button" class="qty-btn qty-plus" onclick="changeQuantity(1)">
                                            <i class="fa-solid fa-plus"></i>
                                        </button>
                                    </div>
                                </div>
                                
                                <button type="submit" class="btn-add-cart">
                                    <i class="fa-solid fa-cart-shopping me-2"></i>
                                    <span>Sepete Ekle</span>
                                </button>
                            </form>
                    <?php else: ?>
                        <div class="alert alert-danger mt-3">
                            <i class="fa-solid fa-triangle-exclamation me-2"></i> Bu ürün şu anda stokta yok. Lütfen daha sonra tekrar deneyin.
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <div class="alert alert-warning mt-3">
                        <i class="fa-solid fa-circle-info me-2"></i> Ürünü satın almak için lütfen <a href="<?= routeUrl('login') ?>" class="alert-link">giriş yapın</a>.
                    </div>
                <?php endif; ?>

                <div class="detail-notice mt-4">
                    <i class="fa-solid fa-lock"></i>
                    <div>
                        <strong>Güvenli alışveriş</strong>
                        <span>Ödeme tamamlandıktan sonra kodlarınız anında "Siparişlerim" sayfanıza düşer.</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<?php if (!empty($similarProducts)): ?>
<section class="similar-products-section">
    <div class="container-wide">
        <div class="section-heading">
            <h2><i class="fa-solid fa-layer-group"></i> Benzer Ürünler</h2>
            <a href="<?= htmlspecialchars(routeUrl('product_category', ['slug' => $product['category_slug']]), ENT_QUOTES, 'UTF-8') ?>"><span>Tümünü Gör</span><i class="fa-solid fa-arrow-right"></i></a>
        </div>
        <div class="home-products-grid">
            <?php foreach ($similarProducts as $similar): ?>
            <div class="home-product-card">
                <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $similar['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__image">
                    <?php $similarImage = htmlspecialchars(assetUrl(resolveCategoryAsset($similar, $categoryMediaMap)), ENT_QUOTES, 'UTF-8'); ?>
                    <img src="<?= $similarImage ?>" alt="<?= sanitize($similar['name']) ?>">
                    <span class="home-product-card__badge home-product-card__badge--new">Benzer</span>
                </a>
                <div class="home-product-card__content">
                    <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $similar['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__title">
                        <?= sanitize($similar['name']) ?>
                    </a>
                    <div class="home-product-card__price">
                        <?= formatPrice($similar['price']) ?>
                    </div>
                </div>
                <div class="home-product-card__actions">
                    <a href="<?= htmlspecialchars(routeUrl('product', ['slug' => $similar['slug'] ?? '']), ENT_QUOTES, 'UTF-8') ?>" class="home-product-card__btn home-product-card__btn--primary">
                        <i class="fa-solid fa-shopping-bag"></i>
                        <span>Satın Al</span>
                    </a>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</section>
<?php endif; ?>

<script>
function changeQuantity(delta) {
    const input = document.getElementById('quantity');
    const min = parseInt(input.min) || 1;
    const max = parseInt(input.max) || 999;
    let currentValue = parseInt(input.value) || 1;
    
    currentValue += delta;
    
    if (currentValue < min) currentValue = min;
    if (currentValue > max) currentValue = max;
    
    input.value = currentValue;
}
</script>

<?php include 'includes/footer.php'; ?>
