<?php
require_once 'config.php';

$categoryFilter = safe_trim($_GET['category'] ?? '');
$searchQuery = safe_trim($_GET['search'] ?? ($_GET['q'] ?? ''));
$sortParam = $_GET['sort'] ?? '';

if (!function_exists('shortenText')) {
    function shortenText(?string $text, int $limit = 110): string
    {
        $text = safe_trim($text);
        if ($text === '') {
            return '';
        }
        if (mb_strlen($text) <= $limit) {
            return $text;
        }
        return safe_rtrim(mb_substr($text, 0, $limit - 3)) . '...';
    }
}

if (!function_exists('resolveCategoryAsset')) {
    function resolveCategoryAsset(array $entity, array $map): string
    {
        if (!empty($entity['image'])) {
            $relative = ltrim((string)$entity['image'], '/\\');
            $fullPath = __DIR__ . '/' . $relative;
            if (is_file($fullPath)) {
                return $relative;
            }
        }

        $slug = $entity['category_slug'] ?? ($entity['slug'] ?? '');
        if (isset($map[$slug])) {
            return $map[$slug];
        }
        foreach ($map as $key => $path) {
            if ($slug !== '' && str_contains($slug, $key)) {
                return $path;
            }
        }
        return 'assets/images/category-oyun.svg';
    }
}

$categoryMediaMap = [
    'oyun-pinleri' => 'assets/images/category-oyun.svg',
    'oyunlar' => 'assets/images/category-oyun.svg',
    'oyun' => 'assets/images/category-oyun.svg',
    'pazaryeri' => 'assets/images/category-market.svg',
    'market' => 'assets/images/category-market.svg',
    'oyun-parasi' => 'assets/images/category-bakiye.svg',
    'bakiye' => 'assets/images/category-bakiye.svg',
];

$categories = getCategories($pdo);
$categoryTree = buildCategoryTree($categories);
$categoryIndexData = buildCategoryIndex($categories);

$activeCategory = null;
$activeParentCategory = null;
$categoryIdsFilter = [];

if ($categoryFilter !== '') {
    foreach ($categories as $cat) {
        if ($cat['slug'] === $categoryFilter) {
            $activeCategory = $cat;
            $categoryIdsFilter = getCategoryDescendantIds($categoryIndexData, (int)$cat['id']);
            break;
        }
    }

    if ($activeCategory && !empty($activeCategory['parent_id'])) {
        $parentId = (int)$activeCategory['parent_id'];
        $activeParentCategory = $categoryIndexData['byId'][$parentId] ?? null;
    }
}

$childCategoryIds = [];
if ($activeCategory) {
    $childCategoryIds = $categoryIndexData['children'][$activeCategory['id']] ?? [];
    if (empty($childCategoryIds) && $activeParentCategory) {
        $childCategoryIds = $categoryIndexData['children'][$activeParentCategory['id']] ?? [];
    }
}

$childCategoriesForDisplay = [];
foreach ($childCategoryIds as $childId) {
    if (isset($categoryIndexData['byId'][$childId])) {
        $childCategoriesForDisplay[] = $categoryIndexData['byId'][$childId];
    }
}

$sql = "SELECT p.*, c.name AS category_name, c.slug AS category_slug
        FROM products p
        JOIN categories c ON p.category_id = c.id
        WHERE p.is_active = 1";
$params = [];

if ($activeCategory && !empty($categoryIdsFilter)) {
    $placeholders = implode(',', array_fill(0, count($categoryIdsFilter), '?'));
    $sql .= " AND c.id IN ($placeholders)";
    $params = array_merge($params, $categoryIdsFilter);
} elseif ($categoryFilter !== '') {
    $sql .= " AND c.slug = ?";
    $params[] = $categoryFilter;
}

if ($searchQuery !== '') {
    $sql .= " AND (p.name LIKE ? OR p.description LIKE ?)";
    $searchTerm = '%' . $searchQuery . '%';
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

if ($sortParam === 'top') {
    $sql .= " ORDER BY p.sales_count DESC, p.created_at DESC";
} elseif ($sortParam === 'new') {
    $sql .= " ORDER BY p.created_at DESC";
} else {
    $sql .= " ORDER BY p.featured DESC, p.created_at DESC";
}

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$products = $stmt->fetchAll();

$heroImage = null;
if ($activeCategory && !empty($activeCategory['banner_image'])) {
    $heroImage = $activeCategory['banner_image'];
} elseif ($activeParentCategory && !empty($activeParentCategory['banner_image'])) {
    $heroImage = $activeParentCategory['banner_image'];
}

if ($activeCategory) {
    $pageTitle = $activeCategory['meta_title'] ?? $activeCategory['name'];
    $pageDescription = $activeCategory['meta_description'] ?? ($activeCategory['description'] ?? '');
} elseif ($searchQuery !== '') {
    $pageTitle = 'Arama Sonuçları';
    $pageDescription = "'$searchQuery' için sonuçlar";
} else {
    $pageTitle = 'Tüm Ürünler';
    $pageDescription = 'Oyun kodları, pazaryeri ilanları ve oyun parasını keşfedin.';
}

$activeTopLevelSlug = '';
if ($activeCategory) {
    if (!empty($activeCategory['parent_id']) && $activeParentCategory) {
        $activeTopLevelSlug = $activeParentCategory['slug'];
    } else {
        $activeTopLevelSlug = $activeCategory['slug'];
    }
}

$topLevelCategories = $categoryTree;

include 'includes/header.php';
?>

<?php
// Build breadcrumbs
$breadcrumbs = [
    ['label' => 'Ana Sayfa', 'url' => routeUrl('home')],
    ['label' => 'Ürünler', 'url' => routeUrl('products')],
];

if ($activeParentCategory) {
    $breadcrumbs[] = [
        'label' => $activeParentCategory['name'],
        'url' => routeUrl('product_category', ['slug' => $activeParentCategory['slug']])
    ];
}

if ($activeCategory) {
    $breadcrumbs[] = [
        'label' => $activeCategory['name'],
        'url' => ''
    ];
}

$heroEyebrow = $activeCategory ? 'Kategori' : 'Ürün Kataloğu';
if ($searchQuery !== '') {
    $heroEyebrow = 'Arama';
}

$heroHeadline = $activeCategory['name'] ?? ($searchQuery !== '' ? "'" . $searchQuery . "' Araması" : 'Tüm Ürünler');
$heroSummary = $pageDescription !== '' ? shortenText($pageDescription, 200) : 'Popüler oyun kodları, hediye kartları ve dijital içerikleri tek panelde keşfedin.';

// Banner görseli kontrolü
$categoryBanner = '';
$hasBanner = false;
if ($activeCategory && !empty($activeCategory['banner_image'])) {
    $categoryBanner = assetUrl($activeCategory['banner_image']);
    $hasBanner = true;
}
$bannerStyle = $hasBanner ? ' style="background-image: url(' . htmlspecialchars($categoryBanner, ENT_QUOTES, 'UTF-8') . ');"' : '';
$heroClasses = 'page-hero page-hero--catalog' . ($hasBanner ? ' page-hero--with-banner' : '');
?>

<section class="<?= $heroClasses ?>"<?= $bannerStyle ?>>
    <div class="container-wide">
        <nav class="page-breadcrumb" aria-label="breadcrumb">
            <?php foreach ($breadcrumbs as $index => $crumb): ?>
                <?php if ($crumb['url'] !== '' && $index < count($breadcrumbs) - 1): ?>
                    <a href="<?= htmlspecialchars($crumb['url'], ENT_QUOTES, 'UTF-8') ?>"><?= sanitize($crumb['label']) ?></a>
                    <span class="page-breadcrumb__divider">/</span>
                <?php else: ?>
                    <span class="page-breadcrumb__current"><?= sanitize($crumb['label']) ?></span>
                <?php endif; ?>
            <?php endforeach; ?>
        </nav>
        <div class="page-hero__content">
            <span class="page-hero__eyebrow"><i class="fa-solid fa-layer-group"></i><?= sanitize($heroEyebrow) ?></span>
            <h1 class="page-hero__title"><?= sanitize($heroHeadline) ?></h1>
            <?php if ($heroSummary !== ''): ?>
                <p class="page-hero__subtitle"><?= sanitize($heroSummary) ?></p>
            <?php endif; ?>
        </div>
    </div>
</section>

<?php if (!empty($topLevelCategories)): ?>
<section class="catalog-filter">
    <div class="container-wide">
        <div class="category-cards-grid">
            <a href="<?= routeUrl('products') ?>" class="category-filter-card <?= $categoryFilter === '' ? 'active' : '' ?>">
                <div class="category-filter-card__icon-wrapper">
                    <i class="fa-solid fa-layer-group"></i>
                </div>
                <div class="category-filter-card__content">
                    <h3 class="category-filter-card__title">Tümü</h3>
                    <p class="category-filter-card__desc">Tüm kategoriler</p>
                </div>
            </a>
            <?php foreach ($topLevelCategories as $topCategory): ?>
                <?php 
                    $isActiveTop = ($activeTopLevelSlug !== '' && $activeTopLevelSlug === $topCategory['slug']);
                    $catIcon = $topCategory['icon'] ?? 'fa-tag';
                    $catImage = !empty($topCategory['image']) ? assetUrl($topCategory['image']) : null;
                    $catDesc = !empty($topCategory['description']) ? shortenText($topCategory['description'], 50) : 'Popüler ürünler';
                ?>
                <a href="<?= htmlspecialchars(routeUrl('product_category', ['slug' => $topCategory['slug']]), ENT_QUOTES, 'UTF-8') ?>" 
                   class="category-filter-card <?= $isActiveTop ? 'active' : '' ?><?= $catImage ? ' has-bg-image' : '' ?>"
                   <?= $catImage ? 'style="background-image: url(' . htmlspecialchars($catImage, ENT_QUOTES, 'UTF-8') . ');"' : '' ?>>
                    <div class="category-filter-card__icon-wrapper">
                        <i class="fa-solid <?= sanitize($catIcon) ?>"></i>
                    </div>
                    <div class="category-filter-card__content">
                        <h3 class="category-filter-card__title"><?= sanitize($topCategory['name']) ?></h3>
                        <p class="category-filter-card__desc"><?= sanitize($catDesc) ?></p>
                    </div>
                </a>
            <?php endforeach; ?>
        </div>

        <?php if (!empty($childCategoriesForDisplay)): ?>
        <div class="subcategory-section-heading">
            <h3 class="subcategory-section-title">
                <i class="fa-solid fa-layer-group"></i> Alt Kategoriler
            </h3>
            <p class="subcategory-section-subtitle">Aradığınız ürüne daha hızlı ulaşmak için alt kategorilerden birini seçin</p>
        </div>
        <div class="subcategory-cards-container">
            <?php foreach ($childCategoriesForDisplay as $child): ?>
                <?php 
                    $isChildActive = $categoryFilter === $child['slug']; 
                    $childIcon = $child['icon'] ?? 'fa-tag';
                    $childImage = !empty($child['image']) ? assetUrl($child['image']) : null;
                    $hasImage = $childImage !== null;
                ?>
                <a class="subcategory-card-mini <?= $isChildActive ? 'active' : '' ?><?= $hasImage ? ' has-image' : '' ?>" 
                   href="<?= htmlspecialchars(routeUrl('product_category', ['slug' => $child['slug']]), ENT_QUOTES, 'UTF-8') ?>">
                    <div class="subcategory-card-mini__icon">
                        <?php if ($hasImage): ?>
                            <img src="<?= htmlspecialchars($childImage, ENT_QUOTES, 'UTF-8') ?>" alt="<?= sanitize($child['name']) ?>" class="subcategory-card-mini__image">
                        <?php else: ?>
                            <i class="fa-solid <?= sanitize($childIcon) ?>"></i>
                        <?php endif; ?>
                    </div>
                    <span class="subcategory-card-mini__name"><?= sanitize($child['name']) ?></span>
                </a>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
</section>
<?php endif; ?>

<section class="catalog-section">
    <div class="container-wide">
        <?php if (empty($products)): ?>
            <div class="catalog-empty">
                <i class="fa-solid fa-inbox"></i>
                <h3>Ürün bulunamadı</h3>
                <p>Aradığınız kriterlere uygun ürün bulunamadı. Diğer kategorilere göz atmayı deneyebilirsiniz.</p>
                <a href="<?= routeUrl('products') ?>" class="btn-primary"><i class="fa-solid fa-rotate"></i>Ürünleri yeniden listele</a>
            </div>
        <?php else: ?>
            <div class="home-products-grid">
                <?php foreach ($products as $product): ?>
                    <?php
                        $imagePath = resolveCategoryAsset($product, $categoryMediaMap);
                        $imageUrl = htmlspecialchars(assetUrl($imagePath), ENT_QUOTES, 'UTF-8');
                        $productUrl = htmlspecialchars(routeUrl('product', ['slug' => $product['slug'] ?? '']), ENT_QUOTES, 'UTF-8');
                        $categoryUrl = htmlspecialchars(routeUrl('product_category', ['slug' => $product['category_slug']]), ENT_QUOTES, 'UTF-8');
                    ?>
                    <article class="home-product-card">
                        <div class="home-product-card__image">
                            <?php if ((int)$product['featured'] === 1): ?>
                                <span class="home-product-card__badge"><i class="fa-solid fa-star"></i>Öne çıkan</span>
                            <?php elseif ((int)$product['stock'] <= 5): ?>
                                <span class="home-product-card__badge home-product-card__badge--warning">
                                    <i class="fa-solid fa-battery-quarter"></i>Son <?= max(0, (int)$product['stock']) ?>
                                </span>
                            <?php endif; ?>
                            <img src="<?= $imageUrl ?>" alt="<?= sanitize($product['name']) ?>">
                        </div>
                        <div class="home-product-card__content">
                            <a class="home-product-card__category" href="<?= $categoryUrl ?>">
                                <?= sanitize($product['category_name']) ?>
                            </a>
                            <h3 class="home-product-card__title">
                                <a href="<?= $productUrl ?>"><?= sanitize($product['name']) ?></a>
                            </h3>
                            <?php if (!empty($product['description'])): ?>
                                <p class="home-product-card__description"><?= sanitize(shortenText($product['description'], 80)) ?></p>
                            <?php endif; ?>
                            <div class="home-product-card__meta">
                                <span><i class="fa-solid fa-box"></i><?= max(0, (int)$product['stock']) ?> stok</span>
                                <span><i class="fa-solid fa-fire"></i><?= (int)$product['sales_count'] ?> satış</span>
                            </div>
                        </div>
                        <div class="home-product-card__footer">
                            <div class="home-product-card__price"><?= formatPrice($product['price']) ?></div>
                        </div>
                        <div class="home-product-card__actions">
                            <a href="<?= $productUrl ?>" class="home-product-card__btn home-product-card__btn--primary">
                                <i class="fa-solid fa-shopping-bag"></i>
                                <span>Satın Al</span>
                            </a>
                        </div>
                    </article>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</section>

<?php include 'includes/footer.php'; ?>
