<?php
require_once 'config.php';
requireLogin();

$user = getUserInfo($pdo, $_SESSION['user_id']);
$profileSuccess = '';
$profileErrors = [];
$withdrawSuccess = '';
$withdrawErrors = [];
$withdrawAmountInput = '';

$payoutAccountName = trim((string)($user['payout_account_name'] ?? ''));
$payoutBankName = trim((string)($user['payout_bank_name'] ?? ''));
$payoutIban = trim((string)($user['payout_iban'] ?? ''));

$formatIbanDisplay = static function (?string $iban): string {
    $clean = strtoupper(preg_replace('/[^A-Z0-9]/', '', (string)$iban));
    return $clean === '' ? '' : trim(chunk_split($clean, 4, ' '));
};

$activeTab = $_GET['tab'] ?? 'deposits';
$allowedTabs = ['deposits', 'marketplace', 'orders', 'withdraw'];
if (!in_array($activeTab, $allowedTabs, true)) {
    $activeTab = 'deposits';
}

$formAction = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $formAction = $_POST['form_action'] ?? 'update_profile';
    if ($formAction === 'create_withdraw') {
        $activeTab = 'withdraw';
    }

    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        if ($formAction === 'create_withdraw') {
            $withdrawErrors[] = 'Güvenlik hatası.';
        } else {
            $profileErrors[] = 'Güvenlik hatası.';
        }
    } else {
        switch ($formAction) {
            case 'update_profile':
                $full_name = trim($_POST['full_name'] ?? '');
                $email = trim($_POST['email'] ?? '');
                $phone = trim($_POST['phone'] ?? '');
                $current_password = $_POST['current_password'] ?? '';
                $new_password = $_POST['new_password'] ?? '';
                $new_password_confirm = $_POST['new_password_confirm'] ?? '';
                $payout_account_name = trim($_POST['payout_account_name'] ?? '');
                $payout_bank_name = trim($_POST['payout_bank_name'] ?? '');
                $payout_iban_input = strtoupper(preg_replace('/[^A-Z0-9]/', '', (string)($_POST['payout_iban'] ?? '')));

                $payoutAccountName = $payout_account_name;
                $payoutBankName = $payout_bank_name;
                $payoutIban = $payout_iban_input;

                if ($full_name === '') {
                    $profileErrors[] = 'Ad soyad gereklidir.';
                }

                if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    $profileErrors[] = 'Geçerli bir e-posta adresi giriniz.';
                }

                if ($email !== ($user['email'] ?? '')) {
                    $emailCheck = $pdo->prepare('SELECT id FROM users WHERE email = ? AND id != ?');
                    $emailCheck->execute([$email, $_SESSION['user_id']]);
                    if ($emailCheck->fetch()) {
                        $profileErrors[] = 'Bu e-posta adresi zaten kullanılıyor.';
                    }
                }

                if ($new_password !== '') {
                    if ($current_password === '') {
                        $profileErrors[] = 'Mevcut şifrenizi giriniz.';
                    } elseif (!password_verify($current_password, $user['password'])) {
                        $profileErrors[] = 'Mevcut şifreniz hatalı.';
                    } elseif (strlen($new_password) < 6) {
                        $profileErrors[] = 'Yeni şifre en az 6 karakter olmalıdır.';
                    } elseif ($new_password !== $new_password_confirm) {
                        $profileErrors[] = 'Yeni şifreler eşleşmiyor.';
                    }
                }

                if ($payout_iban_input !== '' && !preg_match('/^[A-Z]{2}[A-Z0-9]{8,32}$/', $payout_iban_input)) {
                    $profileErrors[] = 'Geçerli bir IBAN giriniz.';
                }

                if (empty($profileErrors)) {
                    try {
                        if ($new_password !== '') {
                            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                            $update = $pdo->prepare('
                                UPDATE users
                                SET full_name = ?, email = ?, phone = ?, password = ?, payout_account_name = ?, payout_bank_name = ?, payout_iban = ?
                                WHERE id = ?
                            ');
                            $update->execute([$full_name, $email, $phone, $hashed_password, $payout_account_name, $payout_bank_name, $payout_iban_input, $_SESSION['user_id']]);
                        } else {
                            $update = $pdo->prepare('
                                UPDATE users
                                SET full_name = ?, email = ?, phone = ?, payout_account_name = ?, payout_bank_name = ?, payout_iban = ?
                                WHERE id = ?
                            ');
                            $update->execute([$full_name, $email, $phone, $payout_account_name, $payout_bank_name, $payout_iban_input, $_SESSION['user_id']]);
                        }

                        $_SESSION['email'] = $email;
                        $user = getUserInfo($pdo, $_SESSION['user_id']);
                        $payoutAccountName = trim((string)($user['payout_account_name'] ?? ''));
                        $payoutBankName = trim((string)($user['payout_bank_name'] ?? ''));
                        $payoutIban = trim((string)($user['payout_iban'] ?? ''));
                        $profileSuccess = 'Profiliniz başarıyla güncellendi.';
                    } catch (PDOException $e) {
                        $profileErrors[] = 'Güncelleme sırasında bir hata oluştu.';
                    }
                }
                break;

            case 'create_withdraw':
                $withdrawAmountInput = str_replace(',', '.', trim((string)($_POST['withdraw_amount'] ?? '')));
                if ($withdrawAmountInput === '' || !is_numeric($withdrawAmountInput)) {
                    $withdrawErrors[] = 'Geçerli bir tutar giriniz.';
                }

                $amount = $withdrawAmountInput !== '' && is_numeric($withdrawAmountInput) ? (float)$withdrawAmountInput : 0.0;
                $amount = round($amount, 2);
                $amountValue = number_format($amount, 2, '.', '');
                if ($withdrawAmountInput !== '' && is_numeric($withdrawAmountInput)) {
                    $withdrawAmountInput = $amountValue;
                }

                if ($amount <= 0) {
                    $withdrawErrors[] = 'Tutar 0\'dan büyük olmalıdır.';
                }

                $balanceAvailable = (float)($user['balance'] ?? 0);
                if ($amount > $balanceAvailable) {
                    $withdrawErrors[] = 'Yeterli bakiyeniz bulunmuyor.';
                }

                if ($payoutAccountName === '' || $payoutBankName === '' || $payoutIban === '') {
                    $withdrawErrors[] = 'Önce banka bilgilerinizi kaydediniz.';
                }

                if (empty($withdrawErrors)) {
                    $pendingStmt = $pdo->prepare('SELECT COUNT(*) FROM withdraw_requests WHERE user_id = ? AND status = "pending"');
                    $pendingStmt->execute([$_SESSION['user_id']]);
                    if ((int)$pendingStmt->fetchColumn() > 0) {
                        $withdrawErrors[] = 'Bekleyen bir çekim talebiniz bulunuyor.';
                    }
                }

                if (empty($withdrawErrors)) {
                    $insert = $pdo->prepare('INSERT INTO withdraw_requests (user_id, amount, status, account_name, bank_name, iban, created_at) VALUES (?, ?, "pending", ?, ?, ?, NOW())');
                    $insert->execute([
                        $_SESSION['user_id'],
                        $amountValue,
                        $payoutAccountName,
                        $payoutBankName,
                        $payoutIban
                    ]);
                    $withdrawSuccess = 'Çekim talebiniz oluşturuldu. Onaylandığında bakiyenizden düşülecektir.';
                    $withdrawAmountInput = '';
                }
                break;

            default:
                $profileErrors[] = 'Geçersiz işlem.';
        }
    }
}

$totalOrdersStmt = $pdo->prepare('SELECT COUNT(*) FROM orders WHERE user_id = ?');
$totalOrdersStmt->execute([$_SESSION['user_id']]);
$totalOrders = (int)$totalOrdersStmt->fetchColumn();

$totalSpentStmt = $pdo->prepare("SELECT COALESCE(SUM(total_amount), 0) FROM orders WHERE user_id = ? AND status = 'completed'");
$totalSpentStmt->execute([$_SESSION['user_id']]);
$totalSpent = (float)$totalSpentStmt->fetchColumn();

$marketplaceTotals = [
    'buy_volume' => 0.0,
    'sell_volume' => 0.0,
    'buy_count' => 0,
    'sell_count' => 0
];
$marketplaceTotalsStmt = $pdo->prepare("SELECT order_type, COUNT(*) AS cnt, COALESCE(SUM(total_amount),0) AS total FROM game_marketplace_orders WHERE user_id = ? AND status IN ('completed','approved') GROUP BY order_type");
$marketplaceTotalsStmt->execute([$_SESSION['user_id']]);
foreach ($marketplaceTotalsStmt->fetchAll() as $row) {
    if ($row['order_type'] === 'buy') {
        $marketplaceTotals['buy_volume'] = (float)$row['total'];
        $marketplaceTotals['buy_count'] = (int)$row['cnt'];
    } elseif ($row['order_type'] === 'sell') {
        $marketplaceTotals['sell_volume'] = (float)$row['total'];
        $marketplaceTotals['sell_count'] = (int)$row['cnt'];
    }
}
$totalSpentIncludingMarketplace = $totalSpent + $marketplaceTotals['buy_volume'];
$marketplaceNet = $marketplaceTotals['sell_volume'] - $marketplaceTotals['buy_volume'];

$lastOrderStmt = $pdo->prepare('SELECT created_at FROM orders WHERE user_id = ? ORDER BY created_at DESC LIMIT 1');
$lastOrderStmt->execute([$_SESSION['user_id']]);
$lastOrder = $lastOrderStmt->fetchColumn();

$mpPerPage = 10;
$mpPage = isset($_GET['mp_page']) ? max(1, (int)$_GET['mp_page']) : 1;

$mpCountStmt = $pdo->prepare('SELECT COUNT(*) FROM game_marketplace_orders WHERE user_id = ?');
$mpCountStmt->execute([$_SESSION['user_id']]);
$mpTotalOrders = (int)$mpCountStmt->fetchColumn();
$mpTotalPages = max(1, (int)ceil($mpTotalOrders / $mpPerPage));
if ($mpPage > $mpTotalPages) {
    $mpPage = $mpTotalPages;
}
$mpOffset = ($mpPage - 1) * $mpPerPage;

$recentMarketplaceStmt = $pdo->prepare('SELECT o.*, s.name AS server_name, c.name AS category_name, m.name AS marketplace_name,
    (SELECT GROUP_CONCAT(mc.code ORDER BY mc.id SEPARATOR ", ") FROM game_marketplace_codes mc WHERE mc.order_id = o.id) AS delivered_codes
    FROM game_marketplace_orders o
    LEFT JOIN game_marketplace_servers s ON s.id = o.server_id
    LEFT JOIN game_marketplace_categories c ON c.id = o.category_id
    LEFT JOIN game_marketplaces m ON m.id = o.marketplace_id
    WHERE o.user_id = :user_id
    ORDER BY o.created_at DESC
    LIMIT :offset, :limit');
$recentMarketplaceStmt->bindValue(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);
$recentMarketplaceStmt->bindValue(':offset', $mpOffset, PDO::PARAM_INT);
$recentMarketplaceStmt->bindValue(':limit', $mpPerPage, PDO::PARAM_INT);
$recentMarketplaceStmt->execute();
$recentMarketplaceOrders = $recentMarketplaceStmt->fetchAll();

$depositSummaryStmt = $pdo->prepare('
    SELECT
        COUNT(*) AS total_count,
        SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'completed\' THEN 1 ELSE 0 END) AS completed_count,
        SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'pending\' THEN 1 ELSE 0 END) AS pending_count,
        SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'cancelled\' THEN 1 ELSE 0 END) AS cancelled_count,
        COALESCE(SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'completed\' THEN t.amount ELSE 0 END), 0) AS completed_total,
        COALESCE(SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'completed\'
            THEN CASE
                WHEN o.payable_amount IS NOT NULL THEN o.payable_amount
                ELSE t.amount + COALESCE(o.commission_amount, 0)
            END
            ELSE 0
        END), 0) AS completed_payable,
        COALESCE(SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'completed\'
            THEN CASE
                WHEN o.commission_amount IS NOT NULL THEN o.commission_amount
                WHEN o.payable_amount IS NOT NULL THEN o.payable_amount - t.amount
                ELSE 0
            END
            ELSE 0
        END), 0) AS completed_commission,
        COALESCE(SUM(CASE WHEN COALESCE(o.status, \'completed\') = \'pending\'
            THEN CASE
                WHEN o.payable_amount IS NOT NULL THEN o.payable_amount
                ELSE t.amount + COALESCE(o.commission_amount, 0)
            END
            ELSE 0
        END), 0) AS pending_total
    FROM transactions t
    LEFT JOIN orders o ON o.id = t.order_id
    WHERE t.user_id = :user_id AND t.type = \'deposit\'
');
$depositSummaryStmt->execute([':user_id' => $_SESSION['user_id']]);
$depositSummaryRow = $depositSummaryStmt->fetch(PDO::FETCH_ASSOC) ?: [];
$depositSummary = [
    'total_count' => (int)($depositSummaryRow['total_count'] ?? 0),
    'completed_count' => (int)($depositSummaryRow['completed_count'] ?? 0),
    'pending_count' => (int)($depositSummaryRow['pending_count'] ?? 0),
    'cancelled_count' => (int)($depositSummaryRow['cancelled_count'] ?? 0),
    'completed_total' => (float)($depositSummaryRow['completed_total'] ?? 0),
    'completed_payable' => (float)($depositSummaryRow['completed_payable'] ?? 0),
    'completed_commission' => (float)($depositSummaryRow['completed_commission'] ?? 0),
    'pending_total' => (float)($depositSummaryRow['pending_total'] ?? 0)
];

$depositPerPage = 10;
$depositPage = isset($_GET['dep_page']) ? max(1, (int)$_GET['dep_page']) : 1;
$depositCountStmt = $pdo->prepare('SELECT COUNT(*) FROM transactions WHERE user_id = ? AND type = \'deposit\'');
$depositCountStmt->execute([$_SESSION['user_id']]);
$depositTotalOrders = (int)$depositCountStmt->fetchColumn();
$depositTotalPages = max(1, (int)ceil($depositTotalOrders / $depositPerPage));
if ($depositPage > $depositTotalPages) {
    $depositPage = $depositTotalPages;
}
$depositOffset = ($depositPage - 1) * $depositPerPage;

$depositOrdersStmt = $pdo->prepare('
    SELECT
        t.*,
        o.id AS order_id,
        o.order_number,
        o.payment_method,
        o.status AS order_status,
        o.payable_amount,
        o.commission_amount,
        o.total_amount AS order_total,
        o.created_at AS order_created_at
    FROM transactions t
    LEFT JOIN orders o ON o.id = t.order_id
    WHERE t.user_id = :user_id AND t.type = \'deposit\'
    ORDER BY t.created_at DESC
    LIMIT :offset, :limit
');
$depositOrdersStmt->bindValue(':user_id', $_SESSION['user_id'], PDO::PARAM_INT);
$depositOrdersStmt->bindValue(':offset', $depositOffset, PDO::PARAM_INT);
$depositOrdersStmt->bindValue(':limit', $depositPerPage, PDO::PARAM_INT);
$depositOrdersStmt->execute();
$depositOrders = $depositOrdersStmt->fetchAll();

$withdrawStmt = $pdo->prepare('SELECT wr.*, au.username AS admin_username FROM withdraw_requests wr LEFT JOIN users au ON au.id = wr.admin_id WHERE wr.user_id = ? ORDER BY wr.created_at DESC');
$withdrawStmt->execute([$_SESSION['user_id']]);
$withdrawRequests = $withdrawStmt->fetchAll();
$hasPendingWithdraw = false;
foreach ($withdrawRequests as $withdrawRow) {
    if (isset($withdrawRow['status']) && strtolower((string)$withdrawRow['status']) === 'pending') {
        $hasPendingWithdraw = true;
        break;
    }
}

$depositQueryBase = $_GET;
unset($depositQueryBase['dep_page']);
$buildDepositPageUrl = static function (int $page) use ($depositQueryBase): string {
    $params = $depositQueryBase;
    if ($page > 1) {
        $params['dep_page'] = $page;
    } else {
        unset($params['dep_page']);
    }
    $query = http_build_query($params);
    $base = routeUrl('profile');
    return $base . ($query ? '?' . $query : '');
};

$mpQueryBase = $_GET;
unset($mpQueryBase['mp_page']);
$buildMpPageUrl = static function (int $page) use ($mpQueryBase): string {
    $params = $mpQueryBase;
    if ($page > 1) {
        $params['mp_page'] = $page;
    } else {
        unset($params['mp_page']);
    }
    $query = http_build_query($params);
    $base = routeUrl('profile');
    return $base . ($query ? '?' . $query : '');
};

$ordersStmt = $pdo->prepare('
    SELECT o.*, COUNT(oi.id) AS item_count
    FROM orders o
    LEFT JOIN order_items oi ON o.id = oi.order_id
    WHERE o.user_id = ?
    GROUP BY o.id
    ORDER BY o.created_at DESC
');
$ordersStmt->execute([$_SESSION['user_id']]);
$profileOrders = $ordersStmt->fetchAll();

$orderItemsMap = [];
if (!empty($profileOrders)) {
    $orderItemStmt = $pdo->prepare('
        SELECT oi.*, pc.code
        FROM order_items oi
        LEFT JOIN product_codes pc ON oi.product_code_id = pc.id
        WHERE oi.order_id = ?
    ');
    foreach ($profileOrders as $orderRow) {
        $orderItemStmt->execute([$orderRow['id']]);
        $orderItemsMap[$orderRow['id']] = $orderItemStmt->fetchAll();
    }
}

$fullNameDisplay = trim((string)($user['full_name'] ?? ''));
$phoneDisplay = trim((string)($user['phone'] ?? ''));
$registeredAt = !empty($user['created_at']) ? date('d.m.Y', strtotime((string)$user['created_at'])) : '';
$lastOrderDateText = $lastOrder ? date('d.m.Y', strtotime($lastOrder)) : '';
$lastOrderTimeText = $lastOrder ? date('H:i', strtotime($lastOrder)) : '';

$pageTitle = 'Profilim';
include 'includes/header.php';
?>

<section class="profile-view">
    <div class="container-wide">
        <div class="profile-header-card">
            <div class="profile-header-card__top">
                <div class="profile-header-card__identity">
                    <div class="profile-avatar profile-avatar--xl">
                        <i class="fa-solid fa-user"></i>
                    </div>
                    <div class="profile-header-card__text">
                        <span class="profile-header-card__eyebrow">Hesap özeti</span>
                        <h1 class="profile-header-card__title"><?= sanitize($user['username']) ?></h1>
                        <?php if ($fullNameDisplay !== ''): ?>
                            <p class="profile-header-card__subtitle"><?= sanitize($fullNameDisplay) ?></p>
                        <?php else: ?>
                            <p class="profile-header-card__subtitle profile-header-card__subtitle--muted">Profil bilgilerinizi hemen güncelleyebilirsiniz.</p>
                        <?php endif; ?>
                        <div class="profile-chip-list">
                            <span class="profile-chip"><i class="fa-solid fa-at"></i><?= sanitize($user['email']) ?></span>
                            <span class="profile-chip"><i class="fa-solid fa-id-badge"></i>ID <?= (int)$user['id'] ?></span>
                            <?php if ($phoneDisplay !== ''): ?>
                                <span class="profile-chip"><i class="fa-solid fa-phone"></i><?= sanitize($phoneDisplay) ?></span>
                            <?php endif; ?>
                            <?php if ($registeredAt !== ''): ?>
                                <span class="profile-chip"><i class="fa-solid fa-calendar-check"></i><?= sanitize($registeredAt) ?></span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <div class="profile-balance-card">
                    <span class="profile-balance-card__label">Bakiyeniz</span>
                    <strong class="profile-balance-card__value"><?= formatPrice($user['balance']) ?></strong>
                    <a class="profile-balance-card__action" href="<?= routeUrl('balance') ?>">
                        <i class="fa-solid fa-wallet"></i>Bakiye Yükle
                    </a>
                </div>
            </div>
            <div class="profile-header-card__stats">
                <div class="profile-stat-card">
                    <span class="profile-stat-card__label">Toplam Sipariş</span>
                    <strong class="profile-stat-card__value"><?= $totalOrders ?></strong>
                    <span class="profile-stat-card__meta">Siparişlerim sekmesinde detaylar</span>
                </div>
                <div class="profile-stat-card">
                    <span class="profile-stat-card__label">Toplam Harcama</span>
                    <strong class="profile-stat-card__value"><?= formatPrice($totalSpentIncludingMarketplace) ?></strong>
                    <span class="profile-stat-card__meta">Mağaza + Marketplace satın alımları</span>
                </div>
                <div class="profile-stat-card">
                    <span class="profile-stat-card__label">Marketplace Gelir</span>
                    <strong class="profile-stat-card__value"><?= formatPrice($marketplaceTotals['sell_volume']) ?></strong>
                    <span class="profile-stat-card__meta"><?= $marketplaceTotals['sell_count'] ?> satış tamamlandı</span>
                </div>
                <div class="profile-stat-card">
                    <span class="profile-stat-card__label">Marketplace Net</span>
                    <strong class="profile-stat-card__value<?= $marketplaceNet >= 0 ? '' : ' text-warning' ?>"><?= formatPrice($marketplaceNet) ?></strong>
                    <span class="profile-stat-card__meta">Satış - Alım farkı</span>
                </div>
            </div>
        </div>

        <div class="profile-content-grid">
            <div class="profile-settings-card">
                <div class="profile-card-heading">
                    <span class="profile-card-heading__eyebrow">Profil</span>
                    <h2>Hesap Bilgileri</h2>
                    <p>İletişim ve giriş bilgilerinizi güvenle güncelleyin.</p>
                </div>

                <?php if (!empty($profileErrors)): ?>
                    <div class="alert alert-danger">
                        <i class="fa-solid fa-circle-exclamation"></i>
                        <div class="alert-messages">
                            <?php foreach ($profileErrors as $error): ?>
                                <div><?= sanitize($error) ?></div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <?php if ($profileSuccess !== ''): ?>
                    <div class="alert alert-success">
                        <i class="fa-solid fa-circle-check"></i><?= sanitize($profileSuccess) ?>
                    </div>
                <?php endif; ?>

                <form class="profile-settings-form" method="POST">
                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                    <input type="hidden" name="form_action" value="update_profile">

                    <div class="profile-settings-form__grid">
                        <div class="profile-field">
                            <label class="form-label">Kullanıcı Adı</label>
                            <input type="text" class="form-control" value="<?= sanitize($user['username']) ?>" disabled>
                            <small>Kullanıcı adı değiştirilemez.</small>
                        </div>
                        <div class="profile-field">
                            <label class="form-label">E-posta</label>
                            <input type="email" class="form-control" name="email" value="<?= sanitize($user['email']) ?>" required>
                        </div>
                        <div class="profile-field">
                            <label class="form-label">Ad Soyad</label>
                            <input type="text" class="form-control" name="full_name" value="<?= sanitize($fullNameDisplay) ?>" required>
                        </div>
                        <div class="profile-field">
                            <label class="form-label">Telefon</label>
                            <input type="text" class="form-control" name="phone" value="<?= sanitize($phoneDisplay) ?>" placeholder="5XX XXX XX XX">
                        </div>
                    </div>

                    <div class="profile-divider"></div>

                    <div class="profile-card-heading profile-card-heading--sub">
                        <span class="profile-card-heading__eyebrow">Ödemeler</span>
                        <h3>Banka Bilgilerim</h3>
                        <p>Bakiye çekim taleplerinizin sorunsuz tamamlanması için hesap bilgilerinizi güncel tutun.</p>
                    </div>

                    <div class="profile-settings-form__grid">
                        <div class="profile-field">
                            <label class="form-label">Hesap Sahibi</label>
                            <input type="text" class="form-control" name="payout_account_name" value="<?= sanitize($payoutAccountName) ?>" placeholder="Ad Soyad">
                        </div>
                        <div class="profile-field">
                            <label class="form-label">Banka Adı</label>
                            <input type="text" class="form-control" name="payout_bank_name" value="<?= sanitize($payoutBankName) ?>" placeholder="Banka / Şube">
                        </div>
                        <div class="profile-field">
                            <label class="form-label">IBAN</label>
                            <input type="text" class="form-control" name="payout_iban" value="<?= sanitize($formatIbanDisplay($payoutIban)) ?>" placeholder="TR00 0000 0000 0000 0000 0000 00">
                            <small>IBAN başında ülke kodu (ör. TR) olacak şekilde boşluksuz veya dört hanede bir boşlukla yazabilirsiniz.</small>
                        </div>
                    </div>

                    <div class="profile-divider"></div>

                    <div class="profile-card-heading profile-card-heading--sub">
                        <span class="profile-card-heading__eyebrow">Güvenlik</span>
                        <h3>Şifre Değişikliği</h3>
                        <p>Şifrenizi güncellemek istemiyorsanız alanları boş bırakabilirsiniz.</p>
                    </div>

                    <div class="profile-settings-form__grid profile-settings-form__grid--three">
                        <div class="profile-field">
                            <label class="form-label">Mevcut Şifre</label>
                            <input type="password" class="form-control" name="current_password" autocomplete="current-password">
                        </div>
                        <div class="profile-field">
                            <label class="form-label">Yeni Şifre</label>
                            <input type="password" class="form-control" name="new_password" autocomplete="new-password">
                        </div>
                        <div class="profile-field">
                            <label class="form-label">Yeni Şifre Tekrar</label>
                            <input type="password" class="form-control" name="new_password_confirm" autocomplete="new-password">
                        </div>
                    </div>

                    <div class="profile-settings-form__actions">
                        <button type="submit" class="btn-primary">
                            <i class="fa-solid fa-floppy-disk"></i>Kaydet
                        </button>
                    </div>
                </form>
            </div>

            <aside class="profile-side-card">
                <div class="profile-card-heading">
                    <span class="profile-card-heading__eyebrow">Hızlı İşlemler</span>
                    <h3>Kontrol Paneli</h3>
                    <p>En sık kullanılan sayfalara tek dokunuşla ulaşın.</p>
                </div>
                <ul class="profile-side-card__list">
                    <li class="profile-side-card__item">
                        <a href="<?= routeUrl('deposits') ?>">
                            <span class="profile-side-card__icon"><i class="fa-solid fa-wallet"></i></span>
                            <div>
                                <strong>Bakiye Yüklemelerim</strong>
                                <small><?= $depositSummary['completed_count'] ?> tamamlandı / <?= formatPrice($depositSummary['completed_total']) ?></small>
                            </div>
                            <i class="fa-solid fa-chevron-right"></i>
                        </a>
                    </li>
                    <li class="profile-side-card__item">
                        <a href="<?= routeUrl('marketplace_orders') ?>">
                            <span class="profile-side-card__icon"><i class="fa-solid fa-cart-shopping"></i></span>
                            <div>
                                <strong>Marketplace İşlemlerim</strong>
                                <small><?= $marketplaceTotals['buy_count'] + $marketplaceTotals['sell_count'] ?> işlem / Net: <?= formatPrice($marketplaceNet) ?></small>
                            </div>
                            <i class="fa-solid fa-chevron-right"></i>
                        </a>
                    </li>
                    <li class="profile-side-card__item">
                        <a href="<?= routeUrl('orders') ?>">
                            <span class="profile-side-card__icon"><i class="fa-solid fa-bag-shopping"></i></span>
                            <div>
                                <strong>Siparişlerim</strong>
                                <small><?= $totalOrders ?> sipariş / <?= formatPrice($totalSpent) ?> harcama</small>
                            </div>
                            <i class="fa-solid fa-chevron-right"></i>
                        </a>
                    </li>
                    <li class="profile-side-card__item">
                        <a href="<?= routeUrl('withdrawals') ?>">
                            <span class="profile-side-card__icon"><i class="fa-solid fa-money-bill-transfer"></i></span>
                            <div>
                                <strong>Bakiye Çekimlerim</strong>
                                <small><?= count($withdrawRequests) ?> talep geçmişi</small>
                            </div>
                            <i class="fa-solid fa-chevron-right"></i>
                        </a>
                    </li>
                    <li class="profile-side-card__item">
                        <a href="<?= routeUrl('balance') ?>">
                            <span class="profile-side-card__icon profile-side-card__icon--primary"><i class="fa-solid fa-plus"></i></span>
                            <div>
                                <strong>Bakiye Yükle</strong>
                                <small>Hızlı ve güvenli yükleme</small>
                            </div>
                            <i class="fa-solid fa-arrow-up-right-from-square"></i>
                        </a>
                    </li>
                    <li class="profile-side-card__item">
                        <a href="<?= routeUrl('market') ?>">
                            <span class="profile-side-card__icon profile-side-card__icon--success"><i class="fa-solid fa-store"></i></span>
                            <div>
                                <strong>Market</strong>
                                <small>Ürün al</small>
                            </div>
                            <i class="fa-solid fa-arrow-up-right-from-square"></i>
                        </a>
                    </li>
                    <li class="profile-side-card__item">
                        <a href="<?= routeUrl('support') ?>">
                            <span class="profile-side-card__icon profile-side-card__icon--info"><i class="fa-solid fa-life-ring"></i></span>
                            <div>
                                <strong>Destek</strong>
                                <small>7/24 yardım hattı</small>
                            </div>
                            <i class="fa-solid fa-arrow-up-right-from-square"></i>
                        </a>
                    </li>
                </ul>
                <div class="profile-side-card__note">
                    <i class="fa-solid fa-shield-heart"></i>
                    <div>
                        <strong>Güvenlik İpucu</strong>
                        <p>Şifrenizi paylaşmayın, e-posta adresinizi güncel tutun ve şüpheli bir işlem gördüğünüzde hemen bizimle iletişime geçin.</p>
                    </div>
                </div>
                <div class="profile-side-card__note profile-side-card__note--alt">
                    <i class="fa-solid fa-chart-line"></i>
                    <div>
                        <strong>Marketplace Özeti</strong>
                        <p>Toplam Alım: <?= $marketplaceTotals['buy_count'] ?> adet / <?= formatPrice($marketplaceTotals['buy_volume']) ?><br>Toplam Satış: <?= $marketplaceTotals['sell_count'] ?> adet / <?= formatPrice($marketplaceTotals['sell_volume']) ?></p>
                    </div>
                </div>
            </aside>
        </div>
    </div>
</section>

<section class="profile-history-section">
    <div class="container-wide">
        <div class="profile-history-head">
            <div class="profile-card-heading">
                <span class="profile-card-heading__eyebrow">Geçmiş</span>
                <h2>İşlem Kayıtları</h2>
                <p>Marketplace ve sipariş hareketlerinizi buradan takip edin.</p>
            </div>
        </div>

        <ul class="nav nav-pills profile-tabs" id="profileTabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link <?= $activeTab === 'deposits' ? 'active' : '' ?>" id="tab-deposits-trigger" data-bs-toggle="pill" data-bs-target="#tab-deposits" type="button" role="tab" aria-controls="tab-deposits" aria-selected="<?= $activeTab === 'deposits' ? 'true' : 'false' ?>">
                    <i class="fa-solid fa-wallet"></i>Bakiye Yüklemelerim
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?= $activeTab === 'marketplace' ? 'active' : '' ?>" id="tab-marketplace-trigger" data-bs-toggle="pill" data-bs-target="#tab-marketplace" type="button" role="tab" aria-controls="tab-marketplace" aria-selected="<?= $activeTab === 'marketplace' ? 'true' : 'false' ?>">
                    <i class="fa-solid fa-cart-shopping"></i>Marketplace İşlemlerim
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?= $activeTab === 'orders' ? 'active' : '' ?>" id="tab-orders-trigger" data-bs-toggle="pill" data-bs-target="#tab-orders" type="button" role="tab" aria-controls="tab-orders" aria-selected="<?= $activeTab === 'orders' ? 'true' : 'false' ?>">
                    <i class="fa-solid fa-bag-shopping"></i>Siparişlerim
                </button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?= $activeTab === 'withdraw' ? 'active' : '' ?>" id="tab-withdraw-trigger" data-bs-toggle="pill" data-bs-target="#tab-withdraw" type="button" role="tab" aria-controls="tab-withdraw" aria-selected="<?= $activeTab === 'withdraw' ? 'true' : 'false' ?>">
                    <i class="fa-solid fa-money-bill-transfer"></i>Bakiye Çekimlerim
                </button>
            </li>
        </ul>

        <div class="tab-content profile-tab-content" id="profileTabsContent">
            <div class="tab-pane fade <?= $activeTab === 'deposits' ? 'show active' : '' ?>" id="tab-deposits" role="tabpanel" aria-labelledby="tab-deposits-trigger">
                <div class="profile-metric-strip">
                    <div class="profile-metric-card">
                        <span>Toplam İşlem</span>
                        <strong><?= number_format($depositSummary['total_count'], 0, ',', '.') ?></strong>
                        <small>Tüm bakiye yüklemeleriniz</small>
                    </div>
                    <div class="profile-metric-card">
                        <span>Tamamlanan Tutar</span>
                        <strong><?= formatPrice($depositSummary['completed_total']) ?></strong>
                        <small><?= $depositSummary['completed_count'] ?> onaylı işlem</small>
                    </div>
                    <div class="profile-metric-card">
                        <span>Ödenen Tutar</span>
                        <strong class="text-info"><?= formatPrice($depositSummary['completed_payable']) ?></strong>
                        <small>Komisyon dahil toplam ödeme</small>
                    </div>
                    <div class="profile-metric-card">
                        <span>Bekleyen</span>
                        <strong class="text-warning"><?= formatPrice($depositSummary['pending_total']) ?></strong>
                        <small><?= $depositSummary['pending_count'] ?> işlem beklemede</small>
                    </div>
                </div>
                <?php if (empty($depositOrders)): ?>
                    <div class="empty-state">
                        <i class="fa-solid fa-wallet"></i>
                        <p>Henüz bakiye yükleme işleminiz bulunmuyor.</p>
                    </div>
                <?php else: ?>
                    <div class="profile-history__table">
                        <div class="table-responsive">
                            <table class="profile-table">
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th>Tarih</th>
                                        <th>Ödeme</th>
                                        <th>Sipariş No</th>
                                        <th>Yüklenen</th>
                                        <th>Komisyon</th>
                                        <th>Ödenen</th>
                                        <th>Durum</th>
                                        <th>Açıklama</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php
                                    $depositStatusMap = [
                                        'pending' => ['class' => 'status-badge status-pending', 'label' => 'Beklemede'],
                                        'completed' => ['class' => 'status-badge status-completed', 'label' => 'Tamamlandı'],
                                        'cancelled' => ['class' => 'status-badge status-cancelled', 'label' => 'İptal'],
                                    ];
                                    ?>
                                    <?php foreach ($depositOrders as $deposit): ?>
                                    <?php
                                        $orderStatusKey = strtolower((string)($deposit['order_status'] ?? 'completed'));
                                        $statusInfo = $depositStatusMap[$orderStatusKey] ?? ['class' => 'status-badge status-completed', 'label' => $deposit['order_status'] ?? 'Tamamlandı'];
                                        $depositDate = $deposit['created_at'] ?? $deposit['order_created_at'] ?? null;
                                        $paymentMethodRaw = trim((string)($deposit['payment_method'] ?? ''));
                                        $paymentMethod = $paymentMethodRaw !== '' ? strtoupper($paymentMethodRaw) : ($deposit['description'] ? 'MANUEL' : '-');
                                        $netAmount = (float)($deposit['amount'] ?? 0);
                                        $orderPayable = isset($deposit['payable_amount']) ? (float)$deposit['payable_amount'] : null;
                                        $orderCommission = isset($deposit['commission_amount']) ? (float)$deposit['commission_amount'] : null;
                                        if ($orderPayable === null && $orderCommission !== null) {
                                            $orderPayable = $netAmount + $orderCommission;
                                        }
                                        if ($orderCommission === null && $orderPayable !== null) {
                                            $orderCommission = max($orderPayable - $netAmount, 0);
                                        }
                                        $grossAmount = $orderPayable ?? ($netAmount + ($orderCommission ?? 0));
                                        $commissionAmount = $orderCommission ?? max($grossAmount - $netAmount, 0);
                                        $description = $deposit['description'] ?? '';
                                        $orderNumber = $deposit['order_number'] ?? '';
                                        $identifier = $deposit['order_id'] ?? $deposit['id'];
                                    ?>
                                    <tr>
                                        <td><?= (int)$identifier ?></td>
                                        <td><?= $depositDate ? date('d.m.Y H:i', strtotime($depositDate)) : '-' ?></td>
                                        <td>
                                            <span class="profile-tag profile-tag--deposit"><i class="fa-solid fa-credit-card"></i><?= sanitize($paymentMethod) ?></span>
                                        </td>
                                        <td>
                                            <?php if ($orderNumber !== ''): ?>
                                                <?= sanitize($orderNumber) ?>
                                            <?php else: ?>
                                                <span class="profile-muted">—</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><strong class="text-success">+<?= formatPrice($netAmount) ?></strong></td>
                                        <td class="text-warning">+<?= formatPrice($commissionAmount) ?></td>
                                        <td><strong class="text-info"><?= formatPrice($grossAmount) ?></strong></td>
                                        <td><span class="<?= $statusInfo['class'] ?>"><?= sanitize($statusInfo['label']) ?></span></td>
                                        <td>
                                            <?php if ($description !== ''): ?>
                                                <?= sanitize($description) ?>
                                            <?php else: ?>
                                                <span class="profile-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <?php if ($depositTotalPages > 1): ?>
                    <nav class="profile-pagination" aria-label="Bakiye yüklemeleri sayfalama">
                        <ul class="pagination pagination-sm justify-content-end">
                            <li class="page-item<?= $depositPage <= 1 ? ' disabled' : '' ?>">
                                <a class="page-link" href="<?= $depositPage <= 1 ? '#' : htmlspecialchars($buildDepositPageUrl($depositPage - 1), ENT_QUOTES, 'UTF-8') ?>" tabindex="<?= $depositPage <= 1 ? '-1' : '0' ?>">Önceki</a>
                            </li>
                            <?php for ($i = 1; $i <= $depositTotalPages; $i++): ?>
                                <li class="page-item<?= $i === $depositPage ? ' active' : '' ?>">
                                    <a class="page-link" href="<?= htmlspecialchars($buildDepositPageUrl($i), ENT_QUOTES, 'UTF-8') ?>"><?= $i ?></a>
                                </li>
                            <?php endfor; ?>
                            <li class="page-item<?= $depositPage >= $depositTotalPages ? ' disabled' : '' ?>">
                                <a class="page-link" href="<?= $depositPage >= $depositTotalPages ? '#' : htmlspecialchars($buildDepositPageUrl($depositPage + 1), ENT_QUOTES, 'UTF-8') ?>" tabindex="<?= $depositPage >= $depositTotalPages ? '-1' : '0' ?>">Sonraki</a>
                            </li>
                        </ul>
                    </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>

            <div class="tab-pane fade <?= $activeTab === 'marketplace' ? 'show active' : '' ?>" id="tab-marketplace" role="tabpanel" aria-labelledby="tab-marketplace-trigger">
                <div class="profile-metric-strip">
                    <div class="profile-metric-card">
                        <span>Alım Sayısı</span>
                        <strong><?= $marketplaceTotals['buy_count'] ?></strong>
                        <small><?= formatPrice($marketplaceTotals['buy_volume']) ?></small>
                    </div>
                    <div class="profile-metric-card">
                        <span>Satış Sayısı</span>
                        <strong><?= $marketplaceTotals['sell_count'] ?></strong>
                        <small><?= formatPrice($marketplaceTotals['sell_volume']) ?></small>
                    </div>
                    <div class="profile-metric-card">
                        <span>Marketplace Net</span>
                        <strong class="<?= $marketplaceNet >= 0 ? 'text-success' : 'text-warning' ?>"><?= formatPrice($marketplaceNet) ?></strong>
                        <small>Satış - Alım</small>
                    </div>
                </div>
                <?php if (empty($recentMarketplaceOrders)): ?>
                    <div class="empty-state">
                        <i class="fa-solid fa-inbox"></i>
                        <p>Henüz marketplace işleminiz bulunmuyor.</p>
                    </div>
                <?php else: ?>
                    <div class="profile-history__table">
                        <div class="table-responsive">
                            <table class="profile-table">
                                <thead>
                                    <tr>
                                        <th>#</th>
                                        <th>Tarih</th>
                                        <th>Oyun / Kategori</th>
                                        <th>Server</th>
                                        <th>Tür</th>
                                        <th>Miktar</th>
                                        <th>Birim</th>
                                        <th>Toplam</th>
                                        <th>Durum</th>
                                        <th></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php
                                    $marketplaceStatusMap = [
                                        'pending' => ['class' => 'status-badge status-pending', 'label' => 'Beklemede'],
                                        'completed' => ['class' => 'status-badge status-completed', 'label' => 'Tamamlandı'],
                                        'cancelled' => ['class' => 'status-badge status-cancelled', 'label' => 'İptal'],
                                    ];
                                    ?>
                                    <?php foreach ($recentMarketplaceOrders as $mpOrder): ?>
                                    <?php
                                        $statusKey = strtolower((string)$mpOrder['status']);
                                        $statusInfo = $marketplaceStatusMap[$statusKey] ?? ['class' => 'status-badge status-pending', 'label' => $mpOrder['status']];
                                    ?>
                                    <tr>
                                        <td><?= $mpOrder['id'] ?></td>
                                        <td><?= date('d.m.Y H:i', strtotime($mpOrder['created_at'])) ?></td>
                                        <td>
                                            <strong><?= sanitize($mpOrder['marketplace_name'] ?? 'Silindi') ?></strong>
                                            <div class="profile-muted"><?= sanitize($mpOrder['category_name'] ?? '-') ?></div>
                                        </td>
                                        <td><?= sanitize($mpOrder['server_name'] ?? '-') ?></td>
                                        <td>
                                            <?php if ($mpOrder['order_type'] === 'sell'): ?>
                                                <span class="profile-tag profile-tag--sell">Bize Sat</span>
                                            <?php else: ?>
                                                <span class="profile-tag profile-tag--buy">Satın Al</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= number_format((float)$mpOrder['quantity'], 2, ',', '.') ?></td>
                                        <td><?= formatPrice($mpOrder['price_per_unit']) ?></td>
                                        <td><?= formatPrice($mpOrder['total_amount']) ?></td>
                                        <td><span class="<?= $statusInfo['class'] ?>"><?= sanitize($statusInfo['label']) ?></span></td>
                                        <td class="text-end">
                                            <button type="button" class="btn-outline profile-detail-btn" data-order-detail
                                                data-order-id="<?= htmlspecialchars((string)$mpOrder['id'], ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-date="<?= htmlspecialchars(date('d.m.Y H:i', strtotime($mpOrder['created_at'])), ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-game="<?= htmlspecialchars($mpOrder['marketplace_name'] ?? '-', ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-category="<?= htmlspecialchars($mpOrder['category_name'] ?? '-', ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-server="<?= htmlspecialchars($mpOrder['server_name'] ?? '-', ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-type="<?= htmlspecialchars($mpOrder['order_type'], ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-status="<?= htmlspecialchars($mpOrder['status'], ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-quantity="<?= htmlspecialchars(number_format((float)$mpOrder['quantity'], 2, ',', '.'), ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-price="<?= htmlspecialchars(formatPrice($mpOrder['price_per_unit']), ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-total="<?= htmlspecialchars(formatPrice($mpOrder['total_amount']), ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-character="<?= htmlspecialchars($mpOrder['character_name'] ?? '-', ENT_QUOTES, 'UTF-8') ?>"
                                                data-order-codes="<?= htmlspecialchars($mpOrder['delivered_codes'] ?? '-', ENT_QUOTES, 'UTF-8') ?>">
                                                Detay
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <?php if ($mpTotalPages > 1): ?>
                    <nav class="profile-pagination" aria-label="Marketplace sayfalama">
                        <ul class="pagination pagination-sm justify-content-end">
                            <li class="page-item<?= $mpPage <= 1 ? ' disabled' : '' ?>">
                                <a class="page-link" href="<?= $mpPage <= 1 ? '#' : htmlspecialchars($buildMpPageUrl($mpPage - 1), ENT_QUOTES, 'UTF-8') ?>" tabindex="<?= $mpPage <= 1 ? '-1' : '0' ?>">Önceki</a>
                            </li>
                            <?php for ($i = 1; $i <= $mpTotalPages; $i++): ?>
                                <li class="page-item<?= $i === $mpPage ? ' active' : '' ?>">
                                    <a class="page-link" href="<?= htmlspecialchars($buildMpPageUrl($i), ENT_QUOTES, 'UTF-8') ?>"><?= $i ?></a>
                                </li>
                            <?php endfor; ?>
                            <li class="page-item<?= $mpPage >= $mpTotalPages ? ' disabled' : '' ?>">
                                <a class="page-link" href="<?= $mpPage >= $mpTotalPages ? '#' : htmlspecialchars($buildMpPageUrl($mpPage + 1), ENT_QUOTES, 'UTF-8') ?>" tabindex="<?= $mpPage >= $mpTotalPages ? '-1' : '0' ?>">Sonraki</a>
                            </li>
                        </ul>
                    </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>

            <div class="tab-pane fade <?= $activeTab === 'orders' ? 'show active' : '' ?>" id="tab-orders" role="tabpanel" aria-labelledby="tab-orders-trigger">
                <?php if (empty($profileOrders)): ?>
                    <div class="empty-state">
                        <i class="fa-solid fa-box-open"></i>
                        <p>Henüz siparişiniz bulunmuyor.</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($profileOrders as $orderRow): ?>
                        <?php
                        $orderStatusClass = 'status-badge status-pending';
                        $statusLabel = 'Beklemede';
                        if ($orderRow['status'] === 'completed') {
                            $orderStatusClass = 'status-badge status-completed';
                            $statusLabel = 'Tamamlandı';
                        } elseif ($orderRow['status'] === 'cancelled') {
                            $orderStatusClass = 'status-badge status-cancelled';
                            $statusLabel = 'İptal';
                        }
                        $items = $orderItemsMap[$orderRow['id']] ?? [];
                        ?>
                        <article class="order-card">
                            <div class="order-head">
                                <div>
                                    <h5><i class="fa-solid fa-hashtag me-2"></i><?= sanitize($orderRow['order_number']) ?></h5>
                                    <div class="order-meta">
                                        <span><i class="fa-solid fa-calendar me-1"></i><?= date('d.m.Y H:i', strtotime($orderRow['created_at'])) ?></span>
                                        <span><i class="fa-solid fa-box me-1"></i><?= (int)$orderRow['item_count'] ?> ürün</span>
                                    </div>
                                </div>
                                <div class="text-end">
                                    <span class="<?= $orderStatusClass ?>"><?= $statusLabel ?></span>
                                    <div class="order-total"><?= formatPrice($orderRow['total_amount']) ?></div>
                                </div>
                            </div>
                            <div class="order-body">
                                <div class="table-responsive">
                                    <table class="order-items-table">
                                        <thead>
                                            <tr>
                                                <th>Ürün</th>
                                                <th class="text-center">Adet</th>
                                                <th class="text-center">Fiyat</th>
                                                <th>Kod</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($items as $item): ?>
                                            <tr>
                                                <td><strong><?= sanitize($item['product_name']) ?></strong></td>
                                                <td class="text-center"><?= (int)$item['quantity'] ?></td>
                                                <td class="text-center"><?= formatPrice($item['price']) ?></td>
                                                <td>
                                                    <?php if (!empty($item['code'])): ?>
                                                        <span class="code-badge"><?= sanitize($item['code']) ?></span>
                                                        <button type="button" class="copy-action" data-copy-code="<?= sanitize($item['code']) ?>">
                                                            <i class="fa-solid fa-copy"></i>
                                                        </button>
                                                    <?php else: ?>
                                                        <span class="profile-muted">Kod henüz tanımlanmadı</span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </article>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>

            <div class="tab-pane fade <?= $activeTab === 'withdraw' ? 'show active' : '' ?>" id="tab-withdraw" role="tabpanel" aria-labelledby="tab-withdraw-trigger">
                <?php if ($withdrawSuccess !== ''): ?>
                    <div class="alert alert-success">
                        <i class="fa-solid fa-circle-check"></i><?= sanitize($withdrawSuccess) ?>
                    </div>
                <?php endif; ?>
                <?php if (!empty($withdrawErrors)): ?>
                    <div class="alert alert-danger">
                        <i class="fa-solid fa-circle-exclamation"></i>
                        <div class="alert-messages">
                            <?php foreach ($withdrawErrors as $error): ?>
                                <div><?= sanitize($error) ?></div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <div class="profile-content-grid">
                    <div class="profile-settings-card">
                        <div class="profile-card-heading profile-card-heading--sub">
                            <span class="profile-card-heading__eyebrow">Bakiye Çekimi</span>
                            <h3>Yeni Çekim Talebi</h3>
                            <p>Bakiyenizi banka hesabınıza aktarmak için talep oluşturun.</p>
                        </div>
                        <form method="post" class="profile-settings-form">
                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                            <input type="hidden" name="form_action" value="create_withdraw">
                            <div class="profile-settings-form__grid">
                                <div class="profile-field">
                                    <label class="form-label">Mevcut Bakiye</label>
                                    <input type="text" class="form-control" value="<?= formatPrice($user['balance']) ?>" readonly>
                                </div>
                                <div class="profile-field">
                                    <label class="form-label">Çekilecek Tutar</label>
                                    <div class="input-group">
                                        <input type="number" class="form-control" name="withdraw_amount" min="10" step="0.01" value="<?= sanitize($withdrawAmountInput) ?>" placeholder="Örn. 250"<?= $hasPendingWithdraw ? ' disabled' : '' ?>>
                                        <span class="input-group-text"><?= CURRENCY ?></span>
                                    </div>
                                </div>
                            </div>

                            <?php if ($payoutAccountName === '' || $payoutBankName === '' || $payoutIban === ''): ?>
                                <div class="alert alert-warning mt-3">
                                    <i class="fa-solid fa-circle-exclamation me-2"></i>
                                    Çekim talebi gönderebilmek için banka bilgilerinizi yukarıdan kaydetmelisiniz.
                                </div>
                            <?php endif; ?>

                            <?php if ($hasPendingWithdraw): ?>
                                <div class="alert alert-info mt-3">
                                    <i class="fa-solid fa-clock me-2"></i>
                                    Bekleyen bir çekim talebiniz mevcut. Yeni talep gönderebilmek için mevcut talebin sonuçlanmasını bekleyin.
                                </div>
                            <?php else: ?>
                                <div class="alert alert-light mt-3 text-muted">
                                    <ul class="mb-0">
                                        <li>Kayıtlı banka: <strong><?= $payoutAccountName !== '' ? sanitize($payoutAccountName) : 'Henüz eklenmedi' ?></strong> / <strong><?= $payoutBankName !== '' ? sanitize($payoutBankName) : 'Henüz eklenmedi' ?></strong></li>
                                        <li>IBAN: <strong><?= $payoutIban !== '' ? sanitize($formatIbanDisplay($payoutIban)) : 'Henüz eklenmedi' ?></strong></li>
                                        <li>Onaylanan talepler bakiyenizden düşülür ve kayıt altına alınır.</li>
                                    </ul>
                                </div>
                            <?php endif; ?>

                            <div class="profile-settings-form__actions">
                                <button type="submit" class="btn-primary"<?= $hasPendingWithdraw || $payoutAccountName === '' || $payoutBankName === '' || $payoutIban === '' ? ' disabled' : '' ?>>
                                    <i class="fa-solid fa-paper-plane"></i>Talep Gönder
                                </button>
                            </div>
                        </form>
                    </div>
                    <aside class="profile-side-card">
                        <div class="profile-card-heading">
                            <span class="profile-card-heading__eyebrow">Geçmiş</span>
                            <h3>Çekim Talepleri</h3>
                            <p>Onay süreci ve ödemelerinizi buradan takip edin.</p>
                        </div>
                        <?php if (empty($withdrawRequests)): ?>
                            <div class="empty-state">
                                <i class="fa-solid fa-money-bill-wave"></i>
                                <p>Henüz bakiye çekim talebiniz bulunmuyor.</p>
                            </div>
                        <?php else: ?>
                            <div class="profile-history__table">
                                <div class="table-responsive">
                                    <table class="profile-table">
                                        <thead>
                                            <tr>
                                                <th>#</th>
                                                <th>Tarih</th>
                                                <th>Tutar</th>
                                                <th>Durum</th>
                                                <th>Not</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($withdrawRequests as $request): ?>
                                                <?php
                                                    $status = strtolower((string)$request['status']);
                                                    $statusBadge = 'status-badge status-pending';
                                                    $statusLabel = 'Beklemede';
                                                    if ($status === 'approved') {
                                                        $statusBadge = 'status-badge status-completed';
                                                        $statusLabel = 'Tamamlandı';
                                                    } elseif ($status === 'rejected') {
                                                        $statusBadge = 'status-badge status-cancelled';
                                                        $statusLabel = 'Reddedildi';
                                                    }
                                                ?>
                                                <tr>
                                                    <td><?= (int)$request['id'] ?></td>
                                                    <td><?= date('d.m.Y H:i', strtotime($request['created_at'])) ?></td>
                                                    <td><strong class="text-danger">-<?= formatPrice($request['amount']) ?></strong></td>
                                                    <td><span class="<?= $statusBadge ?>"><?= sanitize($statusLabel) ?></span></td>
                                                    <td>
                                                        <?php if (!empty($request['admin_note'])): ?>
                                                            <?= sanitize($request['admin_note']) ?>
                                                        <?php else: ?>
                                                            <span class="profile-muted">-</span>
                                                        <?php endif; ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        <?php endif; ?>
                    </aside>
                </div>
            </div>
        </div>
    </div>
</section>

<div class="modal fade" id="marketplaceOrderModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered modal-lg">
        <div class="modal-content">
            <div class="modal-header border-0">
                <h5 class="modal-title"><i class="fa-solid fa-receipt me-2"></i>İşlem Detayı</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Kapat"></button>
            </div>
            <div class="modal-body">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="profile-muted d-block">İşlem No</label>
                        <strong data-field="id">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Tarih</label>
                        <strong data-field="date">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Oyun</label>
                        <strong data-field="game">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Kategori</label>
                        <strong data-field="category">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Server</label>
                        <strong data-field="server">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Karakter Adı</label>
                        <strong data-field="character">—</strong>
                    </div>
                </div>
                <hr>
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="profile-muted d-block">Tür</label>
                        <span class="badge" data-field="type">—</span>
                    </div>
                    <div class="col-md-4">
                        <label class="profile-muted d-block">Durum</label>
                        <span class="badge bg-secondary" data-field="status">—</span>
                    </div>
                    <div class="col-md-4">
                        <label class="profile-muted d-block">Miktar</label>
                        <strong data-field="quantity">—</strong>
                    </div>
                </div>
                <div class="row g-3 mt-2">
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Birim Fiyat</label>
                        <strong data-field="price">—</strong>
                    </div>
                    <div class="col-md-6">
                        <label class="profile-muted d-block">Toplam Tutar</label>
                        <strong data-field="total">—</strong>
                    </div>
                </div>
                <div class="mt-3">
                    <label class="profile-muted d-block">Teslim Edilen Kodlar</label>
                    <div class="badge bg-secondary" data-field="codes">—</div>
                </div>
            </div>
            <div class="modal-footer border-0">
                <button type="button" class="btn-outline" data-bs-dismiss="modal">Kapat</button>
            </div>
        </div>
    </div>
</div>

<?php
$scriptVersionMarketplaceModal = file_exists(__DIR__ . '/assets/js/marketplace-modal.js') ? filemtime(__DIR__ . '/assets/js/marketplace-modal.js') : time();
$scriptVersionProfileMarketplace = file_exists(__DIR__ . '/assets/js/profile-marketplace.js') ? filemtime(__DIR__ . '/assets/js/profile-marketplace.js') : time();
?>
<script>window.__MARKETPLACE_MODAL_OVERRIDE = true;</script>
<script src="<?= htmlspecialchars(publicUrl('assets/js/marketplace-modal.js'), ENT_QUOTES, 'UTF-8') ?>?v=<?= $scriptVersionMarketplaceModal ?>"></script>
<script src="<?= htmlspecialchars(publicUrl('assets/js/profile-marketplace.js'), ENT_QUOTES, 'UTF-8') ?>?v=<?= $scriptVersionProfileMarketplace ?>"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Tab scroll navigation
    const tabScrollLinks = document.querySelectorAll('[data-tab-scroll]');
    
    tabScrollLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const targetTab = this.getAttribute('data-tab-scroll');
            const tabTrigger = document.querySelector('#tab-' + targetTab + '-trigger');
            const tabsSection = document.querySelector('.profile-history-section');
            
            if (tabTrigger && tabsSection) {
                // Activate the tab using Bootstrap 5 API
                if (typeof bootstrap !== 'undefined' && bootstrap.Tab) {
                    const tab = new bootstrap.Tab(tabTrigger);
                    tab.show();
                } else {
                    // Fallback: manual tab activation
                    document.querySelectorAll('.profile-tabs .nav-link').forEach(t => {
                        t.classList.remove('active');
                        t.setAttribute('aria-selected', 'false');
                    });
                    tabTrigger.classList.add('active');
                    tabTrigger.setAttribute('aria-selected', 'true');
                    
                    document.querySelectorAll('.profile-tab-content .tab-pane').forEach(p => {
                        p.classList.remove('show', 'active');
                    });
                    const targetPane = document.querySelector('#tab-' + targetTab);
                    if (targetPane) {
                        targetPane.classList.add('show', 'active');
                    }
                }
                
                // Smooth scroll to tabs section with offset for fixed header
                setTimeout(() => {
                    const headerOffset = 120;
                    const elementPosition = tabsSection.getBoundingClientRect().top;
                    const offsetPosition = elementPosition + window.pageYOffset - headerOffset;
                    
                    window.scrollTo({
                        top: offsetPosition,
                        behavior: 'smooth'
                    });
                }, 150);
            }
        });
    });
    
    // Debug: Log when script is loaded
    console.log('Profile tab scroll script loaded. Found ' + tabScrollLinks.length + ' tab scroll links.');
});
</script>
<?php include 'includes/footer.php'; ?>
