<?php
require_once 'config.php';

if (isLoggedIn()) {
    redirect('index.php');
}

$errors = [];
$requireVerification = emailVerificationRequired();
$recaptchaEnabledRegister = recaptchaEnabled('register');
$recaptchaSiteKey = getSetting('recaptcha_site_key', RECAPTCHA_SITE_KEY);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Güvenlik hatası. Lütfen tekrar deneyin.';
    } else {
        if ($recaptchaEnabledRegister && $recaptchaSiteKey) {
            $recaptchaToken = safe_trim($_POST['g-recaptcha-response'] ?? '');
            if (!verifyRecaptcha($recaptchaToken)) {
                $errors[] = 'Doğrulama başarısız. Lütfen reCAPTCHA kontrolünü tamamlayın.';
            }
        }
        if (empty($errors)) {
            $username = safe_trim($_POST['username'] ?? '');
            $email = safe_trim($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';
            $passwordConfirm = $_POST['password_confirm'] ?? '';
            $full_name = safe_trim($_POST['full_name'] ?? '');
            $phone = safe_trim($_POST['phone'] ?? '');

            if ($username === '' || $email === '' || $password === '' || $passwordConfirm === '') {
                $errors[] = 'Lütfen tüm zorunlu alanları doldurun.';
            }

            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'Geçerli bir e-posta adresi girin.';
            }

            if ($password !== $passwordConfirm) {
                $errors[] = 'Şifreler uyuşmuyor.';
            }

            if (strlen($password) < 6) {
                $errors[] = 'Şifreniz en az 6 karakter olmalıdır.';
            }

            if (empty($errors)) {
                $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
                $stmt->execute([$username, $email]);
                if ($stmt->fetch()) {
                    $errors[] = 'Bu kullanıcı adı veya e-posta zaten kayıtlı.';
                    logUserActivity($pdo, 'auth_register_duplicate', ['username' => $username, 'email' => $email]);
                }
            }

            if (empty($errors)) {
                try {
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    $initialVerified = $requireVerification ? 0 : 1;
                    $verifiedAt = $requireVerification ? null : date('Y-m-d H:i:s');
                    $stmt = $pdo->prepare('
                        INSERT INTO users (username, email, password, full_name, phone, balance, email_verified, email_verified_at)
                        VALUES (?, ?, ?, ?, ?, 0.00, ?, ?)
                    ');
                    $stmt->execute([$username, $email, $hashed_password, $full_name, $phone, $initialVerified, $verifiedAt]);

                    $newUserId = (int)$pdo->lastInsertId();
                    logUserActivity($pdo, 'auth_register_success', ['username' => $username, 'email' => $email, 'user_id' => $newUserId, 'email_verification_required' => $requireVerification]);

                    if ($requireVerification) {
                        $verificationSent = false;
                        $verification = prepareEmailVerification($pdo, $newUserId, true);
                        if ($verification && isset($verification['token'], $verification['user'])) {
                            $verificationUrl = getEmailVerificationUrl($verification['token']);
                            if ($email !== '') {
                                $verificationSent = sendEmailVerificationEmail($verification['user'], $verificationUrl);
                            }
                            logUserActivity($pdo, 'auth_register_verification_sent', ['user_id' => $newUserId, 'email' => $email, 'sent' => $verificationSent]);
                        }

                        if ($verificationSent) {
                            flashMessage('success', 'Kayıt başarılı! E-posta adresine gönderdiğimiz doğrulama bağlantısını onayladıktan sonra giriş yapabilirsin.');
                        } else {
                            flashMessage('warning', 'Kayıt başarılı ancak doğrulama e-postası gönderilemedi. Lütfen destek ekibimizle iletişime geçin.');
                        }
                    } else {
                        if ($email !== '') {
                            $subject = siteName() . ' - Hoş geldiniz';
                            $body = '<p>Merhaba ' . sanitize($username) . ',</p>'
                                . '<p>' . sanitize(siteName()) . ' ailesine katıldığınız için teşekkür ederiz. Giriş yaparak hemen kullanmaya başlayabilirsiniz.</p>'
                                . '<p>Sevgiler,<br>' . sanitize(siteName()) . ' Ekibi</p>';
                            @sendEmail($email, $subject, $body, strip_tags($body));
                        }
                        flashMessage('success', 'Kayıt başarılı! Şimdi giriş yapabilirsiniz.');
                    }

                    redirect('login.php');
                } catch (PDOException $e) {
                    $errors[] = 'Kayıt sırasında bir hata oluştu. Lütfen tekrar deneyin.';
                    logUserActivity($pdo, 'auth_register_error', ['username' => $username, 'email' => $email, 'error' => $e->getMessage()]);
                }
            }
        }
    }
}

$pageTitle = 'Kayıt Ol';
include 'includes/header.php';
?>

<section class="auth-section">
    <div class="container-wide">
        <div class="auth-layout">
            <div class="auth-highlight card-glass">
                <div class="auth-branding">
                    <i class="fa-solid fa-user-plus auth-icon"></i>
                    <h1 class="auth-brand-title"><?= sanitize(siteName()) ?></h1>
                </div>
                <h2 class="auth-highlight-title">Hemen Başlayın!</h2>
                <p class="auth-highlight-desc">
                    <?= sanitize(siteName()) ?> ile oyun dünyasının en avantajlı fiyatlarına erişin.
                    Kayıt ücretsiz ve hızlıdır.
                </p>
                <ul class="auth-benefits">
                    <li><i class="fa-solid fa-shield-halved"></i>%100 güvenli ödeme altyapısı</li>
                    <li><i class="fa-solid fa-bolt"></i>Anında teslimat garantisi</li>
                    <li><i class="fa-solid fa-gift"></i>Özel kampanya ve bonuslar</li>
                    <li><i class="fa-solid fa-crown"></i>VIP müşteri desteği</li>
                </ul>
                <div class="auth-cta">
                    <p class="auth-cta-text">Zaten hesabınız var mı?</p>
                    <a href="<?= routeUrl('login') ?>" class="btn btn-outline-primary btn-lg">
                        <i class="fa-solid fa-right-to-bracket me-2"></i>Giriş Yapın
                    </a>
                </div>
            </div>
            <div class="auth-card card-glass">
                <h3>Kayıt Ol</h3>
                <?php if (!empty($errors)): ?>
                <div class="alert alert-danger">
                    <i class="fa-solid fa-circle-exclamation me-2"></i><?= implode('<br>', array_map('sanitize', $errors)) ?>
                </div>
                <?php endif; ?>
                <form method="post" class="auth-form"<?= $recaptchaEnabledRegister && $recaptchaSiteKey ? ' data-recaptcha-form' : '' ?> autocomplete="off">
                    <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label for="username" class="form-label">Kullanıcı Adı *</label>
                            <input type="text" id="username" name="username" class="form-control" value="<?= sanitize($_POST['username'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label for="email" class="form-label">E-posta *</label>
                            <input type="email" id="email" name="email" class="form-control" value="<?= sanitize($_POST['email'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label for="password" class="form-label">Şifre *</label>
                            <input type="password" id="password" name="password" class="form-control" required>
                        </div>
                        <div class="col-md-6">
                            <label for="password_confirm" class="form-label">Şifre (Tekrar) *</label>
                            <input type="password" id="password_confirm" name="password_confirm" class="form-control" required>
                        </div>
                        <div class="col-md-6">
                            <label for="full_name" class="form-label">Ad Soyad</label>
                            <input type="text" id="full_name" name="full_name" class="form-control" value="<?= sanitize($_POST['full_name'] ?? '') ?>">
                        </div>
                        <div class="col-md-6">
                            <label for="phone" class="form-label">Telefon</label>
                            <input type="text" id="phone" name="phone" class="form-control" value="<?= sanitize($_POST['phone'] ?? '') ?>">
                        </div>
                    </div>
                    <div class="form-check mt-3">
                        <input class="form-check-input" type="checkbox" value="1" id="kvkk" required>
                        <label class="form-check-label" for="kvkk">
                            Kayıt olarak <a href="#">kullanıcı sözleşmesini</a> kabul ediyorum.
                        </label>
                    </div>
                    <?php if ($recaptchaEnabledRegister && $recaptchaSiteKey): ?>
                        <div class="mt-3" data-recaptcha data-site-key="<?= sanitize($recaptchaSiteKey) ?>"></div>
                    <?php endif; ?>
                    <button type="submit" class="btn btn-primary w-100 mt-4">
                        <i class="fa-solid fa-user-plus me-2"></i>Kayıt Ol
                    </button>
                </form>
            </div>
        </div>
    </div>
</section>

<?php include 'includes/footer.php'; ?>
