<?php
require_once 'config.php';

if (!function_exists('resolveCategoryAsset')) {
    function resolveCategoryAsset(array $entity, array $map): string
    {
        if (!empty($entity['image_path'])) {
            $relative = ltrim((string)$entity['image_path'], '/\\');
            $fullPath = __DIR__ . '/' . $relative;
            if (is_file($fullPath)) {
                return $relative;
            }
        }
        if (!empty($entity['image'])) {
            $relative = ltrim((string)$entity['image'], '/\\');
            $fullPath = __DIR__ . '/' . $relative;
            if (is_file($fullPath)) {
                return $relative;
            }
        }

        $slug = $entity['category_slug'] ?? ($entity['slug'] ?? '');
        if (isset($map[$slug])) {
            return $map[$slug];
        }
        foreach ($map as $key => $path) {
            if ($slug !== '' && str_contains($slug, $key)) {
                return $path;
            }
        }
        return 'assets/images/category-oyun.svg';
    }
}

$categoryMediaMap = [
    'oyun-pinleri' => 'assets/images/category-oyun.svg',
    'oyunlar' => 'assets/images/category-oyun.svg',
    'oyun' => 'assets/images/category-oyun.svg',
    'pazaryeri' => 'assets/images/category-market.svg',
    'market' => 'assets/images/category-market.svg',
    'oyun-parasi' => 'assets/images/category-bakiye.svg',
    'bakiye' => 'assets/images/category-bakiye.svg',
];

$searchQuery = trim($_GET['q'] ?? '');
$pageTitle = !empty($searchQuery) ? "'{$searchQuery}' Araması" : 'Arama';
$metaDescription = !empty($searchQuery) ? "{$searchQuery} için arama sonuçları" : 'Oyun kodları, marketplace ürünleri ve daha fazlasını arayın';

// Ürünlerde arama
$products = [];
if ($searchQuery !== '') {
    try {
        $sql = "SELECT p.*, c.name AS category_name, c.slug AS category_slug
                FROM products p
                LEFT JOIN categories c ON p.category_id = c.id
                WHERE p.is_active = 1
                AND (p.name LIKE ? OR (p.description IS NOT NULL AND p.description LIKE ?))
                ORDER BY p.featured DESC, p.created_at DESC
                LIMIT 50";
        $searchTerm = '%' . $searchQuery . '%';
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$searchTerm, $searchTerm]);
        $products = $stmt->fetchAll();
        if ($products === false) {
            $products = [];
        }
    } catch (Throwable $th) {
        // Hata durumunda logla ama boş array döndür
        error_log('Search products error: ' . $th->getMessage());
        $products = [];
    }
}

// Marketplace'de arama (Server bazlı)
$marketplaceItems = [];
$marketplaceDebug = '';
if ($searchQuery !== '') {
    try {
        // Önce tabloların var olup olmadığını kontrol et
        $tablesExist = true;
        try {
            $pdo->query("SELECT 1 FROM game_marketplace_servers LIMIT 1");
            $pdo->query("SELECT 1 FROM game_marketplace_categories LIMIT 1");
            $pdo->query("SELECT 1 FROM game_marketplaces LIMIT 1");
        } catch (Throwable $e) {
            $tablesExist = false;
            $marketplaceDebug = 'Marketplace tabloları bulunamadı: ' . $e->getMessage();
        }
        
        if ($tablesExist) {
            // Basitleştirilmiş arama: Tüm terimleri tek seferde ara
            $searchTerm = '%' . $searchQuery . '%';
            
            $sql = "SELECT 
                    ms.id AS server_id,
                    ms.name AS server_name,
                    ms.slug AS server_slug,
                    ms.description AS server_description,
                    ms.image AS server_image,
                    ms.sell_price,
                    ms.buy_price,
                    ms.sell_stock,
                    ms.min_sell_quantity,
                    ms.max_sell_quantity,
                    ms.min_buy_quantity,
                    ms.max_buy_quantity,
                    gm.id AS game_id,
                    gm.name AS game_name,
                    gm.slug AS game_slug,
                    cat.id AS category_id,
                    cat.name AS category_name,
                    cat.slug AS category_slug,
                    cat.image AS category_image,
                    cat.banner_image AS category_banner_image
                FROM game_marketplace_servers ms 
                INNER JOIN game_marketplace_categories cat ON ms.category_id = cat.id 
                INNER JOIN game_marketplaces gm ON cat.marketplace_id = gm.id 
                WHERE ms.is_active = 1 
                AND cat.is_active = 1 
                AND gm.is_active = 1 
                AND (
                    gm.name LIKE ? 
                    OR gm.slug LIKE ? 
                    OR cat.name LIKE ? 
                    OR cat.slug LIKE ?
                    OR ms.name LIKE ? 
                    OR ms.slug LIKE ?
                    OR ms.description LIKE ?
                )
                ORDER BY ms.display_order ASC, ms.name ASC
                LIMIT 50";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm]);
            $marketplaceItems = $stmt->fetchAll();
            
            if ($marketplaceItems === false) {
                $marketplaceItems = [];
            }
            
            // Debug: Kaç sonuç bulundu?
            $marketplaceDebug = 'Marketplace sonuç sayısı: ' . count($marketplaceItems) . ' | Arama terimi: "' . $searchQuery . '"';
        }
        
    } catch (Throwable $th) {
        // Hata durumunda logla ama boş array döndür
        error_log('Search marketplace error: ' . $th->getMessage());
        error_log('Search marketplace SQL: ' . ($sql ?? 'N/A'));
        $marketplaceItems = [];
        $marketplaceDebug = 'Hata: ' . $th->getMessage();
    }
}

$totalResults = count($products) + count($marketplaceItems);

include 'includes/header.php';
?>

<section class="page-section">
    <div class="container-wide">
        <div class="page-header">
            <div>
                <nav aria-label="breadcrumb" class="breadcrumb-nav">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="<?= routeUrl('home') ?>">Ana Sayfa</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Arama</li>
                    </ol>
                </nav>
                <h1 class="page-title">
                    <?php if (!empty($searchQuery)): ?>
                        '<?= sanitize($searchQuery) ?>' Araması
                    <?php else: ?>
                        Arama
                    <?php endif; ?>
                </h1>
                <?php if (!empty($searchQuery)): ?>
                    <p class="text-muted">'<?= sanitize($searchQuery) ?>' için sonuçlar</p>
                <?php endif; ?>
            </div>
        </div>

        <?php if (empty($searchQuery)): ?>
            <div class="alert alert-info">
                <i class="fa-solid fa-info-circle me-2"></i>
                Arama yapmak için yukarıdaki arama kutusuna bir şeyler yazın.
            </div>
        <?php elseif ($totalResults === 0): ?>
            <?php if (!empty($marketplaceDebug) && isAdmin()): ?>
            <div class="alert alert-warning">
                <strong>Debug (Sadece Admin Görür):</strong> <?= sanitize($marketplaceDebug) ?>
            </div>
            <?php endif; ?>
            <div class="card">
                <div class="card-body text-center py-5">
                    <i class="fa-solid fa-bag-shopping text-muted mb-3" style="font-size: 4rem; opacity: 0.3;"></i>
                    <h3>Ürün bulunamadı</h3>
                    <p class="text-muted mb-4">
                        Aradığınız kriterlere uygun ürün bulunamadı. Diğer kategorilere göz atmayı deneyebilirsiniz.
                    </p>
                    <a href="<?= routeUrl('products') ?>" class="btn btn-primary">
                        <i class="fa-solid fa-rotate-right me-2"></i>Ürünleri yeniden listele
                    </a>
                </div>
            </div>
        <?php else: ?>
            <!-- Ürünler -->
            <?php if (!empty($products)): ?>
                <div class="mb-5">
                    <div class="section-heading mb-4">
                        <h2>
                            <i class="fa-solid fa-box"></i>
                            Ürünler
                        </h2>
                        <span class="badge bg-primary"><?= count($products) ?> sonuç</span>
                    </div>
                    <div class="product-catalog">
                        <?php foreach ($products as $product): ?>
                            <?php
                                $productUrl = routeUrl('product', ['slug' => $product['slug']]);
                                $imagePath = resolveCategoryAsset($product, $categoryMediaMap);
                                $imageUrl = htmlspecialchars(assetUrl($imagePath), ENT_QUOTES, 'UTF-8');
                                $hasStock = $product['stock'] === null || (int)$product['stock'] > 0;
                            ?>
                            <article class="product-card<?= !$hasStock ? ' product-card--out-of-stock' : '' ?>">
                                <a href="<?= $productUrl ?>" class="product-card__image">
                                    <img src="<?= $imageUrl ?>" alt="<?= sanitize($product['name']) ?>" loading="lazy">
                                </a>
                                <div class="product-card__body">
                                    <?php if (!empty($product['category_name'])): ?>
                                        <a href="<?= routeUrl('product_category', ['slug' => $product['category_slug']]) ?>" class="product-card__category">
                                            <?= sanitize($product['category_name']) ?>
                                        </a>
                                    <?php endif; ?>
                                    <h3 class="product-card__title">
                                        <a href="<?= $productUrl ?>"><?= sanitize($product['name']) ?></a>
                                    </h3>
                                    <?php if (!empty($product['description'])): ?>
                                        <p class="product-card__description"><?= sanitize(mb_substr($product['description'], 0, 80)) ?>...</p>
                                    <?php endif; ?>
                                    <div class="product-card__footer">
                                        <span class="product-card__price"><?= formatPrice($product['price']) ?></span>
                                        <?php if ($hasStock): ?>
                                            <a href="<?= $productUrl ?>" class="btn btn-sm btn-primary">
                                                <i class="fa-solid fa-basket-shopping me-1"></i>İncele
                                            </a>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Stokta Yok</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </article>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Marketplace -->
            <?php if (!empty($marketplaceItems)): ?>
                <div>
                    <div class="section-heading mb-4">
                        <h2>
                            <i class="fa-solid fa-store"></i>
                            Marketplace
                        </h2>
                        <span class="badge bg-primary"><?= count($marketplaceItems) ?> sonuç</span>
                    </div>
                    <div class="product-catalog">
                        <?php foreach ($marketplaceItems as $server): ?>
                            <?php
                                // Marketplace server URL'i category üzerinden oluştur
                                $itemUrl = routeUrl('marketplace_category', ['game' => $server['game_slug'], 'category' => $server['category_slug']]);
                                
                                // Resim belirleme
                                $serverImage = !empty($server['server_image']) ? assetUrl($server['server_image']) : null;
                                $fallbackImage = !empty($server['category_image']) ? $server['category_image'] : (!empty($server['category_banner_image']) ? $server['category_banner_image'] : 'assets/images/category-market.svg');
                                if (is_array($fallbackImage)) {
                                    $fallbackImage = 'assets/images/category-market.svg';
                                }
                                $displayImage = $serverImage ?: assetUrl($fallbackImage);
                                
                                $hasBuyPrice = isset($server['buy_price']) && (float)$server['buy_price'] > 0;
                            ?>
                            <article class="product-card marketplace-card">
                                <div class="product-card__media">
                                    <a href="<?= $itemUrl ?>">
                                        <img src="<?= htmlspecialchars($displayImage, ENT_QUOTES, 'UTF-8') ?>" alt="<?= sanitize($server['server_name']) ?>" loading="lazy">
                                    </a>
                                    <span class="product-card__badge"><i class="fa-solid fa-server"></i><?= sanitize($server['server_name']) ?></span>
                                </div>
                                <div class="product-card__body">
                                    <a href="<?= routeUrl('marketplace_game', ['game' => $server['game_slug']]) ?>" class="product-card__category">
                                        <?= sanitize($server['game_name']) ?>
                                    </a>
                                    <h5 class="product-card__title">
                                        <a href="<?= $itemUrl ?>"><?= sanitize($server['server_name']) ?></a>
                                    </h5>
                                    <?php if (!empty($server['server_description'])): ?>
                                        <p class="product-card__summary"><?= sanitize(mb_substr($server['server_description'], 0, 140)) ?></p>
                                    <?php endif; ?>
                                    <div class="marketplace-specs">
                                        <div class="marketplace-spec marketplace-spec--sell">
                                            <span class="marketplace-spec__label"><i class="fa-solid fa-bolt"></i>Satış</span>
                                            <strong><?= formatPrice($server['sell_price']) ?></strong>
                                        </div>
                                        <?php if ($hasBuyPrice): ?>
                                        <div class="marketplace-spec marketplace-spec--buy">
                                            <span class="marketplace-spec__label"><i class="fa-solid fa-hand-holding-dollar"></i>Bize Sat</span>
                                            <strong><?= formatPrice($server['buy_price']) ?></strong>
                                        </div>
                                        <?php endif; ?>
                                        <div class="marketplace-spec marketplace-spec--stock">
                                            <span class="marketplace-spec__label"><i class="fa-solid fa-box"></i>Stok</span>
                                            <strong><?= number_format((float)$server['sell_stock'], 0, ',', '.') ?></strong>
                                        </div>
                                    </div>
                                </div>
                                <div class="product-card__actions">
                                    <a href="<?= $itemUrl ?>" class="btn btn-primary w-100">
                                        <i class="fa-solid fa-shopping-bag me-1"></i>Satın Al
                                    </a>
                                </div>
                            </article>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</section>

<?php include 'includes/footer.php'; ?>

