<?php
require_once 'config.php';
requireLogin();

$userId = (int)$_SESSION['user_id'];
$statusMap = getSupportTicketStatusMap();
$statusFilter = isset($_GET['status']) ? strtolower(trim((string)$_GET['status'])) : '';
if ($statusFilter !== '' && !isValidSupportTicketStatus($statusFilter)) {
    $statusFilter = '';
}

$ticketId = isset($_GET['id']) ? max(0, (int)$_GET['id']) : 0;
$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $errors[] = 'Güvenlik doğrulaması başarısız. Lütfen tekrar deneyin.';
    } else {
        $action = $_POST['action'] ?? '';

        if ($action === 'create_ticket') {
            $subject = trim($_POST['subject'] ?? '');
            $messageBody = trim($_POST['message'] ?? '');
            $attachmentPath = null;

            if ($subject === '') {
                $errors[] = 'Talep konusu zorunludur.';
            }
            if ($messageBody === '') {
                $errors[] = 'Talep mesajı boş olamaz.';
            }

            if (empty($errors) && isset($_FILES['attachment'])) {
                try {
                    $attachmentPath = uploadImageFile($_FILES['attachment'], 'tickets');
                } catch (Throwable $th) {
                    $errors[] = 'Dosya yüklenemedi: ' . $th->getMessage();
                }
            }

            if (empty($errors)) {
                try {
                    $pdo->beginTransaction();
                    $ticketStmt = $pdo->prepare('INSERT INTO support_tickets (user_id, subject, status, last_status_changed_at, last_message_at) VALUES (:user_id, :subject, :status, NOW(), NOW())');
                    $ticketStmt->execute([
                        ':user_id' => $userId,
                        ':subject' => $subject,
                        ':status' => 'open',
                    ]);
                    $newTicketId = (int)$pdo->lastInsertId();

                    $messageStmt = $pdo->prepare('INSERT INTO support_ticket_messages (ticket_id, sender_type, sender_id, message, attachment_path) VALUES (:ticket_id, :sender_type, :sender_id, :message, :attachment)');
                    $messageStmt->execute([
                        ':ticket_id' => $newTicketId,
                        ':sender_type' => 'user',
                        ':sender_id' => $userId,
                        ':message' => $messageBody,
                        ':attachment' => $attachmentPath,
                    ]);

                    $pdo->commit();

                    logUserActivity($pdo, 'support_ticket_created', ['ticket_id' => $newTicketId, 'subject' => $subject]);
                    flashMessage('success', 'Destek talebiniz oluşturuldu. En kısa sürede dönüş yapılacaktır.');
                    redirect('support.php?id=' . $newTicketId);
                } catch (Throwable $th) {
                    $pdo->rollBack();
                    $errors[] = 'Talep oluşturulurken bir hata oluştu: ' . $th->getMessage();
                }
            }
        }

        if ($action === 'reply_ticket') {
            $replyTicketId = isset($_POST['ticket_id']) ? (int)$_POST['ticket_id'] : 0;
            $messageBody = trim($_POST['message'] ?? '');
            $attachmentPath = null;

            if ($replyTicketId <= 0) {
                $errors[] = 'Geçersiz talep numarası.';
            }
            if ($messageBody === '') {
                $errors[] = 'Yanıt mesajı boş olamaz.';
            }

            $ticketRow = null;
            if (empty($errors)) {
                $ticketStmt = $pdo->prepare('SELECT * FROM support_tickets WHERE id = :id AND user_id = :user_id LIMIT 1');
                $ticketStmt->execute([':id' => $replyTicketId, ':user_id' => $userId]);
                $ticketRow = $ticketStmt->fetch();
                if (!$ticketRow) {
                    $errors[] = 'Destek talebi bulunamadı.';
                }
            }

            if (empty($errors) && isset($_FILES['attachment'])) {
                try {
                    $attachmentPath = uploadImageFile($_FILES['attachment'], 'tickets');
                } catch (Throwable $th) {
                    $errors[] = 'Dosya yüklenemedi: ' . $th->getMessage();
                }
            }

            if (empty($errors) && $ticketRow) {
                $currentStatus = $ticketRow['status'];
                $newStatus = $currentStatus;
                if (in_array($currentStatus, ['answered', 'resolved'], true)) {
                    $newStatus = 'customer_reply';
                } elseif ($currentStatus !== 'open') {
                    $newStatus = 'open';
                }

                try {
                    $pdo->beginTransaction();

                    $messageStmt = $pdo->prepare('INSERT INTO support_ticket_messages (ticket_id, sender_type, sender_id, message, attachment_path) VALUES (:ticket_id, :sender_type, :sender_id, :message, :attachment)');
                    $messageStmt->execute([
                        ':ticket_id' => $replyTicketId,
                        ':sender_type' => 'user',
                        ':sender_id' => $userId,
                        ':message' => $messageBody,
                        ':attachment' => $attachmentPath,
                    ]);

                    if ($newStatus !== $currentStatus) {
                        $updateStmt = $pdo->prepare('UPDATE support_tickets SET status = :status, last_status_changed_at = NOW(), last_message_at = NOW(), updated_at = NOW() WHERE id = :id AND user_id = :user_id');
                        $updateStmt->execute([
                            ':status' => $newStatus,
                            ':id' => $replyTicketId,
                            ':user_id' => $userId,
                        ]);
                    } else {
                        $updateStmt = $pdo->prepare('UPDATE support_tickets SET last_message_at = NOW(), updated_at = NOW() WHERE id = :id AND user_id = :user_id');
                        $updateStmt->execute([
                            ':id' => $replyTicketId,
                            ':user_id' => $userId,
                        ]);
                    }

                    $pdo->commit();

                    logUserActivity($pdo, 'support_ticket_replied', ['ticket_id' => $replyTicketId, 'status' => $newStatus]);
                    flashMessage('success', 'Yanıtınız gönderildi. Destek ekibimiz kısa süre içinde dönüş yapacaktır.');
                    redirect('support.php?id=' . $replyTicketId);
                } catch (Throwable $th) {
                    $pdo->rollBack();
                    $errors[] = 'Yanıt gönderilirken bir hata oluştu: ' . $th->getMessage();
                }
            }
        }
    }
}

$ticketList = [];
// Pagination
$perPage = 10;
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;

try {
    // Get total count
    if ($statusFilter !== '') {
        $countStmt = $pdo->prepare('SELECT COUNT(*) FROM support_tickets WHERE user_id = :user_id AND status = :status');
        $countStmt->execute([':user_id' => $userId, ':status' => $statusFilter]);
    } else {
        $countStmt = $pdo->prepare('SELECT COUNT(*) FROM support_tickets WHERE user_id = :user_id');
        $countStmt->execute([':user_id' => $userId]);
    }
    $totalTickets = (int)$countStmt->fetchColumn();
    $totalPages = max(1, (int)ceil($totalTickets / $perPage));
    if ($page > $totalPages) {
        $page = $totalPages;
    }
    $offset = ($page - 1) * $perPage;

    // Get tickets with pagination
    if ($statusFilter !== '') {
        $stmt = $pdo->prepare('SELECT * FROM support_tickets WHERE user_id = :user_id AND status = :status ORDER BY updated_at DESC LIMIT :limit OFFSET :offset');
        $stmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
        $stmt->bindValue(':status', $statusFilter, PDO::PARAM_STR);
        $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
    } else {
        $stmt = $pdo->prepare('SELECT * FROM support_tickets WHERE user_id = :user_id ORDER BY updated_at DESC LIMIT :limit OFFSET :offset');
        $stmt->bindValue(':user_id', $userId, PDO::PARAM_INT);
        $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
    }
    $ticketList = $stmt->fetchAll() ?: [];
} catch (Throwable $th) {
    $ticketList = [];
    $totalPages = 1;
}

$activeTicket = null;
$ticketMessages = [];
if ($ticketId > 0) {
    $ticketStmt = $pdo->prepare('SELECT * FROM support_tickets WHERE id = :id AND user_id = :user_id LIMIT 1');
    $ticketStmt->execute([':id' => $ticketId, ':user_id' => $userId]);
    $activeTicket = $ticketStmt->fetch();

    if ($activeTicket) {
        $messagesStmt = $pdo->prepare('SELECT m.*, m.sender_type, m.sender_id, m.message, m.attachment_path, m.created_at, u.username AS admin_username FROM support_ticket_messages m LEFT JOIN users u ON (m.sender_type = "admin" AND m.sender_id = u.id) WHERE m.ticket_id = :ticket_id ORDER BY m.created_at ASC');
        $messagesStmt->execute([':ticket_id' => $ticketId]);
        $ticketMessages = $messagesStmt->fetchAll() ?: [];
    }
}

$statusBadgeMap = [
    'open' => 'badge bg-warning text-dark',
    'answered' => 'badge bg-primary',
    'customer_reply' => 'badge bg-info text-dark',
    'resolved' => 'badge bg-success',
];

$pageTitle = 'Destek Taleplerim';
include 'includes/header.php';
?>

<section class="section-padding">
    <div class="container-wide">
        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <ul class="mb-0">
                    <?php foreach ($errors as $error): ?>
                        <li><?= sanitize($error) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
        <div class="row g-4">
            <div class="col-lg-4">
                <div class="card card-glass">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="fa-solid fa-life-ring me-2"></i>Destek Taleplerim</h5>
                        <span class="badge bg-secondary"><?= count($ticketList) ?></span>
                    </div>
                    <div class="card-body">
                        <a href="<?= routeUrl('support') ?>" class="btn btn-primary w-100 mb-3" data-bs-toggle="collapse" data-bs-target="#newTicketForm" aria-expanded="<?= empty($errors) ? 'false' : 'true' ?>" aria-controls="newTicketForm">
                            <i class="fa-solid fa-plus me-2"></i>Yeni Talep Oluştur
                        </a>

                        <div class="collapse<?= (!empty($errors) && ($_POST['action'] ?? '') === 'create_ticket') ? ' show' : '' ?>" id="newTicketForm">
                            <form method="post" enctype="multipart/form-data" class="card card-glass mb-3" style="padding: 1.25rem;">
                                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                <input type="hidden" name="action" value="create_ticket">
                                <div class="mb-3">
                                    <label class="form-label">Konu *</label>
                                    <input type="text" name="subject" class="form-control" value="<?= sanitize((($_POST['action'] ?? '') === 'create_ticket') ? ($_POST['subject'] ?? '') : '') ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Mesaj *</label>
                                    <textarea name="message" class="form-control" rows="4" required><?= sanitize((($_POST['action'] ?? '') === 'create_ticket') ? ($_POST['message'] ?? '') : '') ?></textarea>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Görsel (isteğe bağlı)</label>
                                    <input type="file" name="attachment" class="form-control" accept="image/png,image/jpeg,image/webp,image/gif">
                                    <small class="text-muted">Maksimum 4MB, PNG/JPG/WEBP/GIF</small>
                                </div>
                                <button type="submit" class="btn btn-success w-100"><i class="fa-solid fa-paper-plane me-2"></i>Talebi Gönder</button>
                            </form>
                        </div>

                        <div class="list-group">
                            <?php if (empty($ticketList)): ?>
                                <div class="text-center text-muted py-4">Henüz destek talebiniz bulunmuyor.</div>
                            <?php else: ?>
                                <?php foreach ($ticketList as $ticket): ?>
                                    <?php
                                        $statusKey = $ticket['status'];
                                        $badgeClass = $statusBadgeMap[$statusKey] ?? 'badge bg-secondary';
                                        $isActive = $activeTicket && (int)$activeTicket['id'] === (int)$ticket['id'];
                                    ?>
                                    <a class="list-group-item list-group-item-action<?= $isActive ? ' active' : '' ?>" href="<?= htmlspecialchars(routeUrl('support_ticket', ['id' => (int)$ticket['id']]), ENT_QUOTES, 'UTF-8') ?>">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <strong><?= sanitize($ticket['subject']) ?></strong>
                                                <div class="small text-muted">#<?= (int)$ticket['id'] ?> • <?= date('d.m.Y H:i', strtotime($ticket['created_at'])) ?></div>
                                            </div>
                                            <span class="<?= $badgeClass ?>"><?= sanitize(getSupportTicketStatusLabel($statusKey)) ?></span>
                                        </div>
                                    </a>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                        
                        <?php if ($totalPages > 1): ?>
                            <div class="card-footer">
                                <nav class="pagination-nav">
                                    <ul class="pagination justify-content-center mb-0">
                                        <?php if ($page > 1): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="<?= routeUrl('support') ?>?page=<?= $page - 1 ?><?= $statusFilter ? '&status=' . urlencode($statusFilter) : '' ?>">
                                                    <i class="fa-solid fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>

                                        <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                            <li class="page-item<?= $i === $page ? ' active' : '' ?>">
                                                <a class="page-link" href="<?= routeUrl('support') ?>?page=<?= $i ?><?= $statusFilter ? '&status=' . urlencode($statusFilter) : '' ?>"><?= $i ?></a>
                                            </li>
                                        <?php endfor; ?>

                                        <?php if ($page < $totalPages): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="<?= routeUrl('support') ?>?page=<?= $page + 1 ?><?= $statusFilter ? '&status=' . urlencode($statusFilter) : '' ?>">
                                                    <i class="fa-solid fa-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div class="col-lg-8">
                <div class="card card-glass h-100">
                    <?php if ($activeTicket): ?>
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <div>
                                <h5 class="mb-0">#<?= (int)$activeTicket['id'] ?> • <?= sanitize($activeTicket['subject']) ?></h5>
                                <small class="text-muted">Durum: <span class="<?= $statusBadgeMap[$activeTicket['status']] ?? 'badge bg-secondary' ?>"><?= sanitize(getSupportTicketStatusLabel($activeTicket['status'])) ?></span></small>
                            </div>
                            <small class="text-muted">Oluşturulma: <?= date('d.m.Y H:i', strtotime($activeTicket['created_at'])) ?></small>
                        </div>
                        <div class="card-body">
                            <div class="ticket-messages mb-4">
                                <?php foreach ($ticketMessages as $message): ?>
                                    <?php
                                        $isUser = $message['sender_type'] === 'user';
                                        $bubbleClass = $isUser ? 'bg-primary text-white' : 'support-admin-message';
                                        $senderName = $isUser ? sanitize($_SESSION['username']) : sanitize($message['admin_username'] ?? 'Destek Ekibi');
                                        $senderClass = $isUser ? '' : 'text-success';
                                    ?>
                                    <div class="mb-3">
                                        <div class="d-flex justify-content-between align-items-center mb-1">
                                            <strong class="<?= $senderClass ?>"><?= $senderName ?></strong>
                                            <span class="text-muted small"><?= date('d.m.Y H:i', strtotime($message['created_at'])) ?></span>
                                        </div>
                                        <div class="p-3 rounded <?= $bubbleClass ?> mt-1">
                                            <p class="mb-0"><?= nl2br(sanitize($message['message'])) ?></p>
                                            <?php if (!empty($message['attachment_path'])): ?>
                                                <?php $attachmentUrl = assetUrl($message['attachment_path']); ?>
                                                <div class="mt-2">
                                                    <a href="<?= htmlspecialchars($attachmentUrl, ENT_QUOTES, 'UTF-8') ?>" target="_blank" class="btn btn-sm <?= $isUser ? 'btn-outline-light' : 'btn-outline-primary' ?>">
                                                        <i class="fa-solid fa-image me-1"></i>Ekli Görseli Görüntüle
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>

                            <form method="post" enctype="multipart/form-data" class="ticket-reply-form">
                                <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                                <input type="hidden" name="action" value="reply_ticket">
                                <input type="hidden" name="ticket_id" value="<?= (int)$activeTicket['id'] ?>">
                                <div class="mb-3">
                                    <label class="form-label">Yanıtınız</label>
                            <textarea name="message" class="form-control" rows="4" placeholder="Mesajınızı yazın" required><?= sanitize((($_POST['action'] ?? '') === 'reply_ticket') ? ($_POST['message'] ?? '') : '') ?></textarea>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Görsel (isteğe bağlı)</label>
                                    <input type="file" name="attachment" class="form-control" accept="image/png,image/jpeg,image/webp,image/gif">
                                </div>
                                <div class="d-flex justify-content-between">
                                    <a href="<?= routeUrl('support') ?>" class="btn btn-outline-primary"><i class="fa-solid fa-arrow-left me-2"></i>Talep Listesine Dön</a>
                                    <button type="submit" class="btn btn-success"><i class="fa-solid fa-paper-plane me-2"></i>Yanıt Gönder</button>
                                </div>
                            </form>
                        </div>
                    <?php else: ?>
                        <div class="card-body text-center py-5">
                            <div class="mb-3" style="font-size:3rem;"><i class="fa-solid fa-inbox"></i></div>
                            <h4 class="mb-2">Destek talebi seçilmedi</h4>
                            <p class="text-muted">Sol taraftan mevcut taleplerinizi görüntüleyebilir veya yeni bir talep oluşturabilirsiniz.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</section>

<?php include 'includes/footer.php'; ?>

