<?php
require_once 'config.php';
requireLogin();

$user = getUserInfo($pdo, $_SESSION['user_id']);
$success = '';
$errors = [];
$withdrawAmountInput = '';

$payoutAccountName = trim((string)($user['payout_account_name'] ?? ''));
$payoutBankName = trim((string)($user['payout_bank_name'] ?? ''));
$payoutIban = trim((string)($user['payout_iban'] ?? ''));

$formatIbanDisplay = static function (?string $iban): string {
    $clean = strtoupper(preg_replace('/[^A-Z0-9]/', '', (string)$iban));
    return $clean === '' ? '' : trim(chunk_split($clean, 4, ' '));
};

// Handle withdrawal request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Güvenlik hatası.';
    } else {
        $withdrawAmountInput = str_replace(',', '.', trim((string)($_POST['withdraw_amount'] ?? '')));
        if ($withdrawAmountInput === '' || !is_numeric($withdrawAmountInput)) {
            $errors[] = 'Geçerli bir tutar giriniz.';
        }

        $amount = $withdrawAmountInput !== '' && is_numeric($withdrawAmountInput) ? (float)$withdrawAmountInput : 0.0;
        $amount = round($amount, 2);
        $amountValue = number_format($amount, 2, '.', '');
        if ($withdrawAmountInput !== '' && is_numeric($withdrawAmountInput)) {
            $withdrawAmountInput = $amountValue;
        }

        if ($amount <= 0) {
            $errors[] = 'Tutar 0\'dan büyük olmalıdır.';
        }

        $balanceAvailable = (float)($user['balance'] ?? 0);
        if ($amount > $balanceAvailable) {
            $errors[] = 'Yeterli bakiyeniz bulunmuyor.';
        }

        if ($payoutAccountName === '' || $payoutBankName === '' || $payoutIban === '') {
            $errors[] = 'Önce banka bilgilerinizi kaydediniz.';
        }

        if (empty($errors)) {
            $pendingStmt = $pdo->prepare('SELECT COUNT(*) FROM withdraw_requests WHERE user_id = ? AND status = "pending"');
            $pendingStmt->execute([$_SESSION['user_id']]);
            if ((int)$pendingStmt->fetchColumn() > 0) {
                $errors[] = 'Bekleyen bir çekim talebiniz bulunuyor.';
            }
        }

        if (empty($errors)) {
            try {
                $pdo->beginTransaction();

                // Bakiyeden hemen düş
                $updateBalance = $pdo->prepare('UPDATE users SET balance = balance - ? WHERE id = ?');
                $updateBalance->execute([$amountValue, $_SESSION['user_id']]);

                // Çekim talebini oluştur
                $insert = $pdo->prepare('INSERT INTO withdraw_requests (user_id, amount, status, account_name, bank_name, iban, created_at) VALUES (?, ?, "pending", ?, ?, ?, NOW())');
                $insert->execute([
                    $_SESSION['user_id'],
                    $amountValue,
                    $payoutAccountName,
                    $payoutBankName,
                    $payoutIban
                ]);

                // Transaction kaydı ekle
                $requestId = $pdo->lastInsertId();
                $transactionStmt = $pdo->prepare('INSERT INTO transactions (user_id, type, amount, description, created_at) VALUES (?, "withdraw_request", ?, ?, NOW())');
                $description = 'Çekim talebi #' . $requestId . ' (Beklemede)';
                $transactionStmt->execute([$_SESSION['user_id'], $amountValue, $description]);

                $pdo->commit();

                // Kullanıcı bilgisini güncelle
                $user = getUserInfo($pdo, $_SESSION['user_id']);

                $success = 'Çekim talebiniz oluşturuldu ve ' . formatPrice($amountValue) . ' bakiyenizden düşüldü.';
                $withdrawAmountInput = '';
            } catch (Throwable $th) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $errors[] = 'İşlem sırasında hata oluştu: ' . $th->getMessage();
            }
        }
    }
}

// Pagination for withdrawal requests
$perPage = 5;
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;

// Get total count
$countStmt = $pdo->prepare('SELECT COUNT(*) FROM withdraw_requests WHERE user_id = ?');
$countStmt->execute([$_SESSION['user_id']]);
$totalRequests = (int)$countStmt->fetchColumn();
$totalPages = max(1, (int)ceil($totalRequests / $perPage));
if ($page > $totalPages) {
    $page = $totalPages;
}
$offset = ($page - 1) * $perPage;

// Get withdrawal requests with pagination
$withdrawStmt = $pdo->prepare('SELECT wr.*, au.username AS admin_username FROM withdraw_requests wr LEFT JOIN users au ON au.id = wr.admin_id WHERE wr.user_id = ? ORDER BY wr.created_at DESC LIMIT ? OFFSET ?');
$withdrawStmt->bindValue(1, $_SESSION['user_id'], PDO::PARAM_INT);
$withdrawStmt->bindValue(2, $perPage, PDO::PARAM_INT);
$withdrawStmt->bindValue(3, $offset, PDO::PARAM_INT);
$withdrawStmt->execute();
$withdrawRequests = $withdrawStmt->fetchAll();

// Check for pending withdrawal
$hasPendingWithdraw = false;
$pendingCheckStmt = $pdo->prepare('SELECT COUNT(*) FROM withdraw_requests WHERE user_id = ? AND status = "pending"');
$pendingCheckStmt->execute([$_SESSION['user_id']]);
$hasPendingWithdraw = (int)$pendingCheckStmt->fetchColumn() > 0;

// Statistics - get all records for statistics
$allWithdrawalsStmt = $pdo->prepare('SELECT status, amount FROM withdraw_requests WHERE user_id = ?');
$allWithdrawalsStmt->execute([$_SESSION['user_id']]);
$allWithdrawals = $allWithdrawalsStmt->fetchAll();

$totalWithdrawn = 0;
$pendingAmount = 0;
$approvedCount = 0;
$rejectedCount = 0;

foreach ($allWithdrawals as $request) {
    $status = strtolower((string)$request['status']);
    if ($status === 'approved') {
        $totalWithdrawn += (float)$request['amount'];
        $approvedCount++;
    } elseif ($status === 'pending') {
        $pendingAmount += (float)$request['amount'];
    } elseif ($status === 'rejected') {
        $rejectedCount++;
    }
}

$pageTitle = 'Bakiye Çekimlerim';
include 'includes/header.php';
?>

<section class="page-section">
    <div class="container-wide">
        <div class="page-header">
            <div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="<?= routeUrl('profile') ?>"><i class="fa-solid fa-user me-1"></i>Profilim</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Bakiye Çekimlerim</li>
                    </ol>
                </nav>
                <h1 class="page-title"><i class="fa-solid fa-money-bill-transfer me-2"></i>Bakiye Çekimlerim</h1>
                <p class="page-description">Bakiyenizi banka hesabınıza aktarmak için çekim talebi oluşturabilirsiniz.</p>
            </div>
        </div>

        <?php if ($success !== ''): ?>
            <div class="alert alert-success">
                <i class="fa-solid fa-circle-check me-2"></i><?= sanitize($success) ?>
            </div>
        <?php endif; ?>
        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <i class="fa-solid fa-circle-exclamation me-2"></i>
                <div class="alert-messages">
                    <?php foreach ($errors as $error): ?>
                        <div><?= sanitize($error) ?></div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <div class="profile-metric-strip">
            <div class="profile-metric-card">
                <span>Mevcut Bakiye</span>
                <strong class="text-success"><?= formatPrice($user['balance']) ?></strong>
                <small>Çekilebilir bakiye</small>
            </div>
            <div class="profile-metric-card">
                <span>Toplam Çekilen</span>
                <strong><?= formatPrice($totalWithdrawn) ?></strong>
                <small><?= $approvedCount ?> onaylandı</small>
            </div>
            <div class="profile-metric-card">
                <span>Bekleyen</span>
                <strong class="text-warning"><?= formatPrice($pendingAmount) ?></strong>
                <small>İnceleme aşamasında</small>
            </div>
            <div class="profile-metric-card">
                <span>Toplam Talep</span>
                <strong><?= count($withdrawRequests) ?></strong>
                <small><?= $rejectedCount ?> reddedildi</small>
            </div>
        </div>

        <div class="row">
            <div class="col-lg-5 mb-4">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fa-solid fa-plus me-2"></i>Yeni Çekim Talebi</h5>
                    </div>
                    <div class="card-body">
                        <form method="post">
                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                            
                            <div class="mb-3">
                                <label class="form-label">Mevcut Bakiye</label>
                                <input type="text" class="form-control" value="<?= formatPrice($user['balance']) ?>" readonly>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Çekilecek Tutar</label>
                                <div class="input-group">
                                    <input type="number" class="form-control" name="withdraw_amount" min="10" step="0.01" value="<?= sanitize($withdrawAmountInput) ?>" placeholder="Örn. 250"<?= $hasPendingWithdraw ? ' disabled' : '' ?>>
                                    <span class="input-group-text"><?= CURRENCY ?></span>
                                </div>
                            </div>

                            <?php if ($payoutAccountName === '' || $payoutBankName === '' || $payoutIban === ''): ?>
                                <div class="alert alert-warning">
                                    <i class="fa-solid fa-circle-exclamation me-2"></i>
                                    Çekim talebi gönderebilmek için <a href="<?= routeUrl('profile') ?>">profil sayfanızdan</a> banka bilgilerinizi kaydetmelisiniz.
                                </div>
                            <?php endif; ?>

                            <?php if ($hasPendingWithdraw): ?>
                                <div class="alert alert-info">
                                    <i class="fa-solid fa-clock me-2"></i>
                                    Bekleyen bir çekim talebiniz mevcut. Yeni talep gönderebilmek için mevcut talebin sonuçlanmasını bekleyin.
                                </div>
                            <?php else: ?>
                                <div class="alert alert-light text-muted">
                                    <ul class="mb-0">
                                        <li>Kayıtlı banka: <strong><?= $payoutAccountName !== '' ? sanitize($payoutAccountName) : 'Henüz eklenmedi' ?></strong> / <strong><?= $payoutBankName !== '' ? sanitize($payoutBankName) : 'Henüz eklenmedi' ?></strong></li>
                                        <li>IBAN: <strong><?= $payoutIban !== '' ? sanitize($formatIbanDisplay($payoutIban)) : 'Henüz eklenmedi' ?></strong></li>
                                        <li>Onaylanan talepler bakiyenizden düşülür ve kayıt altına alınır.</li>
                                    </ul>
                                </div>
                            <?php endif; ?>

                            <button type="submit" class="btn btn-primary w-100"<?= $hasPendingWithdraw || $payoutAccountName === '' || $payoutBankName === '' || $payoutIban === '' ? ' disabled' : '' ?>>
                                <i class="fa-solid fa-paper-plane me-2"></i>Talep Gönder
                            </button>
                        </form>
                    </div>
                </div>
            </div>

            <div class="col-lg-7">
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fa-solid fa-history me-2"></i>Çekim Talepleri Geçmişi</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($withdrawRequests)): ?>
                            <div class="empty-state">
                                <i class="fa-solid fa-money-bill-wave"></i>
                                <p>Henüz bakiye çekim talebiniz bulunmuyor.</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="profile-table">
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Tarih</th>
                                            <th>Tutar</th>
                                            <th>Durum</th>
                                            <th>Not</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($withdrawRequests as $request): ?>
                                            <?php
                                                $status = strtolower((string)$request['status']);
                                                $statusBadge = 'status-badge status-pending';
                                                $statusLabel = 'Beklemede';
                                                if ($status === 'approved') {
                                                    $statusBadge = 'status-badge status-completed';
                                                    $statusLabel = 'Tamamlandı';
                                                } elseif ($status === 'rejected') {
                                                    $statusBadge = 'status-badge status-cancelled';
                                                    $statusLabel = 'Reddedildi';
                                                }
                                            ?>
                                            <tr>
                                                <td><?= (int)$request['id'] ?></td>
                                                <td><?= date('d.m.Y H:i', strtotime($request['created_at'])) ?></td>
                                                <td><strong class="text-danger">-<?= formatPrice($request['amount']) ?></strong></td>
                                                <td><span class="<?= $statusBadge ?>"><?= sanitize($statusLabel) ?></span></td>
                                                <td>
                                                    <?php if (!empty($request['admin_note'])): ?>
                                                        <?= sanitize($request['admin_note']) ?>
                                                    <?php else: ?>
                                                        <span class="profile-muted">-</span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <?php if ($totalPages > 1): ?>
                                <nav class="pagination-nav mt-3">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($page > 1): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="<?= routeUrl('withdrawals') ?>?page=<?= $page - 1 ?>">
                                                    <i class="fa-solid fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>

                                        <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                            <li class="page-item<?= $i === $page ? ' active' : '' ?>">
                                                <a class="page-link" href="<?= routeUrl('withdrawals') ?>?page=<?= $i ?>"><?= $i ?></a>
                                            </li>
                                        <?php endfor; ?>

                                        <?php if ($page < $totalPages): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="<?= routeUrl('withdrawals') ?>?page=<?= $page + 1 ?>">
                                                    <i class="fa-solid fa-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<?php include 'includes/footer.php'; ?>

