<?php
/**
 * Domain Takip Sistemi - Cron Job: Süresi Yaklaşan Domainleri Kontrol Et
 * 
 * Bu script günde bir kez çalıştırılmalıdır.
 * Crontab örneği: 0 9 * * * /usr/bin/php /path/to/domain-tracker/cron/check_expiring_domains.php
 */

// Script'in CLI'dan çalıştırıldığından emin ol
if (php_sapi_name() !== 'cli') {
    die("Bu script sadece komut satırından çalıştırılabilir.\n");
}

// Proje kök dizinini belirle
$projectRoot = dirname(__DIR__);

// Gerekli dosyaları dahil et
require_once $projectRoot . '/config/config.php';
require_once $projectRoot . '/includes/notification_functions.php';
require_once $projectRoot . '/includes/domain_functions.php';

// Log fonksiyonu
function logMessage($message) {
    $timestamp = date('Y-m-d H:i:s');
    echo "[$timestamp] $message\n";
    
    // Log dosyasına da yaz
    $logFile = dirname(__DIR__) . '/logs/cron.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

try {
    logMessage("Domain süre kontrolü başlatıldı");
    
    // Bildirim yöneticisini başlat
    $notificationManager = new NotificationManager($pdo);
    
    // 1. Önce tüm domainlerin bilgilerini güncelle (opsiyonel)
    $updateAll = isset($argv[1]) && $argv[1] === '--update-all';
    
    if ($updateAll) {
        logMessage("Tüm domain bilgileri güncelleniyor...");
        $updated = DomainInfo::updateAllDomains($pdo);
        logMessage("$updated domain bilgisi güncellendi");
    }
    
    // 2. Süresi yaklaşan domainleri kontrol et ve bildirim gönder
    logMessage("Süresi yaklaşan domainler kontrol ediliyor...");
    $result = $notificationManager->checkExpiringDomains();
    
    if ($result['success']) {
        logMessage($result['message']);
        if (isset($result['domains_checked']) && isset($result['notifications_sent'])) {
            logMessage("Kontrol edilen domain sayısı: " . $result['domains_checked']);
            logMessage("Gönderilen bildirim sayısı: " . $result['notifications_sent']);
        }
    } else {
        logMessage("HATA: " . $result['message']);
    }
    
    // 3. Eski logları temizle (30 günden eski)
    $stmt = $pdo->prepare("DELETE FROM notification_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)");
    $stmt->execute();
    $deletedLogs = $stmt->rowCount();
    
    if ($deletedLogs > 0) {
        logMessage("$deletedLogs eski bildirim logu temizlendi");
    }
    
    logMessage("Domain süre kontrolü tamamlandı");
    
} catch (Exception $e) {
    logMessage("KRITIK HATA: " . $e->getMessage());
    logMessage("Stack trace: " . $e->getTraceAsString());
    exit(1);
}

exit(0);
?>

