<?php
require_once 'config/config.php';
require_once 'includes/domain_functions.php';

// Giriş kontrolü
requireLogin();

// Toplu silme işlemi: POST ile bulk_delete gelirse hemen sil
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_delete'])) {
    $selectedIds = $_POST['selected_domains'] ?? [];
    $deleted = 0;
    if (!empty($selectedIds) && is_array($selectedIds)) {
        foreach ($selectedIds as $domainId) {
            $domainId = intval($domainId);
            if ($domainId > 0) {
                $stmt = $pdo->prepare("DELETE FROM domains WHERE id = ?");
                if ($stmt->execute([$domainId])) {
                    $deleted++;
                }
            }
        }
    }
    $message = "$deleted domain silindi.";
    $messageType = 'success';
    $action = 'list';
}

$action = $_GET['action'] ?? 'list';
$domainId = $_GET['id'] ?? null;
$message = '';
$messageType = '';

// Domain ekleme işlemi
if ($_POST && $action === 'add') {
    $domainName = strtolower(trim($_POST['domain_name'] ?? ''));
    $notes = trim($_POST['notes'] ?? '');
    $autoRenew = isset($_POST['auto_renew']) ? 1 : 0;
    
    if (empty($domainName)) {
        $message = 'Domain adı gereklidir.';
        $messageType = 'danger';
    } elseif (!filter_var($domainName, FILTER_VALIDATE_DOMAIN, FILTER_FLAG_HOSTNAME)) {
        $message = 'Geçerli bir domain adı girin.';
        $messageType = 'danger';
    } else {
        try {
            // Domain zaten var mı kontrol et
            $stmt = $pdo->prepare("SELECT id FROM domains WHERE domain_name = ?");
            $stmt->execute([$domainName]);
            
            if ($stmt->fetch()) {
                $message = 'Bu domain zaten kayıtlı.';
                $messageType = 'warning';
            } else {
                // Domain'i ekle
                $stmt = $pdo->prepare("
                    INSERT INTO domains (domain_name, notes, auto_renew, created_at) 
                    VALUES (?, ?, ?, NOW())
                ");
                $result = $stmt->execute([$domainName, $notes, $autoRenew]);
                $newDomainId = $pdo->lastInsertId();
                if ($result && $newDomainId) {
                    // WHOIS bilgilerini çek ve güncelle
                    $whoisInfo = DomainInfo::getWhoisInfo($domainName);
                    if ($whoisInfo) {
                        $stmt = $pdo->prepare("
                            UPDATE domains SET 
                                registrar = ?,
                                registration_date = ?,
                                expiration_date = ?,
                                nameservers = ?,
                                status = ?,
                                last_checked = NOW(),
                                updated_at = NOW()
                            WHERE id = ?
                        ");
                        $nameservers = !empty($whoisInfo['nameservers']) ? implode("\n", $whoisInfo['nameservers']) : null;
                        $stmt->execute([
                            $whoisInfo['registrar'],
                            $whoisInfo['registration_date'],
                            $whoisInfo['expiration_date'],
                            $nameservers,
                            $whoisInfo['status'],
                            $newDomainId
                        ]);
                        $message = 'Domain başarıyla eklendi ve bilgileri güncellendi.';
                        $messageType = 'success';
                    } else {
                        // WHOIS alınamazsa otomatik güncelleme fonksiyonu çağır
                        if (DomainInfo::updateDomainInfo($pdo, $newDomainId)) {
                            $message = 'Domain eklendi, bilgiler sonradan güncellendi.';
                            $messageType = 'success';
                        } else {
                            $message = 'Domain eklendi fakat WHOIS bilgileri alınamadı.';
                            $messageType = 'warning';
                        }
                    }
                    $action = 'list'; // Listeye dön
                } else {
                    $message = 'Domain eklenemedi.';
                    $messageType = 'danger';
                }
            }
        } catch (PDOException $e) {
            $message = 'Domain eklenirken hata oluştu.';
            $messageType = 'danger';
            error_log("Add domain error: " . $e->getMessage());
        }
    }
}

// Domain düzenleme işlemi
if ($_POST && $action === 'edit' && $domainId) {
    $notes = trim($_POST['notes'] ?? '');
    $autoRenew = isset($_POST['auto_renew']) ? 1 : 0;
    
    try {
        $stmt = $pdo->prepare("UPDATE domains SET notes = ?, auto_renew = ?, updated_at = NOW() WHERE id = ?");
        $stmt->execute([$notes, $autoRenew, $domainId]);
        $message = 'Domain bilgileri güncellendi.';
        $messageType = 'success';
        $action = 'list';
    } catch (PDOException $e) {
        $message = 'Domain güncellenirken hata oluştu.';
        $messageType = 'danger';
        error_log("Update domain error: " . $e->getMessage());
    }
}

// Domain silme işlemi
if ($action === 'delete' && $domainId) {
    try {
        $stmt = $pdo->prepare("DELETE FROM domains WHERE id = ?");
        $stmt->execute([$domainId]);
        
        $message = 'Domain başarıyla silindi.';
        $messageType = 'success';
        $action = 'list';
    } catch (PDOException $e) {
        $message = 'Domain silinirken hata oluştu.';
        $messageType = 'danger';
        error_log("Delete domain error: " . $e->getMessage());
    }
}

// Domain bilgilerini yenile
if ($action === 'refresh' && $domainId) {
    if (DomainInfo::updateDomainInfo($pdo, $domainId)) {
        $message = 'Domain bilgileri güncellendi.';
        $messageType = 'success';
    } else {
        $message = 'Domain bilgileri güncellenirken hata oluştu.';
        $messageType = 'danger';
    }
    $action = 'list';
}

// Toplu güncelleme
if ($_POST && $action === 'bulk_update') {
    $selectedDomains = $_POST['selected_domains'] ?? [];
    $updated = 0;
    
    foreach ($selectedDomains as $domainId) {
        if (DomainInfo::updateDomainInfo($pdo, $domainId)) {
            $updated++;
        }
        // Rate limiting
        sleep(1);
    }
    
    $message = "$updated domain güncellendi.";
    $messageType = 'success';
    $action = 'list';
}

// Domain listesi için verileri çek
if ($action === 'list') {
    $search = $_GET['search'] ?? '';
    $status = $_GET['status'] ?? '';
    
    $whereClause = '';
    $params = [];
    
    if (!empty($search)) {
        $whereClause .= " WHERE domain_name LIKE ?";
        $params[] = "%$search%";
    }
    
        if (!empty($status)) {
        $whereClause .= ($whereClause ? " AND" : " WHERE");
        switch ($status) {
            case 'expired':
                $whereClause .= " TIMESTAMPDIFF(DAY, NOW(), expiration_date) < 0";
                break;
            case 'critical':
                $whereClause .= " TIMESTAMPDIFF(DAY, NOW(), expiration_date) BETWEEN 0 AND 7";
                break;
            case 'warning':
                $whereClause .= " TIMESTAMPDIFF(DAY, NOW(), expiration_date) BETWEEN 8 AND 30";
                break;
            case 'ok':
                $whereClause .= " TIMESTAMPDIFF(DAY, NOW(), expiration_date) > 30";
                break;
        }
    }
    
    $stmt = $pdo->prepare("
        SELECT *, 
               TIMESTAMPDIFF(DAY, NOW(), expiration_date) as days_until_expiry,
               CASE 
                   WHEN expiration_date IS NULL THEN 'unknown'
                   WHEN TIMESTAMPDIFF(DAY, NOW(), expiration_date) < 0 THEN 'expired'
                   WHEN TIMESTAMPDIFF(DAY, NOW(), expiration_date) <= 7 THEN 'critical'
                   WHEN TIMESTAMPDIFF(DAY, NOW(), expiration_date) <= 30 THEN 'warning'
                   ELSE 'ok'
               END as expiry_status
        FROM domains 
        $whereClause 
        ORDER BY 
            CASE 
                WHEN expiration_date IS NULL THEN 999999
                WHEN TIMESTAMPDIFF(DAY, NOW(), expiration_date) < 0 THEN -TIMESTAMPDIFF(DAY, NOW(), expiration_date)
                ELSE TIMESTAMPDIFF(DAY, NOW(), expiration_date)
            END ASC,
            domain_name ASC
    ");
    $stmt->execute($params);
    $domains = $stmt->fetchAll();
}

// Düzenleme için domain bilgilerini çek
if ($action === 'edit' && $domainId) {
    $stmt = $pdo->prepare("SELECT * FROM domains WHERE id = ?");
    $stmt->execute([$domainId]);
    $editDomain = $stmt->fetch();
    
    if (!$editDomain) {
        $message = 'Domain bulunamadı.';
        $messageType = 'danger';
        $action = 'list';
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= SITE_NAME ?> - Domain Yönetimi</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/admin.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <?php if ($message): ?>
                    <div class="alert alert-<?= $messageType ?> alert-dismissible fade show mt-3" role="alert">
                        <?= sanitize($message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if ($action === 'list'): ?>
                    <!-- Domain Listesi -->
                    <form id="bulkDeleteForm" method="POST" action="domains.php">
                        <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                            <h1 class="h2"><i class="fas fa-globe me-2"></i>Domain Yönetimi</h1>
                            <div class="btn-toolbar mb-2 mb-md-0">
                                <div class="btn-group me-2">
                                    <a href="?action=add" class="btn btn-primary">
                                        <i class="fas fa-plus me-1"></i>Yeni Domain
                                    </a>
                                </div>
                                <input type="hidden" name="bulk_delete" value="1">
                                <button type="submit" id="bulkDeleteBtn" class="btn btn-danger ms-2" disabled>
                                    <i class="fas fa-trash me-1"></i>Seçilenleri Sil
                                </button>
                            </div>
                        </div>
                    
                    <!-- Arama ve Filtreler -->
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-search"></i></span>
                                <input type="text" class="form-control" id="domainSearch" 
                                       placeholder="Domain ara..." value="<?= sanitize($search) ?>">
                            </div>
                        </div>
                        <div class="col-md-3">
                            <select class="form-select" id="statusFilter">
                                <option value="">Tüm Durumlar</option>
                                <option value="expired" <?= $status === 'expired' ? 'selected' : '' ?>>Süresi Geçmiş</option>
                                <option value="critical" <?= $status === 'critical' ? 'selected' : '' ?>>Kritik (≤7 gün)</option>
                                <option value="warning" <?= $status === 'warning' ? 'selected' : '' ?>>Uyarı (≤30 gün)</option>
                                <option value="ok" <?= $status === 'ok' ? 'selected' : '' ?>>Normal (>30 gün)</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <button type="button" class="btn btn-outline-primary" onclick="location.reload()">
                                <i class="fas fa-sync me-1"></i>Yenile
                            </button>
                        </div>
                    </div>
                    
                    <!-- Toplu İşlemler -->
                    <?php if (!empty($domains)): ?>
                    <!-- Toplu silme formu kaldırıldı -->
                    <?php endif; ?>
                    
                    <!-- Domain Tablosu -->
                    <div class="card shadow">
                        <div class="card-body">
                            <?php if (empty($domains)): ?>
                                <div class="text-center py-5">
                                    <i class="fas fa-globe fa-3x text-muted mb-3"></i>
                                    <h5 class="text-muted">Henüz domain eklenmemiş</h5>
                                    <p class="text-muted">İlk domain'inizi eklemek için aşağıdaki butona tıklayın.</p>
                                    <a href="?action=add" class="btn btn-primary">
                                        <i class="fas fa-plus me-1"></i>İlk Domain'i Ekle
                                    </a>
                                </div>
                            <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover" id="domainsTable">
                                            <thead>
                                                <tr>
                                                    <th>
                                                        <input type="checkbox" id="selectAll" class="form-check-input">
                                                    </th>
                                                    <th>Domain</th>
                                                    <th>Registrar</th>
                                                    <th>Kayıt Tarihi</th>
                                                    <th>Bitiş Tarihi</th>
                                                    <th>Durum</th>
                                                    <th>Son Kontrol</th>
                                                    <th>İşlemler</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($domains as $domain): ?>
                                                    <tr data-status="<?= $domain['expiry_status'] ?>">
                                                        <td>
                                                        <input type="checkbox" class="form-check-input domain-select" name="selected_domains[]" value="<?= $domain['id'] ?>">
                                                        </td>
                                                        <td>
                                                            <strong><?= sanitize($domain['domain_name']) ?></strong>
                                                            <?php if ($domain['auto_renew']): ?>
                                                                <i class="fas fa-sync-alt text-success ms-1" 
                                                                   title="Otomatik yenileme aktif"></i>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td><?= sanitize($domain['registrar']) ?: '-' ?></td>
                                                        <td><?= formatDate($domain['registration_date']) ?></td>
                                                        <td><?= formatDate($domain['expiration_date']) ?></td>
                                                        <td>
                                                            <?php
                                                            $statusClass = [
                                                                'ok' => 'success',
                                                                'warning' => 'warning',
                                                                'critical' => 'danger',
                                                                'expired' => 'dark',
                                                                'unknown' => 'secondary'
                                                            ];
                                                            $statusText = [
                                                                'ok' => $domain['days_until_expiry'] . ' gün',
                                                                'warning' => $domain['days_until_expiry'] . ' gün',
                                                                'critical' => $domain['days_until_expiry'] . ' gün',
                                                                'expired' => 'Süresi geçmiş',
                                                                'unknown' => 'Bilinmiyor'
                                                            ];
                                                            ?>
                                                            <span class="badge bg-<?= $statusClass[$domain['expiry_status']] ?>">
                                                                <?= $statusText[$domain['expiry_status']] ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <?= $domain['last_checked'] ? formatDate($domain['last_checked']) : 'Hiç' ?>
                                                        </td>
                                                        <td>
                                                            <div class="btn-group btn-group-sm">
                                                                <a href="?action=refresh&id=<?= $domain['id'] ?>" 
                                                                   class="btn btn-outline-primary" 
                                                                   title="Bilgileri Yenile">
                                                                    <i class="fas fa-sync"></i>
                                                                </a>
                                                                <a href="?action=edit&id=<?= $domain['id'] ?>" 
                                                                   class="btn btn-outline-secondary" 
                                                                   title="Düzenle">
                                                                    <i class="fas fa-edit"></i>
                                                                </a>
                                                                <a href="?action=delete&id=<?= $domain['id'] ?>" 
                                                                   class="btn btn-outline-danger" 
                                                                   title="Sil"
                                                                   data-action="delete" 
                                                                   data-domain="<?= sanitize($domain['domain_name']) ?>">
                                                                    <i class="fas fa-trash"></i>
                                                                </a>
                                                            </div>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                    </form>
                                    <div class="d-flex justify-content-end mt-3">
                                        <button type="submit" id="bulkDeleteBtn" class="btn btn-danger" disabled>
                                            <i class="fas fa-trash me-1"></i>Seçilenleri Sil
                                        </button>
                                    </div>
                                </form>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                <?php elseif ($action === 'add'): ?>
                    <!-- Domain Ekleme Formu -->
                    <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                        <h1 class="h2"><i class="fas fa-plus me-2"></i>Yeni Domain Ekle</h1>
                        <div class="btn-toolbar mb-2 mb-md-0">
                            <a href="domains.php" class="btn btn-outline-secondary">
                                <i class="fas fa-arrow-left me-1"></i>Geri Dön
                            </a>
                        </div>
                    </div>
                    
                    <div class="row justify-content-center">
                        <div class="col-md-8">
                            <div class="card shadow">
                                <div class="card-body">
                                    <form id="domainForm" method="POST" action="?action=add">
                                        <div class="mb-3">
                                            <label for="domain_name" class="form-label">Domain Adı *</label>
                                            <input type="text" class="form-control" id="domain_name" name="domain_name" 
                                                   placeholder="example.com" required>
                                            <div class="form-text">
                                                Domain adını www olmadan girin (örn: example.com)
                                            </div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="notes" class="form-label">Notlar</label>
                                            <textarea class="form-control" id="notes" name="notes" rows="3" 
                                                      placeholder="Bu domain hakkında notlarınız..."></textarea>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="auto_renew" name="auto_renew">
                                                <label class="form-check-label" for="auto_renew">
                                                    Otomatik yenileme aktif
                                                </label>
                                            </div>
                                        </div>
                                        
                                        <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                            <a href="domains.php" class="btn btn-outline-secondary me-md-2">İptal</a>
                                            <button type="submit" class="btn btn-primary">
                                                <i class="fas fa-save me-1"></i>Domain Ekle
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                <?php elseif ($action === 'edit' && isset($editDomain)): ?>
                    <!-- Domain Düzenleme Formu -->
                    <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                        <h1 class="h2"><i class="fas fa-edit me-2"></i>Domain Düzenle</h1>
                        <div class="btn-toolbar mb-2 mb-md-0">
                            <a href="domains.php" class="btn btn-outline-secondary">
                                <i class="fas fa-arrow-left me-1"></i>Geri Dön
                            </a>
                        </div>
                    </div>
                    
                    <div class="row justify-content-center">
                        <div class="col-md-8">
                            <div class="card shadow">
                                <div class="card-header">
                                    <h5 class="mb-0"><?= sanitize($editDomain['domain_name']) ?></h5>
                                </div>
                                <div class="card-body">
                                    <div class="mb-3 p-3 bg-light rounded border">
                                        <strong>Domain Bilgileri</strong><br>
                                        <b>Domain:</b> <?= sanitize($editDomain['domain_name']) ?><br>
                                        <b>Registrar:</b> <?= sanitize($editDomain['registrar']) ?: '-' ?><br>
                                        <b>Kayıt Tarihi:</b> <?= formatDate($editDomain['registration_date']) ?><br>
                                        <b>Bitiş Tarihi:</b> <?= formatDate($editDomain['expiration_date']) ?><br>
                                        <b>Durum:</b> <?= sanitize($editDomain['status']) ?: 'Bilinmiyor' ?><br>
                                        <b>Nameserver:</b> <br>
                                        <?php if (!empty($editDomain['nameservers'])):
                                            $nsList = explode("\n", $editDomain['nameservers']);
                                            foreach ($nsList as $ns): ?>
                                                <span class="badge bg-secondary me-1"><?= sanitize($ns) ?></span>
                                            <?php endforeach;
                                        else: ?>
                                            <span class="text-muted">-</span>
                                        <?php endif; ?>
                                    </div>
                                    <form id="editDomainForm" method="POST" action="?action=edit&id=<?= $editDomain['id'] ?>">
                                        <div class="mb-3">
                                            <label class="form-label">Notlar</label>
                                            <textarea class="form-control" name="notes" rows="3" placeholder="Notlarınızı yazın..."><?= sanitize($editDomain['notes']) ?></textarea>
                                        </div>
                                        <div class="mb-3">
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" name="auto_renew" id="auto_renew_edit" <?= $editDomain['auto_renew'] ? 'checked' : '' ?>>
                                                <label class="form-check-label" for="auto_renew_edit">Otomatik yenileme aktif</label>
                                            </div>
                                        </div>
                                        <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                            <a href="domains.php" class="btn btn-outline-secondary me-md-2">İptal</a>
                                            <button type="submit" class="btn btn-primary">
                                                <i class="fas fa-save me-1"></i>Değişiklikleri Kaydet
                                            </button>
                                            <a href="?action=refresh&id=<?= $editDomain['id'] ?>" class="btn btn-info ms-2">Bilgileri Yenile</a>
                                        </div>
                                    </form>
                                </div>

                                    <!-- Gereksiz ikinci form ve butonlar kaldırıldı -->
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toplu silme butonunu aktif et ve seçili id'leri gizli inputa yaz
        document.addEventListener('DOMContentLoaded', function() {
            function updateBulkDeleteBtn() {
                var checkedCount = document.querySelectorAll('input[name="selected_domains[]"]:checked').length;
                var bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
                if (bulkDeleteBtn) {
                    bulkDeleteBtn.disabled = checkedCount === 0;
                }
            }
            document.querySelectorAll('input[name="selected_domains[]"]').forEach(function(checkbox) {
                checkbox.addEventListener('change', updateBulkDeleteBtn);
            });
            var selectAll = document.getElementById('selectAll');
            if (selectAll) {
                selectAll.addEventListener('change', function() {
                    var checked = this.checked;
                    document.querySelectorAll('input[name="selected_domains[]"]').forEach(function(cb) {
                        cb.checked = checked;
                    });
                    updateBulkDeleteBtn();
                });
            }
            updateBulkDeleteBtn();
        });
// Toplu silme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bulk_delete'])) {
    $action = 'bulk_delete';
}

if ($action === 'bulk_delete' && $_POST) {
    $selectedIds = $_POST['selected_domains'] ?? [];
    $deleted = 0;
    if (!empty($selectedIds) && is_array($selectedIds)) {
        foreach ($selectedIds as $domainId) {
            $domainId = intval($domainId);
            if ($domainId > 0) {
                $stmt = $pdo->prepare("DELETE FROM domains WHERE id = ?");
                if ($stmt->execute([$domainId])) {
                    $deleted++;
                }
            }
        }
    }
    $message = "$deleted domain silindi.";
    $messageType = 'success';
    $action = 'list';
}
    </script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        function updateBulkDeleteBtn() {
            var checkedCount = document.querySelectorAll('input[name="selected_domains[]"]:checked').length;
            var bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
            if (bulkDeleteBtn) {
                bulkDeleteBtn.disabled = checkedCount === 0;
            }
        }
        document.querySelectorAll('input[name="selected_domains[]"]').forEach(function(checkbox) {
            checkbox.addEventListener('change', updateBulkDeleteBtn);
        });
        var selectAll = document.getElementById('selectAll');
        if (selectAll) {
            selectAll.addEventListener('change', function() {
                var checked = this.checked;
                document.querySelectorAll('input[name="selected_domains[]"]').forEach(function(cb) {
                    cb.checked = checked;
                });
                updateBulkDeleteBtn();
            });
        }
        updateBulkDeleteBtn();
    });
    </script>
    <script src="assets/js/admin.js"></script>
    <script>
    // Domain arama kutusu ile tablo filtreleme
    document.addEventListener('DOMContentLoaded', function() {
        const searchInput = document.getElementById('domainSearch');
        const table = document.getElementById('domainsTable');
        if (!searchInput || !table) return;
        searchInput.addEventListener('input', function() {
            const term = this.value.trim().toLowerCase();
            table.querySelectorAll('tbody tr').forEach(function(row) {
                const domainCell = row.querySelector('td:nth-child(2)');
                if (!domainCell) return;
                const domainText = domainCell.innerText.trim().toLowerCase();
                row.style.display = domainText.includes(term) ? '' : 'none';
            });
        });
    });
    </script>
    <script>
    // Tablo başlıklarına tıklayınca sıralama
    document.addEventListener('DOMContentLoaded', function() {
        const table = document.getElementById('domainsTable');
        if (!table) return;
        const getCellValue = (tr, idx) => tr.children[idx].innerText.trim();
        const comparer = (idx, asc, isDate = false) => (a, b) => {
            let v1 = getCellValue(asc ? a : b, idx);
            let v2 = getCellValue(asc ? b : a, idx);
            if (isDate) {
                v1 = Date.parse(v1.replace(/\./g, '-')) || 0;
                v2 = Date.parse(v2.replace(/\./g, '-')) || 0;
            }
            return v1.localeCompare(v2, 'tr', {numeric: true});
        };
        table.querySelectorAll('th').forEach((th, idx) => {
            th.style.cursor = 'pointer';
            th.addEventListener('click', function() {
                const isDate = th.innerText.includes('Tarihi') || th.innerText.includes('Kontrol');
                const tbody = table.querySelector('tbody');
                Array.from(tbody.querySelectorAll('tr'))
                    .sort(comparer(idx, this.asc = !this.asc, isDate))
                    .forEach(tr => tbody.appendChild(tr));
            });
        });
    });
    </script>
</body>
</html>

