<?php
/**
 * Domain Takip Sistemi - Domain Bilgileri Fonksiyonları
 */

class DomainInfo {
    
    /**
     * Domain WHOIS bilgilerini çeker
     */
    public static function getWhoisInfo($domain) {
        $domain = strtolower(trim($domain));
        
        // Domain formatını kontrol et
        if (!self::isValidDomain($domain)) {
            return false;
        }
        
        try {
            // WHOIS sunucusunu belirle
            $whoisServer = self::getWhoisServer($domain);
            if (!$whoisServer) {
                return false;
            }
            
            // WHOIS sorgusu yap
            $whoisData = self::queryWhois($domain, $whoisServer);
            if (!$whoisData) {
                return false;
            }
            
            // WHOIS verisini parse et
            return self::parseWhoisData($whoisData, $domain);
            
        } catch (Exception $e) {
            error_log("WHOIS Error for $domain: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Domain DNS bilgilerini çeker
     */
    public static function getDnsInfo($domain) {
        $domain = strtolower(trim($domain));
        if (!self::isValidDomain($domain)) {
            return false;
        }
        try {
            $nameservers = [];
            // NS kayıtlarını çek
            $nsRecords = dns_get_record($domain, DNS_NS);
            if ($nsRecords) {
                foreach ($nsRecords as $record) {
                    if (isset($record['target'])) {
                        $nameservers[] = strtolower($record['target']);
                    }
                }
            }
            return [
                'nameservers' => $nameservers,
                'ns_count' => count($nameservers)
            ];
        } catch (Exception $e) {
            error_log("DNS Error for $domain: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Domain geçerliliğini kontrol eder
     */
    private static function isValidDomain($domain) {
        return filter_var($domain, FILTER_VALIDATE_DOMAIN, FILTER_FLAG_HOSTNAME);
    }
    
    /**
     * Domain için uygun WHOIS sunucusunu belirler
     */
    private static function getWhoisServer($domain) {
        $tld = substr(strrchr($domain, '.'), 1);
        
        $whoisServers = [
            'com' => 'whois.verisign-grs.com',
            'net' => 'whois.verisign-grs.com',
            'org' => 'whois.pir.org',
            'info' => 'whois.afilias.net',
            'biz' => 'whois.neulevel.biz',
            'us' => 'whois.nic.us',
            'uk' => 'whois.nic.uk',
            'de' => 'whois.denic.de',
            'fr' => 'whois.afnic.fr',
            'it' => 'whois.nic.it',
            'nl' => 'whois.domain-registry.nl',
            'be' => 'whois.dns.be',
            'tr' => 'whois.nic.tr',
            'ru' => 'whois.tcinet.ru',
            'pl' => 'whois.dns.pl',
            'br' => 'whois.registro.br',
            'au' => 'whois.aunic.net',
            'jp' => 'whois.jprs.jp',
            'cn' => 'whois.cnnic.net.cn',
            'in' => 'whois.inregistry.net',
            'ca' => 'whois.cira.ca',
            'eu' => 'whois.eu',
            'me' => 'whois.nic.me',
            'co' => 'whois.nic.co',
            'tv' => 'whois.nic.tv',
            'cc' => 'whois.nic.cc'
        ];
        
        return isset($whoisServers[$tld]) ? $whoisServers[$tld] : 'whois.iana.org';
    }
    
    /**
     * WHOIS sorgusu yapar
     */
    private static function queryWhois($domain, $server, $port = 43) {
        $fp = fsockopen($server, $port, $errno, $errstr, 10);
        if (!$fp) {
            return false;
        }
        
        fputs($fp, "$domain\r\n");
        $data = '';
        while (!feof($fp)) {
            $data .= fgets($fp, 128);
        }
        fclose($fp);
        
        return $data;
    }
    
    /**
     * WHOIS verisini parse eder
     */
    private static function parseWhoisData($whoisData, $domain) {
        $lines = explode("\n", $whoisData);
        $result = [
            'domain' => $domain,
            'registrar' => null,
            'registration_date' => null,
            'expiration_date' => null,
            'status' => null,
            'nameservers' => []
        ];
        
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line) || strpos($line, ':') === false) continue;

            list($key, $value) = explode(':', $line, 2);
            $key = strtolower(trim($key));
            $value = trim($value);

            // Log anahtar ve değer (geçici)
            error_log("WHOIS KEY: $key => $value");

            // Registrar bilgisi
            if (in_array($key, ['registrar', 'registrar name', 'sponsoring registrar'])) {
                $result['registrar'] = $value;
            }

            // Kayıt tarihi
            elseif (in_array($key, ['creation date', 'created', 'registered', 'registration time'])) {
                $result['registration_date'] = self::parseDate($value);
            }

            // Bitiş tarihi
            elseif (in_array($key, ['expiry date', 'expiration date', 'expires', 'registry expiry date'])) {
                $result['expiration_date'] = self::parseDate($value);
            }

            // Domain durumu
            elseif (in_array($key, ['status', 'domain status'])) {
                $result['status'] = $value;
            }

            // Name servers
            elseif (in_array($key, ['name server', 'nameserver', 'nserver'])) {
                $result['nameservers'][] = strtolower($value);
            }
        }
        
        // Eğer WHOIS'ten nameserver alınamazsa DNS'ten çek
        if (empty($result['nameservers'])) {
            $dnsInfo = self::getDnsInfo($domain);
            if ($dnsInfo && !empty($dnsInfo['nameservers'])) {
                $result['nameservers'] = $dnsInfo['nameservers'];
            }
        }
        
        return $result;
    }
    
    /**
     * Tarih formatını parse eder
     */
    private static function parseDate($dateString) {
        if (empty($dateString)) return null;
        
        // Farklı tarih formatlarını dene
        $formats = [
            'Y-m-d\TH:i:s\Z',
            'Y-m-d H:i:s',
            'Y-m-d',
            'd.m.Y',
            'd/m/Y',
            'm/d/Y',
            'Y/m/d'
        ];
        
        foreach ($formats as $format) {
            $date = DateTime::createFromFormat($format, $dateString);
            if ($date !== false) {
                return $date->format('Y-m-d');
            }
        }
        
        // Strtotime ile dene
        $timestamp = strtotime($dateString);
        if ($timestamp !== false) {
            return date('Y-m-d', $timestamp);
        }
        
        return null;
    }
    
    /**
     * Domain bilgilerini günceller
     */
    public static function updateDomainInfo($pdo, $domainId) {
        try {
            // Domain bilgisini al
            $stmt = $pdo->prepare("SELECT domain_name FROM domains WHERE id = ?");
            $stmt->execute([$domainId]);
            $domain = $stmt->fetch();
            
            if (!$domain) {
                return false;
            }
            
            // WHOIS bilgilerini çek
            $whoisInfo = self::getWhoisInfo($domain['domain_name']);
            if (!$whoisInfo) {
                return false;
            }
            
            // Veritabanını güncelle
            $stmt = $pdo->prepare("
                UPDATE domains SET 
                    registrar = ?,
                    registration_date = ?,
                    expiration_date = ?,
                    nameservers = ?,
                    status = ?,
                    days_until_expiry = ?,
                    last_checked = NOW(),
                    updated_at = NOW()
                WHERE id = ?
            ");
            
            $nameservers = !empty($whoisInfo['nameservers']) ? implode("\n", $whoisInfo['nameservers']) : null;
            // days_until_expiry hesapla
            $daysUntilExpiry = null;
            if (!empty($whoisInfo['expiration_date'])) {
                $exp = strtotime($whoisInfo['expiration_date']);
                $now = strtotime(date('Y-m-d'));
                $daysUntilExpiry = ($exp - $now) / 86400;
            }

            return $stmt->execute([
                $whoisInfo['registrar'],
                $whoisInfo['registration_date'],
                $whoisInfo['expiration_date'],
                $nameservers,
                $whoisInfo['status'],
                $daysUntilExpiry,
                $domainId
            ]);
            
        } catch (Exception $e) {
            error_log("Update Domain Info Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Tüm domainlerin bilgilerini günceller
     */
    public static function updateAllDomains($pdo) {
        try {
            $stmt = $pdo->query("SELECT id FROM domains WHERE 1=1");
            $domains = $stmt->fetchAll();
            
            $updated = 0;
            foreach ($domains as $domain) {
                if (self::updateDomainInfo($pdo, $domain['id'])) {
                    $updated++;
                }
                // Rate limiting - 1 saniye bekle
                sleep(1);
            }
            
            return $updated;
            
        } catch (Exception $e) {
            error_log("Update All Domains Error: " . $e->getMessage());
            return false;
        }
    }
}
?>

