<?php
/**
 * Domain Takip Sistemi - Bildirim Fonksiyonları
 */

class NotificationManager {
    
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * SMTP portunu otomatik belirle
     */
    private function getAutoPort($smtpSecure) {
        switch (strtolower($smtpSecure)) {
            case 'ssl':
                return 465;
            case 'tls':
                return 587;
            case 'none':
            default:
                return 25;
        }
    }
    
    /**
     * PHPMailer yapılandırması
     */
    private function configurePHPMailer($mail, $settings) {
        $mail->isSMTP();
        $mail->Host = $settings['smtp_host'] ?? '';
        $mail->SMTPAuth = true;
        $mail->Username = $settings['smtp_username'] ?? '';
        $mail->Password = $settings['smtp_password'] ?? '';
        
        // Güvenlik ayarı
        $smtpSecure = strtolower($settings['smtp_secure'] ?? 'tls');
        if ($smtpSecure === 'none') {
            $mail->SMTPSecure = false;
            $mail->SMTPAutoTLS = false;
        } else {
            $mail->SMTPSecure = $smtpSecure;
        }
        
        // Port otomatik belirleme
        $port = $settings['smtp_port'] ?? $this->getAutoPort($smtpSecure);
        $mail->Port = $port;
        
        // Hata ayıklama (geliştirme için)
        $mail->SMTPDebug = 0; // 0 = kapalı, 2 = detaylı
        $mail->Debugoutput = function($str, $level) {
            error_log("SMTP Debug: $str");
        };
        
        // Diğer ayarlar
        $mail->CharSet = 'UTF-8';
        $mail->Encoding = 'base64';
        $mail->Timeout = 30;
        
        return $mail;
    }
    
    /**
     * E-posta bildirimi gönder (PHPMailer ile SMTP)
     */
    public function sendEmailNotification($to, $subject, $message, $isHtml = true) {
        try {
            require_once __DIR__ . '/PHPMailerAutoload.php';
            $settings = $this->getNotificationSettings();
            
            if (empty($settings['smtp_host']) || empty($settings['smtp_username'])) {
                error_log('SMTP ayarları eksik');
                return false;
            }
            
            $mail = new PHPMailer\PHPMailer\PHPMailer(true);
            $mail = $this->configurePHPMailer($mail, $settings);
            
            $mail->setFrom($settings['smtp_username'], 'Domain Takip Sistemi');
            $mail->addAddress($to);
            $mail->Subject = $subject;
            $mail->isHTML($isHtml);
            $mail->Body = $message;
            $mail->AltBody = strip_tags($message);
            
            if ($mail->send()) {
                return true;
            } else {
                error_log('SMTP Error: ' . $mail->ErrorInfo);
                return false;
            }
        } catch (Exception $e) {
            error_log('Email send error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Domain süre uyarısı bildirimi gönder
     */
    private function sendDomainExpiryNotification($domain, $settings) {
        $daysUntilExpiry = $domain['days_until_expiry'];
        
        // Aciliyet seviyesini belirle
        if ($daysUntilExpiry <= 0) {
            $urgency = 'EXPIRED';
            $urgencyClass = 'danger';
        } elseif ($daysUntilExpiry <= $settings['critical_days']) {
            $urgency = 'CRITICAL';
            $urgencyClass = 'danger';
        } else {
            $urgency = 'WARNING';
            $urgencyClass = 'warning';
        }
        
        // E-posta şablonu oluştur
        $emailBody = $this->generateEmailTemplate($domain, $urgency, $urgencyClass, $settings);
        
        // Konu satırı
        $subject = "Domain Uyarısı: {$domain['domain_name']} - " . 
                   ($daysUntilExpiry <= 0 ? 'Süresi Doldu' : 
                    ($daysUntilExpiry <= $settings['critical_days'] ? 'Kritik Durum' : 'Uyarı'));
        
        // E-posta gönder
        return $this->sendEmailNotification(
            $settings['notification_email'],
            $subject,
            $emailBody,
            true
        );
    }
    
    /**
     * Süresi yaklaşan domainleri kontrol et ve bildirim gönder
     */
    public function checkExpiringDomains() {
        try {
            // Bildirim ayarlarını al
            $settings = $this->getNotificationSettings();
            if (!$settings['email_notifications_enabled']) {
                return ['success' => false, 'message' => 'E-posta bildirimleri devre dışı'];
            }
            // Süresi yaklaşan domainleri bul
            $expiringDomains = $this->getExpiringDomains($settings);
            if (empty($expiringDomains)) {
                return ['success' => true, 'message' => 'Bildirim gönderilecek domain yok'];
            }
            $sentCount = 0;
            foreach ($expiringDomains as $domain) {
                if ($this->shouldSendNotification($domain, $settings)) {
                    if ($this->sendDomainExpiryNotification($domain, $settings)) {
                        $this->logNotification($domain['id'], 'email', 'expiry_warning');
                        $sentCount++;
                    }
                }
            }
            return [
                'success' => true, 
                'message' => "$sentCount bildirim gönderildi"
            ];
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Hata: ' . $e->getMessage()];
        }
    }

    /**
     * Süresi yaklaşan domainleri getir
     */
    private function getExpiringDomains($settings) {
        $warningDays = $settings['warning_days'];
        $criticalDays = $settings['critical_days'];
        
        $stmt = $this->pdo->prepare("
            SELECT *, 
                   DATEDIFF(expiration_date, CURDATE()) as days_until_expiry
            FROM domains 
            WHERE expiration_date IS NOT NULL 
            AND DATEDIFF(expiration_date, CURDATE()) BETWEEN 0 AND ?
            ORDER BY days_until_expiry ASC
        ");
        $stmt->execute([$warningDays]);
        
        return $stmt->fetchAll();
    }
    
    /**
     * Bildirim gönderilmeli mi kontrol et
     */
    private function shouldSendNotification($domain, $settings) {
        $daysUntilExpiry = $domain['days_until_expiry'];
        // Son bildirim zamanını kontrol et
        $stmt = $this->pdo->prepare("
            SELECT created_at 
            FROM notification_logs 
            WHERE domain_id = ? AND type = 'email' AND event = 'expiry_warning'
            ORDER BY created_at DESC 
            LIMIT 1
        ");
        $stmt->execute([$domain['id']]);
        $lastNotification = $stmt->fetch();
        // Eğer daha önce bildirim gönderilmişse, tekrar gönderme aralığını kontrol et
        if ($lastNotification) {
            $lastNotificationTime = strtotime($lastNotification['created_at']);
            $hoursSinceLastNotification = (time() - $lastNotificationTime) / 3600;
            // Kritik durumda (7 gün ve altı) günde bir, diğer durumlarda haftada bir bildirim gönder
            $minHours = $daysUntilExpiry <= $settings['critical_days'] ? 24 : 168; // 24 saat veya 7 gün
            if ($hoursSinceLastNotification < $minHours) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * E-posta şablonu oluştur
     */
    private function generateEmailTemplate($domain, $urgency, $urgencyClass, $settings) {
        $domainName = $domain['domain_name'];
        $daysUntilExpiry = $domain['days_until_expiry'];
        $expirationDate = date('d.m.Y', strtotime($domain['expiration_date']));
        $registrar = $domain['registrar'] ?: 'Bilinmiyor';
        
        $urgencyText = [
            'EXPIRED' => 'SÜRESİ DOLDU',
            'CRITICAL' => 'KRİTİK DURUM',
            'WARNING' => 'UYARI'
        ];
        
        $urgencyColors = [
            'danger' => '#dc3545',
            'warning' => '#ffc107'
        ];
        
        $color = $urgencyColors[$urgencyClass] ?? '#ffc107';
        
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                body { font-family: Arial, sans-serif; margin: 0; padding: 20px; background-color: #f8f9fa; }
                .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 10px; overflow: hidden; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
                .header { background: $color; color: white; padding: 20px; text-align: center; }
                .content { padding: 30px; }
                .domain-info { background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0; }
                .footer { background: #6c757d; color: white; padding: 15px; text-align: center; font-size: 12px; }
                .btn { display: inline-block; padding: 12px 24px; background: $color; color: white; text-decoration: none; border-radius: 5px; margin: 10px 0; }
                .status { font-size: 18px; font-weight: bold; color: $color; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>🌐 Domain Takip Sistemi</h1>
                    <h2>{$urgencyText[$urgency]}</h2>
                </div>
                
                <div class='content'>
                    <h3>Domain Bilgileri</h3>
                    
                    <div class='domain-info'>
                        <p><strong>Domain Adı:</strong> $domainName</p>
                        <p><strong>Bitiş Tarihi:</strong> $expirationDate</p>
                        <p><strong>Kalan Süre:</strong> <span class='status'>" . 
                        ($daysUntilExpiry <= 0 ? 'Süresi doldu' : "$daysUntilExpiry gün") . "</span></p>
                        <p><strong>Registrar:</strong> $registrar</p>
                    </div>
                    
                    <h3>Yapılması Gerekenler:</h3>
                    <ul>
                        <li>Domain yenileme işlemini hemen başlatın</li>
                        <li>Registrar firması ile iletişime geçin</li>
                        <li>Ödeme bilgilerinizi kontrol edin</li>
                        <li>Otomatik yenileme ayarlarını gözden geçirin</li>
                    </ul>
                    
                    <p style='text-align: center;'>
                        <a href='{$settings['dashboard_url']}' class='btn'>Admin Paneline Git</a>
                    </p>
                </div>
                
                <div class='footer'>
                    <p>Bu e-posta Domain Takip Sistemi tarafından otomatik olarak gönderilmiştir.</p>
                    <p>Bildirim ayarlarını değiştirmek için admin panelini ziyaret edin.</p>
                </div>
            </div>
        </body>
        </html>";
    }
    
    /**
     * Bildirim logunu kaydet
     */
    private function logNotification($domainId, $type, $event) {
        $stmt = $this->pdo->prepare("
            INSERT INTO notification_logs (domain_id, type, event, created_at) 
            VALUES (?, ?, ?, NOW())
        ");
        return $stmt->execute([$domainId, $type, $event]);
    }
    
    /**
     * Bildirim ayarlarını getir
     */
    public function getNotificationSettings() {
        $stmt = $this->pdo->query("SELECT * FROM notification_settings LIMIT 1");
        $settings = $stmt->fetch();
        
        // Varsayılan ayarlar
        if (!$settings) {
            return [
                'email_notifications_enabled' => true,
                'notification_email' => 'admin@example.com',
                'warning_days' => 30,
                'critical_days' => 7,
                'dashboard_url' => 'http://localhost/domain-tracker/',
                'smtp_host' => '',
                'smtp_port' => 587,
                'smtp_username' => '',
                'smtp_password' => '',
                'smtp_secure' => 'tls'
            ];
        }
        
        return $settings;
    }
    
    /**
     * Bildirim ayarlarını güncelle
     */
    public function updateNotificationSettings($settings) {
        try {
            // Port otomatik belirleme
            if (empty($settings['smtp_port']) || $settings['smtp_port'] == 0) {
                $settings['smtp_port'] = $this->getAutoPort($settings['smtp_secure'] ?? 'tls');
            }
            
            // Mevcut ayarları kontrol et
            $stmt = $this->pdo->query("SELECT COUNT(*) as count FROM notification_settings");
            $exists = $stmt->fetch()['count'] > 0;
            
            if ($exists) {
                $stmt = $this->pdo->prepare("
                    UPDATE notification_settings SET 
                        email_notifications_enabled = ?,
                        notification_email = ?,
                        warning_days = ?,
                        critical_days = ?,
                        dashboard_url = ?,
                        smtp_host = ?,
                        smtp_port = ?,
                        smtp_username = ?,
                        smtp_password = ?,
                        smtp_secure = ?,
                        updated_at = NOW()
                ");
            } else {
                $stmt = $this->pdo->prepare("
                    INSERT INTO notification_settings 
                    (email_notifications_enabled, notification_email, warning_days, critical_days, dashboard_url, smtp_host, smtp_port, smtp_username, smtp_password, smtp_secure, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
                ");
            }

            return $stmt->execute([
                $settings['email_notifications_enabled'] ? 1 : 0,
                $settings['notification_email'],
                $settings['warning_days'],
                $settings['critical_days'],
                $settings['dashboard_url'],
                $settings['smtp_host'] ?? '',
                $settings['smtp_port'] ?? 587,
                $settings['smtp_username'] ?? '',
                $settings['smtp_password'] ?? '',
                $settings['smtp_secure'] ?? 'tls'
            ]);
            
        } catch (Exception $e) {
            error_log("Update notification settings error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Test bildirimi gönder
     */
    public function sendTestNotification($email) {
        try {
            require_once __DIR__ . '/PHPMailerAutoload.php';
            $settings = $this->getNotificationSettings();
            
            if (empty($settings['smtp_host']) || empty($settings['smtp_username'])) {
                return 'SMTP ayarları eksik. Lütfen SMTP bilgilerini girin.';
            }
            
            $mail = new PHPMailer\PHPMailer\PHPMailer(true);
            $mail = $this->configurePHPMailer($mail, $settings);
            
            $mail->setFrom($settings['smtp_username'], 'Domain Takip Sistemi');
            $mail->addAddress($email);
            $mail->Subject = 'Domain Takip Sistemi - Test Bildirimi';
            $mail->isHTML(true);
            $mail->Body = "<html><body style='font-family: Arial, sans-serif; padding: 20px;'><h2>🧪 Test Bildirimi</h2><p>Bu bir test bildirimidir. E-posta ayarlarınız doğru şekilde yapılandırılmış.</p><p><strong>Gönderim Zamanı:</strong> " . date('d.m.Y H:i:s') . "</p><p><strong>SMTP Sunucu:</strong> " . htmlspecialchars($settings['smtp_host']) . "</p><p><strong>Port:</strong> " . htmlspecialchars($settings['smtp_port']) . "</p><p><strong>Güvenlik:</strong> " . strtoupper($settings['smtp_secure']) . "</p><hr><p><em>Domain Takip Sistemi</em></p></body></html>";
            $mail->AltBody = 'Bu bir test bildirimidir. E-posta ayarlarınız doğru şekilde yapılandırılmış. Gönderim Zamanı: ' . date('d.m.Y H:i:s');
            
            if ($mail->send()) {
                return 'Test mail başarıyla gönderildi.';
            } else {
                return 'SMTP Hatası: ' . $mail->ErrorInfo;
            }
        } catch (Exception $e) {
            return 'Hata: ' . $e->getMessage();
        }
    }
}
